#include <string.h>
#include <errno.h>
#include <iconv.h>
#include "conv.h"
#include "debug.h"

typedef struct _LRCONV {
  iconv_t    icLR;
  iconv_t    icRL;
  char       acRemote[1];
} LRCONV;


static size_t _conv(iconv_t ic, size_t *pcbStr, char *pcStr,
                    size_t cbBuf, char *pcBuf, int *pfError)
{
  char    *pcDst = pcBuf;
  size_t  cbDst = cbBuf - 1;

  *pfError = 0;
  while( ( *pcbStr > 0 ) && ( cbDst > 0 ) )
  {
    if ( iconv( ic, (const char **)&pcStr, pcbStr,
                &pcDst, &cbDst ) == (size_t)-1 )
    {
      if ( ( errno == EILSEQ ) && ( *pcbStr != 0 ) )
      {
        // Try to skip invalid character
        pcStr++;
        (*pcbStr)--;
        continue;
      }

      debug( "err:%u:%s, stored %lu, left %lu bytes",
             errno, strerror( errno ), pcDst - pcBuf, *pcbStr );
      *pfError = 1;
      break;
    }
  }

  // Push out the remaining characters in the *pIC object
  iconv( ic, NULL, 0, &pcDst, &cbDst );
  // Trailing zero
  *pcDst = '\0';

  return pcDst - pcBuf;
}


PLRCONV cnvLRCreate(char *pcRemote)
{
  PLRCONV  pLRConv;

  pLRConv = malloc( sizeof(LRCONV) + strlen( pcRemote ) );
  if ( pLRConv == NULL )
    return NULL;

  pLRConv->icLR = iconv_open( (const char *)pcRemote, (const char *)"" );
  if ( pLRConv->icLR == ((iconv_t)(-1)) )
  {
    free( pLRConv );
    return NULL;
  }

  pLRConv->icRL = iconv_open( (const char *)"", (const char *)pcRemote );
  if ( pLRConv->icRL == ((iconv_t)(-1)) )
  {
    iconv_close( pLRConv->icLR );
    free( pLRConv );
    return NULL;
  }

  strcpy( pLRConv->acRemote, pcRemote );

  return pLRConv;
}

void cnvLRDestroy(PLRCONV pLRConv)
{
  if ( pLRConv == NULL )
    return;

  if ( pLRConv->icLR != ((iconv_t)(-1)) )
    iconv_close( pLRConv->icLR );

  if ( pLRConv->icRL != ((iconv_t)(-1)) )
    iconv_close( pLRConv->icRL );

  free( pLRConv );
}

size_t cnvLRConv(PLRCONV pLRConv, unsigned uDirection,
                 int cbStr, char *pcStr, size_t cbBuf, char *pcBuf)
{
  iconv_t  *pIC;
  int      fError = 0;

  if ( cbBuf == 0 )
    return 0;

  if ( cbStr < 0 )
    cbStr = pcStr == NULL ? 0 : strlen( pcStr );

  if ( pLRConv == NULL )
  {
cnvLRConv_l00:
    cbBuf = cbBuf > cbStr ? cbStr : (cbBuf - 1);
    memcpy( pcBuf, pcStr, cbBuf );
    pcBuf[cbBuf] = '\0';
    return cbBuf;
  }

  pIC = uDirection == CNV_LOCALTOREMOTE ? &pLRConv->icLR : &pLRConv->icRL;

  if ( *pIC == ((iconv_t)(-1)) )
  {
    char  *pcTo, *pcFrom;

    if ( uDirection == CNV_LOCALTOREMOTE )
    {
      pcTo = pLRConv->acRemote;
      pcFrom = "";
    }
    else
    {
      pcTo = "";
      pcFrom = pLRConv->acRemote;
    }

    debug( "iconv_open(\"%s\",\"%s\")...", pcTo, pcFrom );
    *pIC = iconv_open( (const char *)pcTo, (const char *)pcFrom );
    if ( *pIC == ((iconv_t)(-1)) )
    {
      debug( "iconv_open(\"%s\",\"%s\") failed", pcTo, pcFrom );
      goto cnvLRConv_l00;
    }
  }  // if ( *pIC == ((iconv_t)(-1)) )

  cbBuf = _conv( *pIC, (size_t *)&cbStr, pcStr, cbBuf, pcBuf, &fError );

  if ( fError || ( cbStr != 0 ) )
  {
    // The object will be reopened the next time the function is called
    iconv_close( *pIC );
    *pIC = ((iconv_t)(-1));
  }

  return cbBuf;
}

char *cnvLRConvNew(PLRCONV pLRConv, unsigned uDirection,
                     int cbStr, char *pcStr, size_t *pcbLength)
{
  char    *pcBuf;
  size_t  cbBuf;

  if ( pcStr == NULL )
    return NULL;

  if ( cbStr < 0 )
    cbStr = strlen( pcStr );

  cbBuf = (cbStr * 4) + 1;
  pcBuf = malloc( cbBuf );
  if ( pcBuf == NULL )
    return NULL;

  cbBuf = cnvLRConv( pLRConv, uDirection, cbStr, pcStr, cbBuf, pcBuf );
  if ( pcbLength != NULL )
    *pcbLength = cbBuf;

  pcStr = realloc( pcBuf, cbBuf + 1 /* keep trailing zero */ );

  return pcStr == NULL ? pcBuf : pcStr;
}

char *cnvConvNew(char *pcTo, char *pcFrom, int cbStr, char *pcStr,
                 size_t *pcbLength)
{
  iconv_t  ic;
  size_t   cbBuf;
  char     *pcBuf;
  int      fError;

  if ( pcStr == NULL )
    return NULL;

  if ( cbStr < 0 )
    cbStr = strlen( pcStr );

  cbBuf = (cbStr * 4) + 1;
  pcBuf = malloc( cbBuf );
  if ( pcBuf == NULL )
    return NULL;

  ic = iconv_open( (const char *)pcTo, (const char *)pcFrom );
  if ( ic == ((iconv_t)(-1)) )
  {
    debug( "iconv_open(\"%s\",\"%s\") failed", pcTo, pcFrom );
    free( pcBuf );
    return NULL;
  }

  cbBuf = _conv( ic, (size_t *)&cbStr, pcStr, cbBuf, pcBuf, &fError );

  iconv_close( ic );

  if ( pcbLength != NULL )
    *pcbLength = cbBuf;

  pcStr = realloc( pcBuf, cbBuf + 1 /* keep trailing zero */ );

  return pcStr == NULL ? pcBuf : pcStr;
}
