/*
 * telnet.c - telnet communication handler for Shifty
 */

#define	INCL_BASE
#define	INCL_DOSDEVICES

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <os2.h>
#include "shifty.h"

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/select.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

/*
 * Commnucation Control Block for Network Device
 */
 
UCHAR   netName[256] ;
BOOL    netInit(INT ac, CHAR *av[]) ;
VOID    netTerm(VOID) ;
VOID    netStat(VOID) ;
INT     netRecv(PUCHAR buff, INT len) ;
INT     netSend(PUCHAR buff, INT len) ;
VOID    netBreak(VOID) ;
VOID    netFlush(VOID) ;

COMREC	netDevice = {
    netName,
    netInit,
    netTerm,
    netStat,
    netRecv,
    netSend,
    netBreak,
    netFlush
} ;

/*
 * Network Communication Control Data
 */

static  struct  sockaddr_in netHost = { 0 } ;
static  int                 netSock         ;
static	BOOL                netConn = FALSE ;

/*
 * netInit - Initialize Network Communication Device
 *  
 *      -net <host>[:<port>]
 *          <host> is TCP/IP host name with string or digits.
 *          <port> is connecting TCP port number with string
 *          (service name) or digit (port number).
 */

BOOL    netInit(INT ac, CHAR *av[])
{
    INT     i ;
    UCHAR   *argSpec = NULL ;
    UCHAR   *argHost = NULL ;
    UCHAR   *argPort = NULL ;
    ULONG   numHost = 0  ;
    USHORT  numPort = 0  ;
    struct hostent  *hp  ;
    struct servent  *sp  ;
    struct in_addr  addr ;
    UCHAR           *cp  ;

    /*
     * Get host:port spec on argument
     */

    for (i = 1 ; i < ac ; i++) {
        if (stricmp(av[i], "-net") == 0) {
	    if ((i + 1) < ac) {
	        argSpec = av[i+1] ;
		break ;
	    }
	}
    }
    if (argSpec == NULL) {
        fprintf(stderr, "%s : no host name specified\n", av[0]) ;
	return(FALSE) ;
    }
    
    /*
     * Parse 'host' and 'port' in argument
     */
    
    argHost = argSpec ;
    if ((argPort = strchr(argSpec, ':')) != NULL) {
        *argPort++ = '\0' ;
    }
    if (argPort == NULL || *argPort == '\0') {
        argPort = "23" ;
    }

    if (isdigit(*argHost)) {
        numHost = inet_addr(argHost) ;
    } else if ((hp = gethostbyname(argHost)) != NULL) {
        numHost = *((ULONG *) hp->h_addr) ;
    } else {
        numHost = 0 ;
    }
    if (isdigit(*argPort)) {
        numPort = atoi(argPort) ;
    } else if ((sp = getservbyname(argPort, "tcp")) != NULL) {
        numPort = ntohs(sp->s_port) ;
    } else {
        numPort = 0 ;
    }
    if (numHost == 0 || numPort == 0) {
        fprintf(stderr, "%s : invalid host:port specification\n", av[0]) ;
	return(FALSE) ;
    }
    addr.s_addr = numHost ;
    cp = inet_ntoa(addr)  ;
    sprintf(netName, "%s:%d", cp, numPort) ;

    /*
     * Connect to specified host:port
     */

    if ((netSock = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
        fprintf(stderr, "%s : failed to create socket\n", av[0]) ;
        return(FALSE) ;
    }

    memset(&netHost, 0, sizeof(netHost)) ;
    netHost.sin_family      = AF_INET ;
    netHost.sin_addr.s_addr = numHost ;
    netHost.sin_port = htons(numPort) ;
    
    if (connect(netSock, (struct sockaddr *) &netHost, sizeof(netHost)) != 0) {
        fprintf(stderr, "%s : failed to connect\n", av[0]) ;
	close(netSock) ;
	return(FALSE) ;
    }
    netConn = TRUE ;
    return(TRUE) ;
}

VOID    netTerm(VOID)
{
    close(netSock) ;
    netConn = FALSE  ;
}

VOID    netStat(VOID)
{
    system("netstat -s") ;
}

/*
 * Telnet State Machine
 */

#define SDATA   0       /* Normal State, IAC introduce SIAC */
#define SIAC    1       /* Followed by IAC                  */
#define SOPT    2       /* Option Negotiation               */
#define	SSUB    3       /* in Sub-Netotiation Parameter     */

static  INT     telStat = SDATA ;
static  INT     telSave = SDATA ;
static  INT     optNego = 0     ;   /* WILL, WONT, DO, DONT         */
static	UCHAR	subNego[80]     ;   /* Hold Sub-negotiation param   */
static  INT     idxNego = 0     ;   /* Index to subNego buffer      */

#define TSE     240
#define TNOP    241
#define TDM     242
#define TBRK    243
#define TIP     244
#define TAO     245
#define TAYT    246
#define TEC     247
#define TEL     248
#define TGA     249
#define TSB     250
#define TWILL   251
#define TWONT   252
#define TDO     253
#define TDONT   254
#define TIAC    255

static	UCHAR	rbuff[256] ;    /* Buffer for Recv Data */
static  UCHAR   obuff[256] ;    /* Buffer for Send Data */

static  void    subInit(void)
{
    idxNego = 0 ;
}

static  void    subDone(void)
{
    int     len ;
    
    if (idxNego == 0) {
        return ;
    }
    switch (subNego[0]) {
    case 24 :       /* TERMINAL-TYPE */
        if (idxNego >= 2 && subNego[1] == 1) {  /* SEND */
	    len = 0 ;
	    obuff[len++] = TIAC ;
	    obuff[len++] = TSB  ;
	    obuff[len++] = 24   ;   /* TERMINAL-TYPE */
	    obuff[len++] = 0    ;   /* IS            */
	    strcpy(&obuff[4], emuDevice->emuType) ;
	    len += strlen(emuDevice->emuType) ;
	    obuff[len++] = TIAC ;
	    obuff[len++] = TSE  ;
	    send(netSock, obuff, len, 0) ;
        }
	break ;
    default :       /* ignore others */
        break ;
    }
    return ;
}

INT     netRecv(PUCHAR buff, INT len)
{
    INT     n, l, i, cnt ;
    UCHAR   *sp  ;
    UCHAR   *dp  ;
    int     maxfd ;
    fd_set  rfds  ;
    struct timeval  tv ;

    if (netConn == FALSE) {
	return(-1) ;
    }

    FD_ZERO(&rfds) ;
    FD_SET(netSock, &rfds) ;
    maxfd = netSock + 1 ;
    tv.tv_sec  = 1 ;
    tv.tv_usec = 0 ;
    
    if ((n = select(maxfd, &rfds, NULL, NULL, &tv)) < 0) {
        return (-1) ;
    } else if (n == 0) {
        return (0) ;
    }
    
    l = min(len, 256) ;
    n = recv(netSock, rbuff, l, 0) ;

    if (n < 0) {
	netConn = FALSE ;
	return(-1) ;
    }    
    sp = rbuff ;
    dp = buff  ;
    
    for (i = 0, cnt = 0 ; i < n ; i++, sp++) {
        switch (telStat) {
        case SDATA :
	    if (*sp != TIAC) {
	        *dp++ = *sp ; cnt += 1 ;
	    } else {
	        telSave = telStat ;
	        telStat = SIAC ;
	    }
	    break ;
        case SSUB :
	    if (*sp != TIAC) {
	        subNego[idxNego++] = *sp ;
	    } else {
	        telSave = telStat ;
	        telStat = SIAC ;
	    }
	    break ;
	case SIAC :
            switch (*sp) {
            case TIAC :
	        if (telSave == SDATA) {
	            *dp++ = *sp ; cnt += 1 ;
	        } else {
		    subNego[idxNego++] = *sp ;
		}
		telStat = telSave ;
		break ;
            case TWILL :
	    case TWONT :
	    case TDO   :
	    case TDONT :
	        optNego = *sp  ;
	        telStat = SOPT ;
		break ;
	    case TSB  :     /* start of sub negotiation */
                subInit() ;
	        telStat = SSUB ;
		break ;
            case TSE  :     /* end of sub negotiation */
                subDone() ;
	        telStat = SDATA ;
		break ;
	    case TNOP :
            case TDM  :
            case TBRK :
            case TIP  :
            case TAO  :
            case TAYT :
            case TEC  :
            case TEL  :
            case TGA  :
		telStat = SDATA ;
                break ;
            default :
	        break ;
	    }
	    break ;
	case SOPT :
	    switch (*sp) {      /* Option Code      */
	    case 24 :           /* TERMINAL-TYPE    */
	        if (optNego == TWILL) {
		    obuff[0] = TIAC  ;
		    obuff[1] = TDONT ;  /* don't receive terminal type */
		    obuff[2] = 24    ;
		    send(netSock, obuff, 3, 0) ;
	        } else if (optNego == TDO) {
		    obuff[0] = TIAC  ;
		    obuff[1] = TWILL ;  /* willing to send terminal type */
		    obuff[2] = 24    ;
		    send(netSock, obuff, 3, 0) ;
		}
		break ;
	    case 1  :           /* ECHO             */
	        if (optNego == TWILL) {
		    obuff[0] = TIAC  ;
		    obuff[1] = TDO   ;
		    obuff[2] = *sp   ;
		    send(netSock, obuff, 3, 0) ;
		}
		break ;
	    case 0  :           /* TRANSMIT-BINARY  */
	    case 3  :           /* Suppress GA      */
	    default :
	        if (optNego == TWILL) {
		    obuff[0] = TIAC  ;
		    obuff[1] = TDONT ;
		    obuff[2] = *sp   ;
		    send(netSock, obuff, 3, 0) ;
	        } else if (optNego == TDO) {
		    obuff[0] = TIAC  ;
		    obuff[1] = TWONT ;
		    obuff[2] = *sp   ;
		    send(netSock, obuff, 3, 0) ;
		}
	    }
	    telStat = SDATA ;
	    break ;
        default :
	    break ;
	}
    }
    return(cnt) ;
}

INT     netSend(PUCHAR buff, INT len)
{
    INT     n, cnt ;

    if (netConn == FALSE) {
	return(-1) ;
    }
    
    cnt = 0 ;
    while (len > 0) {
	if ((n = send(netSock, buff, len, 0)) < 0) {
            break ;
        }
	len  -= n ;
	buff += n ;
	cnt  += n ;
    }
    return(cnt) ;
}

VOID    netBreak(VOID)
{
}

VOID    netFlush(VOID)
{
}

