/*
 * Shifty - A shifty terminal emulator for OS/2
 */

#define	INCL_BASE
#define	INCL_DOSDEVICES

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <malloc.h>
#include <os2.h>
#include "shifty.h"

UCHAR	Version[] = "Shifty Term for OS/2, Version 2.03" ;
UCHAR   Current[256] ;

/*
 * Execution Control Variables
 */

USHORT	ExecDone = FALSE ;	/* TRUE if Program Terminate	*/
UCHAR	KeyAttn  = 0x1d  ;	/* Attention Key Code		*/
FILE    *LogFP = NULL    ;      /* Loggin session               */
BOOL    CtrlPrty = FALSE ;      /* Controls Threads Priotiry    */

/*
 * g e t D e v i c e  - which comm. device to use
 *
 *      -port COMX      use COMX device to communicate
 *      -net host:port 	use 'host' host's 'port' TCP port
 */

extern  COMREC  serDevice ;
extern  COMREC  netDevice ;

static  COMPTR  getDevice(int ac, char *av[])
{
    int     i   ;
    COMPTR  dev = &serDevice ;  /* as default device type   */
    
    for (i = 1 ; i < ac ; i++) {
        if (stricmp(av[i], "-port") == 0) {
	    dev = &serDevice ;
	} else if (stricmp(av[i], "-net") == 0) {
            dev = &netDevice ;
	}
    }
    return(dev) ;
}

/*
 * g e t E m u l a t i o n - which emulation ?
 *
 *      -sjis | -euc | -jis7 | - jis8
 */

extern  EMUREC  vt100sjis ;
extern  EMUREC  vt100euc  ;
extern  EMUREC  vt100jis7 ;
extern  EMUREC  vt100jis8 ;

static  EMUPTR  getEmulation(int ac, char *av[])
{
    int     i   ;
    EMUPTR  emu = &vt100sjis ;  /* as default emulation */
    
    for (i = 1 ; i < ac ; i++) {
        if (stricmp(av[i], "-sjis") == 0) {
            emu = &vt100sjis ;
	} else if (stricmp(av[i], "-euc") == 0) {
            emu = &vt100euc  ;
        } else if (stricmp(av[i], "-jis") == 0) {
	    emu = &vt100jis7 ;
        } else if (stricmp(av[i], "-jis7") == 0) {
	    emu = &vt100jis7 ;
        } else if (stricmp(av[i], "-jis8") == 0) {
	    emu = &vt100jis8 ;
        } else if (stricmp(av[i], "-log") == 0) {
	    if (LogFP == NULL) {
	        LogFP = fopen("shifty.log", "w") ;
	    }
	}
    }
    return(emu) ;
}

/*
 * e x i t t e r m  -  exit funtion on the termination.
 */

void exitterm(USHORT what)
{
    if (comDevice != NULL) {
        (*comDevice->comTerm) () ;
    }
    scrTerm(FALSE) ;
    
    if (LogFP != NULL) {
        fclose(LogFP) ;
    }
    DosExitList(EXLST_EXIT, 0L);
}

/*
 * use two threads for Receiving and Displaying
 */
 
#define	STACKSIZE	16384

TID	TidCom ;	/* thread for reading comm-device   */
TID	TidEmu ;	/* thread for emulating             */

/*
 * doDisp & doKeyin shares screen device, and doRecv and doKeyin
 * shares communication device. Use following semaphore for
 * mutual exclusition.
 */
 
HMTX    DispSem ;               /* mutex sem for Display    */
HMTX    CommSem ;               /* mutex sem for Comm.      */
BOOL	DispSusp = FALSE ;      /* TRUE if Suspended	    */

/*
 * doRecv - Receving Thread, read data from Comm.Dev and put to queue
 */

static  UCHAR   recvbuff[1024] ;
static	UCHAR	lastmesg[] = "connection closed\r\n" ;

void doRecv(void)
{
    INT     len ;
    
    while (! ExecDone) {
        DosRequestMutexSem(CommSem, -1L) ;
        len = (*comDevice->comRecv) (recvbuff, sizeof(recvbuff)) ;
	DosReleaseMutexSem(CommSem) ;
	if (len < 0) {
	    break ;
	} else if (len > 0) {
	    quePut(recvbuff, len) ;
	}
    }
    quePut(lastmesg, strlen(lastmesg)) ;

    ExecDone = TRUE ;

    _endthread() ;
}

/*
 * doDisp - Display Thread, read data from queue and put to screen
 */
 
static  UCHAR   dispbuff[1024] ;

void doDisp(void)
{
    INT     len ;

    DosRequestMutexSem(DispSem, -1L) ;
    DispSusp = FALSE ;
    (*emuDevice->emuRst) () ;
    DosReleaseMutexSem(DispSem) ;
    
    while (! ExecDone) {
	len = queGet(dispbuff, sizeof(dispbuff)) ;
	if (len > 0) {
            DosRequestMutexSem(DispSem, -1L) ;
            (*emuDevice->emuScr) (dispbuff, len) ;
	    scrShow() ;
	    DosReleaseMutexSem(DispSem) ;
        }
    }

    ExecDone = TRUE ;

    _endthread() ;
}

/*
 * doKeyin - work as main thread
 *
 *	Usually, input key data will send to communication device,
 *	but if "Attention Key" was pressed, it does local operations.
 *	When doing local operations, this program takes Execution Semaphore
 *	to suspend emulator display thread.
 */

static  KBDINFO	KeyModeNew = { sizeof(KBDINFO), 0x0006, 0 } ;

void    doKeyin(void)
{
    KBDKEYINFO	key   ;
    USHORT	c     ;
    UCHAR	buff[4] ;
    BOOL        reset   ;

    KbdSetStatus(&KeyModeNew, 0) ;
    
    while (! ExecDone) {
	KbdCharIn(&key, 0, 0) ;
	c = (USHORT) key.chChar ;

	/*
	 * Usually (except KeyAttn) send key data to host
	 */
		
	if (c != KeyAttn) {
            if (DispSusp) {         /* Release Screen   */
		DosReleaseMutexSem(DispSem) ;
		DispSusp = FALSE ;
	    }
	    (*emuDevice->emuKey) (key.chChar, key.chScan, key.fsState) ;
	    continue ;
	}

        /*
         * Attention key introduce local operation
	 */

	if (! DispSusp) {       /* Aquire Screen for Local Functions    */
	    DosRequestMutexSem(DispSem , -1L) ;
	    DispSusp = TRUE ;
	}
	KbdCharIn(&key, 0, 0) ; c = (USHORT) key.chChar ;
        reset = FALSE ;
		
	if (c == '?') {
	    c = showHelp(Current) ;
	}
        if (c == KeyAttn) {
	    buff[0] = (UCHAR ) c ;
	    (*comDevice->comSend) (buff, 1) ;
	} else if (c == '0') {
	    buff[0] = 0 ;
	    (*comDevice->comSend) (buff, 1) ;
	} else if ((c == 'b') || (c == 'B')) {
	    (*comDevice->comBreak) () ;
	} else if ((c == 'f') || (c == 'F')) {
	    (*comDevice->comFlush) () ;
	} else if ((c == 'c') || (c == 'C')) {
	    ExecDone = TRUE ;
	} else if (c == 0x03) {
	    (*emuDevice->emuRst) () ;
            scrInit(0, NULL, "reset terminal") ;
	} else if ((c == 'u') || (c == 'U')) {
	    upLoading() ;
	} else if ((c == 'd') || (c == 'D')) {
	    dnLoading() ;
	} else if ((c == 's') || (c == 'S')) {
	    saveScreen(c == 'S') ;
	} else if (c == '%') {
	    dirChange() ;
	} else if ((c == 'l') || (c == 'L')) {
	    dirListing() ;
	} else if (c == '!') {
	    execCommand() ;
	    reset = TRUE ;
	} else if ((c == 'p') || (c == 'P')) {
	    execShell() ;
	    reset = TRUE ;
	} else if (c == '=') {
	    showStatus() ;
	    reset = TRUE ;
	}
	if (reset) {
	    /* executing external command requires reset KBD setting */
            KbdSetStatus(&KeyModeNew, 0) ;
        }
	if (! DispSusp) {       /* Release Screen from Local Functions  */
            DosReleaseMutexSem(DispSem) ;
	}			
    }

    if (DispSusp) {             /* Release Screen on Termination    */
	DosReleaseMutexSem(DispSem) ;
    }
}

/*
 * m a i n  -  do several setup procedures and start emulating threads
 */

COMPTR	comDevice = NULL ;
EMUPTR	emuDevice = NULL ;

void main(int ac, char *av[])
{
    APIRET  stat, stat1, stat2 ;

    emuDevice = getEmulation(ac, av) ;
    comDevice = getDevice(ac, av)    ;

    if ((*comDevice->comInit) (ac, av) != TRUE) {
        fprintf(stderr, "%s : failed to setup communication\n", av[0]) ;
	exit(1) ;
    }
    sprintf(Current, " %s - %s - %s ",
                Version, comDevice->comName, emuDevice->emuName) ;
    if (scrInit(ac, av, Current) != TRUE) {
        fprintf(stderr, "%s : failed to setup display\n", av[0]) ;
	exit(1) ;
    }
    if (queInit() != TRUE) {
        fprintf(stderr, "%s : failed to setup queue\n", av[0]) ;
	exit(1) ;
    }

    /*
     * Regist Exit Routine
     */

    if ((stat = DosExitList(EXLST_ADD, (PFNEXITLIST) exitterm)) != 0) {
	fprintf(stderr, "%s : cannot set exit routine\n", av[0]) ;
	exit(1) ;
    }

    /*
     * Create Mutex Sem for Display/Communication access
     */

    stat1 = DosCreateMutexSem(NULL, &DispSem, 0, FALSE) ;
    stat2 = DosCreateMutexSem(NULL, &CommSem, 0, FALSE) ;

    if (stat1 != 0 || stat2 != 0) {
        fprintf(stderr, "%s : failed to create Disp/Comm Semaphore\n", av[0]) ;
	exit(2) ;
    }

    /*
     * Start Threads for Receiving and Display
     */

    TidCom = _beginthread(doRecv, NULL, STACKSIZE, NULL) ;
    TidEmu = _beginthread(doDisp, NULL, STACKSIZE, NULL) ;

    if (TidCom < 0 || TidEmu < 0) {
        fprintf(stderr, "%s : failed to start Recv/Disp threads\n", av[0]) ;
	exit(2) ;
    }

    /*
     * Now Start Terminal Emulation
     */

    doKeyin() ;
}
