/*
 * screen.c - Screen handling module for Shifty Term
 */

#define	INCL_BASE

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <malloc.h>
#include <os2.h>
#include "shifty.h"
#include "screen.h"

/*
 * Screen Control Data
 */

static	VIOMODEINFO	ScrModeSav  ;
static	VIOMODEINFO	ScrModeNow  ;

static	PUCHAR	InitMsg = "" ;

INT		ScrLin = 24 ;	    /* Size of the Screen	*/
INT		ScrCol = 80 ;
BOOL		ScrSmooth = FALSE ; /* Flag for Smooth Scroll	*/

static	INT	ScrTop = 0  ;	    /* Scroll Region	*/
static	INT	ScrBot = 23 ;

static	INT	CurLin = 0 ;	    /* Current Position	*/
static	INT	CurCol = 0 ;

static	UCHAR	CurAttr    ;
static  USHORT  CurCell    ;
static  USHORT  ScrSpac    ;	    /* Space fill for scroll	*/

#define setAttr(x)  \
    CurAttr = (x) ;\
    CurCell = (USHORT) CurAttr << 8 ;\
    ScrSpac = CurCell | 0x20 ;

/*
 * Variables for Logical Video Buffer
 */

static	PUSHORT	LvbBase    ;	/* Base address of LVB	        */
static	USHORT	LvbLeng    ;	/* Length of the LVB	        */
static	USHORT	LvbTop     ;	/* Top    of Modification	*/
static	USHORT	LvbBot     ;	/* Bottom of Modification	*/

/*
 * Screen Attributes
 */
 
/*
 * Display Attribute, map to
 *	NORM, BOLD, LITE, ITAL, ULIN, HIGH, BLNK, REVS,
 */

typedef	struct {
    UCHAR   *name   ;
    UCHAR   attr[8] ;
} ATTR_TAB ;

ATTR_TAB	DispAttrTab[] = {
{ "PCAT", 0xf5, 0xf9, 0xf9, 0x9e, 0x9e, 0x9e, 0x9e, 0x9e } ,
{ "PS55", 0x00, 0x40, 0x40, 0x08, 0xa0, 0x40, 0x80, 0x04 } ,
{ "FMR",  0x07, 0x0a, 0x03, 0x03, 0x05, 0x0c, 0x87, 0x70 } ,
{ NULL,   0xf5, 0xf9, 0xf9, 0x9e, 0x9e, 0x9e, 0x9e, 0x9e }
} ;

UCHAR	DispAttr[8] = { 0xf5, 0xf9, 0xf9, 0x9e, 0x9e, 0x9e, 0x9e, 0x9e } ;

/*
 * g e t C o l o r  - change attribute table if ENV was defined
 */

static	void getColor(UCHAR *cmap)
{
    SHORT   i, val ;
    UCHAR   *p ;

    for (i = 0, p = cmap ; i < 8 ; i++) {
        if (isdigit(*p)) {
            val = (*p - '0') * 16 ; p++ ;
	} else if (isxdigit(*p)) {
	    val = (toupper(*p) - 'A' + 10) * 16 ; p++ ;
	} else {
	    return ;
	}
	if (isdigit(*p)) {
	    val += (*p - '0') ; p++ ;
	} else if (isxdigit(*p)) {
	    val += (toupper(*p) - 'A' + 10) ; p++ ;
	} else {
	    return ;
	}
	DispAttr[i] = (UCHAR) val ;
    }
}

/*
 * s c r I n i t  -  Initialize Screen
 *
 *	Set terminal screen size to specified size, and
 *	initialize control variables.
 *
 *      -line #         Number of Lines
 *      -cols #         Number of Columns
 *      -smooth         Turn ON Smooth Scroll
 *	-colors $	Mapping colors into attributes
 */

BOOL    scrInit(int ac, char *av[], char *msg)
{
    INT	    i, j, len, lin, col ;
    UCHAR   *env ;

    /*
     * Scan Env. for Attribute Setting
     */

    if ((env = getenv("MACHINE")) != NULL) {
    	for (i = 0 ; DispAttrTab[i].name ; i++) {
            if (stricmp(DispAttrTab[i].name, env) == 0) {
		break ;
	    }
	}
	for (j = 0 ; j < 8 ; j++) {
	    DispAttr[j] = DispAttrTab[i].attr[j] ;
	}
    }
    if ((env = getenv("SHIFTYCL")) != NULL) {
        getColor(env) ;
    }

    /*
     * Parse Command Line Arguments for Screen Setting
     */
     
    lin = col = 0 ;
    
    for (i = 1 ; i < ac ; i++) {
        if (stricmp(av[i], "-line") == 0) {
	    if ((i + 1) < ac) {
	        lin = atoi(av[i+=1]) ;
	    }
	}
        if (stricmp(av[i], "-cols") == 0) {
	    if ((i + 1) < ac) {
	        col = atoi(av[i+=1]) ;
	    }
	}
	if (stricmp(av[i], "-smooth") == 0) {
	    ScrSmooth = TRUE ;
	}
	if (stricmp(av[i], "-colors") == 0) {
            if ((i + 1) < ac) {
	        getColor(av[i+=1]) ;
	    }
	}
    }
    
    /*
     * Save initial screen mode
     */
    ScrModeSav.cb = sizeof(ScrModeSav) ;
    VioGetMode(&ScrModeSav, 0) ;

    if (lin == 0) {
        lin = ScrModeSav.row ;
    }
    if (col == 0) {
        col = ScrModeSav.col ;
    }

    /*
     * Set required screen mode
     */
    ScrModeNow.cb = sizeof(ScrModeNow) ;
    VioGetMode(&ScrModeNow, 0) ;
    ScrModeNow.row = lin ;
    ScrModeNow.col = col ;

    if (VioSetMode(&ScrModeNow, 0) == 0) {
	ScrLin = lin ;
	ScrCol = col ;
    } else if ((lin <= (INT) ScrModeSav.row) && (col <= (INT) ScrModeSav.col)) {
	ScrLin = (INT) ScrModeSav.row ;
	ScrCol = (INT) ScrModeSav.col ;
    } else {
	return(FALSE) ;
    }

    /*
     * Clear terminal screen and show start-up message
     */
	 
    setAttr(DispAttr[0]) ;
    VioScrollUp(0, 0, -1, -1, -1, (PUCHAR) &ScrSpac, 0) ;
	
    setAttr(DispAttr[7]) ;
    len = strlen(msg)     ;
    VioWrtCharStrAtt(msg, len, 0, (ScrCol - len) / 2, &CurAttr, 0) ;
    InitMsg = msg ;

    /*
     * Initialize screen control variables
     */
	
    ScrTop = 0 ;
    ScrBot = ScrLin - 1 ;

    setAttr(DispAttr[0]) ;

    CurLin = 1 ;	/* Below start-up message	*/
    CurCol = 0 ;	/* left most of the line	*/
	
    VioSetCurPos(CurLin, CurCol, 0) ;
	
    /*
     * Set up Logical Video Buffer
     */

    VioGetBuf((PULONG) &LvbBase, &LvbLeng, 0) ;
    LvbBase = (PUSHORT) _emx_16to32((_far16ptr) LvbBase) ;

    LvbTop = LvbLeng ;
    LvbBot = 0       ;
	
    return(TRUE) ;
}

/*
 * s c r T e r m  -  Clean up on Terminal Screen
 */

VOID	scrTerm(BOOL restore)
{
    if (restore) {
	VioSetMode(&ScrModeSav, 0) ;
    	VioSetCurPos(ScrModeSav.row - 1, 0, 0) ;
    } else {
    	VioSetCurPos(ScrLin - 1, 0, 0) ;
    }
}

/*
 * s c r S h o w  -  Display current LVB and Cursor
 */

VOID	scrShow(VOID)
{
    USHORT  from, leng ;

    if (LvbTop < LvbBot) {
        from = LvbTop * 2 ;
	leng = (LvbBot - LvbTop + 1) * 2 ;
        VioShowBuf(from, leng, 0) ;
    }

    VioSetCurPos(CurLin, CurCol, 0) ;
    LvbTop = LvbLeng ;
    LvbBot = 0       ;
}

/*
 * Buffer for Save/Restore
 */

static	USHORT	saveleng ;
static	PUCHAR	savebuff ;
static	INT	prevlin  ;
static	INT	prevcol  ;
static	UCHAR	prevattr ;

/*
 * s c r S a v e  - Save current terminal screen
 */

BOOL	scrSave(VOID)
{
    saveleng = ScrLin * ScrCol * 2 ;
	
    if ((savebuff = (PUCHAR) malloc(saveleng)) == NULL) {
	DosBeep(220, 100) ;
	return(FALSE) ;
    }
    VioReadCellStr(savebuff, &saveleng, 0, 0, 0) ;
    VioSetCurPos((ScrLin - 1), 0, 0) ;
    return(TRUE) ;
}

/*
 * s c r R e s t  -  Restore previous terminal screen
 */

BOOL	scrRest(VOID)
{
    VIOMODEINFO	mode ;

    if (savebuff == NULL) {
	return(FALSE) ;
    }

    mode.cb = sizeof(mode) ;
    VioGetMode(&mode, 0)   ;
	
    if ((mode.row != (USHORT) ScrLin) || (mode.col != (USHORT) ScrCol)) {
	ScrLin = mode.row ;
	ScrCol = mode.col ;
	ScrTop = 0          ;
	ScrBot = ScrLin - 1 ;
		
	if (CurLin  >= ScrLin) CurLin  = ScrLin - 1 ;
	if (prevlin >= ScrLin) prevlin = ScrLin - 1 ;
	if (CurCol  >= ScrCol) CurCol  = ScrCol - 1 ;
	if (prevcol >= ScrCol) prevcol = ScrCol - 1 ;
	if (ScrTop  >= ScrLin) ScrTop  = ScrLin - 1 ;
	if (ScrBot  >= ScrLin) ScrBot  = ScrLin - 1 ;

	if (saveleng > (USHORT) (ScrLin * ScrCol * 2)) {
            saveleng = (USHORT) (ScrLin * ScrCol * 2) ;
	}
    }
    VioWrtCellStr(savebuff, saveleng, 0, 0, 0) ;
    VioSetCurPos(CurLin, CurCol, 0) ;
    free(savebuff) ;
	
    return(TRUE) ;
}

/*
 * S c r o l l U p  -  Scroll Up Display
 */

static  VOID	ScrollUp(INT top, INT bot)
{
    INT	    i, j, len ;
    USHORT  pos, fpos, tpos ;

    top = max(top, (ScrTop + 1)) ;
    bot = min(bot, ScrBot) ;

    /*
     * Scroll Up
     */

    for (i = top, j = top - 1 ; i <= bot ; i++, j++) {
        fpos = i * ScrCol ;
	tpos = j * ScrCol ;
	for (len = ScrCol ; len > 0 ; len--) {
	    LvbBase[tpos++] = LvbBase[fpos++] ;
	}
    }
	
    /*
     * Fill Bottom line with spaces
     */

    pos = bot * ScrCol ;
    for (len = ScrCol ; len > 0 ; len--) {
        LvbBase[pos++] = ScrSpac ;
    }

    /*
     * Mark modifications
     */	

    pos = (USHORT) ((top - 1) * ScrCol) ;
    LvbTop = min(LvbTop, pos) ;
    pos = (USHORT) ((bot + 1) * ScrCol) ;
    LvbBot = max(LvbBot, pos) ;
	
    if (ScrSmooth) {
	scrShow() ;
    }
}

/*
 * S c r o l l D n  - Scroll down Display
 */

static  VOID	ScrollDn(INT top, INT bot)
{
    INT	    i, j, len ;
    USHORT  pos, fpos, tpos ;

    top = max(top, ScrTop) ;
    bot = min(bot, (ScrBot - 1)) ;

    /*
     * Scroll Down
     */

    for (i = bot, j = bot + 1 ; i >= top ; i--, j--) {
        fpos = i * ScrCol ;
	tpos = j * ScrCol ;
	for (len = ScrCol ; len > 0 ; len--) {
	    LvbBase[tpos++] = LvbBase[fpos++] ;
	}
    }
	
    /*
     * Fill Top line with spaces
     */

    pos = top * ScrCol ;
    for (len = ScrCol ; len > 0 ; len--) {
        LvbBase[pos++] = ScrSpac ;
    }

    /*
     * Mark Modifications
     */

    pos = (USHORT) (top * ScrCol) ;
    LvbTop = min(LvbTop, pos) ;
    pos = (USHORT) ((bot + 2) * ScrCol) ;
    LvbBot = max(LvbBot, pos) ;
	
    if (ScrSmooth) {
	scrShow() ;
    }
}

/*****************************************************************/
/* Follwoing functions are used for ANSI screen control		 */
/*****************************************************************/

INT	InterF = 0 ;
INT	InterC = 0 ;
INT	InterV[MAXPARM] = { 0 } ;

INT	ParamF = 0 ;
INT	ParamC = 0 ;
INT	ParamQ = 0 ;
INT	ParamV[MAXPARM] = { 0 } ;

INT	ModeFollow = 1 ;

/*
 * d o R E S E T  -  Reset Display
 */

VOID	doRESET(VOID)
{
    if (InitMsg) {
        scrInit(0, NULL, InitMsg) ;
    } else {
        scrInit(0, NULL, "ShiftyTerm") ;
    }
}

/*
 * d o A N K  -  Output ANK character and proceed cursor
 */

VOID	doANK(USHORT c)
{
    USHORT  pos ;

    if ((CurCol + 1) > ScrCol) {
	CurLin += 1 ;
	CurCol = 0 ;
    }
    if (CurLin > ScrBot) {
	ScrollUp(ScrTop, ScrBot) ;
	CurLin = ScrBot ;
    }
    pos = CurCol + ScrCol * CurLin ;

    LvbTop = min(LvbTop, pos) ;
    LvbBase[pos++] = CurCell | c ;
    LvbBot = max(LvbBot, pos) ;

    CurCol += 1 ;
}

/*
 * d o K A N J I  -  Output KANJI character (in SHIFT-JIS code)
 */

VOID	doKANJI(USHORT c1, USHORT c2)
{
    USHORT  pos ;

    if ((CurCol + 2) > ScrCol) {
	CurLin += 1 ;
	CurCol = 0  ;
    }
    if (CurLin > ScrBot) {
	ScrollUp(ScrTop, ScrBot) ;
	CurLin = ScrBot ;
    }
    pos = CurCol + ScrCol * CurLin ;

    LvbTop = min(LvbTop, pos) ;
    LvbBase[pos++] = CurCell | c1 ;
    LvbBase[pos++] = CurCell | c2 ;
    LvbBot = max(LvbBot, pos) ;

    CurCol += 2 ;
}

/*
 * d o B E L  - ring bell
 */

VOID	doBEL(VOID)
{
    DosBeep(1000, 100) ;
}

/*
 * d o B S  -  back space (non-destructive)
 */

VOID	doBS(VOID)
{
    if (CurCol == 0) {
	return ;
    }
    CurCol -= 1 ;
}

/*
 * d o H T  - Horizontal Tabulation
 */

VOID	doHT(VOID)
{
    while (CurCol < (ScrCol - 1)) {
	CurCol += 1 ;
	if ((CurCol % 8) == 0) {
	    break ;
	}
    }
}

/*
 * d o C R  - Carrige Return
 */

VOID	doCR(VOID)
{
    CurCol = 0 ;
}

/*
 * d o L F  -  Line Feed
 */

VOID	doLF(VOID)
{
    CurLin += 1 ;

    if (CurLin > ScrBot) {
	ScrollUp(ScrTop, ScrBot) ;
	CurLin = ScrBot ;
    }
}

/*
 * doINDEX - Index Cursor
 */

VOID	doINDEX(VOID)
{
    CurLin += 1 ;

    if (CurLin > ScrBot) {
	ScrollUp(ScrTop, ScrBot) ;
	CurLin = ScrBot ;
    }
}

/*
 * doREVIN - Reverse Index
 */

VOID	doREVIN(VOID)
{
    CurLin -= 1 ;

    if (CurLin < ScrTop) {
	ScrollDn(ScrTop, ScrBot) ;
	CurLin = ScrTop ;
    }
}

/*
 * doCURUP - Cursor Up
 */

VOID	doCURUP(VOID)
{
    INT	    cnt ;

    cnt = ParamV[0] ? ParamV[0] : 1 ;
    CurLin = max(ScrTop, (CurLin - cnt)) ;
}

/*
 * doCURDN - Cursor Down
 */

VOID	doCURDN(VOID)
{
    INT	    cnt ;

    cnt = ParamV[0] ? ParamV[0] : 1 ;
    CurLin = min(ScrBot, (CurLin + cnt)) ;
}

/*
 * doCURFW - Cursor Forward (Right)
 */

VOID	doCURFW(VOID)
{
    INT	    cnt ;

    cnt = ParamV[0] ? ParamV[0] : 1 ;
    CurCol = min((ScrCol - 1), (CurCol + cnt)) ;
}

/*
 * doCURBK - Cursor Backward (Left)
 */

VOID	doCURBK(VOID)
{
    INT	    cnt ;

    cnt = ParamV[0] ? ParamV[0] : 1 ;
    CurCol = max(0, (CurCol - cnt)) ;
}

/*
 * doCURPOS - Cursor Position
 */

VOID	doCURPOS(VOID)
{
    INT	    row, col ;

    row = ParamV[0] ? ParamV[0] : 1 ;
    col = ParamV[1] ? ParamV[1] : 1 ;
    row = (row > ScrLin) ? ScrLin : row ;
    col = (col > ScrCol) ? ScrCol : col ;

    CurLin = row - 1 ;
    CurCol = col - 1 ;
}

/*
 * doERLIN - Erase on Line
 */

VOID	doERLIN(VOID)
{
    scrShow() ;

    switch(ParamV[0]) {
    case 0 :
	VioWrtNCell((PUCHAR) &ScrSpac, (ScrCol - CurCol), CurLin, CurCol, 0) ;
	break ;
    case 1 :
	VioWrtNCell((PUCHAR) &ScrSpac, CurCol, CurLin, 0, 0) ;
	break ;
    default :
	VioWrtNCell((PUCHAR) &ScrSpac, ScrCol, CurLin, 0, 0) ;
	break ;
    }
}

/*
 * doERDSP - Erase on Display
 */

VOID	doERDSP(VOID)
{
    INT	    nums ;

    scrShow() ;

    switch (ParamV[0]) {
    case 0 :
	nums = (ScrCol - CurCol) + (ScrCol * (ScrLin - CurLin)) ;
	VioWrtNCell((PUCHAR) &ScrSpac, nums, CurLin, CurCol, 0) ;
	break ;
    case 1 :
	nums = (ScrCol * (CurLin - 1)) + CurCol ;
	VioWrtNCell((PUCHAR) &ScrSpac, nums, 0, 0, 0) ;
	break ;
    default :
	nums = ScrCol * ScrLin ;
	VioWrtNCell((PUCHAR) &ScrSpac, nums, 0, 0, 0) ;
	break ;
    }
}

/*
 * doATTR  -  Change Attributes
 */

VOID	doATTR(VOID)
{
    INT	    i, attr ;

    if (ParamC == 0) {
        setAttr(DispAttr[0]) ;
	return ;
    }

    for (i = 0 ; i < ParamC ; i++) {
	attr = ParamV[i] ;
	if (attr > 7) {
	    setAttr(attr) ;
	} else {
	    setAttr(DispAttr[attr]) ;
	}
    }
}

/*
 * doREGION  -  set scroll region
 */

VOID	doREGION(VOID)
{
    INT	    lin1, lin2 ;

    if (ParamC == 0) {
	ScrTop = 0 ;
	ScrBot = ScrLin - 1 ;
    } else {
    	lin1 = max(ParamV[0], 1) ;
	lin2 = min(ParamV[1], ScrLin) ;
	ScrTop = min(lin1, lin2) - 1 ;
	ScrBot = max(lin1, lin2) - 1 ;
    }
}

/*
 * doINSERT - Insert Line
 */

VOID	doINSERT(VOID)
{
    INT	    i, cnt ;

    cnt = max(1, ParamV[0]) ;

    if (ModeFollow) {
	if (CurLin >= ScrBot) {
	    scrShow() ;
	    VioWrtNCell((PUCHAR) &ScrSpac, ScrCol, CurLin, 0, 0) ;
	} else {
	    for (i = 0 ; i < cnt ; i++) {
		ScrollDn(CurLin, ScrBot) ;
	    }
    	}
    } else {
	if (CurLin <= ScrTop) {
	    scrShow() ;
	    VioWrtNCell((PUCHAR) &ScrSpac, ScrCol, CurLin, 0, 0) ;
	} else {
	    for (i = 0 ; i < cnt ; i++) {
		ScrollUp(ScrTop, CurLin) ;
	    }
	}
    }
}

/*
 * doDELETE  -  delete line
 */

VOID	doDELETE(VOID)
{
    INT	    i, cnt ;

    cnt = max(1, ParamV[0]) ;

    if (ModeFollow) {
	if (CurLin >= ScrBot) {
	    scrShow() ;
	    VioWrtNCell((PUCHAR) &ScrSpac, ScrCol, ScrBot, 0, 0) ;
	} else {
	    for (i = 0 ; i < cnt ; i++) {
	    	ScrollUp((CurLin + 1), ScrBot) ;
	    }
	}
    } else {
	if (CurLin <= ScrTop) {
	    scrShow() ;
	    VioWrtNCell((PUCHAR) &ScrSpac, ScrCol, ScrTop, 0, 0) ;
	} else {
	    for (i = 0 ; i < cnt ; i++) {
		ScrollDn(ScrTop, (CurLin - 1)) ;
	    }
	}
    }
}

/*
 * doSETMD  -  set modes
 */

VOID	doSETMD(VOID)
{

}

/*
 * doRSTMD  -  reset modes
 */

VOID	doRSTMD(VOID)
{

}

/*
 * doREPST  -  report terminal status
 */

static	UCHAR	statbuff[16] ;

VOID	doREPST(VOID)
{
    USHORT  len ;
    INT     num, d, r ;

    if (ParamC == 0) {
	return ;
    }
    statbuff[0] = '\0' ;

    DosEnterCritSec() ;

    if (ParamV[0] == 5) {
	/*
	 * Report Terminal Status
	 */
	len = 0 ;
	statbuff[len++] = 0x1b ;
	statbuff[len++] = '['  ;
	statbuff[len++] = '0'  ;
	statbuff[len++] = 'n'  ;
    } else if (ParamV[0] == 6) {
	/* 
	 * Report Cursor Position
	 */
	len = 0 ;
	statbuff[len++] = 0x1b ;
	statbuff[len++] = '[' ;
	for (num = CurLin + 1, r = 100 ; num >= 10 ; num /= r, r /= 10) {
            if ((d = num / r) > 0) {
	        statbuff[len++] = (UCHAR) ('0' + d) ;
	    }
	}
        statbuff[len++] = (UCHAR) ('0' + num) ;
	statbuff[len++] = ';';
	for (num = CurLin + 1, r = 100 ; num >= 10 ; num /= r, r /= 10) {
            if ((d = num / r) > 0) {
	        statbuff[len++] = (UCHAR) ('0' + d) ;
	    }
	}
        statbuff[len++] = (UCHAR) ('0' + num) ;
	statbuff[len++] = 'R';
    }
    if ((len = strlen(statbuff))> 0) {
        (*comDevice->comSend) (statbuff, len) ;
    }
    DosExitCritSec() ;
    return ;
}

/*
 * doSAVE  -  save cursor
 */

VOID	doSAVE(VOID)
{
    prevlin  = CurLin  ;
    prevcol  = CurCol  ;
    prevattr = CurAttr ;
}

/*
 * doRESTORE  -  restore cursor
 */

VOID	doRESTORE(VOID)
{
    CurLin     = prevlin  ;
    CurCol     = prevcol  ;
    
    setAttr(prevattr) ;
}
