/*
 * page.c - page layout on Printing Paper
 */

#include <stdio.h>
#include <stdlib.h>

#include "nup.h"
#include "page.h"

/*
 * Page Variables
 */

SIZEL   formSize ;              /* Size of Paper            */
RECTL   formClip ;              /* Clip of Paper            */
SIZEL   drawSize ;              /* Size of Drawing Area     */
RECTL   drawRect ;              /* Drawing Area             */

BOOL    pageFrame  ;            /* TRUE if draw page frame  */
BOOL    pageMarker ;            /* TRUE if draw center mark */
int     widFrame  ;             /* Slack for Page Frame     */
int     widMarker ;             /* Length for Center Marker */

int     pagePerForm ;           /* Pages in a paper     */
RECTL   pageRect[MAXPAGES] ;    /* Page Regions         */
SIZEL   pageSize    ;           /* Page Size            */
int     pageLins    ;           /* Number of lines      */
int     pageCols    ;           /* Number of columns    */

SIZEL   cellSize1 ;         /* Size of SBCS Cell    */
SIZEL   cellSize2 ;         /* Size of DBCS Cell    */
SIZEL   charSize  ;         /* Size of Char Box     */

/*
 * mmtotwips - convert 'mm' to 'twips'
 */

static  LONG    mmtotwips(LONG mm)
{
    return (LONG) (((float) mm) / 25.4 * 1440.0) ;
}

/*
 * getFormSize - get Size of Printer's Default Form
 *
 *      formSize - size of paper
 *      formClip - clip on paper
 */

static  HCINFO  aForms[32] ;
static  int     nForms = 0 ;

static  void    getFormSize(void)
{
    int     i   ;
    PHCINFO pfm ;
    
    /*
     * query printer's current form
     */
     
    nForms = DevQueryHardcopyCaps(hdcPrinter, 0, 32, aForms) ;
    
    for (i = 0, pfm = NULL ; i < nForms ; i++) {
        if ((aForms[i].flAttributes & HCAPS_CURRENT) != 0) {
	    pfm = &aForms[i] ;
	    break ;
        }
    }
    
    if (pfm != NULL) {
        formSize.cx      = mmtotwips(pfm->cx)          ;
	formSize.cy      = mmtotwips(pfm->cy)          ;
	formClip.xLeft   = mmtotwips(pfm->xLeftClip)   ;
	formClip.xRight  = mmtotwips(pfm->xRightClip)  ;
	formClip.yTop    = mmtotwips(pfm->yTopClip)    ;
	formClip.yBottom = mmtotwips(pfm->yBottomClip) ;
    } else {
        formSize.cx = mmtotwips(210) ;      /* A4 as default form   */
	formSize.cy = mmtotwips(297) ;      /*                      */
	formClip.xLeft   = 0           ;
	formClip.xRight  = formSize.cx ;
	formClip.yTop    = formSize.cy ;
	formClip.yBottom = 0           ;
    }
    
#ifdef  DEBUG
    printf("form Size %d x %d, Clip %d - %d, %d - %d\n",
            formSize.cx, formSize.cy,
	    formClip.xLeft, formClip.xRight,
	    formClip.yBottom, formClip.yTop) ;
    fflush(stdout) ;
#endif
}

/*
 * getDrawArea - calc. drawing area on the paper
 *
 *      drawing are wiil restricted with margins (in layout) and formClip.
 *      result will set to
 *
 *      drawRect - drawing area
 *      drawSize - size of drawing area
 */

static  void    getDrawArea(LAYPTR lay)
{
    LONG    val, cx, cy ;

    cx = formSize.cx ;
    cy = formSize.cy ;
    
    /*
     * calc. drawing area from Margins
     */

    drawRect.xLeft   = mmtotwips(lay->lmargin)      ;
    drawRect.xRight  = cx - mmtotwips(lay->rmargin) ;
    drawRect.yTop    = cy - mmtotwips(lay->tmargin) ;
    drawRect.yBottom = mmtotwips(lay->bmargin)      ;

#ifdef  DEBUG
    printf("margins Left %d, Right %d, Top %d, Bottom %d\n",
                mmtotwips(lay->lmargin), mmtotwips(lay->rmargin),
		mmtotwips(lay->tmargin), mmtotwips(lay->bmargin)) ;
    printf("drawarea %d - %d, %d - %d\n",
                drawRect.xLeft, drawRect.xRight, 
		drawRect.yBottom, drawRect.yTop) ;
    fflush(stdout) ;
#endif

    /*
     * adjust with form clip
     */

    if (drawRect.xLeft > formClip.xLeft) {
        drawRect.xLeft  -= formClip.xLeft ;
	drawRect.xRight -= formClip.xLeft ;
    } else {
        val = formClip.xLeft - drawRect.xLeft ;
        drawRect.xLeft  += val ;
	drawRect.xRight += val ;
    }
    if (drawRect.yBottom > formClip.yBottom) {
        drawRect.yBottom -= formClip.yBottom ;
	drawRect.yTop    -= formClip.yBottom ;
    } else {
        val = formClip.yBottom - drawRect.yBottom ;
        drawRect.yBottom -= val ;
	drawRect.yTop    -= val ;
    }
    if ((drawRect.xRight - drawRect.xLeft) > cx) {
        val = drawRect.xRight - drawRect.xLeft - cx ;
	drawRect.xRight -= val ;
    }
    if ((drawRect.yTop - drawRect.yBottom) > cy) {
        val = drawRect.yTop - drawRect.yBottom - cy ;
	drawRect.yTop -= val ;
    }

    /*
     * set size of drawing area
     */
    
    drawSize.cx = drawRect.xRight - drawRect.xLeft ;
    drawSize.cy = drawRect.yTop - drawRect.yBottom ;

#ifdef  DEBUG
    printf("drawing area Size %d x %d, Rect %d - %d, %d - %d\n",
            drawSize.cx, drawSize.cy,
	    drawRect.xLeft, drawRect.xRight,
	    drawRect.yBottom, drawRect.yTop) ;
    fflush(stdout) ;
#endif
}

/*
 * rotateForm - rotate form if page orientation not matches
 */

static  void    rotateForm(void)
{
    SIZEL       siz ;
    RECTL       rct ;
    POINTL      pt  ;
    MATRIXLF    mat ;

    /*
     * Swap horz/virt of Form, Drawing Are
     */
     
    siz = formSize ;
    formSize.cx = siz.cy ;
    formSize.cy = siz.cx ;
    siz = drawSize ;
    drawSize.cx = siz.cy ;
    drawSize.cy = siz.cx ;

    rct = formClip ;
    formClip.xLeft   = rct.yBottom ;
    formClip.yBottom = rct.xLeft   ;
    formClip.xRight  = rct.yTop    ;
    formClip.yTop    = rct.xRight  ;
    rct = drawRect ;
    drawRect.xLeft   = rct.yBottom ;
    drawRect.yBottom = rct.xLeft   ;
    drawRect.xRight  = rct.yTop    ;
    drawRect.yTop    = rct.xRight  ;
    
    /*
     * set rotate matrix on printer's presentaion space
     */

    GpiQueryDefaultViewMatrix(hpsPrinter, 9, &mat) ;
    pt.x = pt.y = 0 ;
    GpiRotate(hpsPrinter, &mat, TRANSFORM_ADD, MAKEFIXED(90, 0), &pt) ;
    pt.x = formClip.yTop - formClip.yBottom ;
    pt.y = 0 ;
    GpiTranslate(hpsPrinter, &mat, TRANSFORM_ADD, &pt) ;
    GpiSetDefaultViewMatrix(hpsPrinter, 9, &mat, TRANSFORM_REPLACE) ;
}
 
/*
 * set page layouts for each layout styles. result will set to
 *
 *      pagePerForm - number of pages on a form
 *      pageRect[]  - region for each pages
 *      pageSize    - size of page
 */

static  void    dumpLayout(void)
{
    int     i ;
    
    printf("Page Size %d x %d\n", pageSize.cx, pageSize.cy) ;
    for (i = 0 ; i < pagePerForm ; i++) {
        printf("Page %d, Horz %d - %d, Virt %d - %d\n", i,
	            pageRect[i].xLeft, pageRect[i].xRight,
	            pageRect[i].yBottom, pageRect[i].yTop) ;
    }
    fflush(stdout) ;
}

static  void    layoutPort1(LAYPTR lay)
{
    if (formSize.cx > formSize.cy) {    /* form was Landscape   */
        rotateForm() ;                  /* rotate to Portrait   */
    }
    
    /*
     * PORT1    +-------+
     *          |       |
     *          |       |
     *          |       |
     *          |       |
     *          +-------+
     */
     
    pagePerForm = 1 ;
    pageSize.cx = drawSize.cx ;
    pageSize.cy = drawSize.cy ;
    
    pageRect[0].xLeft   = drawRect.xLeft   + widFrame ;
    pageRect[0].xRight  = drawRect.xRight  - widFrame ;
    pageRect[0].yTop    = drawRect.yTop    - widFrame ;
    pageRect[0].yBottom = drawRect.yBottom + widFrame ;

    pageSize.cx -= (widFrame * 2) ;
    pageSize.cy -= (widFrame * 2) ;
    
#ifdef  DEBUG
    dumpLayout() ;
#endif
}

static  void    layoutPort2(LAYPTR lay)
{
    LONG    hspace ;
    
    if (formSize.cy > formSize.cx) {    /* form was Portrait    */
        rotateForm() ;                  /* rotate to Landscape  */
    }
    
    /*
     * PORT2    +-----+-----+
     *          |     |     |
     *          |     |     |
     *          |     |     |
     *          +-----+-----+
     */

    pagePerForm = 2 ;
    hspace = mmtotwips(lay->hspace) ;
    
    pageSize.cx = (drawSize.cx - hspace) / 2 ;
    pageSize.cy = drawSize.cy ;
    
    pageRect[0].xLeft   = drawRect.xLeft                + widFrame ;
    pageRect[0].xRight  = drawRect.xLeft + pageSize.cx  - widFrame ;
    pageRect[0].yTop    = drawRect.yTop                 - widFrame ;
    pageRect[0].yBottom = drawRect.yBottom              + widFrame ;

    pageRect[1].xLeft   = drawRect.xRight - pageSize.cx + widFrame ;
    pageRect[1].xRight  = drawRect.xRight               - widFrame ;
    pageRect[1].yTop    = drawRect.yTop                 - widFrame ;
    pageRect[1].yBottom = drawRect.yBottom              + widFrame ;

    pageSize.cx -= (widFrame * 2) ;
    pageSize.cy -= (widFrame * 2) ;

#ifdef  DEBUG
    dumpLayout() ;
#endif
}

static  void    layoutPort4Row(LAYPTR lay)
{
    LONG    vspace, hspace ;

    if (formSize.cx > formSize.cy) {    /* form was Landscape   */
        rotateForm() ;                  /* rotate to Portrait   */
    }
    
    /*
     * PORT4-ROW-FIRST
     *          +---+---+
     *          | 0 | 1 |
     *          +---+---+
     *          | 2 | 3 | 
     *          +---+---+
     */

    pagePerForm = 4 ;
    vspace = mmtotwips(lay->vspace) ;
    hspace = mmtotwips(lay->hspace) ;
    
    pageSize.cx = (drawSize.cx - hspace) / 2 ;
    pageSize.cy = (drawSize.cy - vspace) / 2 ;
    
    pageRect[0].xLeft   = drawRect.xLeft                 + widFrame ;
    pageRect[0].xRight  = drawRect.xLeft   + pageSize.cx - widFrame ;
    pageRect[0].yTop    = drawRect.yTop                  - widFrame ;
    pageRect[0].yBottom = drawRect.yTop    - pageSize.cy + widFrame ;

    pageRect[1].xLeft   = drawRect.xRight  - pageSize.cx + widFrame ;
    pageRect[1].xRight  = drawRect.xRight                - widFrame ;
    pageRect[1].yTop    = drawRect.yTop                  - widFrame ;
    pageRect[1].yBottom = drawRect.yTop    - pageSize.cy + widFrame ;

    pageRect[2].xLeft   = drawRect.xLeft                 + widFrame ;
    pageRect[2].xRight  = drawRect.xLeft   + pageSize.cx - widFrame ;
    pageRect[2].yTop    = drawRect.yBottom + pageSize.cy - widFrame ;
    pageRect[2].yBottom = drawRect.yBottom               + widFrame ;

    pageRect[3].xLeft   = drawRect.xRight  - pageSize.cx + widFrame ;
    pageRect[3].xRight  = drawRect.xRight                - widFrame ;
    pageRect[3].yTop    = drawRect.yBottom + pageSize.cy - widFrame ;
    pageRect[3].yBottom = drawRect.yBottom               + widFrame ;

    pageSize.cx -= (widFrame * 2) ;
    pageSize.cy -= (widFrame * 2) ;

#ifdef  DEBUG
    dumpLayout() ;
#endif
}

static  void    layoutPort4Col(LAYPTR lay)
{
    LONG    vspace, hspace ;

    if (formSize.cx > formSize.cy) {    /* form was Landscape   */
        rotateForm() ;                  /* rotate to Portrait   */
    }
    
    /*
     * PORT4-COL-FIRST
     *          +---+---+
     *          | 0 | 2 |
     *          +---+---+
     *          | 1 | 3 | 
     *          +---+---+
     */

    pagePerForm = 4 ;
    vspace = mmtotwips(lay->vspace) ;
    hspace = mmtotwips(lay->hspace) ;
    
    pageSize.cx = (drawSize.cx - hspace) / 2 ;
    pageSize.cy = (drawSize.cy - vspace) / 2 ;
    
    pageRect[0].xLeft   = drawRect.xLeft                 + widFrame ;
    pageRect[0].xRight  = drawRect.xLeft   + pageSize.cx - widFrame ;
    pageRect[0].yTop    = drawRect.yTop                  - widFrame ;
    pageRect[0].yBottom = drawRect.yTop    - pageSize.cy + widFrame ;

    pageRect[1].xLeft   = drawRect.xLeft                 + widFrame ;
    pageRect[1].xRight  = drawRect.xLeft   + pageSize.cx - widFrame ;
    pageRect[1].yTop    = drawRect.yBottom + pageSize.cy - widFrame ;
    pageRect[1].yBottom = drawRect.yBottom               + widFrame ;

    pageRect[2].xLeft   = drawRect.xRight  - pageSize.cx + widFrame ;
    pageRect[2].xRight  = drawRect.xRight                - widFrame ;
    pageRect[2].yTop    = drawRect.yTop                  - widFrame ;
    pageRect[2].yBottom = drawRect.yTop    - pageSize.cy + widFrame ;

    pageRect[3].xLeft   = drawRect.xRight  - pageSize.cx + widFrame ;
    pageRect[3].xRight  = drawRect.xRight                - widFrame ;
    pageRect[3].yTop    = drawRect.yBottom + pageSize.cy - widFrame ;
    pageRect[3].yBottom = drawRect.yBottom               + widFrame ;

    pageSize.cx -= (widFrame * 2) ;
    pageSize.cy -= (widFrame * 2) ;

#ifdef  DEBUG
    dumpLayout() ;
#endif
}

static  void    layoutLand1(LAYPTR lay)
{
    if (formSize.cy > formSize.cx) {    /* form was Portrait    */
        rotateForm() ;                  /* rotate to Landscape  */
    }
    
    /*
     * LAND1    +-----------+
     *          |           |
     *          |           |
     *          |           |
     *          +-----------+
     */

    pagePerForm = 1 ;
    pageSize.cx = drawSize.cx ;
    pageSize.cy = drawSize.cy ;
    
    pageRect[0].xLeft   = drawRect.xLeft   + widFrame ;
    pageRect[0].xRight  = drawRect.xRight  - widFrame ;
    pageRect[0].yTop    = drawRect.yTop    - widFrame ;
    pageRect[0].yBottom = drawRect.yBottom + widFrame ;

    pageSize.cx -= (widFrame * 2) ;
    pageSize.cy -= (widFrame * 2) ;

#ifdef  DEBUG
    dumpLayout() ;
#endif
}

static  void    layoutLand2(LAYPTR lay)
{
    LONG    vspace ;
    
    if (formSize.cx > formSize.cy) {    /* form was Portrait    */
        rotateForm() ;                  /* rotate to Landscape  */
    }
    
    /*
     * LAND2    +-------+
     *          |       |
     *          |       |
     *          +-------+
     *          |       |
     *          |       |
     *          +-------+
     */

    pagePerForm = 2 ;
    vspace = mmtotwips(lay->vspace) ;
    
    pageSize.cx = drawSize.cx ;
    pageSize.cy = (drawSize.cy - vspace) / 2 ;
    
    pageRect[0].xLeft   = drawRect.xLeft               + widFrame ;
    pageRect[0].xRight  = drawRect.xLeft + pageSize.cx - widFrame ;
    pageRect[0].yTop    = drawRect.yTop                - widFrame ;
    pageRect[0].yBottom = drawRect.yTop  - pageSize.cy + widFrame ;

    pageRect[1].xLeft   = drawRect.xLeft                 + widFrame ;
    pageRect[1].xRight  = drawRect.xLeft + pageSize.cx   - widFrame ;
    pageRect[1].yTop    = drawRect.yBottom + pageSize.cy - widFrame ;
    pageRect[1].yBottom = drawRect.yBottom               + widFrame ;

    pageSize.cx -= (widFrame * 2) ;
    pageSize.cy -= (widFrame * 2) ;

#ifdef  DEBUG
    dumpLayout() ;
#endif
}

static  void    layoutLand4Row(LAYPTR lay)
{
    LONG    vspace, hspace ;

    if (formSize.cy > formSize.cx) {    /* form was Portrait    */
        rotateForm() ;                  /* rotate to Landscape  */
    }
    
    /*
     * LAND4-ROW-FIRST
     *          +---+---+
     *          | 0 | 1 |
     *          +---+---+
     *          | 2 | 3 | 
     *          +---+---+
     */

    pagePerForm = 4 ;
    vspace = mmtotwips(lay->vspace) ;
    hspace = mmtotwips(lay->hspace) ;
    
    pageSize.cx = (drawSize.cx - hspace) / 2 ;
    pageSize.cy = (drawSize.cy - vspace) / 2 ;
    
    pageRect[0].xLeft   = drawRect.xLeft                 + widFrame ;
    pageRect[0].xRight  = drawRect.xLeft   + pageSize.cx - widFrame ;
    pageRect[0].yTop    = drawRect.yTop                  - widFrame ;
    pageRect[0].yBottom = drawRect.yTop    - pageSize.cy + widFrame ;

    pageRect[1].xLeft   = drawRect.xRight  - pageSize.cx + widFrame ;
    pageRect[1].xRight  = drawRect.xRight                - widFrame ;
    pageRect[1].yTop    = drawRect.yTop                  - widFrame ;
    pageRect[1].yBottom = drawRect.yTop    - pageSize.cy + widFrame ;

    pageRect[2].xLeft   = drawRect.xLeft                 + widFrame ;
    pageRect[2].xRight  = drawRect.xLeft   + pageSize.cx - widFrame ;
    pageRect[2].yTop    = drawRect.yBottom + pageSize.cy - widFrame ;
    pageRect[2].yBottom = drawRect.yBottom               + widFrame ;

    pageRect[3].xLeft   = drawRect.xRight  - pageSize.cx + widFrame ;
    pageRect[3].xRight  = drawRect.xRight                - widFrame ;
    pageRect[3].yTop    = drawRect.yBottom + pageSize.cy - widFrame ;
    pageRect[3].yBottom = drawRect.yBottom               + widFrame ;

    pageSize.cx -= (widFrame * 2) ;
    pageSize.cy -= (widFrame * 2) ;

#ifdef  DEBUG
    dumpLayout() ;
#endif
}

static  void    layoutLand4Col(LAYPTR lay)
{
    LONG    vspace, hspace ;

    if (formSize.cy > formSize.cx) {    /* form was Portrait    */
        rotateForm() ;                  /* rotate to Landscape  */
    }
    
    /*
     * LAND4-COL-FIRST
     *          +---+---+
     *          | 0 | 2 |
     *          +---+---+
     *          | 1 | 3 | 
     *          +---+---+
     */

    pagePerForm = 4 ;
    vspace = mmtotwips(lay->vspace) ;
    hspace = mmtotwips(lay->hspace) ;
    
    pageSize.cx = (drawSize.cx - hspace) / 2 ;
    pageSize.cy = (drawSize.cy - vspace) / 2 ;
    
    pageRect[0].xLeft   = drawRect.xLeft                 + widFrame ;
    pageRect[0].xRight  = drawRect.xLeft   + pageSize.cx - widFrame ;
    pageRect[0].yTop    = drawRect.yTop                  - widFrame ;
    pageRect[0].yBottom = drawRect.yTop    - pageSize.cy + widFrame ;

    pageRect[1].xLeft   = drawRect.xLeft                 + widFrame ;
    pageRect[1].xRight  = drawRect.xLeft   + pageSize.cx - widFrame ;
    pageRect[1].yTop    = drawRect.yBottom + pageSize.cy - widFrame ;
    pageRect[1].yBottom = drawRect.yBottom               + widFrame ;

    pageRect[2].xLeft   = drawRect.xRight  - pageSize.cx + widFrame ;
    pageRect[2].xRight  = drawRect.xRight                - widFrame ;
    pageRect[2].yTop    = drawRect.yTop                  - widFrame ;
    pageRect[2].yBottom = drawRect.yTop    - pageSize.cy + widFrame ;

    pageRect[3].xLeft   = drawRect.xRight  - pageSize.cx + widFrame ;
    pageRect[3].xRight  = drawRect.xRight                - widFrame ;
    pageRect[3].yTop    = drawRect.yBottom + pageSize.cy - widFrame ;
    pageRect[3].yBottom = drawRect.yBottom               + widFrame ;

    pageSize.cx -= (widFrame * 2) ;
    pageSize.cy -= (widFrame * 2) ;

#ifdef  DEBUG
    dumpLayout() ;
#endif
}

/*
 * getFontSize - calc. font size & cell size
 *
 *      pageLins  - number of lines   in a page
 *      pageCols  - number of columns in a page
 *      charSize  - size of character box (square for KANJI)
 *      cellSize1 - cell size for ANK
 *      cellSize2 - cell size for KANJI
 */

static  void    getFontSize(LAYPTR lay)
{
    LONG    cx, cy ;
    float   cf     ;
    
    pageLins = lay->lines   ;
    pageCols = lay->columns ;
    
    /*
     * Calc. char size
     */

    cy = pageSize.cy / pageLins     ;
    cx = pageSize.cx / pageCols * 2 ;

    cf = (float) ((cy < cx) ? cy : cx) ;
    charSize.cx = charSize.cy = (LONG) (cf * (float) lay->ratio / 100.0) ;

    /*
     * Calc. cell size 
     */
     
    cellSize1.cy = cellSize2.cy = cy ;
    cellSize2.cx = (LONG) (((float) charSize.cx) * 1.1) ;
    
    if (cellSize2.cx > cx) {
        cellSize1.cx = cx / 2 ;
	cellSize2.cx = cx     ;
    } else {
        cellSize1.cx = cellSize2.cx / 2 ;
    }
    
    /*
     * Adjust number of columns with new cell width
     */
     
    pageCols = pageSize.cx / cellSize1.cx ;

#ifdef  DEBUG
    printf("Char Size       %d x %d\n", charSize.cx,  charSize.cy ) ;
    printf("Cell Size ANK   %d x %d\n", cellSize1.cx, cellSize1.cy) ;
    printf("Cell Size KANJI %d x %d\n", cellSize2.cx, cellSize2.cy) ;
    fflush(stdout) ;
#endif
}

/*
 * pageLayout - calc. page layouts
 */

void    pageLayout(LAYPTR lay)
{
    if (hdcPrinter == NULLHANDLE || hpsPrinter == NULLHANDLE) {
#ifdef  DEBUG
        printf("pagelayout - printer is not open\n") ; fflush(stdout) ;
#endif
        return ;
    }
    
    pageFrame  = lay->frame  ;
    pageMarker = lay->marker ;

    widFrame  = mmtotwips(2) ;
    widMarker = mmtotwips(4) ;

    getFormSize()    ;
    getDrawArea(lay) ;
    
    switch (lay->index) {
    case LAYOUT_PORT1 :
        layoutPort1(lay) ;
	break ;
    case LAYOUT_PORT2 :
        layoutPort2(lay) ;
	break ;
    case LAYOUT_PORT4 :
        if (lay->pgorder == LAYOUT_ORDER_ROW) {
	    layoutPort4Row(lay) ;
	} else {
	    layoutPort4Col(lay) ;
	}
	break ;
    case LAYOUT_LAND1 :
        layoutLand1(lay) ;
	break ;
    case LAYOUT_LAND2 :
        layoutLand2(lay) ;
	break ;
    case LAYOUT_LAND4 :
        if (lay->pgorder == LAYOUT_ORDER_ROW) {
	    layoutLand4Row(lay) ;
	} else {
	    layoutLand4Col(lay) ;
	}
	break ;
    }

    getFontSize(lay) ;
}
