 /****************************************************************************
 **
 **   lj -- A printing utility for the HP LaserJet
 **
 **
 **   This program prints a series of files on the LaserJet printer.  The
 **   files are printed in a "landscape" font at 17 characters to the inch.
 **   To take advantage of this density, two "pages" of information from
 **   the file are printed on each piece of paper (left and right halves).
 **
 **   Usage is:       lj  file1 file2 file3 ...
 **
 **   Where file# is a valid MS-DOS filename, included on the command line.
 **
 **   Originally written by Joe Barnhart and subsequently modifed by
 **   Ray Duncan and Chip Rabinowitz.  This program has been placed in
 **   the public domain for use without profit.
 **
 **   See the ChangeLog for update information.
 **
 ****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <ctype.h>
#include <fcntl.h>

#ifdef __STDC__
# define _PROTO(x) x
#else
# define _PROTO(x) ()
#endif

int  printfile _PROTO((char  *filename));
int  printpage _PROTO((void));
void header    _PROTO((char  *filename,int  pagenum));
void timestamp _PROTO((char  *timestr));
void datestamp _PROTO((char  *datestr));
void dovert    _PROTO((void));
void usage     _PROTO((void));
void shade     _PROTO((void));

int     MAXLINE=66;        /* maximum lines per page on LaserJet */
int     MAXLINE_RESET=66;  /* maximum lines per page on LaserJet */
#define MAXVERT 69         /* maximum lines per page on LaserJet */

FILE *fp;               /* FILE pointer for the file to be printed */
FILE *lp = stdout;

int line_no = 1;
int pagenum;

int  hdrflag;     /* Header print suppress flag */
int  pgflag;      /* Page separator flag */
int  tab = 4;     /* default value of one tab stop */
int  pgside;      /* left/right half of page indicator  */
int  psprint = 0;   /* proportional using Helv 6 pt. */
int  shade_it = 0;  /* shade every other line */
int  book = 0;      /* indent left 6 additional spaces for punching*/
int  maxcol=85;
int  numbers = 0;
int  duplex=0;
int  formfeed=0;

int do_print=1;     /*flags for doing duplex printing*/

/*************************************************************************

main

*************************************************************************/

main(argc, argv)
int argc;
char *argv[];
{
int filenum;
char *fname;
int first;

#ifdef __EMX__
   _wildcard (&argc, &argv);
#endif

   if (argc <= 1) usage();

   for (first = 1; first < argc; first++)  {
      if (*argv[first] != '-') /* check for flags */
         break;               /* and exit if none */
      argv[first]++;
      switch (tolower(*argv[first])) {
      case 'e':      /* Helv proportional font */
         psprint = 1;
         maxcol=104;
         break;
      case 'h':      /* suppress headers flag */
         hdrflag++;
      MAXLINE=69;
      MAXLINE_RESET=69;
         break;
      case 't':      /* Horizontal tab value */
         tab = atoi(++argv[first]);
         if (tab < 1 || tab > 8) {
             fprintf(stderr, "Invalid tab value specified\n"); /* -- defaulting to 8\n"); */
             /* tab = 8; */
            exit(1);
         }
         break;
      case 'l':
         MAXLINE = atoi(++argv[first]);
         if (MAXLINE < 1 || MAXLINE > MAXLINE_RESET) {
             fprintf(stderr, "Invalid lines/page value specified\n"); /*  -- defaulting to MAX\n"); */
            /* MAXLINE = MAXLINE_RESET; */
            exit(1);
         }
         break;
      case 'p':      /* suppress page eject between files */
         pgflag++;
         break;
      case 's':
         shade_it++;
         break;
      case 'n':
         numbers++;
         break;
      case 'b':
         book++;
         maxcol=79;
         break;
      case 'd':
         duplex=1;
         break;
      case 'i':
         formfeed++;
         break;
      default:
         fprintf(stderr, "Invalid Flag Specified ('%s')\n",  /* -- ignored.....\n", */
               --argv[first]);
         usage();
      }
   }

   if(duplex) {
     if((argc - 1) > first) {
        fprintf(stderr, "Only one file at a time may be printed when using DUPLEX mode.\n");
        exit(1);
     }
   }

   fprintf(stderr, "Lines per page set to %d\n",MAXLINE);

   for(filenum = first; filenum < argc; filenum++ ) {
      fp = fopen(fname = argv[filenum], "r");
      if( fp == NULL ) {
         fprintf(stderr,  "File %s doesn't exist.\n", fname );
      }
      else {
         fprintf(stderr,  "Now printing %s\n", fname );
         if (first) {   /* initialize the LaserJet for landscape, etc. */
            fprintf( lp,"\033E\033&l1O\033(s17H\033&l5.14C\033&l70F\033&l5E" );
            if (psprint)
               fprintf(lp,"\033&l1O\033(0U\033(s1p6vsb4T");
            first = 0;
         }
         printfile( fname );
         fclose( fp );
      }
   }    /* of loop through run-time args */

   if (first == 0) {             /* we found a file */
      if (pgflag && pgside)
         fputc('\f', lp);
      fprintf( lp, "\r\033E" );      /* clear LaserJet */
      if (lp != stdout)
         fclose(lp);
		fputc('\n', stderr);
   }
   return(0);
}

/*************************************************************************

printfile (filename)

*************************************************************************/

printfile(filename)
char *filename;
{
   int retval = 0;
   int printed = 0;
   int pass=0;
   int c2;
   int f_feed=0;
   int phys_pages=0;
   int plus=1;

do_it_again:
   pagenum=1;
   while( (feof(fp)==0) && (retval==0) ) {
      f_feed=0;
      if (pgside == 0) {
         printed = 0;
         c2 = fgetc(fp);
         if(feof(fp)) break;
         switch (c2) {
         case EOF:
         case '\032':
            retval = -1;
            break;
         default:
            printed=1;
            if(do_print) {
					dovert();
            	if(psprint)
	               fprintf(lp,"\033&a0r0c128m0L\r");
   	         else if(book)
      	         fprintf(lp, "\033&a0r0c85m7L\r"); /* set LaserJet to left half */
         	   else
            	   fprintf(lp, "\033&a0r0c85m0L\r"); /* set LaserJet to left half */
            	if(shade_it && do_print)
						shade();
				}
            ungetc(c2,fp);
            header(filename, pagenum++);       /* title top of page */
            retval = printpage();                 /* print one page */
            pgside ^= 1;
         }
      }
      if(feof(fp)==0 && retval==0) {      /* if more to print... */
			if (do_print) {
				if(psprint)
	            fprintf(lp,"\033&a0r0c255m131L\r");
   	      else if(book)
      	      fprintf(lp, "\033&a0r0c175m97L\r");    /* LaserJet to right half */
         	else
	            fprintf(lp, "\033&a0r0c175m90L\r");    /* LaserJet to right half */
			}
         c2 = fgetc(fp);
         if(feof(fp)) break;
         switch (c2) {
         case EOF:
         case '\032':
            retval = -1;
            break;
         default:
            ungetc(c2,fp);
            header(filename, pagenum++);          /* title top of page */
            retval = printpage();                 /* print one page */
            pgside ^= 1;
         }
      }

      if (printed && do_print && (pgflag == 0 || pgside == 0)) {
         fputc('\f', lp);
         pgside = 0;
         f_feed = 1;
         if(plus)
            ++phys_pages;
         else
            --phys_pages;
      }
      if(duplex) do_print ^= 1;
   }

   if(duplex) {
      if(pass) {
         while(phys_pages) {
            fputc('\f',lp);
            --phys_pages;
         }
         return(0);
      }
      plus=0;
      if(!f_feed && retval == 0)
         fputc('\f',lp);
      fflush(lp);
      ++pass;
      rewind(fp);
      retval=0;
      do_print = 0;
      pgside = 0;
      fprintf(stderr, "\nFlip the paper and press any key when ready\n");
      getchar();
      goto do_it_again;
   }
   return(0);
}

/*******************************************************************************

printpage
   print a logical page

*************************************************************************/

printpage()
{
   int c;
   int line,col;
   static int cont = 0;
   static char *cont1 = "---->";

   line = col = 0;
   if(cont)
      {
      if(do_print) fprintf(lp,cont1);
      col = strlen(cont1);
      cont = 0;
      }

   while( line < MAXLINE ) {
      c = fgetc(fp);
      if(feof(fp)) return(-1);

      if(col>maxcol) {
         line++;
         switch(c) {
         case '\n':
         case '\r':
         case '\f':
         case EOF:
         case '\032':
            break;
         default:
            if(line >= MAXLINE) {
               cont = 1;
               ungetc(c,fp);
               return(0);
            }
            if(do_print) fprintf(lp,"\n%s",cont1);
            col = strlen(cont1);
            break;
         }
      }

      if(col == 0) {
         if(do_print) {
            fprintf(stderr, "Printing page %4.4d line %4.4d\r",pagenum-1,line);
         }
         else {
            fprintf(stderr, "Skipping page %4.4d line %4.4d\r",pagenum-1,line);
         }
         if(numbers) {
            if(do_print) fprintf(lp,"%4.4d:",line_no);
            col=5;
         }
         ++line_no;
      }

      switch(c) {
      case '\n':           /* newline found */
         col = 0;          /* zero column and */
         line++;           /* advance line count */
         if( line < MAXLINE )
            if(do_print) fprintf(lp,"\n");
         break;
      case '\r':           /* CR found */
         break;            /* discard it */
      case '\t':           /* TAB found */
         do {
            if(do_print) fputc(' ',lp);
         } while ( (++col % tab) != 0 );
         break;
      case '\f':                      /* Page break or */
         if(formfeed) break;
         if(line != 0)
            line = MAXLINE;      /* force termination of loop */
            break;
      case EOF:            /* EOF mark */
      case '\032':         /* EOF mark */
         return(-1);
      default:              /* no special case */
         if(do_print) fputc(c,lp);       /* print character */
         col++;
         break;
      }
   }
   return(0);
}

/*************************************************************************

header
   print a page header

*************************************************************************/

void header( filename, pagenum )
char *filename;
int pagenum;
{
   char datestr[11], timestr[11];

   if (hdrflag || !do_print) {
      return;     /* skip if flag set */
      }
   timestamp(timestr);
   datestamp(datestr);
   fprintf(lp,"\033&d0D");
   if(book)
      fprintf(lp, "File: %-40s%s   %s  --  Page: %03d \n\n",
        filename,datestr,timestr,pagenum);
    else
      fprintf(lp, "File: %-40s%s   %s  --  Page: %03d       \n\n",
          filename,datestr,timestr,pagenum);
   fprintf(lp,"\033&d@");
}

void timestamp( timestr )
char   *timestr;
{
   struct tm *tod;
   time_t tt;
   tt = time(NULL);
   tod = localtime(&tt);
   sprintf(timestr,"%02d:%02d", tod->tm_hour, tod->tm_min);
   return;
}

void datestamp( datestr )
char  *datestr;
{
   struct tm *tod;
   time_t tt;
   tt = time(NULL);
   tod = localtime(&tt);
   sprintf(datestr,"%02d/%02d/%02d", tod->tm_mon + 1, tod->tm_mday,
   tod->tm_year + 1900);
   return;
}

/*************************************************************************

dovert()
   draw a vertical line down the center of the physical page

*************************************************************************/

void dovert()
{
   int line = 1;

   if(psprint)
      fprintf(lp,"\033&a0r0c131m129L\r|");
   else
      fprintf(lp,"\033&a0r0c90m88L\r|");

   while(line++ < MAXVERT) fprintf(lp,"\n|");
}


/*************************************************************************

usage
   print help and exit

*************************************************************************/

void usage()
{
   fprintf(stderr, "\nUsage: lj2 [flags] file1 [file2 ...]\n\n");
   fprintf(stderr, "\t-e  select Helv proportional 6 PT font\n");
   fprintf(stderr, "\t-h  suppress page headers\n");
   fprintf(stderr, "\t-tx set tab stop value (where x is 1 to 8)\n");
   fprintf(stderr, "\t-p  do not leave blank half pages between files\n");
   fprintf(stderr, "\t-s  shade every other line\n");
   fprintf(stderr, "\t-n  print line numbers\n");
   fprintf(stderr, "\t-b  indent for punching\n");
   fprintf(stderr, "\t-d  for duplex printing\n");
   fprintf(stderr, "\t-lxx set lines/page to x(where x is 1 to 66)\n");
   fprintf(stderr, "\t-i  ignore formfeed characters\n");
   exit(1);
}


void shade()
{
   int i;

   fprintf (lp, "\033&f0S\033*p0x0Y\033*c3300a32b10G");
   for (i=1; i <= 35; i++)
      fprintf (lp, "\033*c2P\033&a+2R");
   fprintf (lp, "\033&f1S");
}
