/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       viptext.c
**     SYSTEM   NAME:       VIP
**     ORIGINAL AUTHOR(S):  Alfred Kayser
**     VERSION  NUMBER:     1.00
**     CREATION DATE:       1992/5/29
**        
** DESCRIPTION: Static Text Module.
**              
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision$
** WORKFILE:    $Workfile$
** LOGINFO:     $Log$
*************************************************************************/

#include "vipinc.h"

#define TABMARGIN 3
#define TL_UPDATE 0x4000
#define SV_UPDATE 0x2000
#define SH_UPDATE 0x1000


PRIVAT LONG VipTextHandler(VIPINFO *wip, USHORT msg, MPARAM mp1, MPARAM mp2);
PRIVAT void VipTextUpdate(VIPINFO *wip, HPS hps, BOOLEAN all);
PRIVAT void VipListUpdate(VIPINFO *wip, HPS hps, BOOLEAN all);
PRIVAT LONG VipListHandler(VIPINFO *wip, USHORT msg, MPARAM mp1, MPARAM mp2);
PRIVAT VOID DrawText(VIPINFO *wip, HPS hps, RECTL *prct, TXTLINE *txt, SHORT fh);
PRIVAT BOOLEAN TextAdjust(VIPINFO *wip, SWP *pswp);
PRIVAT VOID TextVScroll(VIPINFO *wip, VOID *arg, int current);
PRIVAT VOID TextHScroll(VIPINFO *wip, VOID *arg, int current);
PRIVAT VOID ListSelect(VIPINFO *wip, HPS hps, int line);


IMPORT BOOLEAN vipPrinting;

/**************************************************************
** NAME:        VipOpenList                               [API]
** SYNOPSIS:    VIPINFO *VipOpenList(VIPINFO *parent,
**                  int x, int y, int w, int h)
** DESCRIPTION: Open a list window. The text in the
**              cannot be editted, but only be changed via
 **              the VipSetText functions.
** RETURNS:     VIP pointer or NULL.
**************************************************************/
VIPINFO *
VipOpenList(VIPINFO *parent, int x, int y, int w, int h)
{
    VIPINFO *wip;

    if (!(wip = VipOpenText(parent, x,y,w,h)))
        return NULL;

    /* Change Update function to a list draw function... */
    wip->handler=VipListHandler;
    wip->update=VipListUpdate;
    return wip;
}


/**************************************************************
** NAME:        VipOpenText                               [API]
** SYNOPSIS:    VIPINFO *VipOpenText(VIPINFO *parent,
**                  int x, int y, int w, int h)
** DESCRIPTION: Open a static text window. The text in the
**              cannot be editted, but only be changed via
**              the VipSetText functions.
** RETURNS:     VIP pointer or NULL.
**************************************************************/
VIPINFO *
VipOpenText(VIPINFO *parent, int x, int y, int w, int h)
{
    VIPINFO *wip;
	struct _textdata *data;
 
    if (!(data = VipMalloc(sizeof(struct _textdata))))
        return NULL;
    if (!(wip = VipOpenSimple(parent, x,y,w,h)))
    {
        VipFree(data);
        return NULL;
    }
    memset(data,0,sizeof(struct _textdata));
    data->lastActive=-1;

    wip->textdata=data;
    wip->type=T_TEXT;
    wip->border=1;
    wip->btype=VIP_BOX;
    wip->active=-1;
    BACKGROUND(wip)=CLR_WHITE;

    /* Change Update function to a text draw function... */
    wip->update=VipTextUpdate;
    wip->handler=VipTextHandler;
    wip->adjust=TextAdjust;
    return wip;
}


/**************************************************************
** NAME:        VipClearText                              [API]
** SYNOPSIS:    void VipClearText(VIPINFO *wip,
** DESCRIPTION: Clears all the text of <wip>.
** RETURNS:     void
**************************************************************/
EXPORT void
VipClearText(VIPINFO *wip)
{
    int i;
    TYPETEST(wip,T_TEXT,return);
    if (TEXTDATA(wip,text))
    {
        for (i=0;i<TEXTDATA(wip,lines);i++)
            DnpapFree(TEXTLINE(wip,i).text);
        DnpapFree(TEXTDATA(wip,text));
    }
    TEXTDATA(wip,lines)=0;
    TEXTDATA(wip,tsize)=0;
    if (TEXTDATA(wip, vertS))
    {
        SCROLLDATA(TEXTDATA(wip, vertS), size) = 1;
        SCROLLDATA(TEXTDATA(wip, vertS), max) = 1;
        SCROLLDATA(TEXTDATA(wip, vertS), range) = 1;
        SCROLLDATA(TEXTDATA(wip, vertS), current) = 0;
        TEXTDATA(wip, flags)|=SV_UPDATE;
    }
}


/**************************************************************
** NAME:        VipSetTextLine                            [API]
** SYNOPSIS:    BOOLEAN VipSetTextLine(VIPINFO *wip,
**                  const char *text, int line, int flags)
** DESCRIPTION: Sets the text of line <line>.
**              <flags> can be VIP_LEFT, VIP_RIGHT,
**              VIP_CENTER, or VIP_WORDWRAP.
**              Only VIP_LEFT and VIP_WORDWRAP can be
**              combined. In effect this the same as
**              only VIP_WORDWRAP.
** RETURNS:     FALSE, when failed
**              TRUE, when okay
**************************************************************/
BOOLEAN
VipSetTextLine(VIPINFO *wip, const char *text, int line, int flags)
{
    TXTLINE *new;

    TYPETEST(wip,T_TEXT,return FALSE);
    if (TEXTDATA(wip,tsize)<=line)
    {
        int i, len=TEXTDATA(wip,tsize);
        if (!len)
        {
            len=line+1;
            if (len<16) len=16;
            new=DnpapMalloc(len*sizeof(TXTLINE));
        }
        else
        {
            while (len<=line) len*=2;
            new=DnpapRealloc(TEXTDATA(wip, text), len*sizeof(TXTLINE));
        }
        if (!new)
        {
            DnpapMessage(DMC_ERROR, VIPERR_MEMORY,
                "Failed to allocate memory for line %d for text window",
                    line);
            return FALSE; /* Failed to alloc, or resize textdata */
        }

        for (i=TEXTDATA(wip,lines);i<len;i++)
        {
            new[i].text=NULL;
            new[i].flags=0;
            new[i].fcol=FOREGROUND(wip);
            new[i].bcol=BACKGROUND(wip);
        }
        TEXTDATA(wip,tsize)=len;
        TEXTDATA(wip,text)=new;
    }
    if (TEXTDATA(wip,lines)<=line)
    {
        TEXTDATA(wip,lines)=line+1;
        if (TEXTDATA(wip, vertS))
        {
            if (wip->cy)
            {
                int visLines = (int)(wip->cy/VipQueryFontHeight(wip->font));
                if (visLines>line)
                    visLines=line+1;
                SCROLLDATA(TEXTDATA(wip, vertS), size) = visLines;
            }
            SCROLLDATA(TEXTDATA(wip, vertS), max) = line+1;
            SCROLLDATA(TEXTDATA(wip, vertS), range) = line+1;
            TEXTDATA(wip, flags)|=SV_UPDATE;
        }
    }
	VipString(&(TEXTLINE(wip,line).text),text);
	TEXTLINE(wip,line).flags=flags|TL_UPDATE;
    return TRUE;
}


/**************************************************************
** NAME:        VipScrollText                             [API]
** SYNOPSIS:    void VipScrollText(VIPINFO *wip, int dir)
** DESCRIPTION: Scrolls the text up or down.
**              When <dir> is below zero, text scrolls down.
**              When <dir> is above zero, text scrolls up.
** RETURNS:     void
**************************************************************/
void
VipScrollText(VIPINFO *wip, int dir)
{
    int lines, act;
    RECTL rect;

    if (dir==0) return;
    TYPETEST(wip,T_TEXT,return);
    lines=TEXTDATA(wip,lines);
    if (!lines) return;
    act=TEXTDATA(wip,offset)+dir;
    if (act>=lines) act=lines-1;
    if (act<0) act=0;
    TEXTDATA(wip, offset) = act;
    if (wip->cy)
    {
        rect.xLeft   = wip->border;
        rect.yBottom = wip->border;
        rect.xRight  = wip->cx - wip->border;
        rect.yTop    = wip->cy - wip->border;
        WinScrollWindow(wip->win,0,dir*VipQueryFontHeight(wip->font),
            &rect, &rect, NULL, NULL, SW_INVALIDATERGN);
    }
    if (TEXTDATA(wip, vertS))
    {
        SCROLLDATA(TEXTDATA(wip, vertS), current) = act;
        TEXTDATA(wip, flags)|=SV_UPDATE;
    }
}


/**************************************************************
** NAME:        VipSetTextScrollbars                      [API]
** SYNOPSIS:    void VipSetTextScrollbars(VIPINFO *wip,
**                  VIPINFO *vertScroll, *horzScroll);
** DESCRIPTION: Installs scrollbars for a text window.
** RETURNS:     void
**************************************************************/
void
VipSetTextScrollbars(VIPINFO *wip, VIPINFO *vertScroll, VIPINFO *horzScroll)
{
    TYPETEST(wip,T_TEXT,return);

    TEXTDATA(wip,vertS) = vertScroll;
    TEXTDATA(wip,horzS) = horzScroll;
    if (vertScroll)
    {
        VipSetScrollCallBack(vertScroll, TextVScroll, wip);
        SCROLLDATA(TEXTDATA(wip, vertS), size) = 1;
        SCROLLDATA(TEXTDATA(wip, vertS), max) = 1;
        SCROLLDATA(TEXTDATA(wip, vertS), range) = 1;
        SCROLLDATA(TEXTDATA(wip, vertS), current) = 0;
        TEXTDATA(wip, flags)|=SV_UPDATE;
    }
    if (horzScroll)
    {
        VipSetScrollCallBack(horzScroll, TextHScroll, wip);
        SCROLLDATA(TEXTDATA(wip, horzS), size) = 1;
        SCROLLDATA(TEXTDATA(wip, horzS), max) = 1;
        SCROLLDATA(TEXTDATA(wip, horzS), range) = 1;
        SCROLLDATA(TEXTDATA(wip, horzS), current) = 0;
        TEXTDATA(wip, flags)|=SH_UPDATE;
    }
}


/**************************************************************
** NAME:        VipSetTextTabs                            [API]
** SYNOPSIS:    void VipSetTextTabs(VIPINFO *wip,
**                  int tabCount, int *tabPos, int flags);
** DESCRIPTION: Installs tabstops.
**              <tabCount> is the number of tabstops set.
**              <tabPos> is points to an array of <tabCount>
**              values in the range 0 to 1000.
**              <flags> can have the value 0 or VIP_TABLINES.
**              Each tab.position can be ored with VIP_TCENTER,
**              VIP_TRIGHT or VIP_TLEFT. VIP_TLEFT is default.
**              These flags cause the corresponding column
**              to be resp. centered, right aligned or left
**              aligned. The format flag of VipSetTextLine
**              determines the format of the first column.
** EXAMPLE:     int tabs={250|VIP_TLEFT,
**                        500|VIP_TCENTER,
**                        750|VIP_TRIGHT};
**              VipSetTextTabs(textwindow, 3, tabs);
**              VipSetTextLine(textwindow, "a\tb\tc\td", 0, VIP_LEFT);
**              This will create four colums of equal width.
**              Each being a quarter of the window width.
**              The first column contains a 'a' and is left
**              aligned. The second contains a left aligned b.
**              The third contains a centered c, and the last
**              has a right aligned d.
** RETURNS:     void
**************************************************************/
void
VipSetTextTabs(VIPINFO *wip, int tabCount, int *tabPos, int flags)
{
    int i, last=0;
    
    TYPETEST(wip,T_TEXT,return);

    TEXTDATA(wip, tabpos) = VipMalloc(tabCount*sizeof(SHORT));
    if (!TEXTDATA(wip, tabpos)) return;

    for (i=0;i<tabCount;i++)
    {
        if (tabPos[i]<0 || tabPos[i]>1000)                                               
            DnpapMessage(DMC_ERROR, VIPERR_TABS,
                "Tab %d has illegal value: %d\n"
                "The tab position must be in the range 0-1000!",
                    i, tabPos[i]);
        if (tabPos[i]<=last)                                               
            DnpapMessage(DMC_ERROR, VIPERR_TABS,
                "Tab %d has illegal value: %d\n"
                "Each tab position must be bigger than the previous one!",
                    i, tabPos[i]);
        TEXTDATA(wip, tabpos)[i] = last = tabPos[i];
    }
    TEXTDATA(wip, tabs) = tabCount;
    TEXTDATA(wip, flags) = flags;
}


/**************************************************************
** NAME:        VipSetListCallBack                        [API]
** SYNOPSIS:    void VipSetListCallBack(VIPINFO *wip,
**                  VIP_CALLBACK mb, VOID *ptr);
** DESCRIPTION: Installs a listselect callback function.
**              The syntax of the VIP_CALLBACK is:
**                  void VIP_CALLBACK(VIPINFO *wip,
**                            VOID *pointer, int listIndex);
**              This function is called whenever a user has
**              selected a list entry.
** RETURNS:     void
**************************************************************/
void
VipSetListCallBack(VIPINFO *wip, VIP_CALLBACK mb, VOID *ptr)
{
    TYPETEST(wip,T_TEXT,return);
    wip->callback = mb;
    wip->pointer = ptr;
}


PRIVAT LONG
VipListHandler(VIPINFO *wip, USHORT msg, MPARAM mp1, MPARAM mp2)
{
    static BOOLEAN mmouse=FALSE;
    int curr;
    LONG x,y,fh;
    BOOLEAN moved;

	switch(msg)
	{
    case WM_BUTTON1DOWN:
        WinSetFocus(HWND_DESKTOP, wip->win); 
        WinSetCapture(HWND_DESKTOP, wip->win);
        mmouse=TRUE;
        /*FALLTHROUGH*/
    case WM_MOUSEMOVE:
        if (mmouse)
        {
            x = SHORT1FROMMP(mp1);
            y = SHORT2FROMMP(mp1);

            fh = VipQueryFontHeight(wip->font);
            moved=FALSE;

            y = wip->cy - y;
            if (y>=wip->cy)
            {
                moved=TRUE;
                while (y>=wip->cy)
                {
                    /* Move list lines one entry up */
                    curr=TEXTDATA(wip,offset)+1;
                    if (curr==TEXTDATA(wip,lines)) break;
                    TEXTDATA(wip,offset)=curr;
                    if (TEXTDATA(wip, vertS))
                        SCROLLDATA(TEXTDATA(wip, vertS),current)=curr;
                    y-=fh;
                }
            }
            if (y<0)
            {
                moved=TRUE;
                while (y<0)
                {
                    /* Move list lines one entry up */
                    curr=TEXTDATA(wip,offset);
                    if (curr==0) break;
                    curr--;
                    TEXTDATA(wip,offset)=curr;
                    if (TEXTDATA(wip, vertS))
                        SCROLLDATA(TEXTDATA(wip, vertS),current)=curr;
                    y+=fh;
                }
            }

            /* Just to be sure */
            if (y<0) y=0;
            if (y>wip->cy) y=wip->cy;

            curr = (int)(y/fh) + TEXTDATA(wip, offset);
            if (curr>=TEXTDATA(wip,lines))
                curr=TEXTDATA(wip,lines)-1;
            if (curr!=wip->active || moved)
            {
                wip->active=curr;
                VipUpdate(wip, 0);
                if (moved && TEXTDATA(wip, vertS))
                    VipUpdate(TEXTDATA(wip, vertS), 0);
            }                    
            return TRUE;
        }
        break;

    case WM_BUTTON1UP:
        mmouse=FALSE;
        WinSetCapture(HWND_DESKTOP, NULL);

        /* When there is no callback and no submenu, ring a bell! */
        if (!wip->callback)
            VipBell();
        else
            wip->callback(wip, wip->pointer, wip->active);
        return TRUE;
	}
	return VipTextHandler(wip, msg, mp1, mp2);
}


PRIVAT VOID
ListSelect(VIPINFO *wip, HPS hps, int line)
{
    POINTL pp;
    LONG fh;
    int n;

    if (line<0) return;

    n = line - TEXTDATA(wip, offset);
    if (n<0) return;

    fh = VipQueryFontHeight(wip->font);

    GpiSetMix(hps, FM_INVERT); 
    pp.x=wip->border;
    pp.y=wip->cy - n*fh - wip->border - 1;
    GpiMove(hps, &pp);
    pp.x=wip->cx - wip->border - 1;
    pp.y-=fh - 2;
    if (pp.y<wip->border) pp.y=wip->border;
    GpiBox(hps,DRO_FILL,&pp,0,0);
    GpiSetMix(hps, FM_OVERPAINT);
}


PRIVAT LONG
VipTextHandler(VIPINFO *wip, USHORT msg, MPARAM mp1, MPARAM mp2)
{
	switch(msg)
	{
    case UM_SCALE:
    case WM_SHOW:
        VipSimpleHandler(wip, msg, mp1, mp2);
        if (SHORT1FROMMP(mp1))  /* True? */
        {
            if (wip->cy && TEXTDATA(wip, vertS))
            {
                int visLines = (int)(wip->cy/VipQueryFontHeight(wip->font));
                if (visLines>TEXTDATA(wip,lines))
                    visLines=TEXTDATA(wip,lines)+1;
                SCROLLDATA(TEXTDATA(wip, vertS), size) = visLines;
                TEXTDATA(wip, flags) |= SV_UPDATE;
            }
        }
        return FALSE;
	}
	return VipSimpleHandler(wip, msg, mp1, mp2);
}


PRIVAT void
VipListUpdate(VIPINFO *wip, HPS hps, BOOLEAN all)
{
    if (!vipPrinting && TEXTDATA(wip,lastActive)>=0)
    {
        int new=TEXTDATA(wip, offset);
        TEXTDATA(wip,offset)=TEXTDATA(wip,lastOffset);
        ListSelect(wip,hps,TEXTDATA(wip,lastActive));    /* Hide selection box */
        TEXTDATA(wip,offset)=TEXTDATA(wip,lastOffset)=new;
    }
    VipTextUpdate(wip, hps, all);
    TEXTDATA(wip,lastActive)=wip->active;
    if (!vipPrinting && TEXTDATA(wip,lastActive)>=0)
       ListSelect(wip,hps,wip->active);    /* Show selection box */
}


PRIVAT void
VipTextUpdate(VIPINFO *wip, HPS hps, BOOLEAN all)
{
    TXTLINE *text;
    SHORT fh;
    RECTL rect;
    int t, i;

    if (QREDRAW(wip))
    {
        RREDRAW(wip);
        all=TRUE;
    }

    /* The following are coordinaties rel. to this window */
    rect.xLeft   = wip->border;
    rect.yBottom = wip->border;
    rect.xRight  = wip->cx - wip->border;
    rect.yTop    = wip->cy - wip->border;
         
    fh=VipQueryFontHeight(wip->font);
    if (all) VipBorder(wip, hps);

    VipUseFont(hps, wip->font);
    text=TEXTDATA(wip,text)+TEXTDATA(wip,offset);
    for (i=TEXTDATA(wip,offset); i<TEXTDATA(wip,lines); i++)
    {
        if (all || (text->flags&TL_UPDATE))
        {
            text->flags&=~TL_UPDATE;
            DrawText(wip, hps, &rect, text, fh);
        }
        else
            rect.yTop -= fh;
        if (rect.yTop <= rect.yBottom) break;
        text++;
    }
    VipReleaseFont(hps, wip->font);

    if (!all && (rect.yTop>rect.yBottom))
        WinFillRect(hps, &rect, BACKGROUND(wip));

    if (TEXTDATA(wip, flags)&VIP_TABLINES)
    {
        LONG width, pos;
        POINTL pp;

        width=rect.xRight - rect.xLeft - TEXTDATA(wip,tabs) * TABMARGIN;
        rect.yBottom = wip->border;
        rect.yTop    = wip->cy - wip->border;
        GpiSetColor(hps, FOREGROUND(wip));
        for (t=0; t<TEXTDATA(wip,tabs); t++)
        {
            pos = rect.xLeft + ((width*TEXTDATA(wip,tabpos)[t])+500L)/1000L
                             + t*TABMARGIN - TABMARGIN;
            MOVE(pos, rect.yBottom);
            LINE(pos, rect.yTop-1);
        }
    }

    if (TEXTDATA(wip, flags) & SV_UPDATE)
    {
        if (TEXTDATA(wip, vertS))
            VipUpdate(TEXTDATA(wip, vertS),FALSE);
        TEXTDATA(wip, flags) &= ~SV_UPDATE;
    }
    if (TEXTDATA(wip, flags) & SH_UPDATE)
    {
        if (TEXTDATA(wip, horzS))
            VipUpdate(TEXTDATA(wip, horzS),FALSE);
        TEXTDATA(wip, flags) &= ~SH_UPDATE;
    }
}


PRIVAT VOID
DrawText(VIPINFO *wip, HPS hps, RECTL *prct, TXTLINE *txt, SHORT fh)
{
    USHORT f=VIP_LEFT;

    f=DT_TOP|DT_ERASERECT|DT_LEFT;
    switch(txt->flags)
    {
    case VIP_RIGHT:    f=DT_TOP|DT_ERASERECT|DT_RIGHT;break;
    case VIP_CENTER:   f=DT_TOP|DT_ERASERECT|DT_CENTER;break;
    case VIP_WORDWRAP: f=DT_TOP|DT_ERASERECT|DT_LEFT|DT_WORDBREAK;break;
    }

    if (TEXTDATA(wip,tabs))
    {
        int k, l, t, n;
        char *str;
        LONG rightM=prct->xRight;
        LONG topM=prct->yTop;
        LONG lowM=prct->yTop;
        LONG botM=prct->yBottom;
        LONG leftM=prct->xLeft;
        LONG width=rightM-leftM - TEXTDATA(wip,tabs) * TABMARGIN;

        str=txt->text;
        if (!str) str="";
        for (t=0; t<=TEXTDATA(wip,tabs); t++)
        {
            if (t)
            {
                switch(TEXTDATA(wip,tabpos)[t-1]&VIP_TABFORMAT)
                {
                case VIP_TRIGHT:  f=DT_TOP|DT_ERASERECT|DT_RIGHT;break;
                case VIP_TCENTER: f=DT_TOP|DT_ERASERECT|DT_CENTER;break;
                default:          f=DT_TOP|DT_ERASERECT|DT_LEFT;break;
                }
            }
            l = 0;
            if (str[0])
            {
                char *p=strchr(str, '\t');
                l = p ? (p-str) : strlen(str);
            }
            if (t<TEXTDATA(wip, tabs))
                prct->xRight = leftM
                             + ( ( ( width
                                   * (TEXTDATA(wip,tabpos)[t]&~VIP_TABFORMAT))
                                 + 500L)
                               / 1000L)
                             + t*TABMARGIN - TABMARGIN - 2;
            else
                prct->xRight=rightM;
            if (l==0)
            {
                prct->yBottom = prct->yTop - fh;
                if (prct->yBottom<botM) prct->yBottom=botM;
                WinFillRect(hps, prct, txt->bcol);
                prct->yTop = prct->yBottom;
            }
            else for (k=0;k<l;k+=n)
            {
                prct->yBottom = prct->yTop - fh;
                if (prct->yBottom<botM) prct->yBottom=botM;
                n=WinDrawText(hps, l-k, str+k, prct, txt->fcol, txt->bcol, f);
                prct->yTop = prct->yBottom;
                if (n==0) break;
            }
            if (prct->yTop<lowM) lowM=prct->yTop;
            prct->yTop=topM;
            prct->xLeft=prct->xRight + TABMARGIN + 1;
            str+=l;
            if (str[0]=='\t') str++;
        }
        prct->xLeft=leftM;
        prct->xRight=rightM;
        prct->yTop=lowM;
        prct->yBottom=botM;
    }
    else
    {
        LONG botM=prct->yBottom;
        int k, n, l=0;

        if (txt->text) l=strlen(txt->text);
        if (l==0)
        {
            prct->yBottom = prct->yTop - fh;
            if (prct->yBottom<botM) prct->yBottom=botM;
            WinFillRect(hps, prct, txt->bcol);
            prct->yTop = prct->yBottom;
        }
        else for (k=0;k<l;k+=n)
        {
            prct->yBottom=prct->yTop - fh;
            if (prct->yBottom<botM) prct->yBottom=botM;
            n=WinDrawText(hps, l-k, txt->text+k, prct, txt->fcol, txt->bcol, f);
            prct->yTop = prct->yBottom;
            if (n==0) break;
        }
        prct->yBottom=botM;
    }
}


PRIVAT BOOLEAN
TextAdjust(VIPINFO *wip, SWP *pswp)
{
    BOOLEAN changed;
    TXTLINE *text;
    RECTL rect;
    LONG maxX, maxY;
    HPS hps;
    int i, lines, h;
    
    if (wip->w>0 && wip->h>0) return FALSE;

    lines=TEXTDATA(wip,lines);

    changed=FALSE;

    hps=WinGetPS(wip->win);
    VipUseFont(hps, wip->font);

    if (wip->h==0 || TEXTDATA(wip, vertS))
    {
        h = VipQueryFontHeight(wip->font);
        if (wip->h!=0)
            lines = (pswp->cy - 2*wip->border)/h;
        maxY = lines * h + 2*wip->border;
        if ((SHORT)maxY!=pswp->cy) { pswp->cy=(SHORT)maxY; changed=TRUE; }
    }

    if (wip->w==0)
    {
        rect.xLeft=rect.yBottom=0;
        text=TEXTDATA(wip,text);
        for (maxX=i=0;i<lines;i++)
           if (text[i].text)
            {
                rect.xRight=rect.yTop=1000;
                WinDrawText(hps, -1, text[i].text, &rect, 0L,0L,
                    DT_LEFT|DT_BOTTOM|DT_QUERYEXTENT|DT_TEXTATTRS);
                if (rect.xRight>maxX) maxX=rect.xRight;
            }
        maxX = maxX + 2*wip->border;
        if ((SHORT)maxX!=pswp->cx) { pswp->cx=(SHORT)maxX; changed=TRUE; }
    }
    VipReleaseFont(hps, wip->font);
    WinReleasePS(hps);
    return changed;
}


PRIVAT VOID
TextVScroll(VIPINFO *scrol, VOID *arg, int current)
{
    VIPINFO *wip = (VIPINFO *)arg;
    VipScrollText(wip, current - TEXTDATA(wip, offset));
}


PRIVAT VOID
TextHScroll(VIPINFO *scrol, VOID *arg, int current)
{
    VIPINFO *wip = (VIPINFO *)arg;
}
