/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       vipmenu.c
**     SYSTEM   NAME:       VIP
**     ORIGINAL AUTHOR(S):  Alfred Kayser
**     VERSION  NUMBER:     1.00
**     CREATION DATE:       1992/5/29
**
** DESCRIPTION: Dynamic Menu Module.
**              
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision$
** WORKFILE:    $Workfile$
** LOGINFO:     $Log$
*************************************************************************/
#define LIBRARY
#include "vipinc.h"

IMPORT HAB hab;
IMPORT LONG VipLabelHandler(VIPINFO *wip, USHORT msg, MPARAM mp1, MPARAM mp2);

PRIVAT LONG    MenuHandler(VIPINFO *wip, USHORT msg, MPARAM mp1, MPARAM mp2);
PRIVAT void    MenuUpdate(VIPINFO *wip, HPS hps, BOOLEAN all);
PRIVAT BOOLEAN MenuAdjust(VIPINFO *wip, SWP *pswp);

PRIVAT void DrawSelect(VIPINFO *wip, HPS hps, BOOLEAN on);
PRIVAT void ShowSubMenu(VIPINFO *wip, LONG x, LONG y);


#ifdef TEST2
/**************************************************************
** NAME:        VipOpenMenu                               [API]
** SYNOPSIS:    VIPINFO *VipOpenMenu(VIPINFO *parent,
**                  int x, int y, int w, int h)
** DESCRIPTION: Opens a menu window. When <w> or <h> the width
**              resp. height are resized to fit the title
**              exactly.
** RETURNS:     void
**************************************************************/
#endif
VIPINFO *
VipOpenMenu(VIPINFO *parent, int x, int y, int w, int h)
{
    VIPINFO *wip;

    if (!(wip = VipOpenText(parent, x,y,w,h)))
        return NULL;
    
    TEXTDATA(wip, parmenu) = NULL;
    TEXTDATA(wip, submenu) = NULL;
    TEXTDATA(wip, flags) = 0;

    wip->active = -1;
    wip->callback = NULL;
    wip->pointer = NULL;
    wip->border = 1;
    wip->btype = VIP_RISE;
    BACKGROUND(wip)=CLR_PALEGRAY;

    wip->adjust  = MenuAdjust;
    wip->handler = MenuHandler;
    wip->update  = MenuUpdate;
    return wip;
}



/**************************************************************
** NAME:        VipSetMenuEntry                           [API]
** SYNOPSIS:    void VipSetMenuEntry(VIPINFO *wip,
**                CONST char *text, int num, int type, VOID *p)
** DESCRIPTION: Inserts menu entries in a menu. <num> must be
**              be positive, <type> can be VIP_MENUITEM, 
**              VIP_MENUTITLE, VIP_MENUBAR, or VIP_MENUSUB and
**              can be ored with VIP_LEFT, VIP_CENTER or
**              VIP_RIGHT.
**              <p> is used as a pointer to data, which is
**              passed to the callback function. <p> must 
**              point to a menu window when <type> is
**              VIP_MENUSUB. <text> can only be NULL when
**              <type> is VIP_MENUBAR.
** RETURNS:     void
**************************************************************/
void
VipSetMenuEntry(VIPINFO *wip, CONST char *text, int num, int type, VOID *p)
{
    TYPETEST(wip,T_TEXT,return);
    VipSetTextLine(wip, text, num, type);
    if ((type&VIP_MENUF)==VIP_MENUSUB)
    {
        if ((VIPINFO*)p==wip)
        {
            VipError(VIPERR_MENU,"Parameter <pointer> of VipSetMenuEntry can not be"
                "a pointer to the menu self!");
            return;
        }
        if (((VIPINFO*)p)->magic!=VipCookie)
        {
            VipError(VIPERR_MENU,"Parameter <pointer> of VipSetMenuEntry must be"
                "a pointer to a menu window, and not pointing to itself!");
            return;
        }
    }
    TEXTLINE(wip, num).pointer=p;
    SREDRAW(wip);
}


/**************************************************************
** NAME:        VipSetMenuCallBack                        [API]
** SYNOPSIS:    void VipSetMenuCallBack(VIPINFO *wip,
**                  VIP_MENUCALL mb);
** DESCRIPTION: Installs a menu callback function.
**              The syntax of the VIP_MENUCALL is:
**                  void VIP_MENUCALL(VIPINFO *wip,
**                            VOID *pointer, int menuIndex);
**              This function is called whenever a user has
**              selected a menu entry. It is also called when
**              the user selected the menu, but didn't select
**              a valid menu entry.
**              <menuIndex> will be -1 in that case.
** RETURNS:     void
**************************************************************/
void
VipSetMenuCallBack(VIPINFO *wip, VIP_MENUCALL mb)
{
    TYPETEST(wip,T_TEXT,return);
    wip->callback = mb;
}

/**************************************************************
** NAME:        VipSetMenuType                            [API]
** SYNOPSIS:    void VipSetMenuType(VIPINFO *wip, int flags);
** DESCRIPTION: Sets type of menu.
**              Flags can be VIP_HMENU,VIP_FIXMENU or
**              VIP_POPMENU. These can be ored to make
**              combinations.
**              VIP_HMENU makes menu a horizontal menu.
**              VIP_FIXMENU makes the menu unmoveable.
**              VIP_POPMENU makes the menu a popup menu.
** RETURNS:     void
**************************************************************/
void
VipSetMenuType(VIPINFO *wip, int flags)
{
    TYPETEST(wip,T_TEXT,return);
    TEXTDATA(wip, flags) = flags;
    SREDRAW(wip);
}


PRIVAT LONG
MenuHandler(VIPINFO *wip, USHORT msg, MPARAM mp1, MPARAM mp2)
{
    static BOOLEAN mmouse=FALSE;
    HPS hps;
    LONG x,y;
    int curr;
    VIPINFO *tmp;

    switch(msg)
    {
    case WM_BUTTON1DOWN:
        WinSetFocus(HWND_DESKTOP, wip->win); 
        WinSetCapture(HWND_DESKTOP, wip->win);
        wip->active=-2;    /* Force redraw of selection box */
        mmouse=TRUE;
        /*FALLTHROUGH*/
    case WM_MOUSEMOVE:
        if (mmouse)
        {
            x = SHORT1FROMMP(mp1);
            y = SHORT2FROMMP(mp1);
            if ((x<0) || (x>wip->cx)
             || (y<0) || (y>wip->cy) )
                curr=-2;
            else if (TEXTDATA(wip, flags)&VIP_HMENU)
                curr = (int) ((x*TEXTDATA(wip, lines))/wip->cx);
            else
                curr = TEXTDATA(wip,lines)
                        - (int) ((y*TEXTDATA(wip,lines))/wip->cy + 1);
            if (curr>=0)
                if ((TEXTLINE(wip, curr).flags&VIP_MENUBAR)==VIP_MENUBAR)
                    curr=-1;
            if (curr!=wip->active)
            {
                hps=WinGetPS(wip->win);
                DrawSelect(wip, hps, FALSE);
                wip->active = curr;
                DrawSelect(wip, hps, (int)TRUE);
                WinReleasePS(hps);
                if (curr==-2)   /* Outside our menuwindow? */
                {
                    wip->active = -1;
                    if (tmp=TEXTDATA(wip, parmenu))
                    {
                        POINTL pp;
                        pp.x = x;
                        pp.y = y;
                        WinMapWindowPoints(wip->win, tmp->win, &pp, 1);
                        if (pp.x>=0 && pp.y>=0 &&
                            pp.x<=tmp->cx && pp.y <=tmp->cy)
                        {
                            mmouse=FALSE;
                            /* Release mouse capture */
                            WinSetCapture(HWND_DESKTOP, NULL);
                            /* Signal parmenu a button press in its window */
                            WinPostMsg(tmp->win,
                                WM_BUTTON1DOWN, MPFROM2SHORT(x,y), NULL);
                            TEXTDATA(wip, parmenu)=NULL;
                        }
                    }
                    if (tmp=TEXTDATA(wip,submenu))
                    {
                        POINTL pp;
                        pp.x = x;
                        pp.y = y;
                        WinMapWindowPoints(wip->win, tmp->win, &pp, 1);
                        if (pp.x>=0 && pp.y>=0 &&
                            pp.x<=tmp->cx && pp.y <=tmp->cy)
                        {
                            mmouse=FALSE;

                            /* Make a link from submenu into this menu */
                            TEXTDATA(TEXTDATA(wip, submenu), parmenu) = wip;

                            /* Release mouse capture */
                            WinSetCapture(HWND_DESKTOP, NULL);

                            /* Signal submenu a button press in its window */
                            WinPostMsg(TEXTDATA(wip, submenu)->win,
                                WM_BUTTON1DOWN, MPFROM2SHORT(x,y), NULL);

                            /* End of my task */
                            return TRUE;
                        }
                    }
                }
                else if (curr>=0)
                {
                    /* Hide previous submenu */
                    if (TEXTDATA(wip, submenu))
                    {
                        TEXTDATA(TEXTDATA(wip, submenu), parmenu) = NULL;
                        VipHide(TEXTDATA(wip, submenu));
                        TEXTDATA(wip, submenu) = NULL;
                    }
                    /* Draw new submenu */
                    if (MENUFLAG(wip, curr, VIP_MENUSUB))
                        ShowSubMenu(wip, x, y);
                }
            }
            return TRUE;
        }
        break;

    case WM_BUTTON1UP:
        mmouse=FALSE;
        WinSetCapture(HWND_DESKTOP, NULL);
        hps=WinGetPS(wip->win);
        DrawSelect(wip, hps, FALSE);
        WinReleasePS(hps);

        if (TEXTDATA(wip, flags)&VIP_POPMENU)
            for (tmp=wip;
                 tmp && (TEXTDATA(tmp, flags)&VIP_POPMENU);
                 tmp=TEXTDATA(tmp, parmenu))
               VipHide(tmp);

        curr=wip->active;

        /* When there is no callback and no submenu, ring a bell! */
        if (!wip->callback && !TEXTDATA(wip, submenu))
            VipBell();

        /* We're done, so submenu can forget this menu */
        if (TEXTDATA(wip, submenu))
        {
            TEXTDATA(TEXTDATA(wip, submenu), parmenu) = NULL;
            TEXTDATA(wip, submenu)=NULL;
        }

        if (wip->callback)
        {
            if (curr==-1)
                wip->callback(wip, NULL, curr);
            else if (MENUFLAG(wip, curr, VIP_MENUITEM))
                wip->callback(wip, TEXTLINE(wip, curr).pointer, curr);
        }
        return TRUE;

    case WM_SETFOCUS:
	    if (HWNDFROMMP(mp1)==wip->win)
            if (!SHORT1FROMMP(mp2)) /* lost focus, so hide me */
            {
                for (tmp=wip;tmp;tmp=TEXTDATA(tmp, parmenu))
                    if (TEXTDATA(tmp, flags)&VIP_POPMENU)
                        VipHide(tmp);
                    else
                        break;
            }
    	break;
    }
    if (TEXTDATA(wip,flags)&VIP_FIXMENU)
        return VipSimpleHandler(wip, msg, mp1, mp2);
    return VipLabelHandler(wip, msg, mp1, mp2);
}

PRIVAT void
ShowSubMenu(VIPINFO *wip, LONG x, LONG y)
{
    VIPINFO *sub;
    LONG step;
    SWP swp;

    if (wip->active<0) return;
    sub = (VIPINFO*)(TEXTLINE(wip, wip->active).pointer);
    TEXTDATA(wip, submenu) = sub;
    if (TEXTDATA(wip,flags)&VIP_HMENU)
    {
        WinQueryWindowPos(wip->win, &swp);
        step = (wip->cx-2*wip->border)/TEXTDATA(wip, lines);
        x = wip->active*step;
        y = swp.y - sub->cy + wip->border;
        if (wip->parent)
            if (y<0)
                y = swp.y + wip->parent->cy + wip->border;
    }
    else
    {
        WinQueryWindowPos(wip->win, &swp);
        step = (wip->cy-2*wip->border)/TEXTDATA(wip, lines);
        x = swp.x + wip->cx - wip->border;
        y = swp.y + wip->cy - wip->active*step - sub->cy;
        if (wip->parent)
        {
            if (x + sub->cx >= wip->parent->cx)
                x = swp.x - sub->cx - wip->border;
            if (y<0)
                y += sub->cy;
        }
    }
    VipMoveWindow(sub, x, y);
    VipShow(sub);
}


PRIVAT void
DrawSelect(VIPINFO *wip, HPS hps, BOOLEAN on)
{
    RECTL rect;
    LONG step;
        
    if (wip->active<0)
        return;

    if (TEXTDATA(wip,flags)&VIP_HMENU)
    {
        step = (wip->cx-2*wip->border)/TEXTDATA(wip, lines);
        rect.xLeft   = wip->border + (wip->active)*step + 1;
        rect.xRight  = rect.xLeft + step - 1;
        rect.yBottom = wip->border + 1;
        rect.yTop    = wip->cy - rect.yBottom;
    }
    else
    {
        step = (wip->cy-2*wip->border)/TEXTDATA(wip, lines);
        rect.xLeft   = wip->border+1;
        rect.xRight  = wip->cx - rect.xLeft;
        rect.yTop    = wip->cy - (wip->active)*step - wip->border - 1;
        rect.yBottom = rect.yTop - step + 1;
    }
    VipDrawBorder(wip, &rect, hps, on?VIP_LOW:VIP_NONE, 1);
}

PRIVAT void
MenuUpdate(VIPINFO *wip, HPS hps, BOOLEAN all)
{
    USHORT f;
    LONG step;
    TXTLINE *text;
    int i, lines;
    RECTL rect;

    if (QREDRAW(wip) || all)
    {
        RREDRAW(wip);
        if (all) VipBorder(wip, hps);

        VipUseFont(hps, wip->font);
        text=TEXTDATA(wip,text);
        lines=TEXTDATA(wip,lines);

        if (TEXTDATA(wip, flags)&VIP_HMENU)
        {
            step=(wip->cx-2*wip->border)/lines;
            rect.xLeft   = wip->border + 2;
            rect.xRight  = rect.xLeft;
            rect.yBottom = wip->border + 2;
            rect.yTop    = wip->cy - rect.yBottom;

            for (i=0;i<TEXTDATA(wip,lines);i++)
            {
                rect.xRight += step;
                switch(text[i].flags&VIP_TEXTF)
                {
                case VIP_RIGHT:  f=DT_MNEMONIC|DT_VCENTER|DT_RIGHT;  break;
                case VIP_CENTER: f=DT_MNEMONIC|DT_VCENTER|DT_CENTER; break;
                default:         f=DT_MNEMONIC|DT_VCENTER|DT_LEFT;   break;
                }
                if (!all) f|=DT_ERASERECT;
                if (text[i].text)
                    WinDrawText(hps, -1, text[i].text,
                        &rect, FOREGROUND(wip), BACKGROUND(wip), f);
                rect.xLeft += step;
            }
        }
        else
        {
            step=(wip->cy-2*wip->border)/lines;
            rect.yBottom = wip->cy - wip->border;
            rect.yTop    = rect.yBottom;
            rect.xLeft   = wip->border + 2;
            rect.xRight  = wip->cx - rect.xLeft;

            for (i=0;i<TEXTDATA(wip,lines);i++)
            {
                rect.yBottom -= step;
                switch(text[i].flags&VIP_TEXTF)
                {
                case VIP_RIGHT:  f=DT_MNEMONIC|DT_VCENTER|DT_RIGHT;  break;
                case VIP_CENTER: f=DT_MNEMONIC|DT_VCENTER|DT_CENTER; break;
                default:         f=DT_MNEMONIC|DT_VCENTER|DT_LEFT;   break;
                }
                if (!all) f|=DT_ERASERECT;
                if ((text[i].flags&VIP_MENUF)==VIP_MENUBAR)
                {
                    RECTL rct2=rect;
                    rct2.xLeft+=step/4;
                    rct2.xRight-=step/4;
                    rct2.yTop-=step/4;
                    rct2.yBottom+=step/4;
                    VipDrawBorder(wip, &rct2, hps, VIP_RISE, 1);
                }
                else if (text[i].text)
                    WinDrawText(hps, -1, text[i].text,
                        &rect, FOREGROUND(wip), BACKGROUND(wip), f);
                if ((text[i].flags&VIP_MENUF)==VIP_MENUTITLE)
                {
                    RECTL rct2=rect;
                    rct2.xLeft-=3;
                    rct2.xRight+=3;
                    VipDrawBorder(wip, &rct2, hps, VIP_RISE, 1);
                }
                rect.yTop -= step;
            }
        }
        VipReleaseFont(hps, wip->font);
    }
}


PRIVAT BOOLEAN
MenuAdjust(VIPINFO *wip, SWP *pswp)
{
    BOOLEAN changed;
    TXTLINE *text;
    RECTL rect;
    LONG maxX, maxY;
    int i, lines;
    HPS hps;

    if (wip->w>0 && wip->h>0) return FALSE;

    lines=TEXTDATA(wip,lines);
    changed=FALSE;
    if (wip->h==0)
    {
        maxY = VipQueryFontHeight(wip->font);
        if (TEXTDATA(wip, flags)&VIP_HMENU)
            maxY = maxY + 4 + 2*wip->border;
        else
            maxY = lines*(maxY+4) + 2*wip->border;
        if (maxY!=pswp->cy) { pswp->cy=(SHORT)maxY; changed=TRUE; }
    }
            
    if (wip->w==0)
    {
        hps=WinGetPS(wip->win);
        VipUseFont(hps, wip->font);
        rect.xLeft=rect.yBottom=0;
        text=TEXTDATA(wip,text);
        for (maxX=i=0;i<lines;i++)
            if (text[i].text)
            {
                rect.xRight=rect.yTop=1000;
                WinDrawText(hps, -1, text[i].text, &rect, 0L,0L,
                    DT_LEFT|DT_BOTTOM|DT_QUERYEXTENT|DT_TEXTATTRS);
                if (rect.xRight>maxX) maxX=rect.xRight;
            }
        if (TEXTDATA(wip, flags)&VIP_HMENU)
            maxX = lines*(maxX+4) + 2*wip->border;
        else
            maxX = maxX + 6 + 2*wip->border;
        if (maxX!=pswp->cx) { pswp->cx=(SHORT)maxX; changed=TRUE; }
        VipReleaseFont(hps, wip->font);
        WinReleasePS(hps);
    }

    /* Only if window is auto scaled, move it into view */
    if (wip->parent && !(TEXTDATA(wip,flags)&VIP_FIXMENU))
    {
        SHORT s = (SHORT)wip->parent->cx;
        if (pswp->x>0 && pswp->x+pswp->cx>s)
        {
            pswp->x = s - pswp->cx;
            if (pswp->x<0) pswp->x = 0;
            changed=TRUE;
        }
        s = (SHORT)wip->parent->cy;
        if (pswp->y>0 && pswp->y + pswp->cy > s)
        {
            pswp->y = s - pswp->cy;
            if (pswp->y<0) pswp->y = 0;
            changed=TRUE;
        }
    }
    return changed;
}

