/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       api.c
**     SYSTEM   NAME:       mib
**     ORIGINAL AUTHOR(S):  Jan van Oorschot
**     VERSION  NUMBER:     1.00
**     CREATION DATE:       
**
** DESCRIPTION: This file contains the API calls supported my the
**              MIB library.
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision$
** WORKFILE:    $Workfile$
** LOGINFO:     $Log$
*************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dnpap.h>
#include <config.h>
#include "mibinc.h"

#define ERR_ERR DnpapMessage

#ifndef MIBHOME
#define	MIBHOME "/usr/local/lib/mibdir"
#endif

/**************************************************************
** NAME:        MibInit
** SYNOPSIS:    iRet = MibInit();
**              
** DESCRIPTION: Initialises the MIB system
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int MibInit(void)
{
    MibInitialise();
    return 0;
}
/**************************************************************
** NAME:        MibFullMib
** SYNOPSIS:    iRet = MibFullMib();
**              
** DESCRIPTION: Reads the file $MIBHOME/mibs and interprets each
**		line as the name of a MOSI mib file in $MIBHOME
**		Puts these
**		files in the tree.
**		MIBHOME should be something like 
**
**			/usr/local/lib/mibhome/
**
**      this functions will use the following logic to locate
**      $MIBHOME:
**
**          . search the .ini files
**          . search environment variable MIBHOME
**          . use value of macro $(MIBHOME)
**
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int MibFullMib(void)
{
    FILE *fp1,*fp2;
    char  *mibhome;
    char  fname[265],pname[256];
    int len,iret;

    /* search the ini files */
    if( !ConfigGetString("mibdf.mibhome",&mibhome))
    {
        mibhome=NULL;
    }

    /* use the environment variable */
    if (mibhome==NULL)
        mibhome =getenv("MIBHOME");
    
    /* use the macro */
    if (mibhome==NULL)
        mibhome=MIBHOME;
    
    strcpy(pname,mibhome);
    strcat(pname,"/mibs");

    fp1 = fopen(pname,"r");
    if(fp1 == NULL)
    {
	    DnpapMessage(
			DMC_WARNING,
			1,
		    "Could not find MIB catalog 'mibs',\n\
            define mibdf.mibhome in the configuration file,\n\
            or set environment variable MIBHOME\n"
		    );
        return 0;
    }
    while(!feof(fp1))
    {
	    strcpy(pname,mibhome);
	    fgets(fname,256,fp1);
	    len = strlen(fname);
	    if(len > 1)
	    {
	        if(fname[len-1] == '\n')
	        fname[len-1] = '\0';
	        strcat(pname,"/");
	        strcat(pname,fname);
	        fp2 = fopen(pname,"r");
	
	        if(fp2 != NULL)
	        {
		    iret = MibReadMib(fp2);
	        }
	        else
	        {
		    DnpapMessage(
			    DMC_WARNING,
			    1,
			    "could not read mib file <%s>\n",
			    pname
			    );
            exit(1);
	        }
	    }
    }
    return 0;
}

/**************************************************************
** NAME:        MibReadMib
** SYNOPSIS:    iRet = MibReadMib(fp);
**		FILE *fp;
**              
** DESCRIPTION: Reads a MIB definition from file-stream <fp>
**		up to the EOF.  The MIB read from this file
**		is stored in the global MIB tree.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int MibReadMib( FILE *fp)
{
    char line[256];
    MIBNODE *node;

    while( !feof(fp))
    {
	    fgets(line,256,fp);
	    node = MibScanLine(line);
    }
    return 0;
}
/**************************************************************
** NAME:	MibName2ID        
** SYNOPSIS:    int = MibName2ID(id,name);
**		LONG	*id;
**		char	*name;
**              
** DESCRIPTION: Translates the <name> of a MIB variable 
**		in the <id> of that variable.
** RETURNS:     <0 	error
**		else	length of id
**************************************************************/
int MibName2ID(LONG *id, const char *name)
{
    MIBNODE *node;
    /* find the correct node, and trace it back to get it's id */
    node = MibFindNode(name);
    if( node == NULL)
      { 
      	return -1;
      }
    return MibFillID(node,id);
}

/**************************************************************
** NAME:        MibID2Name
** SYNOPSIS:    cPtr = MibID2Name(id,nr);
**		        LONG	*id;
**		        int	    nr;
**              
** DESCRIPTION: Translates the given <id> with length <nr>
**		to the corresponding logical MIB name. Returns
**		a pointer to that name.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
const char * MibID2Name(const LONG *id, const int nr)
{
    
    MIBNODE *node;

    node = MibFindID(id,nr);
    if( node == NULL) return NULL;
    return node->name;
}
/**************************************************************
** NAME:        MibID2Name2
** SYNOPSIS:    cPtr = MibID2Name2(id,nr,nrmatched);
**		LONG	*id;
**		int	    nr;
**		int	    *nrmatched
**              
** DESCRIPTION: Translates the given <id> with length <nr>
**		to the corresponding logical MIB name. Returns
**		a pointer to that name and fills <nrmatched>
**		with the number of matched sub-identifiers
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
const char * MibID2Name2(const LONG *id, const int nr,int *nrmatched)
{
    
    MIBNODE *node;

    node = MibFindID2(id,nr,nrmatched);
    if( node == NULL) return NULL;
    return node->name;
}
/**************************************************************
** NAME:        MibIDType
** SYNOPSIS:    iRet =  MibIDType(id,nr);
**              LONG	*id;
**              int	    iRet,nr;
** DESCRIPTION: Returns the type if the MIB variable identified
**		        by <id> of lenght <nr>
** RETURNS:     <0	error
**              else    mode
**************************************************************/
int MibIDType( const LONG *id, const int nr)
{
    MIBNODE *node;
    node = MibFindID(id,nr);
    if( node == NULL) return -1;
    return node->type;
}

/**************************************************************
** NAME:        MibIDMode
** SYNOPSIS:    iret = MibIDMode(id,nr);
**              LONG *id;
**              int nr,iRet;
** DESCRIPTION: Returns the mode of the MIB variable identified
**		by <id> of length <nr>
** RETURNS:     <0	error
**		mode
**************************************************************/
int MibIDMode( const LONG *id, const int nr)
{
    MIBNODE *node;
    node = MibFindID(id,nr);
    if( node == NULL) return -1;
    return node->mode;
}

/**************************************************************
** NAME:        MibIDStatus
** SYNOPSIS:    iRet = MibIDStatus(id,nr);
**		        LONG 	*id;
**		        int	nr,iRet;
** DESCRIPTION: Returns the status of the MIB varaible 
**		identified by <id> of length <nr>.
** RETURNS:     <0	error
**              status
** RETURNS:     void
**************************************************************/
int MibIDStatus( const LONG *id, const int nr)
{
    MIBNODE *node;
    node = MibFindID(id,nr);
    if( node == NULL) return -1;
    return node->status;
}
/**************************************************************
** NAME:        MibIDInfo
** SYNOPSIS:    iRet = MibIDInfo(id,nr,res);
**		        LONG 	*id;
**		        int	nr,iRet;
**			char    *res;
** DESCRIPTION: stores information about the given
**		MIB <id> in character buffer <res>
** RETURNS:     <0	error
**              status
** RETURNS:     void
**************************************************************/
int MibIDInfo( const LONG *id, const int nr,char *res)
{	   
	char *type, *mode, *status;
	MIBNODE *node;
	
	node = MibFindID(id,nr);
    	if( node == NULL) 
		return -1;
	if( (status = MibEncodeStatus(node->status)) == NULL)
		return -1;
	if( (mode = MibEncodeMode(node->mode)) == NULL)
		return -1;
	if( (type = MibEncodeType(node->type)) == NULL)
		return -1;
	strcpy(res,node->name);
	strcat(res," ");
	strcat(res,type);
	strcat(res," ");
	strcat(res,mode);
	strcat(res," ");
	strcat(res,status);
	return 0;
}

/**************************************************************
** NAME:        MibNameType
** SYNOPSIS:    iRet = MibNameType(name)
**		char	*name;
**
** DESCRIPTION: Returns the type of the MIB varaible 
**		identified by <name>
** RETURNS:     <0	error
**              type
** RETURNS:     void
**************************************************************/
int MibNameType( const char *name)
{
    MIBNODE *node;
    node = MibFindNode(name);
    if( node == NULL) return -1;
    return node->type;
}

/**************************************************************
** NAME:        MibNameMode
** SYNOPSIS:    iRet = MibNameMode(name)
**		char	*name;
**
** DESCRIPTION: Returns the mode of the MIB varaible 
**		identified by <name>
** RETURNS:     <0	error
**              mode
** RETURNS:     void
**************************************************************/
int MibNameMode( const char *name)
{
    MIBNODE *node;
    node = MibFindNode(name);
    if( node == NULL) return -1;
    return node->mode;
}

/**************************************************************
** NAME:        MibNameStatus
** SYNOPSIS:    iRet = MibNameStatus(name)
**		char	*name;
**
** DESCRIPTION: Returns the status of the MIB varaible 
**		identified by <name>
** RETURNS:     <0	error
**              status
** RETURNS:     void
**************************************************************/
int MibNameStatus( const char *name)
{
    MIBNODE *node;
    node = MibFindNode(name);
    if( node == NULL) return -1;
    return node->mode;
}
/**************************************************************
** NAME:        MibGetFirstChild
** SYNOPSIS:    char * MibGetFirstChild(name);
**		char	*name;
** DESCRIPTION: return pointer to name of first 
**		child of <name>
** RETURNS:     NULL	no first child
**              ptr to name first child
** RETURNS:     void
**************************************************************/
const char *MibGetFirstChild(const char *name)
{
    MIBNODE *node,*fchild;

    node = MibFindNode(name);
    if(node==NULL) return NULL;
    fchild=node->child1;
    if(fchild==NULL) return NULL;
    return fchild->name;
}
/**************************************************************
** NAME:        MibGetNextChild
** SYNOPSIS:    char * MibGetNextChild(name);
**		char	*name;
** DESCRIPTION: return pointer to name of next 
**		child from child <name>
** RETURNS:     NULL	no next child
**              ptr to name next child
** RETURNS:     void
**************************************************************/
const char *MibGetNextChild(const char *name)
{
    MIBNODE *node,*fchild;

    node = MibFindNode(name);
    if(node==NULL) return NULL;
    fchild=node->brother;
    if(fchild==NULL) return NULL;
    return fchild->name;
}







