/**********************************************************************
** MODULE INFORMATION*
**********************
**      FILE     NAME:       msgmain.c
**      SYSTEM   NAME:       message
**      ORIGINAL AUTHOR(S):  Alfred Kayser
**      VERSION  NUMBER:     1.0
**      CREATION DATE:       91/09/16
**
** DESCRIPTION: Message functions.
***********************************************************************
** CHANGES INFORMATION **
*************************
** REVISION: $Revision: 1.1 $
** AUTHOR:   $Author: alfred $
** DATE:     $Date: 1991/12/13 15:34:30 $
** LOG:      $Log: message.c,v $
**        Revision 1.1  1991/12/13  15:34:30  alfred
**        First check in.
**
**********************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <stdarg.h>

#define LIBRARY
#include <dnpap.h>
#include <config.h>
#include "message.h"



/*--------------------Local Defines-------------------------*/

#define NUMBEROF2(a)     (sizeof(a)/sizeof(*(a)))
#define ERROR(n,s)      DnpapMessage(DMC_ERROR,MSG_ERROR+n, s)
#define ERROR1(n,s,a)   DnpapMessage(DMC_ERROR,MSG_ERROR+n, s, a)
#define ERROR2(n,s,a,b) DnpapMessage(DMC_ERROR,MSG_ERROR+n, s, a, b)


/*--------------------Local Typedefs------------------------*/

typedef struct _msgdef MSGDEF;
    struct _msgdef                     
    {
        CONST char * moduleName;
        MSGDEF     * next;
        LONG         enable;
        LONG         base;            
        TRANSCRIPT * defTrans;
        TRANSCRIPT * trans[DMC_MAX];
    };


/*--------------------Local Prototypes------------------------*/

PRIVAT void 	Message 	__((int mc, LONG ec, CONST char *msg, VOID * ap));
PRIVAT MSGDEF * FindModule  __((LONG base));
PRIVAT MSGDEF * AddModule   __((LONG base));
PRIVAT void     StringLower __((char *str, CONST char *ins));
IMPORT void     TransInit   __((CONST char *defDevice));


/*--------------------Local Global Vars-----------------------*/

PRIVAT MSGDEF * msgDefs=NULL;
PRIVAT MSGDEF * msgDefault;
PRIVAT CONST char * msgTitle[]= { "fatal", "error", "warning", 
                                  "message", "debug", "trace", "log"};


/**************************************************************
** NAME:        MessageInit                               [API]
** SYNOPSIS:    void MessageInit(void)
** DESCRIPTION: Initializes the message module.
** RETURNS:     void
**************************************************************/
EXPORT void
MessageInit(CONST char *defDevice VAR_PROTO)
{
    va_list vp;
    TRANSDEVICE *device;

    DnpapSetMessageFun(Message);           /* Install the new message function */

    /* First define the devices... */
    VAR_START(vp, defDevice);
    while (device=va_arg(vp, TRANSDEVICE *))
         TransAddDevice(device);
    VAR_END(vp);

    TransInit(defDevice);                  /* Initialize transcript lib */
    MessageConfig(0, "default");           /* Configurate defaults */
    MessageConfig(MSG_ERROR, "message");   /* Configure myself */
}


/**************************************************************
** NAME:        MessageConfig                             [API]
** SYNOPSIS:    void MessageConfig(base, name)
**                LONG base;
**              CONST char *name;
** DESCRIPTION: Configurates the message module. <node> points
**              to the node in the tree of which the children
**              are expected to be 'level' or 'trans'.
**                Each modules (library which uses the message
**                system should call this function to initialize
**                the transcripts of his messages.
** CONFIG:      *.message.trans = name    
**              *.message.fatal.trans = name 
**              *.message.error.trans = name 
**              *.message.warning.trans = name 
**              *.message.message.trans = name    
**              *.message.debug.trans = name 
**              *.message.trace.trans = name 
**              *.message.log.trans = name 
** RETURNS:     void
**************************************************************/
EXPORT void 
MessageConfig(LONG base, CONST char *Name)
{
    char buf[200], name[40];
    MSGDEF *p=AddModule(base);            
    char *tname;
    int i;
    
    p->moduleName=name;
    p->enable=(LONG)-1L;
    p->defTrans=NULL;
    for (i=0;i<NUMBEROF2(p->trans);i++)
        p->trans[i]=NULL;
                            
    StringLower(name,Name);
    sprintf(buf,"%s.message.media", name);
    if (ConfigGetString(buf, &tname))
        if (!(p->defTrans=TransOpen(tname)))
            ERROR1(0,"Failed to open transcript to %s\n", tname);

    for (i=0;i<NUMBEROF2(msgTitle);i++)
    {
        sprintf(buf,"%s.message.%s.media", name, msgTitle[i]);
        if (ConfigGetString(buf, &tname))
            if (!(p->trans[i]=TransOpen(tname)))
                ERROR1(0,"Failed to open transcript to %s\n", tname);
		if (p->trans[i]==NULL)
			p->trans[i]=p->defTrans;
    }    
    if (base==0)
        msgDefault=p;                             
}


/**************************************************************
** NAME:        MessageEnable                             [API]
** SYNOPSIS:    void MessageEnable(LONG base, 
**                    MESSAGECODE dmc, BOOLEAN flag);
** DESCRIPTION: Enables messaging of a module.
** RETURNS:     void
**************************************************************/
EXPORT void
MessageEnable(LONG base, MESSAGECODE dmc, BOOLEAN flag)
{
    MSGDEF *p=FindModule(base);
    if (!p) return;  
    if (flag) 
        p->enable|=(1<<dmc); 
    else
        p->enable&=~(1<<dmc);
}


/**************************************************************
** NAME:        MessageSetTrans                           [API]
** SYNOPSIS:    void MessageSetTrans(
**                      LONG base, 
**                      MESSAGECODE dmc, 
**                      TRANSCRIPT *trans);
** DESCRIPTION: Sets message transcripts of a module.
**				if <dmc> is -1, the transcript of types of 
**				messages are set to <trans>.
** RETURNS:     void
**************************************************************/
EXPORT void
MessageSetTrans(LONG base, MESSAGECODE dmc, TRANSCRIPT *trans)
{											
	int i;
    MSGDEF *p=FindModule(base);
    if (!p) return;
	if (dmc<0)	   
	{
	 	p->defTrans=trans;
		for (i=0;i<NUMBEROF2(p->trans);i++)
			p->trans[i]=trans;
	}
	else					 
	    p->trans[dmc]=trans ? trans : p->defTrans;
}


/**************************************************************
** NAME:        Message                                   
** SYNOPSIS:    void Message( MESSAGECODE mc,
**                            LONG ec, 
**                            CONST char *message, va_list vp);
** DESCRIPTION: Prints an message message.
**                The old name 'DnpapMessage' is macroed to
**                'Message', so it can still be used...
** RETURNS:     void 
**************************************************************/
PRIVAT void 
Message(int mc, LONG ec, CONST char *msg, VOID * ap)
{                                     
    PRIVAT char formatBuf[1024];
    MSGDEF *p;
    char title='?';
       
    p=FindModule(ec);
    if (!p) p=msgDefault;

    vsprintf(formatBuf, msg, (va_list)ap);

    if (mc>=0 && mc<NUMBEROF2(msgTitle))
        title=toupper(msgTitle[mc][0]);       
    if (p) 
    {
        if (p->enable&(1<<mc))
            TransMessage(p->trans[mc], "%c%06ld: %s\n", title, ec, formatBuf);
    }
    else
        TransMessage(NULL, "%c%06ld: %s\n", title, ec, formatBuf);
}


/**************************************************************
** NAME:        AddModule                                 
** SYNOPSIS:    MSGDEF * AddModule(LONG base)
** DESCRIPTION: Finds a module in the modulelist, if it isn't
**              found then it is added to the list.
** RETURNS:     module number.
**************************************************************/
PRIVAT MSGDEF *
AddModule(LONG base)    
{
    int i;
    MSGDEF *p=FindModule(base);                           
    
    if (p==NULL)
    {   
        p=(MSGDEF *)DnpapMalloc(sizeof(MSGDEF));             /* Add new module */
        if (p==NULL) 
        {
            ERROR(1,"Out of memory\n"); 
            return NULL;
        }
        p->moduleName=NULL;
        p->base=base;
        for (i=0;i<NUMBEROF2(p->trans);i++)
            p->trans[i]=0;
        p->next=msgDefs;
        msgDefs=p;
    }
    return p;
}


/**************************************************************
** NAME:        FindModule                                
** SYNOPSIS:    MSGDEF *FindModule(LONG base)
** DESCRIPTION: Finds a module in the modulelist, if it isn't
**              found then it returns -1;
** RETURNS:     module number (or -1).
**************************************************************/
PRIVAT MSGDEF *
FindModule(LONG base)
{
    MSGDEF *p;           
    
    for (p=msgDefs;p!=NULL;p=p->next)
        if (p->base==base)
            return p;
    return NULL;
}


/**************************************************************
** NAME:        StringLower
** SYNOPSIS:    PRIVAT void
**              StringLower(char *str, CONST char *ins)
** DESCRIPTION: Converts the string <str> to lowercase.
** RETURNS:     void
**************************************************************/
PRIVAT void
StringLower(char *str, CONST char *ins)
{
    for (;*ins;ins++)
        *str++=(char)tolower(*ins);
    *str='\0';
}

