/************************************************************************
** MODULE INFORMATION **
************************
** FILE NAME:          mib.c
** SYSTEM NAME:        beholder
** MODULE NAME:        mib
** ORIGINAL AUTHOR(S): M.F.B. de Greeve
** VERSION NUMBER:     1.0
** CREATION DATE:      1992/7/13
** DESCRIPTION:        RMON Management Information Base routines
*************************************************************************/

#include <dnpap.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <snmp.h>
#include "mibsup.h"
#include "mibrqs.h"



MIB_RMON Mib =  {
                    NULL,
                    0,
                    NULL,
                    0,
                };


static SHORT MibCmpObjId (LONG *ObjId, WORD ObjIdLen1, LONG *ObjId2, WORD ObjIdLen2);
static SHORT MibCmpProfil (MIB_COMMUNITY *Prf1, MIB_COMMUNITY *Prf2);
static SHORT MibCmpObject (MIB_ELEMENT *Obj1, MIB_ELEMENT *Obj2);
static WORD MibGetProfil (BYTE *Community, WORD CommunityLen);
static WORD MibGetSetNxt (SNMP_OBJECT *Obj);



/*****************************************************************
** NAME:        MibRequest
** SYNOPSIS:    WORD MibRequest(BYTE *Community, WORD CommunityLen,
**                              SNMP_OBJECT *Obj)
** PARAMETERS:  Community: community name                
**              CommunityLen: length of Prf->Comm
**              *Obj: pointer to MIB-object
**                  Obj->Request: MIB_PDU_GET : get-request
**                                MIB_PDU_NXT : next-request
**                                MIB_PDU_SET : set-request
**                  Obj->Id: object identification
**                  Obj->IdLen: length of Obj->Id
**                  Obj->Type: MIB_NULL: Obj->Syntax.Ptr
**                             MIB_INTEGER: Obj->Syntax.LngInt
**                             MIB_OCTETSTR: Obj->Syntax.BufChr
**                             MIB_DISPLAYSTR: Obj->Syntax.BufChr
**                             MIB_ELEMENTID: Obj->Syntax.BufInt
**                             MIB_IPADDR: Obj->Syntax.LngUns
**                             MIB_COUNTER: Obj->Syntax.LngUns
**                             MIB_GAUGE: Obj->Syntax.LngUns
**                             MIB_TIMETICKS: Obj->Syntax.LngUns
**                             MIB_OPAQUE: Obj->Syntax.BufChr
**                  Obj->Syntax: response (depends on Obj->Type)           
**                  Obj->SyntaxLen: length of Obj->Syntax            
** DESCRIPTION: searches Obj in the MIB and sets/gets its syntax.
**              checks Prf for supported requests: Obj->Request
**              MIB_PDU_SET: checks Obj->Type
**                          sets Obj->Syntax
**              MIB_PDU_GET / MIB_PDU_NXT: inserts Obj->Type
**                                       inserts Obj->Syntax
** REMARKS:     see mib.h & mibrqs.h
** RETURNS:     MIB_NOERROR : no error
**              MIB_NOSUCHNAME: Obj->Id not found in MIB or
**                              request not supported by Prf->Comm
**              MIB_BADVALUE : Obj->Request == MIB_PDU_SET and
**                             wrong Obj->Type / incorrect value
**              MIB_TOOBIG : response too big for 1 packet
**              MIB_GENERROR : general error, dependent on object
*******************************************************************/

WORD MibRequest (BYTE *Community, WORD CommunityLen, SNMP_OBJECT *Obj)
{
    WORD Support;
    WORD Err = SNMP_NOSUCHNAME;

    if ((Support = MibGetProfil (Community, CommunityLen)) == 0)
        return (Err);

    switch (Obj->Request)
    {
        case SNMP_PDU_GET:
        case SNMP_PDU_NEXT:
                if (!(Support & MIB_READ))
                    return (Err);
                break;
        case SNMP_PDU_SET:
                if (!(Support & MIB_WRITE))
                    return (Err);
                break;
    }
    Err = MibGetSetNxt (Obj);
    return (Err);
}

/*****************************************************************
** NAME:        MibInit
** SYNOPSIS:    VOID MibInit (MIB_COMMUNITY *Prf, MIB_ELEMENT *Obj,
**                  WORD PrfSze, WORD ObjSze)
** PARAMETERS:  *Prf: array of MIB-profiles
**                  Prf->Comm: community name
**                  Prf->CommLen: length of Prf->Comm
**                  Prf->Support: MIB_READ
**                                MIB_WRITE
**                                MIB_READ || MIB_WRITE
**              *Obj: array of MIB-objects
**                  Obj->Id: object identification
**                  Obj->IdLen: length of Obj->Id
**                  (*Rqs)(*Obj, IdLen): function for MibGetResp
**                      *Obj: see MibGetResp
**                      IdLen: length of Obj->Id in the MIB
**                  Obj->Type: SNMP_NULL: Obj->Syntax.Ptr
**                             SNMP_INTEGER: Obj->Syntax.LngInt
**                             SNMP_OCTETSTR: Obj->Syntax.BufChr
**                             SNMP_DISPLAYSTR: Obj->Syntax.BufChr
**                             SNMP_ELEMENTID: Obj->Syntax.BufInt
**                             SNMP_IPADDR: Obj->Syntax.LngUns
**                             SNMP_COUNTER: Obj->Syntax.LngUns
**                             SNMP_GAUGE: Obj->Syntax.LngUns
**                             SNMP_TIMETICKS: Obj->Syntax.LngUns
**                             SNMP_OPAQUE: Obj->Syntax.BufChr
**                  Obj->Support: MIB_READ
**                                MIB_WRITE
**                                MIB_READ || MIB_WRITE
**              PrfSize: number of communities    
**              ObjSize: number of objects    
** DESCRIPTION: initializes the MIB.
**              (*Rqs) is a function-pointer to the application-
**                  specicific part of the MIB-variable. When the
**                  Obj->Id is found this function is called to
**                  retrieve the data from the application.
** REMARKS:     see also mib.h & mibsup.h
** RETURNS:     VOID
*******************************************************************/

VOID MibInit (MIB_COMMUNITY *Prf, MIB_ELEMENT *Obj, WORD PrfSze, WORD ObjSze)
{
    Mib.Prf = Prf;
    Mib.Obj = Obj;
    
    Mib.PrfSze = PrfSze;
    Mib.ObjSze = ObjSze;

    qsort (Mib.Prf, Mib.PrfSze, sizeof(MIB_COMMUNITY), MibCmpProfil);
    qsort (Mib.Obj, Mib.ObjSze, sizeof(MIB_ELEMENT), MibCmpObject);

    return;
}


/*****************************************************************
** NAME:        MibSimple
** SYNOPSIS:    BOOLEAN MibSimple (SNMP_OBJECT *Obj,unsigned IdLen)
** PARAMETERS:  *Obj: see MibRequest()
**              IdLen: length of Obj->Id in the MIB
** DESCRIPTION: checks the Obj->Id of a simple object.
** REMARKS:     see also MibRmon(), mib.h & mibsup.h
** RETURNS:     TRUE: good Obj->Id: process further
**              FALSE: bad Obj->Id: return SNMP_NOSUCHNAME
*******************************************************************/

BOOLEAN MibSimple (SNMP_OBJECT *Obj, unsigned IdLen)
{
    if (Obj->Request != SNMP_PDU_NEXT)
    {
        if (Obj->IdLen == IdLen + 1 && Obj->Id[IdLen] == 0)
            return (TRUE);
        return (FALSE);
    }
    if (Obj->IdLen == IdLen)
    {
        Obj->Id[IdLen] = 0;
        Obj->IdLen++;
        return (TRUE);
    }
    return (FALSE);
}

/*****************************************************************
** NAME:        MibRmon
** SYNOPSIS:    MIB_LOCAL * MibRmon (SNMP_OBJECT *Obj,
**                  MIB_LOCAL *Local, WORD IdLen, WORD IdSize)
** PARAMETERS:  *Obj: see MibRequest()
**              *Local: begin of list of local datastructures
**              IdLen: length of Obj->Id in the MIB
**              IdSize: length of Obj->Id after the MIB
** DESCRIPTION: checks the Obj->Id of an Rmon (aggregate) object.
** REMARKS:     see also MibSimple(), mib.h & mibsup.h
** RETURNS:     pointer to found local datastructure.
**                  ->Index: collector index
**                  ->Data: pointer to the application data
**              NULL: local datastructure was not found
*******************************************************************/

MIB_LOCAL * MibRmon
    (SNMP_OBJECT *Obj, MIB_LOCAL *Local, WORD IdLen, WORD IdSize)
{
    if (Local == NULL)
        return NULL;
    if (Obj->Request != SNMP_PDU_NEXT)
    {
        if (Obj->IdLen != IdLen + IdSize)
            return NULL;
        while (Local->Index < Obj->Id[IdLen] && Local->Next != NULL)
            Local = Local->Next;
        if (Local->Index == Obj->Id[IdLen])
            return(Local);
        return NULL;
    }
    if (Obj->IdLen > IdLen + IdSize)        /* bad index */
        return NULL;
    if (Obj->IdLen == IdLen)                /* first instance */
        return (Local);
    while (Local->Index < Obj->Id[IdLen] && Local->Next != NULL)
        Local = Local->Next;
    if (Local->Index >= Obj->Id[IdLen])
        return(Local);
    return NULL;
}

/*****************************************************************
** NAME:        MibInsert
** SYNOPSIS:    MIB_LOCAL * MibInsert (SNMP_OBJECT *Obj,
**                  MIB_LOCAL **local, WORD IdLen, WORD IdSize)
** PARAMETERS:  *Obj: see MibRequest()
**              **local: begin of list of local datastructures
**              IdLen: length of Obj->Id in the MIB
**              IdSize: length of Obj->Id after the MIB
** DESCRIPTION: creates a new Rmon instance in local datastructure.
** REMARKS:     see also MibRemove(), mib.h & mibsup.h
** RETURNS:     pointer to created instance.
**                  Index: collector index
**                  Data: pointer to the application data
**              NULL: local datastructure was not created
*******************************************************************/

MIB_LOCAL * MibInsert
    (SNMP_OBJECT *Obj, MIB_LOCAL **local, WORD IdLen, WORD IdSize)
{
    MIB_LOCAL * Junk;
    MIB_LOCAL * Local = *local;

    if (Obj->IdLen != IdLen + IdSize || Obj->Id[IdLen] < 1)
        return NULL;
    if ((Junk = DnpapMalloc (sizeof(MIB_LOCAL))) == NULL)
        return NULL;
    Junk->Index = Obj->Id[IdLen];
    
    if (Local == NULL || Local->Index > Junk->Index)
    {
        Junk->Next = Local;
        *local = Junk;
        return (Junk);
    }
    while (Local->Next != NULL && Local->Next->Index <= Junk->Index)
        Local = Local->Next;
    if (Local->Index == Junk->Index)
    {
        DnpapFree(Junk);
        return(Local);
    }
    Junk->Next = Local->Next;
    Local->Next = Junk;
    return (Junk);
}

/*****************************************************************
** NAME:        MibRemove
** SYNOPSIS:    BOOLEAN MibRemove (SNMP_OBJECT *Obj,
**                  MIB_LOCAL **local, WORD IdLen, WORD IdSize)
** PARAMETERS:  *Obj: see MibRequest()
**              **local: begin of list of local datastructures
**              IdLen: length of Obj->Id in the MIB
**              IdSize: length of Obj->Id after the MIB
** DESCRIPTION: removes an Rmon instance in local datastructure.
** REMARKS:     see also MibInsert(), mib.h & mibsup.h
** RETURNS:     TRUE: instance was removed
**              FALSE: instance was not found
*******************************************************************/

BOOLEAN MibRemove (SNMP_OBJECT *Obj, MIB_LOCAL **local, WORD IdLen, WORD IdSize)
{
    MIB_LOCAL * Junk;
    MIB_LOCAL * Local = *local;

    if (!Local)
        return FALSE;
    if (Obj->IdLen != IdLen + IdSize)
        return FALSE;
    if (Local->Index == Obj->Id[IdLen])
    {
        Junk = Local->Next;
        DnpapFree(Local);
        *local = Junk;
        return TRUE;
    }
    while (Local->Next != NULL && Local->Next->Index < Obj->Id[IdLen])
        Local = Local->Next;
    if (Local->Next->Index == Obj->Id[IdLen])
    {
        Junk = Local->Next->Next;
        DnpapFree(Local->Next);
        Local->Next = Junk;
        return TRUE;
    }
    return FALSE;
}

SHORT MibCmpProfil (MIB_COMMUNITY *Prf1, MIB_COMMUNITY *Prf2)
{
    return (memcmp (Prf1->Comm, Prf2->Comm, SNMP_SIZE_COMM)); 
}

SHORT MibCmpObject (MIB_ELEMENT *Obj1, MIB_ELEMENT *Obj2)
{
    return (memcmp (Obj1->Id, Obj2->Id, SNMP_SIZE_OBJECTID)); 
}

SHORT MibCmpObjId (LONG *ObjId1, WORD ObjIdLen1, LONG *ObjId2, WORD ObjIdLen2)
{
    while (ObjIdLen1 > 0 && ObjIdLen2 > 0 && *ObjId1 == *ObjId2)
    {
        ObjIdLen1--;
        ObjIdLen2--;
        ObjId1++;
        ObjId2++;
    }
    if (!ObjIdLen1 && !ObjIdLen2)
        return (0);
    if (!ObjIdLen1)
        return (-1);
    if (!ObjIdLen2)
        return (1);
    if (*ObjId1 < *ObjId2)
        return (-2);
    else
        return (2);
}

WORD MibGetProfil (BYTE *Community, WORD CommunityLen)
{
    static WORD Fst, Lst, Ind;

    Fst = 0;
    Lst = Mib.PrfSze-1;

    while (Fst < Lst)
    {
        Ind = (Fst + Lst)/2;
        if (memcmp (Mib.Prf[Ind].Comm, Community, Mib.Prf[Ind].CommLen) < 0)
            Fst = Ind + 1;
        else
            Lst = Ind;
    }
    while (Fst < Mib.PrfSze && !memcmp (Mib.Prf[Fst].Comm, Community, Mib.Prf[Fst].CommLen))
    {
        if (Mib.Prf[Fst].CommLen == CommunityLen)
            return(Mib.Prf[Fst].Support);
        Fst++;
    }
    return(0);
}

WORD MibGetSetNxt (SNMP_OBJECT *Obj)
{
    static WORD Err, Fst, Lst, Ind;

    Err = SNMP_NOSUCHNAME;
    Fst = 0;
    Lst = Mib.ObjSze-1;

    while (Fst < Lst)
    {
        Ind = (Fst + Lst)/2;
        switch (MibCmpObjId (Mib.Obj[Ind].Id, Mib.Obj[Ind].IdLen, Obj->Id, Obj->IdLen))
        {
            case -2:
                Fst = Ind+1;
                break;
            case -1:
            case 0:
                Fst = Lst = Ind;
                break;
            case 1:
            case 2:
                Lst = Ind;
                break;
        }
    }
    switch (Obj->Request)
    {
        case SNMP_PDU_GET:
            if (!(Mib.Obj[Fst].Support & MIB_READ))
                return (Err);
            if (Mib.Obj[Fst].Rqs == NULL)
                return (Err);
            Obj->Type = Mib.Obj[Fst].Type;
            if ((MibCmpObjId (Mib.Obj[Fst].Id, Mib.Obj[Fst].IdLen, Obj->Id, Obj->IdLen)==-1))
                Err = Mib.Obj[Fst].Rqs (Obj, Mib.Obj[Fst].IdLen);
            return (Err);
        case SNMP_PDU_SET:
            if (!(Mib.Obj[Fst].Support & MIB_WRITE))
                return (Err);
            if (Mib.Obj[Fst].Rqs == NULL)
                return (Err);
            if (Mib.Obj[Fst].Type != Obj->Type)
                return (SNMP_BADVALUE);
            if (Obj->Type == SNMP_OBJECTID && Obj->SyntaxLen > SNMP_SIZE_BUFINT)
                return (SNMP_BADVALUE);
            if (Obj->Type == SNMP_OCTETSTR ||
                Obj->Type == SNMP_DISPLAYSTR ||
                Obj->Type == SNMP_OPAQUE)
            {
                if (Obj->SyntaxLen > SNMP_SIZE_BUFCHR)
                    return (SNMP_BADVALUE);
            }
            if ((MibCmpObjId (Mib.Obj[Fst].Id, Mib.Obj[Fst].IdLen, Obj->Id, Obj->IdLen)==-1))
                Err = Mib.Obj[Fst].Rqs (Obj, Mib.Obj[Fst].IdLen);
            return (Err);
        case SNMP_PDU_NEXT:
            switch (MibCmpObjId (Mib.Obj[Fst].Id, Mib.Obj[Fst].IdLen, Obj->Id, Obj->IdLen))
            {
                case -2:
                    return (Err);
                case -1:
                case 0:
                    if (Mib.Obj[Fst].Rqs != NULL && Mib.Obj[Fst].Support & MIB_READ)
                    {
                        Obj->Type = Mib.Obj[Fst].Type;
                        Err = Mib.Obj[Fst].Rqs (Obj, Mib.Obj[Fst].IdLen);
                    }
                    Fst++;
            }
            while (Fst < Mib.ObjSze && Err == SNMP_NOSUCHNAME)
            {
                memcpy (Obj->Id, Mib.Obj[Fst].Id,
                        Mib.Obj[Fst].IdLen * sizeof (LONG));
                Obj->IdLen = Mib.Obj[Fst].IdLen;
                if (Mib.Obj[Fst].Rqs != NULL && Mib.Obj[Fst].Support & MIB_READ)
                {
                    Obj->Type = Mib.Obj[Fst].Type;
                    Err = Mib.Obj[Fst].Rqs (Obj, Mib.Obj[Fst].IdLen);
                }
                Fst++;
            }
            return (Err);
    }
}
