#include <stdlib.h>
#include <time.h>
#include <dnpap.h>
#include <memory.h>
#include <maxmem.h>
#include <config.h>
#include <mac.h>
#include <sys.h>

#include "min.h"

#include "hist_e.h"
#include "hist_c.h"

  

LONG HistoryMaxNrBuckets = 2000;


static ULONG  prvtime;

static ETHER_STATS  etherstats;


static VOID HistoryTimerCallback(TIMER_DESCR *timer, ULONG now, VOID *param);



BOOLEAN HistoryConfigInit(VOID)
{
	ConfigGetLong("beholder.history.maxnrbuckets", &HistoryMaxNrBuckets);

	if (HistoryMaxNrBuckets < 2)
    {
        DnpapMessage(DMC_WARNING, HISTORY_MAX, "historycontrol: beholder.history.maxnrbuckets < 2, setting it to 2");
        HistoryMaxNrBuckets = 2;
    }

	return TRUE;
}


BOOLEAN HistoryCInit(HISTORY_CONTROL *historycontrol)
{
LONG source[] = {1,3,6,1,2,1,2,2,1,1,1};

    historycontrol->Index = 0;
    memcpy(historycontrol->Source, source, sizeof(source));
    historycontrol->SourceLen = sizeof(source)/sizeof(source[0]);
    historycontrol->BucketsRequested = 50;
    historycontrol->BucketsGranted = 0;
    historycontrol->Interval = 1800;
    historycontrol->Owner[0] = '\0';
    historycontrol->OwnerLen = 0;
    historycontrol->Status = SNMP_INVALID;

    historycontrol->Timer = NULL;
    historycontrol->FirstCall = TRUE;

    if ((historycontrol->Iface =
        MacIfaceGet((WORD) historycontrol->Source[historycontrol->SourceLen-1])) == NULL)
    {
        DnpapMessage(DMC_ERROR, HISTORY_NETINIT, "historycontrol: network initialisation failed");
        return (FALSE);
    }

    historycontrol->BucketsGranted = Min(MemoryGetAvail(), MemoryGetMaxChunk())/sizeof(ETHER_HISTORY);
    if (historycontrol->BucketsGranted > Min(historycontrol->BucketsRequested, HistoryMaxNrBuckets))
        historycontrol->BucketsGranted = Min(historycontrol->BucketsRequested, HistoryMaxNrBuckets);
    if ((historycontrol->EtherBuckets = NewEtherHistory(historycontrol->BucketsGranted)) == NULL)
    {
        historycontrol->BucketsGranted = 0;
        historycontrol->BucketsAllocated = 0;
        historycontrol->BucketsPos = 0;
        historycontrol->BucketsWrapped = FALSE;
    }
    else
    {
        historycontrol->BucketsAllocated = historycontrol->BucketsGranted;
        historycontrol->BucketsPos = 0;
        historycontrol->BucketsWrapped = FALSE;
    }

	return TRUE;
}


BOOLEAN HistoryCStart(HISTORY_CONTROL *historycontrol)
{
ULONG now, waittime, nexthour;        
           
    now = (ULONG)difftime(0, time(NULL));
    nexthour = (now/3600+1)*3600;
    waittime = (nexthour-now) - ((nexthour-now)/historycontrol->Interval)*historycontrol->Interval;
    
    if ((historycontrol->Timer = TimerRegister(HistoryTimerCallback, historycontrol, waittime, TIMER_FOREVER, TIMER_TYPE_SKIP)) == NULL)
    {
        DnpapMessage(DMC_ERROR, HISTORY_TIMER, "history: can not register a timer");
        return FALSE;
    }
                        
    historycontrol->FirstCall = TRUE;
    
    return TRUE;
}


BOOLEAN HistoryCStop(HISTORY_CONTROL *historycontrol)
{
    TimerRemove(historycontrol->Timer);

    DelEtherHistory(historycontrol->EtherBuckets);

    return TRUE;
}


VOID HistoryTimerCallback(TIMER_DESCR *timer, ULONG now, VOID *param)
{
ULONG dtime;
MAC_STAT netstat;
ETHER_HISTORY *etherhistory, *etherentries;
HISTORY_CONTROL *historycontrol = param;
                          
    if (historycontrol->FirstCall == TRUE)
    {
        prvtime = TimerNow();
    
        MacStatistics(historycontrol->Iface, &netstat);
    
        etherstats.DropEvents = netstat.LostPkts + netstat.DiscardedPkts;
        etherstats.Octets = netstat.Octets;
        etherstats.Pkts = netstat.Pkts;
        etherstats.BroadcastPkts = netstat.BroadcastPkts; 
        etherstats.MulticastPkts = netstat.MulticastPkts;
        etherstats.CRCAlignErrors = netstat.CRCAlignErrors;
        etherstats.UndersizePkts = netstat.UndersizePkts; 
        etherstats.OversizePkts = netstat.OversizePkts;  
        etherstats.Fragments = netstat.Fragments;     
        etherstats.Jabbers = netstat.Jabbers;       
        etherstats.Collisions = netstat.Collisions;
    
        TimerChange(timer, historycontrol->Interval*1000);
            
        historycontrol->FirstCall = FALSE;
        
        return;
    }

    if (historycontrol->BucketsWrapped == FALSE &&
        historycontrol->BucketsGranted < historycontrol->BucketsRequested &&
        historycontrol->BucketsGranted < HistoryMaxNrBuckets)
    {
        if ((etherentries = ReNewEtherHistory(historycontrol->EtherBuckets, historycontrol->BucketsGranted+1, historycontrol->BucketsAllocated, HistoryMaxNrBuckets)) != NULL)
        {
            historycontrol->EtherBuckets = etherentries;
            historycontrol->BucketsAllocated = ++historycontrol->BucketsGranted;
        }
    }

    if (historycontrol->BucketsAllocated == 0)
    {
        DnpapMessage(DMC_ERROR, HISTORY_NOENTRIES, "history: could not allocate any entries");
        return;
    }

    if (historycontrol->BucketsPos == historycontrol->BucketsGranted)
    {
        historycontrol->BucketsPos = 0;
        historycontrol->BucketsWrapped = TRUE;
    }

    etherhistory = &historycontrol->EtherBuckets[historycontrol->BucketsPos++];

    dtime = now - prvtime;
    prvtime = now;

    MacStatistics(historycontrol->Iface, &netstat);

    etherhistory->IntervalStart = SysTime();
    etherhistory->DropEvents = netstat.LostPkts + netstat.DiscardedPkts - etherstats.DropEvents;
    etherhistory->Octets = netstat.Octets - etherstats.Octets;
    etherhistory->Pkts = netstat.Pkts - etherstats.Pkts;
    etherhistory->BroadcastPkts = netstat.BroadcastPkts - etherstats.BroadcastPkts; 
    etherhistory->MulticastPkts = netstat.MulticastPkts - etherstats.MulticastPkts;
    etherhistory->CRCAlignErrors = netstat.CRCAlignErrors - etherstats.CRCAlignErrors;
    etherhistory->UndersizePkts = netstat.UndersizePkts - etherstats.UndersizePkts; 
    etherhistory->OversizePkts = netstat.OversizePkts - etherstats.OversizePkts;  
    etherhistory->Fragments = netstat.Fragments - etherstats.Fragments;     
    etherhistory->Jabbers = netstat.Jabbers - etherstats.Jabbers;    
    etherhistory->Collisions = netstat.Collisions - etherstats.Collisions;
    etherhistory->Utilization = (etherhistory->Octets + etherhistory->Pkts*24)*8/dtime;

    etherstats.DropEvents = netstat.LostPkts + netstat.DiscardedPkts;
    etherstats.Octets = netstat.Octets;
    etherstats.Pkts = netstat.Pkts;
    etherstats.BroadcastPkts = netstat.BroadcastPkts; 
    etherstats.MulticastPkts = netstat.MulticastPkts;
    etherstats.CRCAlignErrors = netstat.CRCAlignErrors;
    etherstats.UndersizePkts = netstat.UndersizePkts; 
    etherstats.OversizePkts = netstat.OversizePkts;  
    etherstats.Fragments = netstat.Fragments;     
    etherstats.Jabbers = netstat.Jabbers;       
    etherstats.Collisions = netstat.Collisions;
}


ETHER_HISTORY *EtherHistorySearch(HISTORY_CONTROL *historycontrol, SNMP_OBJECT *obj, WORD idlen)
{
    if (obj->Id[idlen+1] > historycontrol->BucketsGranted ||
        (historycontrol->BucketsWrapped == FALSE && obj->Id[idlen+1] > historycontrol->BucketsPos))
        return NULL;
    else
        if (historycontrol->BucketsWrapped == FALSE)
            return &historycontrol->EtherBuckets[obj->Id[idlen+1]-1];
        else
            return &historycontrol->EtherBuckets[(historycontrol->BucketsPos+obj->Id[idlen+1]-1) % historycontrol->BucketsGranted];
}


ETHER_HISTORY *NewEtherHistory(LONG nrentries)
{
    return ReNewEtherHistory(NULL, nrentries, 0, HistoryMaxNrBuckets);
}


ETHER_HISTORY *ReNewEtherHistory(ETHER_HISTORY* ptr, LONG nrentries, LONG oldnrentries, LONG maxnrentries)
{           
LONG n, nold;

	if (nrentries > maxnrentries || nrentries*sizeof(ETHER_HISTORY) > MemoryGetMaxChunk())
		return NULL;
    n = ((nrentries-1)/MINALLOCENTRIES)*MINALLOCENTRIES+MINALLOCENTRIES;
    nold = ((oldnrentries-1)/MINALLOCENTRIES)*MINALLOCENTRIES+MINALLOCENTRIES;
    if (ptr == NULL || abs((WORD)(n-nold)) >= MINALLOCENTRIES)  
    {
        return DnpapRealloc(ptr, Min(n, maxnrentries)*sizeof(ETHER_HISTORY));
    }        
    else
        return ptr;
}


VOID DelEtherHistory(ETHER_HISTORY* ptr)
{
    DnpapFree(ptr);
}
