#include <dnpap.h>
#include <event.h>
#include <memory.h>
#include <message.h>
#include <mibsup.h>
#include <mibrqs.h>
#include <stdlib.h>
#include <string.h>
#include <trap.h>

#include "alarm_e.h"
#include "alarm_d.h"
#include "alarm_c.h"


static CHAR MODULE[] = "AlarmC";


static VOID AlarmCallback(TIMER_DESCR *timer, ULONG now, VOID *param);



BOOLEAN AlarmCInit(RmonAlarm* alarm)
{
    alarm->Index = 0;
    alarm->Interval = 1;
    alarm->Variable[0] = 0;
    alarm->Variable[1] = 0;
    alarm->VarLen = 2;
    alarm->SampleType = ALARM_ABSOLUTE;
    alarm->Value = 0;
    alarm->PrevValue = 0;
    alarm->PrevDeltaValue = 0;
    alarm->StartupAlarm = 0;
    alarm->RisingThreshold = 0;
    alarm->RisingEvent = FALSE;
    alarm->FallingThreshold = 0;
    alarm->FallingEvent = FALSE;
    alarm->RisingEventIndex = 0;
    alarm->FallingEventIndex = 0;
    alarm->Owner[0] = '\0';
    alarm->OwnerLen = 0;
    alarm->Status = SNMP_INVALID;
    alarm->Timer = NULL;

    return TRUE;
}


BOOLEAN AlarmCStart(RmonAlarm* alarm)
{
LONG value;
SNMP_OBJECT obj = { SNMP_PDU_GET, { 0, 0 }, 2, SNMP_INTEGER };

    if (AlarmSample(alarm->Variable, alarm->VarLen, &value, &obj) != SNMP_NOERROR)
    {
    	DnpapMessage(DMC_ERROR, ALARM_SAMPLE, "alarm %ld: can not sample MIB variable", alarm->Index);
        return FALSE;
    }

    if (alarm->SampleType == ALARM_ABSOLUTE)
    {
        if ((alarm->Timer = TimerRegister(AlarmCallback, (VOID*)alarm, alarm->Interval*1000, TIMER_FOREVER, TIMER_TYPE_SKIP)) == NULL)
            return FALSE;
    }
    else
    if (alarm->SampleType == ALARM_DELTA)       /*  sample twice as much for the precise algorithm  */
    {
        if ((alarm->Timer = TimerRegister(AlarmCallback, (VOID*)alarm, alarm->Interval*500, TIMER_FOREVER, TIMER_TYPE_SKIP)) == NULL)
            return FALSE;
        alarm->PrevValue = value;
    }

    if (alarm->SampleType == ALARM_ABSOLUTE)
    {
        switch (alarm->StartupAlarm)
        {
        case ALARM_RISINGFALLING:
            if (value < alarm->FallingThreshold)
                alarm->FallingEvent = GenerateEvent(MODULE, alarm->FallingEventIndex, SNMP_TRAP_ENTSPECIFIC, TRAP_SPECIFIC_FALLINGALARM, &obj, 1);
            if (value > alarm->RisingThreshold)
                alarm->RisingEvent = GenerateEvent(MODULE, alarm->RisingEventIndex, SNMP_TRAP_ENTSPECIFIC, TRAP_SPECIFIC_RISINGALARM, &obj, 1);
            break;
        case ALARM_RISING:
            if (value > alarm->RisingThreshold)
                alarm->RisingEvent = GenerateEvent(MODULE, alarm->RisingEventIndex, SNMP_TRAP_ENTSPECIFIC, TRAP_SPECIFIC_RISINGALARM, &obj, 1);
            break;
        case ALARM_FALLING:
            if (value < alarm->FallingThreshold)
                alarm->FallingEvent = GenerateEvent(MODULE, alarm->FallingEventIndex, SNMP_TRAP_ENTSPECIFIC, TRAP_SPECIFIC_FALLINGALARM, &obj, 1);
            break;
        default:
            break;
        }
    }    
    
    return TRUE;
}


BOOLEAN AlarmCStop(RmonAlarm* alarm)
{
    TimerRemove(alarm->Timer);
    
    return TRUE;
}


WORD AlarmSample(LONG *id, WORD idlen, LONG *value, SNMP_OBJECT *obj)
{
    obj->Request = SNMP_PDU_GET;
	memcpy(obj->Id, id, (obj->IdLen = idlen)*sizeof(id[0]));
	if (MibRequest("public", 6, obj) == SNMP_NOSUCHNAME)
        return SNMP_NOSUCHNAME;
	
	switch (obj->Type)
	{
	case SNMP_INTEGER:
	case SNMP_COUNTER:
	case SNMP_GAUGE:
	case SNMP_TIMETICKS:
		*value = obj->Syntax.LngInt;
		return SNMP_NOERROR;
	default:
		return SNMP_BADVALUE;
	}
}


VOID AlarmCallback(TIMER_DESCR *timer, ULONG now, VOID *param)
{
SNMP_OBJECT obj = { SNMP_PDU_GET, { 0, 0 }, 2, SNMP_INTEGER };
SNMP_OBJECT rmobj = { SNMP_PDU_SET, { MIB_ALARM,MIB_ALARMSTATUSINDEX }, MIB_ALARMLEN+2, SNMP_INTEGER, SNMP_INVALID };
LONG value, delta;
RmonAlarm *alarm = param;

	if (AlarmSample(alarm->Variable, alarm->VarLen, &value, &obj) != SNMP_NOERROR)
    {
    	DnpapMessage(DMC_ERROR, ALARM_SAMPLE, "alarm %ld: can not sample MIB variable", alarm->Index);
        rmobj.Id[MIB_ALARMLEN+1] = alarm->Index;
        if (MibRequest("collector", strlen("collector"), &rmobj) != SNMP_NOERROR)
    	    DnpapMessage(DMC_ERROR, ALARM_INVALID, "can not destroy alarm entry %ld", alarm->Index);
        return;
    }

    if (alarm->SampleType == ALARM_DELTA)
    {
        /*  calculate the sum of the last 2 delta values  */
        alarm->Value = (delta = value - alarm->PrevValue) + alarm->PrevDeltaValue;
        alarm->PrevValue = value;
        alarm->PrevDeltaValue = delta;
    }
    else
    if (alarm->SampleType == ALARM_ABSOLUTE)
    {
        alarm->Value = value;
    }

    if (alarm->Value < alarm->FallingThreshold && alarm->FallingEvent == FALSE)
    {
        alarm->FallingEvent = GenerateEvent(MODULE, alarm->FallingEventIndex, SNMP_TRAP_ENTSPECIFIC, TRAP_SPECIFIC_FALLINGALARM, &obj, 1);
        alarm->RisingEvent = FALSE;
    }
    if (value > alarm->RisingThreshold && alarm->RisingEvent == FALSE)
    {
        alarm->RisingEvent = GenerateEvent(MODULE, alarm->RisingEventIndex, SNMP_TRAP_ENTSPECIFIC, TRAP_SPECIFIC_RISINGALARM, &obj, 1);
        alarm->FallingEvent = FALSE;
    }
}


RmonAlarm* NewRmonAlarm(WORD nralarms)
{
    return DnpapMalloc(nralarms*sizeof(RmonAlarm));
}


RmonAlarm* ReNewRmonAlarm(RmonAlarm* ptr, WORD nralarms)
{
    return DnpapRealloc(ptr, nralarms*sizeof(RmonAlarm));
}


VOID DelRmonAlarm(RmonAlarm* ptr)
{
    DnpapFree(ptr);
}
