/* TraceCtl - Initialize OS/2 Trace using configuration file

   FIXME to doc when need to run from trace directory
   FIXME to optionally use existing settings

   Copyright (c) 2001-2022 Steven Levine and Associates, Inc.
   All rights reserved.

   This program is free software licensed under the terms of the GNU
   General Public License.  The GPL Software License can be found in
   gnugpl2.txt or at http://www.gnu.org/licenses/licenses.html#GPL

   2001-01-11 SHL Baseline
   2015-10-25 SHL Shut down trace before dieing
   2018-03-28 SHL Convert to Globals style
   2018-03-28 SHL Sync with templates
   2018-03-28 SHL Support OPTIONS keyword
   2018-03-29 SHL Avoid spurious buffer size change errors
   2020-12-13 SHL Sync with templates
   2020-12-13 SHL Strip spaces from command line
   2022-08-27 SHL Merge TrcInit and TrcCtl to this
   2022-09-02 SHL More signal Error off/on

   Config file options
     CMD = tracecmd		trace setup commands
     DIEONERROR = yes/no	die on errors
     EXE = exe			exe to trace
     FINDPID = yes/no		find PID for exe
     OPTIONS			override default options (i.e. /b:512)
     PID = pid			pid to trace (overrides exe)

   PID defaults the decimal unless hex or has 0x prefix
   EXE and PID can be occur multiple times

   Notes: tracepoints are set in decimal and displayed in hex
	  pids are set in hex and displayed in hex
	  krnlrfs tracepoints 281 and 283 will trap on 14.93c to 14.100c

*/

signal on Error
signal on Failure name Error
signal on Halt
signal on NotReady name Error
signal on NoValue name Error
signal on Syntax name Error

gVersion = '0.9 2022-08-27'

Globals = 'gCfgFile gCmdName gEnv gTrc. gVersion'

call Initialize

Main:

  say
  say gCmdName 'starting.'

  '@setlocal'

  parse arg cmdLine
  call ScanArgs cmdLine
  drop cmdLine

  call ChkRequired

  if gCfgFile \== '.' then
    call ProcessCfgFile
  else do
    /* Allow for no configuration file */
    gTrc.!ExeList.0 = 0
  end

  call DoRequests

  call Directory oldDir

  exit

/* end main */

/*=== ChkRequired() Check required items accessible ===*/

ChkRequired: procedure expose (Globals)

  /* Requires DieQuiet */

  s = 'trace.exe'
  exe = SysSearchPath('PATH', s)
  if exe == '' then
    call DieQuiet 'Cannot access' s 'in PATH'

  s = 'tracefmt.exe'
  exe = SysSearchPath('PATH', s)
  if exe == '' then
    call DieQuiet 'Cannot access' s 'in PATH'

  say 'FIXME for ChkRequired to be done'
  /* trcdll.dll */

  /* trcformt.dll */

  /* trace.hlp */
  s = 'trace.hlp'
  exe = SysSearchPath('HELP', s)
  if exe == '' then
    call DieQuiet 'Cannot access' s 'in HELP'

  return

/* end ChkRequired */

/*=== ProcessCfgFile() Parse configuration file and set trace options ===*/

ProcessCfgFile: procedure expose (Globals)

  /* Requires DieQuiet */

  call ReadCfg gCfgFile

  /* Build comma separated exe list and find PIDs if requested */
  exeList = ''
  fail = 0
  do iExe = 1 to gTrc.!ExeList.0
    if iExe > 1 then
      exeList = exeList','
    exeList = exeList || gTrc.!ExeList.iExe
    if gTrc.!FindPID then do
      /* Assume process name matches module name */
      s = GetBaseName(gTrc.!ExeList.iExe)
      decPid = GetPidForProcess(s)
      if decPid == '' then do
	call WarnMsgWarble 'Warning:' gTrc.!ExeList.iExe 'not running'
	fail = 1
      end
      else do
	/* Add to PID list */
	pidNum = gTrc.!PidList.0 + 1
	gTrc.!PidList.pidNum = decPid
	gTrc.!PidList.0 = pidNum
      end
    end
  end

  if fail then
    call ScanArgsUsage'Please check configuration file'	/* Time to die */

  /* Build comma separated hex PID list */
  pidList = ''
  do pidNum = 1 to gTrc.!PidList.0
    if pidList \== '' then
      pidList = pidList','
    s = gTrc.!PidList.pidNum
    /* FIXME to know why /P: requires hex */
    if translate(left(s, 2)) \== '0x' then
      s = '0x' || d2x(s)
    pidList = pidList || s
  end

  /* Find boot drive */
  bootDrv = SysBootDrive()

  /* Must run from trace directory to avoid trap setting kernel tracepoints - FIXME to verify still true? */
  oldDir = directory()
  call directory bootDrv || '\os2\system\trace'

  /* Ensure trace initialized and buffer allocated */
  opts = gTrc.!Options
  haveBuf = pos('/B:', translate(opts)) > 0
  signal off Error
  '@trace /q >nul'
  signal on Error
  if RC = 1055 then do
    if \ haveBuf then do
      /* Default to 512KB */
      opts = strip('/b:512' opts)
      haveBuf = 1
    end
  end

  if haveBuf & RC = 0 then do
    /* Deallocate buffer */
    signal off Error
    'trace off /b'
    signal on Error
    if RC \= 0 then
      call Die 'Trace initialzation failed with error' RC
  end

  say 'Initializing trace with options' opts

  signal off Error
  'trace on' opts
  signal on Error
  if haveBuf & RC = 1055 then do
    /* 1055 may be bogus */
    signal off Error
    '@trace /q >nul'
    signal on Error
  end
  if RC \= 0 then
    call Die 'Trace initialzation failed with error' RC

  'trace off'
  'trace /c /s'

  /* Issue configured trace commands */
  do i = 1 to gTrc.!Cmd.0
    if 0 then say gTrc.!Cmd.i
    signal off Error
    gTrc.!Cmd.i
    signal on Error
    if RC \= 0 then do
      call WarnMsgWarble 'Warning: can not set tracepoint(s) - trace reported error' RC'.'
      if gTrc.!DieOnError then do
	/* Clean up */
	signal off Error
	'trace off'
	'trace off /p:all'
	signal on Error
	call DieQuiet 'Die on error requested by configuration.'
      end
    end
  end

  /* Trace specific pids or specific process as requested */
  if pidList \= '' then
    'trace on /p:'pidList
  else if exeList \= '' then
    'trace on /n:'exeList
  else
    'trace on /p:all'			/* Just in case */

  return

/* end ProcessCfgFile */

/*=== DoRequests() Do requests ===*/

DoRequests: procedure expose (Globals)

  /* Find boot drive */
  BootDrv = SysBootDrive()

  /* Must run from trace directory to avoid trap setting kernel tracepoints
     FIXME to know if still true
  */
  oldDir = directory()
  call Directory BootDrv'\os2\system\trace'

  /* Check trace initialized */
  signal off Error
  '@trace /q >nul'
  signal on Error
  if RC = 1055 then do
    say 'Trace not initialized'
    exit RC
  end

  /* Clear trace buffer */
  'trace /s /c'
  'trace /q'
  say

  req = ''
  opt = '/s'
  newOpt = '/r'
  pid = GetPidForProcess('tracefmt')
  fStartTraceFmt = pid == ''

  do while req \= 'Q'

    /* Warn if something not running */

    do iExe = 1 to gTrc.!ExeList.0
      /* Assume process name matches module name */
      s = GetBaseName(gTrc.!ExeList.iExe)
      pid = GetPidForProcess(s)
      if pid == '' then do
	call WarnMsgWarble 'Warning:' gTrc.!ExeList.iExe 'not running'
	fail = 1
      end
    end

    if opt \= newOpt then do

      opt = newOpt
      'trace' opt

      if opt = '/r' then
	say 'Tracing resumed.'
      else do
	say 'Tracing suspended.'
	pid = GetPidForProcess('tracefmt')
	fStartTraceFmt = pid == ''
      end
    end /* if newOpt */

    if fStartTraceFmt then do
      fStartTraceFmt = 0
      pid = GetPidForProcess('tracefmt')
      if pid \= '' then do
	say 'Tracefmt already running - activating'
	'activate "OS/2 System Trace Formatter*" restore'
      end
      else do
	say 'Starting tracefmt'
	signal off Error		/* In case 457 */
	'start tracefmt'
	signal on Error
	call SysSleep 2
	pid = GetPidForProcess('tracefmt')
	if pid == '' then
	  call WarnMsgWarble 'Warning: can not start tracefmt'
      end
    end

    say
    call charout ,'S)uspend R)esume O)ptions F)ormatter Q)uit H)elp ? '
    req = translate(SysGetKey('NOECHO'))
    say
    select
    when req == 'H' then do
      say
      say ' S      suspend tracing'
      say ' R      resume tracing'
      say ' Enter  toggle trace options'
      say ' O      show trace options'
      say ' Q      turn off tracing and quit'
      say ' F      show trace formatter window'
      say ' H      show this message'
      say ' !      invoke shell'
    end
    when req == 'F' then
      fStartTraceFmt = 1
    when req == 'O' then do
      say
      'trace /q'
    end
    when req == 'R' then
      newOpt = '/r'
    when req == 'S' then
      newOpt = '/s'
    when req == x2c('0d') then do
      /* Enter Key - toggle state */
      newOpt = translate(opt, 'sr', 'rs')
    end
    when req == 'Q' then
      nop
    when req == '!' then do
      /* Shell */
      say
      prompt = value('PROMPT',, gEnv)
      shell = value('COMSPEC',, gEnv)
      '@prompt' gCmdName 'shell' prompt
      signal off Error
      shell
      signal on Error
      '@prompt' prompt
    end
    otherwise
      nop				/* Ignore others */
    end /* select */

  end /* forever */

  /* Clean up
     Do not free trace buffer
  */
  say
  'trace /c /s'
  'trace off'
  'trace off /p:all'
  say
  'trace /q'

  call directory oldDir

  exit

/* end DoRequests */

/*=== ReadCfg(cfgFile) Read settings from trcinit.cfg ===*/

ReadCfg: procedure expose (Globals)

  parse arg cfgFile

  if cfgFile = '' then
    cfgFile = gCmdName'.cfg'		/* Default */

  /* Preset */
  gTrc.!Cmd.0 = 0
  gTrc.!DieOnError = 1
  gTrc.!ExeList.0 = 0
  gTrc.!FindPID = 0
  gTrc.!PidList.0 = 0
  gTrc.!Options = ''

  /* Scan and parse */

  say 'Reading' cfgFile

  s = stream(cfgFile, 'C', 'QUERY EXISTS')

  if s == '' then do
    i = lastpos('.', cfgFile)
    j = lastpos('\', cfgFile)
    if i <= j then do
      s = stream(cfgFile'.cfg', 'C', 'QUERY EXISTS')
      if s \== '' then
	cfgFile = s
    end
    if s == '' then
      call ScanArgsUsage 'Cannot access' cfgFile
  end

  fail = 0
  warnmsg = ''
  failmsg = ''
  linenum = 0

  call stream cfgFile, 'C', 'OPEN READ'

  do while lines(cfgFile) \= 0

    line = linein(cfgFile)
    line = strip(line)
    linenum = linenum + 1

    if line = '' then iterate
    if left(line, 1) = ';' then iterate	/* Comment line */

    parse var line req '=' opt
    req = strip(req)
    opt = strip(opt)

    uopt = translate(opt)
    select
    when abbrev('YES', uopt) \= 0 then yesno = 1
    when uopt = 1 then yesno = 1
      when abbrev('NO', uopt) \= 0 then yesno = 0
    when uopt = 0 then yesno = 0
    otherwise yesno = ''
    end

    ureq = translate(req)
    select
    when abbrev('CMD', ureq) \= 0 then do
      if opt = '' then
	failmsg = 'Expected command line for' req
      else do
	i = gTrc.!Cmd.0 + 1
	gTrc.!Cmd.i = opt
	gTrc.!Cmd.0 = i
      end
    end
    when abbrev('DIEONERROR', ureq) \= 0 then do
      if yesno = '' then
	failmsg 'Expected yes/no value for' req
      else
	gTrc.!DieOnError = yesno
    end
    when abbrev('EXE', ureq) \= 0 then do
      if opt = '' then
	failmsg = 'Expected exe name for' req
      else do
	i = lastpos('.', opt)
	j = lastpos('\', opt)
	if i <= j then
	  opt = opt'.exe'
	exe = stream(opt, 'C', 'QUERY EXISTS')
	if exe == '' then
	  exe = SysSearchPath('PATH', opt)
	if exe == '' then do
	  warnmsg = opt 'not found by name or by PATH'
	  exe = opt
	end
	/* Always add to list */
	i = gTrc.!ExeList.0 + 1
	gTrc.!ExeList.i = exe
	gTrc.!ExeList.0 = i
      end
    end
    when abbrev('FINDPID', ureq) \= 0 then do
      if yesno = '' then
	failmsg = 'Expected yes/no value for' req
      else
	gTrc.!FindPID = yesno
    end
    when abbrev('PID', ureq) \= 0 then do
      select
      when datatype(opt, 'X') then
	opt = '0x' || opt		/* provide 0x prefix for trace and others */
      when datatype(opt, 'W') then
	nop
      when translate(left(opt, 2)) == '0X' & datatype(substr(opt, 3), 'X') then
	nop
      otherwise
	failmsg = 'Expected decimal or hexadecimal PID for' req
      end
      if failmsg == '' then do
	exeNameOut = GetProcessForPid(opt)
	if exeNameOut = '' then
	  failmsg = 'PID' opt 'not running'
	else do
	  say 'PID' opt 'is' exeNameOut
	  i = gTrc.!ExeList.0 + 1
	  gTrc.!ExeList.i = exeNameOut
	  gTrc.!PidList.i = opt
	  gTrc.!ExeList.0 = i
	end
      end
    end
    when abbrev('OPTIONS', ureq) \= 0 then do
      if gTrc.!Options \== '' then
	failmsg = 'Options already set to' gTrc.!Options
      else do
	gTrc.!Options = opt
      end
    end
    otherwise
      failmsg = 'Request "' || req || '" unexpected at line' linenum 'of' cfgFile
    end

    if failmsg \= '' then do
      call WarnMsgBeep failmsg
      failmsg = ''
      fail = 1
    end

    if warnmsg \= '' then do
      call WarnMsgBeep warnmsg
      warnmsg = ''
      if gTrc.!DieOnError then
	fail = 1
    end

  end /* while lines */

  call stream cfgFile, 'C', 'CLOSE'

  if gTrc.!PidList.0 > 0 & gTrc.!ExeList.0 > 0 & \ gTrc.!FindPID then do
    call WarnMsgBeep 'PID not allowed with EXE'
    fail = 1
  end

  if fail then
    call ScanArgsUsage 'Please check configuration file'	/* Time to die */

  return

/* end ReadCfg */

/*=== Initialize: Initialize globals ===*/

Initialize: procedure expose (Globals)
  call LoadRexxUtil
  call SetCmdName
  gEnv = 'OS2ENVIRONMENT'
  call LoadRxUtilEx
  return

/* end Initialize */

/*=== ScanArgs(CmdLine) Scan command line ===*/

ScanArgs: procedure expose (Globals)

  gCfgFile = ''

  parse arg cmdTail
  cmdTail = strip(cmdTail)

  do while cmdTail \= ''

    parse var cmdTail curArg cmdTail
    cmdTail = strip(cmdTail)

    select
    when curArg == '-h' | curArg == '-?' then
      call ScanArgsHelp
    when curArg == '-n' then
      gCfgFile = '.'
    when curArg == '-V' then do
      say gCmdName gVersion
      exit
    end
    when left(curArg, 1) = '-' then
      call ScanArgsUsage curArg 'unexpected'
    otherwise
      if gCfgFile \== '' then
	call ScanArgsUsage 'Only one configuration file allowed'
      else
	gCfgFile = curArg		/* Check existance later */
    end
  end /* do */

  return

/* end ScanArgs */

/*=== ScanArgsHelp() Display ScanArgs usage help exit routine ===*/

ScanArgsHelp:
  say
  say 'OS/2 Trace Facility Wrapper.'
  say
  say 'Usage:' gCmdName '[-h] [-n] [-V] [-?] [tracectl-file]'
  say
  say '  -h -?          Display this message'
  say '  -n             Run without tracectl configuration file'
  say '  -V             Display version number and quit'
  say
  say '  tracectl-file  Tracectl configuration file'
  exit 255

/* end ScanArgsHelp */

/*=== ScanArgsUsage(message) Report ScanArgs usage error exit routine ===*/

ScanArgsUsage:
  parse arg msg
  say
  if msg \== '' then
    say msg
  say 'Usage:' gCmdName '[-h] [-n] [-V] [-?] [tracectl-file]'
  exit 255

/* end ScanArgsUsage */

/*==============================================================================*/
/*=== SkelRexxFunc standards - Delete unused - Move modified above this mark ===*/
/*==============================================================================*/

/*=== DieQuietV([message,...]) Write multi-line message to STDOUT omitting SIGL and die quietly ===*/

DieQuiet:
  call lineout 'STDOUT', ''
  do i = 1 to arg()
    call lineout 'STDOUT', arg(i)
  end
  call lineout 'STDOUT', gCmdName 'aborting.'
  call SysSleep 2
  exit 254

/* end DieQuietV */

/*=== GetBaseName(pathName) Return file base name stripping drive path and extension ===*/

GetBaseName: procedure
  parse arg s
  /* return path name with drive, directory and last extension stripped */
  s = filespec('N', s)
  i = lastpos('.', s)
  if i > 1 then
    s = left(s, i - 1)			/* Chop extension */
  return s
/* end GetBaseName */

/*=== GetPidForProcess(procName) Return decimal pid for named process or empty string ===*/

GetPidForProcess: procedure
  /* Requires LoadRxUtilEx */
  parse arg procName
  /* If process name omitted, get own pid */
  if procName \== '' then
    req = 'N'
  else do
    procName = 0
    req = 'P'
  end
  /* Get pid parent-pid process-type priority cpu-time executable-name */
  info = Sys2QueryProcess(procName, req)
  if info == '' then
    decpid = ''				/* Failed */
  else
    parse var info decpid .
  return decpid

/* end GetPidForProcess */

/*=== GetProcessForPid(pid[, radix]) Return fully qualified process name for pid or empty string ===*/

/**
 * pid is decimal pid, unless radix, content or 0x prefix indicates hex
 * radix forces hex regardless of pid value, any non-blank value will do
 */

GetProcessForPid: procedure expose (Globals)
  /* Requires LoadRxUtilEx */
  parse arg pid, radix

  if pid = '' then
    call Die 'PID required'

  select
  when radix \== '' then
    radix = 'H'				/* Assume user knows */
  when  datatype(pid, 'X') then
    radix = 'H'
  when  datatype(pid, 'W') then
    radix = 'P'
  when translate(left(pid, 2)) == '0X' & datatype(substr(pid, 3), 'X') then do
    pid = substr(pid, 3)	/* GetProcessForPid cannot handle 0x prefix */
    radix = 'H'
  end
  otherwise
    call Die pid 'must be a decimal or hex PID number'
  end

  /* Get pid parent-pid process-type priority cpu-time executable-name */
  info = Sys2QueryProcess(pid, radix)

  if info = '' then
    procName = ''
  else do
    parse var info . . . . . procName
    procName = strip(procName)
  end

  return procName

/* end GetProcessForPid */

/*=== LoadRxUtilEx() Load Alex's RxUtilEx functions ===*/

LoadRxUtilEx:
  if RxFuncQuery('Sys2LoadFuncs') then do
    call RxFuncAdd 'Sys2LoadFuncs', 'RXUTILEX', 'Sys2LoadFuncs'
    if RESULT then
      call Die 'Cannot load Sys2LoadFuncs'
    call Sys2LoadFuncs
  end
  return

/* end LoadRxUtilEx */

/*=== WarnMsgBeep(message) Write multi-line warning message to STDERR and beep ===*/

WarnMsgBeep: procedure
  do i = 1 to arg()
    msg = arg(i)
    call lineout 'STDERR', msg
  end
  call beep 400, 300
  return

/* end WarnMsgBeep */

/*=== WarnMsgWarble(message) Write multi-line warning message to STDERR and warble ===*/

WarnMsgWarble: procedure
  do i = 1 to arg()
    msg = arg(i)
    call lineout 'STDERR', msg
  end
  do 10;
    f=random(262,1047)
    d=random(100,200)
    call beep f,d
  end
  return

/* end WarnMsgWarble */

/*==========================================================================*/
/*=== SkelRexx standards - Delete unused - Move modified above this mark ===*/
/*==========================================================================*/

/*=== Die([message,...]) Write multi-line message to STDERR and die ===*/

Die:
  call lineout 'STDERR', ''
  do i = 1 to arg()
    call lineout 'STDERR', arg(i)
  end
  call lineout 'STDERR', gCmdName 'aborting at line' SIGL || '.'
  call beep 200, 300
  call SysSleep 2
  exit 254

/* end Die */

/*=== Error() Set gErrCondition; report to STDOUT; trace and exit or return if called ===*/

Error:
  say
  parse source . . cmd
  gErrCondition = condition('C')
  say gErrCondition 'signaled at line' SIGL 'of' cmd || '.'
  if condition('D') \== '' then
    say 'REXX reason =' condition('D') || '.'
  if gErrCondition == 'SYNTAX' & symbol('RC') == 'VAR' then
    say 'REXX error =' RC '-' errortext(RC) || '.'
  else if symbol('RC') == 'VAR' then
    say 'RC =' RC || '.'
  say 'Source =' sourceline(SIGL)

  if condition('I') \== 'CALL' | gErrCondition == 'NOVALUE' | gErrCondition == 'SYNTAX' then do
    trace '?A'
    say 'Enter REXX commands to debug failure.  Press enter to exit script.'
    nop
    if symbol('RC') \== 'VAR' then
      RC = 255
    exit RC
  end

  return

/* end Error */

/*=== Halt() Report HALT condition to STDOUT and exit ===*/

Halt:
  say
  parse source . . cmd
  say condition('C') 'signaled at' cmd 'line' SIGL || '.'
  say 'Source =' sourceline(SIGL)
  say 'Sleeping for 2 seconds...'
  call SysSleep 2
  exit 253

/* end Halt */

/*=== LoadRexxUtil() Load RexxUtil functions ===*/

LoadRexxUtil:
  if RxFuncQuery('SysLoadFuncs') then do
    call RxFuncAdd 'SysLoadFuncs', 'REXXUTIL', 'SysLoadFuncs'
    if RESULT then
      call Die 'Cannot load SysLoadFuncs.'
    call SysLoadFuncs
  end
  return

/* end LoadRexxUtil */

/*=== SetCmdName() Set gCmdName to short script name ===*/

SetCmdName: procedure expose (Globals)
  parse source . . cmd
  cmd = filespec('N', cmd)		/* Chop path */
  c = lastpos('.', cmd)
  if c > 1 then
    cmd = left(cmd, c - 1)		/* Chop extension */
  gCmdName = translate(cmd, xrange('a', 'z'), xrange('A', 'Z'))	/* Lowercase */
  return

/* end SetCmdName */

/* The end */
