/* DumpTrapScreen.cmd: Dump trap registers screens from dump image.
   Dump image may be diskette 1 of a trap set or a trapdump file on disk

   FIXME to support >2GiB files

   Copyright (c) 2000, 2014 Steven Levine and Associates, Inc.
   All rights reserved.

   This program is free software licensed under the terms of the GNU
   General Public License.  The GPL Software License can be found in
   gnugpl2.txt or at http://www.gnu.org/licenses/licenses.html#GPL

   2000-11-14 SHL Baseline
   2003-03-26 SHL Look for ##[0-9A-F]
   2003-04-10 SHL Handle truncated dumps nicer
   2004-10-12 SHL Show version with dump output
   2006-06-06 SHL Sync with templates
   2013-01-07 SHL Sync with templates
   2014-06-12 SHL Sync with templates
*/

signal on Error
signal on FAILURE name Error
signal on Halt
signal on NOTREADY name Error
signal on NOVALUE name Error
signal on SYNTAX name Error

call Initialize

G.!Version = '0.4'

Main:

  parse arg cmdTail

  if cmdTail = '' then
    call ScanArgsHelp

  G.!fDebug = 0
  G.!fCapture = 0
  G.!DumpFile = ''

  do while cmdTail \= ''
    parse var cmdTail s cmdTail
    cmdTail = strip(cmdTail)
    if left(s, 1) = '-' then do
      s = substr(s, 2)
      select
      when s = '?' | s = 'h' then
	call ScanArgsHelp
      when s = 'd' then
	G.!fDebug = 1
      when s = 'c' then
	G.!fCapture = 1
      when s == 'V' then do
	say G.!CmdName G.!Version
	exit
      end
      otherwise
	call ScanArgsUsage 'Option' s ' unknown'
      end
    end
    else
      G.!DumpFile = s
  end

  if G.!DumpFile = '' then
    call ScanArgsUsage 'Dump file name required'

  G.!Printables = xrange(' ', '7e'x) || '0d'x || '0a'x
  G.!HexDigits = xrange('0', '9') || xrange('A', 'F')
  G.!CrLf = '0d'x || '0a'x
  G.!LfCr = '0a'x || '0d'x

  call ScanDumpFile

  exit

/* end main */

/*=== ScanDumpFile() Read dump file; find trap screens; write to stdout. ===*/

ScanDumpFile:

  if stream(G.!DumpFile, 'C', 'QUERY EXISTS') == '' then do
    call Fatal G.!DumpFile 'not found'
  end

  say G.!CmdName 'v'G.!Version' reading' G.!DumpFile stream(G.!DumpFile, 'C', 'QUERY DATETIME')

  /* Read in 1st 512KB of dump image */

  call stream G.!DumpFile, 'C', 'OPEN READ'

  cBufMax = 512 * 1024

  signal off NOTREADY
  G.!Buf = charin(G.!DumpFile,, cBufMax)
  signal on NOTREADY name Error
  call stream G.!DumpFile, 'C', 'CLOSE'

  cBufMax = length(G.!Buf)		/* In case file short */

  /* If requested capture buffer image and quit */

  if G.!fCapture then do
    sOutFile = 'tmpdump.out'
    call SysFileDelete sOutFile
    call charout sOutFile, G.!Buf
    call charout sOutFile
    say 'Dump file header captured to' sOutFile'.  Exiting.'
    exit
  end

  do forever

    /* Find next tag */
    cSelect = cBufMax
    cPound = pos("##", G.!Buf)
    if cPound > 0 then do
      ch = substr(G.!Buf, cPound + 2, 4)
      if verify(ch, G.!HexDigits) = 0 then
	cSelect = min(cPound, cSelect)
    end
    cException = pos('Exception in', G.!Buf)
    if cException > 0 then
      cSelect = min(cException, cSelect)
    cP1Eq = pos("P1=", G.!Buf)
    if cP1Eq > 0 then
      cSelect = min(cP1Eq, cSelect)
    cC000 = pos("c000", G.!Buf)
    if cC000 > 0 then
      cSelect = min(cC000, cSelect)
    cIRevision = pos("Internal revision", G.!Buf)
    if cIRevision > 0 then
      cSelect = min(cIRevision, cSelect)

    if cSelect = cBufMax then
      leave				/* No more tags */

    G.!Buf = substr(G.!Buf, cSelect)	/* Dump non-printable prefix */

    if G.!fDebug then
      say 'Selected 'cSelect':' """"substr(G.!Buf, 1, 4)""""

    /* Dump line and format special as needed */
    select
    when cSelect = cPound then do
      say
      call DumpLines ''
    end
    when cSelect = cIRevision then do
      say
      call DumpLines ''
    end
    when cSelect = cException then do
      call FixLfCr
      call DumpLines ''
    end
    when cSelect = cC000 then do
      call DumpLines ''
      call DumpLines ''
    end
    otherwise
      call DumpLines ''
    end /* select */

  end /* forever */

  return

/* end ScanDumpFile */

/*=== DumpLines() Write printable area to stdout and find next printable ===*/

DumpLines: procedure expose G.

  c = verify(G.!Buf, G.!Printables)	/* Find next non-printable */

  c = c - 1				/* Size printable range */

  if c < 1 then
   call Fatal 'Can not size print area'

  G.!Area = substr(G.!Buf, 1, c)	/* Isolate printable area */

  /* Ignore empty trap screens */
  if substr(G.!Area, 1, 4) \= 'P1=X' then do
    if pos('CS:EIP', G.!Area) \= 0 then do
      say				/* Give some separation */
    end
    say G.!Area
  end

  G.!Buf = substr(G.!Buf, c + 1)	/* Drop printable area */

  c = verify(G.!Buf, G.!Printables, 'M')	/* Find next printable */
  if c \= 0 then
    G.!Buf = substr(G.!Buf, c)	/* Drop non-printable area */

  return

/* end DumpLines */

/*=== FixLfCr() Correct backwards Lf/Cr ===*/

FixLfCr: procedure expose G.

  c = pos(G.!LfCr, G.!Buf)
  do while c \= 0
    G.!Buf = delstr(G.!Buf, c, 2)
    G.!Buf = insert(G.!CrLf, G.!Buf, c - 1)
    c = pos(G.!LfCr, G.!Buf, c + 2)
  end

  return

/* end FixLfCr */

/*=== Initialize() Initialize globals ===*/

Initialize: procedure expose G.
  call GetCmdname
  call LoadRexxUtil
  return

/* end Initialize */

/*=== ScanArgsUsage(message) Report usage error ===*/

ScanArgsUsage:

  parse arg msg
  say msg
  say 'Usage:' G.!CmdName '[-h] [-d] [-c] [-V] trapdumpfile'
  exit 255

/* end ScanArgsUsage */

/*=== ScanArgsHelp() Display usage help ===*/

ScanArgsHelp:
  say
  say 'Usage:' G.!CmdName '[-h] [-d] [-c] [-V] trapdumpfile'
  say
  say ' -c            Write dump header to tmpdump.out'
  say ' -d            Display debug info'
  say ' -h            Display this message'
  say ' -V            Display version'
  say
  say ' trapdumpfile  Trap dump file to parse'
  exit 255

/* end ScanArgsHelp */

/*==============================================================================*/
/*=== SkelRexxFunc standards - Delete unused - Move modified above this mark ===*/
/*==============================================================================*/

/*==========================================================================*/
/*=== SkelRexx standards - Delete unused - Move modified above this mark ===*/
/*==========================================================================*/

/*=== Error() Report ERROR, FAILURE etc., trace and exit or return if called ===*/

Error:
  say
  parse source . . cmd
  say 'CONDITION'('C') 'signaled at line' SIGL 'of' cmd'.'
  if 'CONDITION'('D') \= '' then say 'REXX reason =' 'CONDITION'('D')'.'
  if 'CONDITION'('C') == 'SYNTAX' & 'SYMBOL'('RC') == 'VAR' then
    say 'REXX error =' RC '-' 'ERRORTEXT'(RC)'.'
  else if 'SYMBOL'('RC') == 'VAR' then
    say 'RC =' RC'.'
  say 'Source =' 'SOURCELINE'(SIGL)

  if 'CONDITION'('I') \== 'CALL' | 'CONDITION'('C') == 'NOVALUE' | 'CONDITION'('C') == 'SYNTAX' then do
    trace '?A'
    say 'Enter REXX commands to debug failure.  Press enter to exit script.'
    call 'SYSSLEEP' 2
    if 'SYMBOL'('RC') == 'VAR' then exit RC; else exit 255
  end

  return

/* end Error */

/*=== Fatal(message) Report fatal error and exit ===*/

Fatal:
  parse arg msg
  call 'LINEOUT' 'STDERR', ''
  call 'LINEOUT' 'STDERR', G.!CmdName':' msg 'at script line' SIGL
  call 'BEEP' 200, 300
  call 'SYSSLEEP' 2
  exit 254

/* end Fatal */

/*=== GetCmdName() Get short script name and set G.!CmdName ===*/

GetCmdName: procedure expose G.
  parse source . . cmd
  cmd = filespec('N', cmd)		/* Chop path */
  c = lastpos('.', cmd)
  if c > 1 then
    cmd = left(cmd, c - 1)		/* Chop extension */
  G.!CmdName = translate(cmd, xrange('a', 'z'), xrange('A', 'Z'))	/* Lowercase */
  return

/* end GetCmdName */

/*=== Halt() Report HALT condition and exit ===*/

Halt:
  say
  parse source . . cmd
  say 'CONDITION'('C') 'signaled at' cmd 'line' SIGL'.'
  say 'Source = ' 'SOURCELINE'(SIGL)
  call 'SYSSLEEP' 2
  say 'Exiting.'
  exit 253

/* end Halt */

/*=== LoadRexxUtil() Load RexxUtil functions ===*/

LoadRexxUtil:
  if RxFuncQuery('SysLoadFuncs') then do
    call RxFuncAdd 'SysLoadFuncs', 'REXXUTIL', 'SysLoadFuncs'
    if RESULT then
      call Fatal 'Cannot load SysLoadFuncs'
    call SysLoadFuncs
  end
  return

/* end LoadRexxUtil */

/* The end */
