/* AttachHLLDebugData - Attach DLL debug data to LX executable
   Best run under 4OS2, but should run OK under CMD.EXE

   Copyright (c) 2014-2021 Steven Levine and Associates, Inc.
   All rights reserved.

   This program is free software licensed under the terms of the GNU
   General Public License, Version 2.  The GPL Software License can be found
   in gnugpl2.txt or at http://www.gnu.org/licenses/licenses.html#GPL

   2014-11-16 SHL Baseline
   2014-11-23 SHL Rework LX offset detect logic
   2014-11-23 SHL Support cmd.exe
   2015-11-07 SHL Sync with templates
   2015-11-14 SHL Avoid call setlocal
   2016-02-20 SHL Correct typo
   2019-06-19 SHL Convert to globals
   2019-06-19 SHL Sync with templates
   2019-06-19 SHL Correct gDbgDataBytes set - oops
   2020-05-17 SHL Sync with templates
   2020-05-17 SHL Default to *.dbg if omitted
   2020-06-15 SHL More progress reporting
   2020-06-15 SHL Catch wildcards
   2021-04-03 SHL Sync with templates
   2021-04-03 SHL Add rest of CMD.EXE support
   2021-04-03 SHL Verify touch.exe available
*/

signal on Error
signal on Failure name Error
signal on Halt
signal on NotReady name Error
signal on NoValue name Error
signal on Syntax name Error

gVersion = '0.2 2021-04-03'

Globals = 'gArgList gCmdName gBackupFile gDbgDataBytes gDbgDataOffset gDbgFile gDbgLvl',
	  'gEnv gExeFile gIs4OS2 gLxHdr gLxHdrOffset gDryRun gTmpDir gVerbose gVersion'

call Initialize

Main:
  parse arg cmdLine
  call ScanArgs cmdLine
  drop cmdLine

  gIs4OS2 = Is4OS2()

  numeric digits 11

  s = 'touch.exe'
  exe = SysSearchPath('PATH', 'touch.exe')
  if exe == '' then
    call Die 'Cannot locate unix style touch in PATH - please install'

  call AttachHLL

  exit

/* end main */

/*=== AppendDebugData() ===*/

AppendDebugData: procedure expose (Globals)

  call VerboseMsg 'Appending' gDbgFile 'to' gExeFile

  cmd = '@copy' || IIF(gIs4OS2,' /q','') '/b' gExeFile || '+' || gDbgFile gExeFile
  say
  if gDryRun then
    say cmd '(DRY-RUN)'
  else
    cmd
  return

/* end AppendDebugData */

/*=== AttachHLL() ===*/

AttachHLL: procedure expose (Globals)

  call ChkLxExecutable
  call ChkHLLDebug
  call MakeBackup
  call AppendDebugData
  call RewriteHeader
  call RestoreTimestamp

  return

/* end AttachHLL */

/*=== ChkHLLDebug() ===*/

ChkHLLDebug: procedure expose (Globals)

  call VerboseMsg 'Validating' gDbgFile

  sig = charin(gDbgFile, 1, 4)
  if sig \== 'NB04' then
    call Die gDbgFile 'header does not contain a valid HLL debug data signature'

  call stream gDbgFile, 'C', 'SEEK <8'
  sig = charin(gDbgFile, , 4)
  s = charin(gDbgFile, , 4)
  offset = c2d(reverse(s))

  call stream gDbgFile, 'C', 'CLOSE'
  size = stream(gDbgFile, 'C', 'QUERY SIZE')

  if sig \== 'NB04' then
    call Die gDbgFile 'trailer does not not contain a valid HLL debug data signature'

  if offset \= size then
    call Die gDbgFile 'trailer offset' offset 'does not match size' size

  gDbgDataBytes = stream(gDbgFile, 'C', 'QUERY SIZE')

  return

/* end ChkHLLDebug */

/*=== ChkLxExecutable() ===*/

ChkLxExecutable: procedure expose (Globals)

  call VerboseMsg 'Validating' gExeFile

  /* MZ header length is 0x40 (64 bytes)
     new exe header follows flag at 0x18 (24) mzhdr.reloc_offset
     new exe header follows flag is is 0x40
     new exe header offset is at 0x3c (60)
     LX header is 0xb0 (176) bytes
     debug info offset is at 0x98 (152)
     debug info size is at 0x9c (156)
   */

  gLxHdrOffset = 0
  call stream gExeFile, 'C', 'OPEN READ'

  /* Read in assumed LX header (assume larger than MZ header) */
  gLxHdr = charin(gExeFile, gLxHdrOffset + 1, 176)
  if left(gLxHdr, 2) == 'MZ' then do
    /* Have MZ header, find LX header location and read in */
    flag = substr(gLxHdr, 24 + 1, 1)
    if flag \== '40'x then
      call Die gExeFile 'missing new header flags at 0x18'
    gLxHdrOffset = c2d(reverse(substr(gLxHdr, 60 + 1, 2)))
    gLxHdr = charin(gExeFile, gLxHdrOffset + 1, 176)
  end
  if left(gLxHdr, 2) \== 'LX' then
    call Die gExeFile 'is not an LX executable'

  s = substr(gLxHdr, 152 + 1, 4)
  offset = c2d(reverse(s))
  if offset \= 0 then
    call Die gExeFile 'debug info offset field is 0x' || d2x(offset), 'probably already have debug data attached'

  s = substr(gLxHdr, 156 + 1, 4)
  bytes = c2d(reverse(s))
  if bytes \= 0 then
    call Die gExeFile 'debug info bytes field is not 0', 'unexpected LX header content'

  call stream gExeFile, 'C', 'SEEK <8'
  sig = charin(gExeFile, , 4)
  s = charin(gExeFile, , 4)
  offset = c2d(reverse(s))

  if sig >= 'NB01' & sig <= 'NB09' then
    call ScanArgsUsage gExeFile 'contains unexpected attached debug data'

  call stream gExeFile, 'C', 'CLOSE'

  gDbgDataOffset = stream(gExeFile, 'C', 'QUERY SIZE')

  return

/* end ChkLxExecutable */

/*=== MakeBackup() Make backup ===*/

MakeBackup: procedure expose (Globals)

  gBackupFile = gExeFile || '-orig'
  if IsFile(gBackupFile) then do
    say
    say gExeFile 'already backed up as' gBackupFile
    cmd = 'dir' || IIF(gIs4OS2, ' /k /m /t ','') gExeFile gBackupFile
    say
    cmd
  end
  else do
    call VerboseMsg 'Backing up' gExeFile 'as' gBackupFile

    cmd = 'copy' gExeFile gBackupFile
    say
    if gDryRun then
      say cmd '(DRY-RUN)'
    else do
      cmd
      if \ IsFile(gBackupFile) then
	call Die 'Cannot backup' gExeFile 'to' gBackupFile
      say
      say gExeFile 'backed up as' gBackupFile
    end
    cmd = 'dir' || IIF(gIs4OS2, ' /k /m /t ','') gExeFile gBackupFile
    if gDryRun & \ IsFile(gBackupFile) then do
      say
      say cmd '(DRY-RUN)'
      cmd = 'dir' || IIF(gIs4OS2, ' /k /m /t ','') gExeFile
    end
    say
    cmd
  end
  return

/* end MakeBackup */

/*=== RestoreTimestamp() Restore file timestamp from backup file and show results ===*/

RestoreTimestamp: procedure expose (Globals)

  call VerboseMsg 'Restoring' gExeFile 'timestamp'

  if gDryRun & \ IsFile(gBackupFile) then
    dateTime = stream(gExeFile, 'C', 'QUERY DATETIME')
  else
    dateTime = stream(gBackupFile, 'C', 'QUERY DATETIME')
  parse var dateTime date time
  time = strip(time)
  unixTouch = DateTimeToUnixTouch(dateTime)

  /* Quotes ensure we use unix touch */
  cmd = '"touch" -t' unixTouch gExeFile	/* unix touch */
  say
  if gDryRun then
    say cmd '(DRY-RUN)'
  else do
    signal off Error
    cmd
    signal on Error
    if RC \= 0 then
      say 'touch command failed with error' RC '- cannot restore timestamp'
  end

  cmd = 'dir' || IIF(gIs4OS2, ' /k /m /t ','') gExeFile gBackupFile gDbgFile
  if gDryRun & \ IsFile(gBackupFile) then do
    say
    say cmd '(DRY-RUN)'
    cmd = 'dir' || IIF(gIs4OS2, ' /k /m /t ','') gExeFile gDbgFile
  end
  say
  cmd
  return

/* end RestoreTimestamp */

/*=== RewriteHeader() ===*/

RewriteHeader: procedure expose (Globals)

  call VerboseMsg 'Updating' gExeFile 'debug info header fields',,
		  'Debug data offset will be' gDbgDataOffset,,
		  'Debug data size will be' gDbgDataBytes

  s = reverse(d2c(gDbgDataOffset, 4))
  gLxHdr = overlay(s, gLxHdr, 152 + 1)	/* Update offset at 0x98 */

  s = reverse(d2c(gDbgDataBytes, 4))
  old = gLxHdr
  gLxHdr = overlay(s, gLxHdr, 156 + 1)	/* Update count at 0x9c */

  if old == gLxHdr then
    call Die

  if \ gDryRun then do
    call stream gExeFile, 'C', 'OPEN'
    call charout gExeFile, gLxHdr, gLxHdrOffset + 1
    call stream gExeFile, 'C', 'CLOSE'

    /* FIXME debug */
    call stream gExeFile, 'C', 'OPEN'
    s = charin(gExeFile, gLxHdrOffset + 1, length(gLxHdr))
    if s \== gLxHdr then
      call Die
    call stream gExeFile, 'C', 'CLOSE'
  end

  return

/* end RewriteHeader */

/*=== Initialize() Initialize globals ===*/

Initialize: procedure expose (Globals)
  call SetCmdName
  call LoadRexxUtil
  gEnv = 'OS2ENVIRONMENT'
  call SetTmpDir
  return

/* end Initialize */

/*=== ScanArgsInit() ScanArgs initialization exit routine ===*/

ScanArgsInit: procedure expose (Globals) cmdTail swCtl keepQuoted

  if cmdTail == '' then
    call ScanArgsHelp

  /* Preset defaults */
  gDbgLvl = 0				/* Display debug messages */
  gDryRun = 0				/* Run in test mode */
  gVerbose = 0				/* Verbose messages */
  gArgList.0 = 0			/* Reset arg count */
  gExeFile = ''
  gDbgFile = ''
  return

/* end ScanArgsInit */

/*=== ScanArgsSwitch() ScanArgs switch option exit routine ===*/

ScanArgsSwitch: procedure expose (Globals) curSw curSwArg

  select
  when curSw == 'd' then
    gDbgLvl = gDbgLvl + 1
  when curSw == 'h' | curSw == '?' then
    call ScanArgsHelp
  when curSw == 'n' then
    gDryRun = 1
  when curSw == 'v' then
    gVerbose = 1
  when curSw == 'V' then do
    say gCmdName gVersion
    exit
  end
  otherwise
    call ScanArgsUsage 'switch '''curSw''' unexpected'
  end /* select */

  return

/* end ScanArgsSwitch */

/*=== ScanArgsArg() ScanArgs argument option exit routine ===*/

ScanArgsArg: procedure expose (Globals) curArg

  if \ IsFile(curArg) then
    call ScanArgsUsage curArg 'file not found'

  if IsWild(curArg) then
    call ScanArgsUsage curArg 'cannot have wildcards'

  if gExeFile == '' then
    gExeFile = curArg
  else if gDbgFile == '' then
    gDbgFile = curArg
  else
    call ScanArgsUsage curArg 'unexpected'

  return

/* end ScanArgsArg */

/*=== ScanArgsTerm() ScanArgs scan end exit routine ===*/

ScanArgsTerm: procedure expose (Globals)

  if gDbgFile = '' then do
    if gExeFile == '' then
      call ScanArgsUsage 'required arguments missing'
      /* Try to default */
      gDbgFile = SetExt(gExeFile, '.dbg')
      if \ IsFile(gDbgFile) then
	call ScanArgsUsage 'Cannot access' gDbgFile
  end
  return

/* end ScanArgsTerm */

/*=== ScanArgsHelp() Display ScanArgs usage help exit routine ===*/

ScanArgsHelp:
  say
  say 'Attach DLL debug data to LX executable.'
  say 'Best run under 4OS2, but expected to run under CMD.EXE.'
  say 'Backs up orginal to *-orig.'
  say
  say 'Usage:' gCmdName '[-d] [-h] [-n] [-v] [-V] [-?] exefile [dbgfile]'
  say
  say '  -d       Enable debug logic, repeat for more verbosity'
  say '  -h -?    Display this message'
  say '  -n       Dry run - no files changed'
  say '  -v       Enable verbose messaging, repeat for more verbosity'
  say '  -V       Display version number and quit'
  say
  say '  exefile  LX executable, no wildcards'
  say '  dbgfile  HLL debug file, defaults to exefile.dbg, no wildcards'

  exit 255

/* end ScanArgsHelp */

/*=== ScanArgsUsage(message) Report ScanArgs usage error exit routine ===*/

ScanArgsUsage:
  parse arg msg
  say
  if msg \== '' then
    say msg
  say 'Usage:' gCmdName '[-d] [-h] [-n] [-v] [-V] [-?] exefile dbgfile'
  exit 255

/* end ScanArgsUsage */

/*==============================================================================*/
/*=== SkelRexxFunc standards - Delete unused - Move modified above this mark ===*/
/*==============================================================================*/

/*=== DateTimeToUnixTouch(d t) Convert OS/2 date/time to unix touch format ([cc]yymmddhhmm.ss) ===*/

DateTimeToUnixTouch: procedure
  parse arg d t				/* mm-dd-yy hh:mm:ss etc */
  t = strip(t)
  if t == '' then
    t = '00:00:00'			/* Default */

  /* Guess input format - convert to unix touch format [[CC]YY]MMDDhhmm[.ss] */
  if length(d) = 8 then do
    /* Assume have mm-dd-yy */
    /* translate(    str,           to,    from) */
    /*          out: yymmddhhmm.ss     in: mm-dd-yy hh:mm:ss */
    out = translate('GHABDEJKMN.PQ', d t, 'ABCDEFGHIJKLMNOPQ')
  end
  else if pos('-', d) = 3 then do
    /* Assume have mm-dd-yyyy */
    /* translate(    str,           to,    from) */
    /*          out: yyyymmddhhmm.ss     in: mm-dd-yyyy hh:mm:ss */
    out = translate('GHIJABDELMOP.RS', d t, 'ABCDEFGHIJKLMNOPQRS')
  end
  else do
    /* Assume have yyyy-mm-dd */
    /* translate(    str,           to,    from) */
    /*          out: yyyymmddhhmm.ss     in: yyyy-mm-dd hh:mm:ss */
    out = translate('ABCDFGIJLMOP.RS', d t, 'ABCDEFGHIJKLMNOPQRS')
  end
  return out

/* end DateTimeToUnixTouch */

/*=== IIF(bool, true, false) Inline if ===*/

IIF: procedure
  parse arg e, t, f
  if e then
    return t
  else
    return f

/* end IIF */

/*=== Is4OS2() Return true if running under 4OS2 else false ===*/

Is4OS2: procedure expose (Globals)
  /* Keep Is4OS2 and Chk4OS2 in sync */
  old = value('_X',, gEnv)		/* In case in use */
  /* 4OS2 sets _X to 0, cmd.exe sets x to @eval[0], rxd leaves X unchanged */
  '@set _X=%@eval[0]'
  new = value('_X',, gEnv)
  '@set _X=' || old			/* Restore */
  yes = new = 0 | old == new		/* Assume 4OS2 if running under rxd */
  return yes				/* if running under 4OS2 */

/* end Is4OS2 */

/*=== IsFile(file) return true if arg is file and file exists ===*/

IsFile: procedure expose (Globals)
  parse arg file
  if file == '' then
    yes = 0
  else do
    /* '.' and '..' returns files in '.' or '..' - so avoid false positives */
    call SysFileTree file, 'files', 'F'
    if RESULT \= 0 then
      call Die 'SysFileTree' file 'failed'
    /* Assume caller knows if arg contains wildcards */
    yes = file \== '.' & file \== '..' & files.0 \= 0
  end
  return yes

/* end IsFile */

/*=== IsWild(pathName) return true if path name contains wild card character ===*/

IsWild: procedure
  return verify(arg(1),'*?', 'M') \= 0

/* end IsWild */

/*=== DbgMsg([minLvl, ]message,...) Optionally write multi-line message to STDERR ===*/

/**
 * Write message if gDbgLvl >= minLvl
 * @param minLvl defaults to 1 if omitted
 * @returns true if message written
 */

DbgMsg: procedure expose (Globals)
  minLvl = arg(1)
  if datatype(minLvl, 'W') then
    start = 2
  else do
    minLvl = 1
    start = 1
  end
  if gDbgLvl >= minLvl then do
    do i = start to arg()
      msg = arg(i)
      if msg \== '' then
	msg = ' *' msg
      call lineout 'STDERR', msg
    end
  end
  return gDbgLvl >= minLvl

/* end DbgMsg */

/*=== MakePath(pathparts,...) Make path name from parts ===*/

MakePath: procedure

  /* All parts optional - code guesses what caller means.
     If last arg begins with dot, it is assumed to be a file extension.
     To avoid this behavior, pass empty arg as last arg.
     Empty args are ignored.
     Automatically converts unix slashes to dos slashes.
     If 1st arg is drive letter, it must have trailing colon.
   */

  argCnt = arg()

  path = ''

  do argNum = 1 to argCnt
    s = arg(argNum)
    s = translate(s, '\', '/')		/* Ensure DOS */
    if s == '' & argNum = argCnt then
      iterate				/* Ignore nul last arg */
    if argNum = 1 then
      path = s
    else do
      lead = left(s, 1)
      tail = right(path, 1)
      if tail == ':' & argNum = 2 then
	path = path || s		/* Append path part to drive spec */
      else if lead == '.' & argNum = argCnt then
	path = path || s		/* Assume extension */
      else if tail == '\' & lead == '\' then
	path = path || substr(s, 2)	/* Drop extra backslash */
      else if path \== '' & tail \== '\' & lead \== '\' then
	path = path || '\' || s		/* Ensure have backslash */
      else
	path = path || s
    end
  end /* for */

  return path

/* end MakePath */

/*=== SetExt(file, ext) Return file name with extension replaced ===*/

SetExt: procedure

  /* Requires MakePath */
  /* Assume called with sane arguments, but avoid returning garbage */
  parse arg file, ext
  if file \== '' & ext \== '' then do
    if left(ext, 1) \== '.' then
      ext = '.' || ext
    iSlash = lastpos('\', file)
    iDot = lastpos('.', file)
    /* Avoid .name */
    if iDot > 1 & iSlash + 1 \= iDot then
      file = left(file, iDot - 1)	/* Strip extension */
    file = MakePath(file, ext)		/* File has no extension */
  end

  return file

/* end SetExt */

/*=== VerboseMsg([level, ]message,...) Write multi-line message to STDERR if verbose ===*/

VerboseMsg: procedure expose (Globals)
  level = arg(1)
  if datatype(level, 'W') then
    start = 2
  else do
    level = 1
    start = 1
  end
   if level <= gVerbose then do
    do i = start to arg()
      call lineout 'STDERR', arg(i)
    end
  end
  return

/* end VerboseMsg */

/*==========================================================================*/
/*=== SkelRexx standards - Delete unused - Move modified above this mark ===*/
/*==========================================================================*/

/*=== Die([message,...]) Write multi-line message to STDERR and die ===*/

Die:
  call lineout 'STDERR', ''
  do i = 1 to arg()
    call lineout 'STDERR', arg(i)
  end
  call lineout 'STDERR', gCmdName 'aborting at line' SIGL || '.'
  call beep 200, 300
  call SysSleep 2
  exit 254

/* end Die */

/*=== Error() Set gErrCondition; report to STDOUT; trace and exit or return if called ===*/

Error:
  say
  parse source . . cmd
  gErrCondition = condition('C')
  say gErrCondition 'signaled at line' SIGL 'of' cmd || '.'
  if condition('D') \= '' then
    say 'REXX reason =' condition('D') || '.'
  if gErrCondition == 'SYNTAX' & symbol('RC') == 'VAR' then
    say 'REXX error =' RC '-' errortext(RC) || '.'
  else if symbol('RC') == 'VAR' then
    say 'RC =' RC || '.'
  say 'Source =' sourceline(SIGL)

  if condition('I') \== 'CALL' | gErrCondition == 'NOVALUE' | gErrCondition == 'SYNTAX' then do
    trace '?A'
    say 'Enter REXX commands to debug failure.  Press enter to exit script.'
    nop
    if symbol('RC') \== 'VAR' then
      RC = 255
    exit RC
  end

  return

/* end Error */

/*=== Halt() Report HALT condition to STDOUT and exit ===*/

Halt:
  say
  parse source . . cmd
  say condition('C') 'signaled at' cmd 'line' SIGL || '.'
  say 'Source =' sourceline(SIGL)
  say 'Sleeping for 2 seconds...'
  call SysSleep 2
  exit 253

/* end Halt */

/*=== LoadRexxUtil() Load RexxUtil functions ===*/

LoadRexxUtil:
  if RxFuncQuery('SysLoadFuncs') then do
    call RxFuncAdd 'SysLoadFuncs', 'REXXUTIL', 'SysLoadFuncs'
    if RESULT then
      call Die 'Cannot load SysLoadFuncs.'
    call SysLoadFuncs
  end
  return

/* end LoadRexxUtil */

/*=== ScanArgs(cmdLine) Scan command line ===*/

ScanArgs: procedure expose (Globals)

  /* Calls user exits to process arguments and switches */

  parse arg cmdTail
  cmdTail = strip(cmdTail)

  call ScanArgsInit

  /* Ensure optional settings initialized */
  if symbol('SWCTL') \== 'VAR' then
    swCtl = ''				/* Switches that take args, append ? if optional */
  if symbol('KEEPQUOTED') \== 'VAR' then
    keepQuoted = 0			/* Set to 1 to keep arguments quoted */

  /* Scan */
  curArg = ''				/* Current arg string */
  curSwList = ''			/* Current switch list */
  /* curSwArg = '' */			/* Current switch argument, if needed */
  noMoreSw = 0				/* End of switches */

  do while cmdTail \== '' | curArg \== '' | curSwList \== ''

    /* If arg buffer empty, refill */
    if curArg == '' then do
      qChar = left(cmdTail, 1)		/* Remember quote */
      if \ verify(qChar,'''"', 'M') then
	parse var cmdTail curArg cmdTail	/* Not quoted */
      else do
	/* Arg is quoted */
	curArg = ''
	do forever
	  /* Parse dropping quotes */
	  parse var cmdTail (qChar)quotedPart(qChar) cmdTail
	  curArg = curArg || quotedPart
	  /* Check for escaped quote within quoted string (i.e. "" or '') */
	  if left(cmdTail, 1) \== qChar then do
	    cmdTail = strip(cmdTail)	/* Strip leading whitespace */
	    leave			/* Done with this quoted arg */
	  end
	  curArg = curArg || qChar	/* Append quote */
	  if keepQuoted then
	    curArg = curArg || qChar	/* Append escaped quote */
	  parse var cmdTail (qChar) cmdTail	/* Strip quote */
	end /* do forever */
	if keepQuoted then
	  curArg = qChar || curArg || qChar	/* requote */
      end /* if quoted */
    end /* if curArg empty */

    /* If switch buffer empty, refill */
    if curSwList == '' & \ noMoreSw then do
      if left(curArg, 1) == '-' & curArg \== '-' then do
	if curArg == '--' then
	  noMoreSw = 1
	else
	  curSwList = substr(curArg, 2)	/* Remember switch string */
	curArg = ''			/* Mark empty */
	iterate				/* Refill arg buffer */
      end /* if switch */
    end /* if curSwList empty */

    /* If switch in progress */
    if curSwList \== '' then do
      curSw = left(curSwList, 1)	/* Next switch */
      curSwList = substr(curSwList, 2)	/* Drop from pending */
      /* Check switch allows argument, avoid matching ? */
      if pos(curSw, translate(swCtl,,'?')) \= 0 then do
	if curSwList \== '' then do
	  curSwArg = curSwList		/* Use rest of switch string for switch argument */
	  curSwList = ''
	end
	else if curArg \== '' & left(curArg, 1) \== '-' then do
	  curSwArg = curArg		/* Arg string is switch argument */
	  curArg = ''			/* Mark arg string empty */
	end
	else if pos(curSw'?', swCtl) = 0 then
	  call ScanArgsUsage 'Switch "-' || curSw || '" requires an argument'
	else
	  curSwArg = ''			/* Optional arg omitted */
      end

      call ScanArgsSwitch		/* Passing curSw and curSwArg */
      drop curSwArg			/* Must be used by now */
    end /* if switch */

    /* If arg */
    else if curArg \== '' then do
      noMoreSw = 1
      call ScanArgsArg			/* Passing curArg */
      curArg = ''
    end

  end /* while not done */

  call ScanArgsTerm

  return

/* end ScanArgs */

/*=== SetCmdName() Set gCmdName to short script name ===*/

SetCmdName: procedure expose (Globals)
  parse source . . cmd
  cmd = filespec('N', cmd)		/* Chop path */
  c = lastpos('.', cmd)
  if c > 1 then
    cmd = left(cmd, c - 1)		/* Chop extension */
  gCmdName = translate(cmd, xrange('a', 'z'), xrange('A', 'Z'))	/* Lowercase */
  return

/* end SetCmdName */

/*=== SetTmpDir() Set gTmpDir to %TMP with trailing backslash or empty string ===*/

SetTmpDir: procedure expose (Globals)
  s = value('TMP',,gEnv)
  if s \= '' & right(s, 1) \= ':' & right(s, 1) \== '\' then
    s = s'\'				/* Stuff backslash */
  gTmpDir = s
  return

/* end SetTmpDir */

/* eof */
