/*
 * readbdf.c - read BDF font file
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "font.h"

/*
 * File access routines for BDF font file
 */

#define MAXLIN  512

/*
 * string handling supports
 *      prefix      checks prefix of a line
 *      streq       checks two strings
 *      name_eq     checks name of Prop.
 *      isinteger   checks if string is a valid integer
 *      hexbyte     make a byte from the first two hex characters in s
 */

static BOOL prefix(char *buf, char *str)
{
    return strncmp(buf, str, strlen(str))? FALSE : TRUE;
}

static BOOL streq(char *a, char *b)
{
    return strcmp(a, b)? FALSE : TRUE;
}

static BOOL name_eq(FontPropPtr pfp, char *s)
{
    return(streq(pfp->name, s)) ;
}

static BOOL isinteger(char *str)
{
    char c;

    c = *str++;
    if( !(isdigit(c) || c=='-' || c=='+') )
	return(FALSE);

    while(c = *str++)
	if( !isdigit(c) )
	    return(FALSE);

    return(TRUE);
}

static unsigned char hexbyte(char *s)
{
    unsigned char           b = 0;
    register unsigned char  c;
    int i;

    for (i=2; i; i--) {
	c = *s++;
	if ((c >= '0') && (c <= '9'))
	    b = (b<<4) + (c - '0');
	else if ((c >= 'A') && (c <= 'F'))
	    b = (b<<4) + 10 + (c - 'A');
	else if ((c >= 'a') && (c <= 'f'))
	    b = (b<<4) + 10 + (c - 'a');
	else
	    return 0;
    } 
    return b;
}

/*
 * get a effective (not COMMENT) line
 */

static char *getline(FILE *fp, char *buf)
{
    char    *p ;

    buf = fgets(buf, MAXLIN, fp) ;
    while (buf) {
        int len = strlen(buf) ;
	
	if ((len == 0) || prefix(buf, "COMMENT")) {
	    buf = fgets(buf, MAXLIN, fp) ;
	} else break ;
    }
    if (buf) {
        for (p = buf ; *p != '\0' ; p++) {
            if (*p == '\r' || *p == '\n') {
	        *p = '\0' ;
            }
        }
    }
    return(buf) ;
}

/*
 * Read BDF Font File and build Font Data Structure
 */

FontPtr ReadBdfFont(FILE *fp)
{
    FontPtr     pFont ;     /* Generating Font Data */
    int         nprops ;
    int         nchars ;
    float       pointSize  ;
    int         xRes, yRes ;
    unsigned    attributes;
    int		digitWidths = 0, digitCount = 0, ex = 0;
    int		char_row, char_col;

    char    fontName[MAXLIN]  ;
    char    linebuf[MAXLIN]   ;
    char    namebuf[MAXLIN]   ;
    char    secondbuf[MAXLIN] ;
    char    thirdbuf[MAXLIN]  ;

    /*
     * Create Font Entry
     */

    if ((pFont = CreateFontInfo()) == NULL) {
        return(NULL) ;
    }

    /*
     * Start Reading BDF File
     */

    getline(fp, linebuf) ;

    /*
     * "STARTFONT" - start of font data, check version
     */

    if ((sscanf(linebuf, "STARTFONT %s", namebuf) != 1) ||
	    !streq(namebuf, "2.1")) {
	FreeFontInfo(pFont) ;
	return(NULL) ;              /* bad STARTFONT */
    }
    getline(fp, linebuf) ;

    /*
     * "FONT" - font name
     */

    if (sscanf(linebuf, "FONT %[^\n]", fontName) != 1) {
        FreeFontInfo(pFont) ;
	return(NULL) ;              /* bad FONT */
    }
    if (SetFontName(pFont, fontName) == NULL) {
        FreeFontInfo(pFont) ;
	return(NULL) ;
    }
    
    getline(fp, linebuf);

    /*
     * "SIZE", point size and resolution
     */

    if (!prefix(linebuf, "SIZE")) {
        FreeFontInfo(pFont) ;
	return(NULL) ;              /* missing SIZE */
    }
    if ((sscanf(linebuf, "SIZE %f%d%d", &pointSize, &xRes, &yRes) != 3)) {
        FreeFontInfo(pFont) ;
	return(NULL) ;              /* bad SIZE */
    }
    if ((pointSize < (float) 1.0) || (xRes < 1) || (yRes < 1)) {
        FreeFontInfo(pFont) ;
	return(NULL) ;              /* SIZE value must be > 0 */
    }
    if (xRes != yRes) {
        FreeFontInfo(pFont) ;
	return(NULL) ;              /* x and y resolution must be equal */
    }
    getline(fp, linebuf);
    
    /*
     * FONTBOUNDINGBOX
     */

    if (!prefix(linebuf, "FONTBOUNDINGBOX")) {
        FreeFontInfo(pFont) ;
	return(NULL) ;              /* missing 'FONTBOUNDINGBOX' */
    }
    getline(fp, linebuf) ;

    /*
     * STARTPROPERTIES - Start of Font Properties
     */

    if (!prefix(linebuf, "STARTPROPERTIES")) {
        FreeFontInfo(pFont) ;
	return(NULL) ;              /* missing 'STARTPROPERTY' */
    }
    if (sscanf(linebuf, "STARTPROPERTIES %d", &nprops) != 1) {
        FreeFontInfo(pFont) ;
        return(NULL) ;              /* bad 'STARTPROPERTIES' */
    }
    AllocFontProp(pFont, nprops) ;
    getline(fp, linebuf);

    while((nprops-- > 0) && !prefix(linebuf, "ENDPROPERTIES")) {

        switch (sscanf(linebuf, "%s%s%s", namebuf, secondbuf, thirdbuf) ) {

        case 1: /* missing required parameter value */
            FreeFontInfo(pFont) ;
            return(NULL) ;          /* missing parameter value */
	    break;

	case 2:
	    /*
	     * Possibilites include:
	     * valid quoted string with no white space
	     * valid integer value 
	     * invalid value
	     */
	    if( secondbuf[0] == '"'){
	        AddStrProp(pFont, namebuf, (linebuf+strlen(namebuf))) ;
	    } else if( isinteger(secondbuf) ){
	        AddIntProp(pFont, namebuf, atol(secondbuf)) ;
	    } else {
	        FreeFontInfo(pFont) ;
	        return(NULL) ;          /* invalid parameter value */
	    }
	    break ;

	case 3:
	    /* 
	     * Possibilites include:
	     * valid quoted string with some white space
	     * invalid value (reject even if second string is integer)
	     */
	    if( secondbuf[0] == '"'){
	        AddStrProp(pFont, namebuf, (linebuf+strlen(namebuf))) ;
	    } else {
	        FreeFontInfo(pFont) ;
		return(NULL) ;          /* invalid parameter value */
	    }
	    break ;
	}
	getline(fp, linebuf);
    }
    if (!prefix(linebuf, "ENDPROPERTIES")) {
        FreeFontInfo(pFont) ;
        return(NULL) ;                  /* missing 'ENDPROPERTIES' */
    }

    /*
     * Check special properties
     */

    if (GetFontProp(pFont, "FONT_ASCENT") == NULL) {
        FreeFontInfo(pFont) ;
	return(NULL) ;                  /* must have 'FONT_ASCENT'  */
    }
    if (GetFontProp(pFont, "FONT_DESCENT") == NULL) {
        FreeFontInfo(pFont) ;
	return(NULL) ;                  /* must have 'FONT_DESCENT'  */
    }
    if (GetFontProp(pFont, "FONT") == NULL) {
        AddStrProp(pFont, "FONT", fontName) ;
    }
    if (GetFontProp(pFont, "POINT_SIZE") == NULL) {
        AddIntProp(pFont, "POINT_SIZE", (long) (pointSize*10.0)) ;
    }
    if (GetFontProp(pFont, "RESOLUTION") == NULL) {
        AddIntProp(pFont, "RESOLUTION", (long) ((xRes*100.0)/72.27)) ;
    }
    getline(fp, linebuf);

    /*
     * CHARS - number of CHARs
     */

    if (sscanf(linebuf, "CHARS %d", &nchars) != 1) {
        FreeFontInfo(pFont) ;
	return(NULL) ;              /* bad 'CHARS' */
    }
    if (nchars < 1) {
        FreeFontInfo(pFont) ;
	return(NULL) ;              /* invalid number of CHARS */
    }
    getline(fp, linebuf);

    /*
     * Character's Metrics and Bitmap (GLYPH)
     */

    while ((nchars-- > 0) && prefix(linebuf, "STARTCHAR")) {
        int	t;
	int	ix;	/* counts bytes in a glyph */
	int	wx;	/* x component of width */
	int	wy;	/* y component of width */
	int	bw;	/* bounding-box width */
	int	bh;	/* bounding-box height */
	int	bl;	/* bounding-box left */
	int	bb;	/* bounding-box bottom */
	int	enc, enc2;	/* encoding */
	char	*p;	/* temp pointer into linebuf */
	int	bytesperrow, row, hexperrow, perrow, badbits;
	char	charName[100];
        int     bytesGlUsed ;
        char    *pGl ;

	if (sscanf(linebuf, "STARTCHAR %s", charName) != 1) {
	    FreeFontInfo(pFont) ;
	    return(NULL) ;          /* bad character name */
	}
	getline(fp, linebuf);

	if ((t=sscanf(linebuf, "ENCODING %d %d", &enc, &enc2)) < 1) {
	    FreeFontInfo(pFont) ;
	    return(NULL) ;          /* bad 'ENCODING' */
	}
	if ((enc < -1) || ((t == 2) && (enc2 < -1))) {
	    FreeFontInfo(pFont) ;
	    return(NULL) ;          /* bad ENCODING value */
	}
	if (t == 2 && enc == -1)
	    enc = enc2;
	if (enc == -1) {
	    do {
	    	char *s = getline(fp, linebuf);
		if (!s) {
		    FreeFontInfo(pFont) ;
		    return(NULL) ;  /* Unexpected EOF */
		}
	    } while (!prefix(linebuf, "ENDCHAR"));
	    getline(fp, linebuf);
	    continue;
	}
	char_row = (enc >> 8) & 0xFF;
	char_col = enc & 0xFF;
	
	getline(fp, linebuf);
	if (sscanf( linebuf, "SWIDTH %d %d", &wx, &wy) != 2) {
	    FreeFontInfo(pFont) ;
	    return(NULL) ;          /* bad 'SWIDTH' */
	}
	if (wy != 0) {
	    FreeFontInfo(pFont) ;
	    return(NULL) ;          /* SWIDTH y value must be zero */
        }
	getline(fp, linebuf);
	if (sscanf( linebuf, "DWIDTH %d %d", &wx, &wy) != 2) {
	    FreeFontInfo(pFont) ;
	    return(NULL) ;          /* bad 'DWIDTH' */
	}
	if (wy != 0) {
	    FreeFontInfo(pFont) ;
	    return(NULL) ;          /* DWIDTH y value must be zero */
	}
	getline(fp, linebuf);

	if (sscanf(linebuf, "BBX %d %d %d %d", &bw, &bh, &bl, &bb) != 4) {
	    FreeFontInfo(pFont) ;
	    return(NULL) ;          /* bad 'BBX' */
	}
	if ((bh < 0) || (bw < 0)) {
	    FreeFontInfo(pFont) ;
	    return(NULL) ;          /* negative sized bitmap */
	}
	getline(fp, linebuf);

	if (prefix(linebuf, "ATTRIBUTES")) {
	    for (p = linebuf + strlen("ATTRIBUTES ");
		(*p == ' ') || (*p == '\t');
		p ++)
		/* empty for loop */ ;
	    attributes = hexbyte(p)<< 8 + hexbyte(p+2);
	    getline(fp, linebuf); /* set up for BITMAP which follows */
	}
	else
	    attributes = 0;

	if (!prefix(linebuf, "BITMAP")) {
	    FreeFontInfo(pFont) ;
	    return(NULL) ;          /* missing 'BITMAP' */
	}

	/* collect data for generated properties */
	if ((strlen(charName) == 1)){
	    if ((charName[0] >='0') && (charName[0] <= '9')) {
		digitWidths += wx;
		digitCount++;
	    } else if (charName[0] == 'x') {
	        ex = (bh+bb)<=0? bh : bh+bb ;
	    }
	}
	
	pGl = AllocCharInfo(pFont, char_row, char_col,
	            bl, bl+bw, bh+bb, -bb, wx, attributes) ;
        if (pGl == NULL) {
	    FreeFontInfo(pFont) ;
	    return(NULL) ;          /* not enough memory */
	}

        bytesGlUsed = 0;            /* index to pGl area */
	badbits = 0;
	bytesperrow = (bw + 7) >> 3 ;
	hexperrow = (bw + 7) >> 3;
	if (hexperrow == 0) hexperrow = 1;

	for (row=0; row < bh; row++) {
	    getline(fp, linebuf);
	    p = linebuf;
	    t = strlen(p);
	    t >>= 1;

	    perrow = min(hexperrow, t);
	    for ( ix=0; ix < perrow; ix++, p+=2, bytesGlUsed++)
	    {
	        pGl[bytesGlUsed] = hexbyte(p);
	    }
	    if (perrow && (hexperrow <= t) && (bw & 7) &&
		(ix = (pGl[bytesGlUsed-1] & (0xff >> (bw & 7))))) {
		pGl[bytesGlUsed-1] &= ~ix;
	    }
	    for ( ix=perrow; ix < bytesperrow; ix++, bytesGlUsed++)
	    {
	        pGl[bytesGlUsed] = 0;
	    }
	}
	getline(fp, linebuf);

	if (!prefix(linebuf, "ENDCHAR")) {
	    FreeFontInfo(pFont) ;
            return(NULL) ;              /* missing 'ENDCHAR' */
	}
	getline(fp, linebuf);		/* get STARTCHAR or ENDFONT */
    }

    if (nchars != -1) {
        FreeFontInfo(pFont) ;
        return(NULL) ;                  /* too few characters */
    }
    if (prefix(linebuf, "STARTCHAR")) {
        FreeFontInfo(pFont) ;
	return(NULL) ;                  /* more characters than specified */
    }
    if (!prefix(linebuf, "ENDFONT")) {
        FreeFontInfo(pFont) ;
        return(NULL) ;                  /* missing 'ENDFONT'  */
    }

    /*
     * XFONT olny requires min/maxboundings
     */
     
    ComputeBoundings(pFont) ;
    
    /*
     * Generate properties
     */

    if (GetFontProp(pFont, "X_HEIGHT") == NULL) {
        AddIntProp(pFont, "X_HEiGHT",
	        (long) (ex ? ex : pFont->minbound.ascent)) ;
    }
    if (GetFontProp(pFont, "QUAD_WIDTH") == NULL) {
        AddIntProp(pFont, "QUAD_WIDTH",
	(digitCount ? (long) ((float)digitWidths/(float)digitCount) :
	(pFont->minbound.characterWidth+pFont->maxbound.characterWidth)/2));
    }
    if (GetFontProp(pFont, "WEIGHT") == NULL) {
        AddIntProp(pFont, "WEIGHT", ComputeWeight(pFont)) ;
    }
    
    return(pFont) ;
}
