/*
 * qpager.c - Entry of the Quick Pager
 */

#include <stdio.h>
#include <stdlib.h>

#define INCL_KBD
#include <os2.h>

#include "qpager.h"
#include "screen.h"

PSZ     progName = "qp" ;

/*
 * Commands for Quick Pager
 */

#define CMD_UNDEF       0xffff      /* undefined command    */
#define CMD_QUIT        0x0000      /* Quit Quick Pager     */
#define CMD_REDRAW      0x0001      /* Redraw Current Page  */
#define CMD_AUTO        0x0002      /* Automatic Command    */
#define CMD_HELP        0x0003      /* Help                 */

#define CMD_FILETOP     0x0010      /* Show First File      */
#define CMD_FILEBOT     0x0011      /* Show Last  File      */
#define CMD_FILEPRV     0x0012      /* Show Prev. File      */
#define CMD_FILENXT     0x0013      /* Show Next  File      */

#define CMD_TEXTTOP     0x0100      /* Move to Top of Text  */
#define CMD_TEXTBOT     0x0101      /* Move to Bot of Text  */

#define CMD_PAGEUP      0x0110      /* Page Above           */
#define CMD_PAGEDN      0x0111      /* Page Below           */
#define CMD_LINEUP      0x0112      /* Scroll Line Up       */
#define CMD_LINEDN      0x0113      /* Scroll Line Down     */
#define CMD_HALFUP      0x0114      /* Scroll Up   Half     */
#define CMD_HALFDN      0x0115      /* Scroll Down Half     */

#define CMD_LOOKF       0x0120      /* Search Forward       */
#define CMD_LOOKB       0x0121      /* Search Backward      */
#define CMD_GOTO        0x0122      /* Goto Line            */
#define CMD_SAVES       0x0130      /* Save Screen          */
#define CMD_SAVEF       0x0131      /* Save File            */

/*
 * getAutomate - automated command execution
 *      at EOF          CMD_PAGEDN
 *      still reading   CMD_REDRAW
 *      not last file   CMD_FILENXT
 *      last file       CMD_QUIT
 */

static  int getAutomate(FILEPTR fptr)
{
    if (dispFull(fptr)) {
        return CMD_PAGEDN ;
    }
    if (fptr->stat == IO_READ) {
        return CMD_REDRAW ;
    }
    if (filesNext(fptr) != NULL) {
        return CMD_FILENXT ;
    } else {
        return CMD_QUIT ;
    }
}

/*
 * getCommand - get Command from Keyboard
 */

static  int     getCommand(FILEPTR fptr)
{
    KBDKEYINFO  keyin  ;
    USHORT      ch, sc ;
    int         cmd    ;

    if (fptr->stat != IO_READ) {
        KbdCharIn(&keyin, 0, 0) ;
        ch = keyin.chChar ;
        sc = keyin.chScan ;
    } else {
        while (1) {
	    KbdCharIn(&keyin, 1, 0) ;
	    if (keyin.fbStatus != 0) {
                ch = keyin.chChar ;
                sc = keyin.chScan ;
	        break ;
            }
	    DosSleep(100)  ;
	    dispPage(fptr) ;	/* dispStat or dispPage ? */
	    if (fptr->stat != IO_READ) {
	        return CMD_UNDEF ;
            }
        }
    }

    /*
     * Determine what to do
     */

    if (ch == ':') {        /* File Command Prefix  */
        KbdCharIn(&keyin, 0, 0) ;
	switch(keyin.chChar) {
    	    case '0' : cmd = CMD_FILETOP ; break ;
	    case '$' : cmd = CMD_FILEBOT ; break ;
	    case 'p' : cmd = CMD_FILEPRV ; break ;
	    case 'P' : cmd = CMD_FILEPRV ; break ;
	    case 'n' : cmd = CMD_FILENXT ; break ;
	    case 'N' : cmd = CMD_FILENXT ; break ;
	    case '?' : cmd = CMD_HELP    ; break ;
	    case 'q' : cmd = CMD_QUIT    ; break ;
	    case 'Q' : cmd = CMD_QUIT    ; break ;
	    default  : cmd = CMD_UNDEF   ; break ;
        }
    } else if ((sc != 0) && (ch == 0x00 || ch == 0xe0)) {   /* EXT KEY  */
        switch (sc) {
	    case 0x47 : cmd = CMD_TEXTTOP ; break ;     /* HOME */
	    case 0x4f : cmd = CMD_TEXTBOT ; break ;     /* END  */
	    case 0x49 : cmd = CMD_PAGEUP  ; break ;     /* PgUp */
	    case 0x51 : cmd = CMD_PAGEDN  ; break ;     /* PgDn */
	    case 0x48 : cmd = CMD_LINEUP  ; break ;     /* UP   */
	    case 0x50 : cmd = CMD_LINEDN  ; break ;     /* DN   */
	    case 0x8d : cmd = CMD_HALFUP  ; break ;     /* C-UP */
	    case 0x91 : cmd = CMD_HALFDN  ; break ;     /* C-DN */
	    case 0x4b : cmd = CMD_FILEPRV ; break ;     /* LF   */
	    case 0x4d : cmd = CMD_FILENXT ; break ;     /* RT   */
	    case 0x52 : cmd = CMD_FILEPRV ; break ;     /* INS  */
	    case 0x53 : cmd = CMD_FILENXT ; break ;     /* DEL  */
	    case 0x3b : cmd = CMD_HELP    ; break ;     /* F1   */
	    default   : cmd = CMD_UNDEF   ; break ; 
        }
    } else {                                           /* NORMAL KEY   */
        switch(ch) {
	    case '\r' : cmd = CMD_AUTO   ; break ;
	    case ' '  : cmd = CMD_AUTO   ; break ;
	    case 'j'  : cmd = CMD_LINEDN ; break ;
	    case 'J'  : cmd = CMD_LINEDN ; break ;
	    case 'k'  : cmd = CMD_LINEUP ; break ;
	    case 'K'  : cmd = CMD_LINEUP ; break ;
	    case 'f'  : cmd = CMD_PAGEDN ; break ;
	    case 'F'  : cmd = CMD_PAGEDN ; break ;
	    case 'b'  : cmd = CMD_PAGEUP ; break ;
	    case 'B'  : cmd = CMD_PAGEUP ; break ;
	    case 'd'  : cmd = CMD_HALFDN ; break ;
	    case 'D'  : cmd = CMD_HALFDN ; break ;
	    case 'u'  : cmd = CMD_HALFUP ; break ;
	    case 'U'  : cmd = CMD_HALFUP ; break ;
	    case 'q'  : cmd = CMD_QUIT   ; break ;
	    case 'Q'  : cmd = CMD_QUIT   ; break ;
	    case 'x'  : cmd = CMD_QUIT   ; break ;
	    case 'X'  : cmd = CMD_QUIT   ; break ;
	    case 'e'  : cmd = CMD_QUIT   ; break ;
	    case 'E'  : cmd = CMD_QUIT   ; break ;
            case '/'  : cmd = CMD_LOOKF  ; break ;
	    case '?'  : cmd = CMD_LOOKB  ; break ;
	    case 'g'  : cmd = CMD_GOTO   ; break ;
	    case 'G'  : cmd = CMD_GOTO   ; break ;
	    case 's'  : cmd = CMD_SAVES  ; break ;
	    case 'S'  : cmd = CMD_SAVEF  ; break ;
	    default   : cmd = CMD_UNDEF  ; break ;
        }
    }
    if (cmd == CMD_AUTO) {
        cmd = getAutomate(fptr) ;
    }
    return cmd ;
}

/*
 * doPager - Execute Commands
 */

static  void    doPager(FILEPTR first)
{
    int     cmd ;
    FILEPTR curFile, nxtFile ;

    if ((curFile = first)  == NULL) {
        return ;
    }
    if (curFile->stat == IO_INIT) {
        filesFill(curFile) ;
    }
    cmdTextTop(curFile) ;

    while ((cmd = getCommand(curFile)) != CMD_QUIT) {
        switch(cmd) {
        case CMD_REDRAW  :
	    nxtFile = cmdDispAgain(curFile) ;
	    break ;
	case CMD_HELP :
	    nxtFile = cmdHelp(curFile) ;
	    break ;
        case CMD_FILETOP :
	    nxtFile = cmdFileTop(curFile) ;
	    break ;
        case CMD_FILEBOT :
	    nxtFile = cmdFileBot(curFile) ;
	    break ;
        case CMD_FILEPRV :
	    nxtFile = cmdFilePrv(curFile) ;
	    break ;
        case CMD_FILENXT : 
            nxtFile = cmdFileNxt(curFile) ;
	    break ;
        case CMD_TEXTTOP :
	    nxtFile = cmdTextTop(curFile) ;
	    break ;
        case CMD_TEXTBOT :
	    nxtFile = cmdTextBot(curFile) ;
	    break ;
        case CMD_PAGEUP  :
	    nxtFile = cmdPageUp(curFile) ;
	    break ;
        case CMD_PAGEDN  :
	    nxtFile = cmdPageDn(curFile) ;
	    break ;
        case CMD_LINEUP  :
	    nxtFile = cmdLineUp(curFile) ;
	    break ;
        case CMD_LINEDN  :
	    nxtFile = cmdLineDn(curFile) ;
            break ;
        case CMD_HALFUP  :
	    nxtFile = cmdHalfUp(curFile) ;
	    break ;
        case CMD_HALFDN  :
	    nxtFile = cmdHalfDn(curFile) ;
	    break ;
        case CMD_LOOKF :
	    nxtFile = cmdSearchForw(curFile) ;
	    break ;
        case CMD_LOOKB :
	    nxtFile = cmdSearchBack(curFile) ;
	    break ;
        case CMD_GOTO  :
	    nxtFile = cmdGotoLine(curFile) ;
	    break ;
        case CMD_SAVES :
	    nxtFile = cmdSaveDisp(curFile) ;
	    break ;
        case CMD_SAVEF :
	    nxtFile = cmdSaveFile(curFile) ;
	    break ;
        default :
	    nxtFile = NULL ;
	    break ;
        }
	if (nxtFile != NULL) {
	    curFile = nxtFile ;
        }
        if (curFile->stat == IO_INIT) {
	    filesFill(curFile) ;
	}
	if (curFile->curln == NULL) {
            cmdTextTop(curFile) ;
        }
    }
}

/*
 * Program Start Here
 */

int     main(int ac, char *av[])
{
    int     i     ;
    FILEPTR first ;
    ULONG   htype, hflag ;
    
    _wildcard(&ac, &av) ;
    
    progName = av[0] ;

    if (filesInit() != 0) {
        fprintf(stderr, "%s : failed to initialize\n", progName) ;
	exit(1) ;
    }
    if (ac > 1) {
        for (i = 1 ; i < ac ; i++) {
	    filesAppend(av[i], NULL) ;
        }
    } else {
        DosQueryHType(0, &htype, &hflag) ;
	if ((htype & 0x07) != 2) {
	    fprintf(stderr, "%s : no redirection\n", progName) ;
	    exit(1) ;
        }
	filesAppend("-stdin-", stdin) ;  
    }
    if ((first = filesTop()) == NULL) {
        fprintf(stderr, "%s : no files to view\n", progName) ;
	exit(1) ;
    }
    if (scrInit("QPATTR") != 0) {
        fprintf(stderr, "%s : failed to setup screen\n", progName) ;
	exit(1) ;
    }
    
    doPager(first) ;
    
    scrDone()   ;
    filesDone() ;

    return 0 ;
}
