/*
 * lines.c - Operation on Lines in a File
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <os2.h>

#include "qpager.h"

/*
 * Both Display/Reader threads access Lines, use MUTEX
 */

#define LOCK(fptr)          DosRequestMutexSem(fptr->mutex, -1L)
#define UNLOCK(fptr)        DosReleaseMutexSem(fptr->mutex)

/*
 * linesInit - initialize lines in a file
 */

int     linesInit(FILEPTR fptr)
{
    DosCreateMutexSem(NULL, &fptr->mutex, 0, FALSE) ;
    fptr->stat = IO_INIT ;
    fptr->stop = FALSE   ;
    fptr->nline = 0      ;
    
    fptr->lines.prev = &fptr->lines ;
    fptr->lines.next = &fptr->lines ;
    fptr->lines.nline = 0 ;

    return 0 ;
}

/*
 * linesDone - dispose lines in a file
 */

int     linesDone(FILEPTR fptr)
{
    LINEPTR p, prev, next ;

    fptr->stop = TRUE ;     /* request to stop reading  */
    
    LOCK(fptr) ;

    while ((p = fptr->lines.next) != &fptr->lines) {
        /*
	 * unlink from Line List
	 */
        prev = p->prev ;
	next = p->next ;
	prev->next = next ;
	next->prev = prev ;

	/*
	 * dispose Line Data
	 */
	free(p->line) ;
	free(p) ;
    }
    UNLOCK(fptr) ;

    DosCloseMutexSem(fptr->mutex) ;
    
    return 0 ;
}

/*
 * linesAppend - append a Line at Bottom
 */

int     linesAppend(FILEPTR fptr, PSZ line)
{
    PUCHAR      dp ;
    LINEPTR     lp, prev, next ;
    
    /*
     * Allocate for new line
     */

    dp = (PUCHAR)  malloc(strlen(line) + 2) ;
    lp = (LINEPTR) malloc(sizeof(LINEREC))  ;
    if (dp == NULL || lp == NULL) {
        if (dp) free(dp) ;
	if (lp) free(lp) ;
        return -1 ;
    }
    for (lp->line = dp ; *line != '\0' ; line++) {
        if (*line == '\r' || *line == '\n') {
	    break ;
	}
        *dp++ = *line ;
    }
    *dp = '\0' ;

    lp->code = kanjiAssume(lp->line) ;
    
    /*
     * link new line into File's Line List
     */

    LOCK(fptr) ;
    next = &fptr->lines     ;
    prev = fptr->lines.prev ;
    lp->prev = prev ;
    lp->next = next ;
    prev->next = lp ;
    next->prev = lp ;
    lp->nline = prev->nline + 1 ;
    fptr->nline += 1 ;
    UNLOCK(fptr) ;

    return 0 ;
}

/*
 * Refer to Lines
 *      linesTop        refer to first line
 *      linesBot        refer to last  line
 *      linesPrev       refer to prev  line
 *      linesNext       refer to next  line
 */

LINEPTR linesTop(FILEPTR fptr)
{
    LINEPTR p ;

    if (fptr == NULL) {
        return NULL ;
    }

    LOCK(fptr) ;
    if ((p = fptr->lines.next) == &fptr->lines) {
        p = NULL ;
    }
    UNLOCK(fptr) ;
    
    return p ;
}

LINEPTR linesBot(FILEPTR fptr)
{
    LINEPTR p ;

    if (fptr == NULL) {
        return NULL ;
    }

    LOCK(fptr) ;
    if ((p = fptr->lines.prev) == &fptr->lines) {
        p = NULL ;
    }
    UNLOCK(fptr) ;
    
    return p ;
}

LINEPTR linesPrev(FILEPTR fptr, LINEPTR lp)
{
    LINEPTR p ;

    if (fptr == NULL || lp == NULL) {
        return NULL ;
    }

    LOCK(fptr) ;
    if ((p = lp->prev) == &fptr->lines) {
        p = NULL ;
    }
    UNLOCK(fptr) ;
    
    return p ;
}

LINEPTR linesNext(FILEPTR fptr, LINEPTR lp)
{
    LINEPTR p ;

    if (fptr == NULL || lp == NULL) {
        return NULL ;
    }

    LOCK(fptr) ;
    if ((p = lp->next) == &fptr->lines) {
        p = NULL ;
    }
    UNLOCK(fptr) ;
    
    return p ;
}

/*
 * linesFill - fill up lines in a File
 */

void    linesFill(void *arg)
{
    FILEPTR fptr = (FILEPTR) arg ;
    UCHAR   buff[1024] ;

    if (fptr == NULL) {
        return ;
    }
    
    /*
     * open file if not yet
     */
    if (fptr->fp == NULL) {
        fptr->fp = fopen(fptr->name, "r") ;
    }
    if (fptr->fp == NULL) {
        fptr->stat = IO_ERROR ;
	return ;
    }
    
    /*
     * read file
     */

    fptr->stat = IO_READ ;
    
    while(fgets(buff, 1024, fptr->fp) != NULL) {
        if (linesAppend(fptr, buff) != 0) {
	    fptr->stat = IO_ERROR ;
            break ;
        }
        if (fptr->stop) {
	    break ;
	}
    }
    if (fptr->stat == IO_READ) {
        fptr->stat = IO_DONE ;
    }
    if (fptr->fp != stdin) {
        fclose(fptr->fp) ;
    }
    return ;
}
