/*
 * kconv.c - Text Conversion, into internal S-JIS encoding
 */

#include <os2.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <iconv.h>
#include "qpager.h"

/*
 * check for UTF-8
 */

static  BOOL    checkOne(PUCHAR p)
{
    if (p[0] == '\0' || (p[0] & 0x80) != 0) {
        return FALSE ;
    }
    if (p[0] == 0x1b) {
        return FALSE ;
    }
    return TRUE ;
}

static  BOOL    checkTwo(PUCHAR p)
{
    if (p[0] == '\0' || (p[0] & 0xe0) != 0xc0) {
        return FALSE ;
    }
    if (p[1] == '\0' || (p[1] & 0xc0) != 0x80) {
        return FALSE ;
    }
    return TRUE ;
}

static  BOOL    checkThree(PUCHAR p)
{
    if (p[0] == '\0' || (p[0] & 0xf0) != 0xe0) {
        return FALSE ;
    }
    if (p[1] == '\0' || (p[1] & 0xc0) != 0x80) {
        return FALSE ;
    }
    if (p[2] == '\0' || (p[2] & 0xc0) != 0x80) {
        return FALSE ;
    }
    return TRUE ;
}

BOOL    checkUTF8(PUCHAR str)
{
    PUCHAR  p ;
    BOOL    asciionly = TRUE ;

    for (p = str ; *p != '\0' ;  ) {
        if ((*p & 0x80) == 0) {
            if (checkOne(p)) {
                p += 1 ;
            } else {
                return FALSE ;
            }
        } else if ((*p & 0xe0) == 0xc0) {
            asciionly = FALSE ;
            if (checkTwo(p)) {
                p += 2 ;
            } else {
                return FALSE ;
            }
        } else if ((*p & 0xf0) == 0xe0) {
            asciionly = FALSE ;
            if (checkThree(p)) {
                p += 3 ;
            } else {
                return FALSE ;
            }
        } else {
	    return FALSE ;
        }
    }
    if (asciionly) {
        return FALSE ;
    } else {
        return TRUE ;
    }
}
    
/*
 * kanjiAssume - assume KANJI encoding
 */

#define ASSUME_NON      0       /* basic status */
#define ASSUME_ESC      1       /* found ESC    */
#define ASSUME_C1       2       /* found C1     */
#define ASSUME_HI       3       /* found G1     */

#define WEIGHT_SHOULD   100
#define WEIGHT_MAYBE    10
#define WEIGHT_POSSIBLE 1

int     kanjiAssume(PUCHAR p)
{
    int     assume ;
    int     stat   ;
    int     pointMax ;
    int     pointSJIS = 0 ;
    int     pointJIS  = 0 ;
    int     pointEUC  = 0 ;
    
    if (checkUTF8(p)) {
        return ENC_UTF8 ;
    }

    for (stat = ASSUME_NON ; *p ; p++) {
        switch (stat) {
            
	case ASSUME_NON :
	    if (*p == 0x1b) {
	        stat = ASSUME_ESC ;
	    } else if (*p < 0x80) {
	        stat = ASSUME_NON ;
	    } else if (*p < 0xa0) {
	        stat = ASSUME_C1 ;
	    } else {
	        stat = ASSUME_HI ;
            }
	    break ;

	case ASSUME_ESC :
	    if (*p == '$' || *p == '(') {
	        pointJIS += WEIGHT_SHOULD ;
		stat = ASSUME_NON ;
            } else if (*p == '[') {
	        stat = ASSUME_NON ;
	    } else if (*p >= 0x40 && *p <= 0x5a) {
	        stat = ASSUME_NON ;
	    } else if (*p >= 0x60 && *p <= 0x7a) {
	        stat = ASSUME_NON ;
	    }
	    break ;

	case ASSUME_C1 :
	    if (*p < 0x80) {
	        pointSJIS += WEIGHT_MAYBE    ;
		pointEUC  += WEIGHT_POSSIBLE ;
            } else {
	        pointSJIS += WEIGHT_SHOULD ;
            } 
	    stat = ASSUME_NON ;
	    break ;
	
	case ASSUME_HI :
	    if (*p < 0xa0) {
	        pointSJIS += WEIGHT_SHOULD ;
            } else {
	        pointEUC  += WEIGHT_POSSIBLE ;
            }
	    stat = ASSUME_NON ;
	    break ;
	}
    }

    assume = ENC_SJIS ;
    pointMax = 0      ;

    if (pointJIS > pointMax) {
        assume = ENC_JIS7    ;
	pointMax = pointJIS  ;
    }
    if (pointSJIS > pointMax) {
        assume = ENC_SJIS    ;
	pointMax = pointSJIS ;
    }
    if (pointEUC > pointMax) {
        assume = ENC_EUC     ;
	pointMax = pointEUC  ;
    }
    return assume ;
}

/*
 * kanjiConv - Convert Kanji Text into SJIS (for OS/2 native code)
 */

int     kanjiConv(PUCHAR sp, PUCHAR dp, int code)
{
    iconv_t cd ;
    char    *iptr, *optr ;
    size_t  ilen, olen, clen ;
    char    obuff[2048]  ;  /* sorry for immediate size */
    
    switch (code) {
    case ENC_JIS7 :
        cd = iconv_open("SJIS", "ISO-2022-JP-1") ;
	break ;
    case ENC_EUC  :
        cd = iconv_open("SJIS", "EUC-JP") ;
	break ;
    case ENC_UTF8 :
        cd = iconv_open("SJIS", "UTF-8") ;
	break ;
    default       :
        cd = NULL ;
        break ;
    }
    
    ilen = strlen(sp) ;
    iptr = sp ;
    olen = sizeof(obuff) ;
    optr = obuff ;
    memset(obuff, 0, sizeof(obuff)) ;
    
    if (cd == NULL) {
        strcpy(obuff, sp) ;
    } else {
        while (*iptr != '\0') {
	    if (iconv(cd, &iptr, &ilen, &optr, &olen) == -1) {
	        *optr++ = *iptr++ ;
		ilen -= 1 ;
		olen -= 1 ;
            }
        }
        iconv_close(cd) ;
    }
    strcpy(dp, obuff) ;
    
    return(strlen(dp)) ;
}    
