/* $Id: ini.c,v 1.2 2004/05/12 23:10:29 root Exp $ */

/* Configuration update utility for WIN.INI/SYSTEM.INI/MMPM2.INI/etc. */

#include <io.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "csp_defs.h"
#include "csp_lib.h"
#include <filerd.h>

#define KILLCHAR                 '^'

/* is*() */

#define iswspace(c) ((c)==32||(c)>=9&&(c)<=13)
#define isoption(str) ((str)[0]=='-'&&(str)[1]!='\0'&&(str)[2]=='\0')

/* File allocation structure - allows for files larger than 64K */

struct fstr;
struct fstr
{
 struct fstr *next;
 char *str;
};

/* Keyword format:
   <tag>=[<value>]
   OR
   ;<tag>[=[<value>]]
   OR
   %Text Message
 */

#define MAX_LINE                      24576
#define MAX_KEYWORDS                    512

struct keyword
{
 unsigned int len;                      /* To facilitate search */
 unsigned int hits;                     /* Usage counter */
 char *locality;                        /* Name of group to look in
                                           (can be NULL) */
 char *tag;                             /* Pure keyword */
 char *replacement;                     /* Complete replacement line */
};

static struct keyword keywords[MAX_KEYWORDS];
static unsigned int total_keywords=0;

/* Flags */

static int no_backup=0;
static int persistent_keywords=0;

/* Prototypes */

static void parse_token(char *str);

/* Error routine */

static int error(char *msg)
{
 puts(msg);
 exit(1);
}

/* malloc_msg() */

static void *malloc_msg(size_t s)
{
 void *rc;

 if((rc=malloc(s))==NULL)
  error("Low memory.");
 return(rc);
}

/* A fopen() with error reporting */

static FILE *file_open(char *name, char *mode)
{
 FILE *rc;

 if((rc=fopen(name, mode))==NULL)
 {
  puts(name);
  error("Can't open.");
 }
 return(rc);
}

/* Checks a single line for keywords and replaces them. Returns 1
   if a replacement takes place. */

static char *fix_str(char *str)
{
 char *p, *e;
 int is_cmt=0;
 int i, l;
 static char *cur_group=NULL;

 p=str;
 /* Determine the starting point (comments won't stop us) */
 while(iswspace(*p))
  p++;
 if(*p=='[')
 {
  if(cur_group!=NULL)
   free(cur_group);
  if((cur_group=strdup(p+1))==NULL)
   error("Can't allocate memory for group ID");
  if((e=strchr(cur_group, ']'))!=NULL)
   *e='\0';
  return(0);
 }
 if(*p==';')
 {
  is_cmt=1;
  p++;
  while(iswspace(*p))
   p++;
 }
 if(*p=='\0')
  return(NULL);
 /* Locate the ending point */
 for(l=1; !iswspace(p[l])&&p[l]!='\0'&&p[l]!='='; l++)
  ;
 /* Now look what's beyond the ending point */
 for(e=p+l; iswspace(*e); e++)
  ;
 if((*e=='\0'&&!is_cmt)||               /* Stray tag. Ignore. */
    (*e!='='&&*e!='\0'&&is_cmt))        /* Freeform comment. Ignore. */
  return(NULL);
 /* Find a keyword that matches */
 for(i=0; i<total_keywords; i++)
 {
  if(keywords[i].len==l&&!memicmp(p, keywords[i].tag, l))
  {
   if(keywords[i].locality==NULL||
      (cur_group!=NULL&&!stricmp(cur_group, keywords[i].locality)))
    break;
  }
 }
 if(i>=total_keywords)
  return(NULL);
 /* Point out the replacement string */
 return(keywords[i].replacement);
}

/* Proceeds through the file image, replacing the keywords.
   Returns the number of keywords found. */

static unsigned long fix_image(struct fstr *first)
{
 struct fstr *c, *prev;
 unsigned long rc=0;
 char *rstr;

 prev=NULL;
 for(c=first; c!=NULL; c=c->next)
 {
  if((rstr=fix_str(c->str))!=NULL)
  {
   free(c->str);
   c->str=strdup(rstr);
   if(c->str[0]==KILLCHAR)
   {
    prev->next=c->next;
    free(c->str);
    free(c);
    c=prev;
   }
   rc++;
  }
  prev=c;
 }
 return(rc);
}

/* Fill-it-up */

static struct fstr *fillitup(FILE *stream)
{
 static char *line_buf;
 struct fstr *first=NULL, *cur=NULL, *t;
 int l;

 line_buf=(char *)malloc_msg(MAX_LINE);
 while(fgets(line_buf, MAX_LINE, stream)!=NULL)
 {
  stripcr(line_buf);
  l=strlen(line_buf);
  t=(struct fstr *)malloc_msg(sizeof(struct fstr));
  t->str=strdup(line_buf);
  if(cur==NULL)
   first=cur=t;
  else
  {
   cur->next=t;
   cur=t;
  }
 }
 free(line_buf);
 cur->next=NULL;
 return(first);
}

/* Parses a keyword entry. Returns <dest> if OK. */

static struct keyword *parse_keyword(struct keyword *dest, char *str)
{
 int l;
 char *r;
 char *s;
 static char m_lowmem[]="Extremely low on memory\n";

 /* Separate the locality from the tag */
 if((r=strchr(str, '/'))!=NULL)
  s=r+1;
 else
  s=str;
 if((dest->replacement=strdup(s))==NULL)
  error(m_lowmem);
 for(dest->tag=dest->replacement; iswspace(*dest->tag); dest->tag++)
  ;
 if(*dest->tag==';'||*dest->tag==KILLCHAR)
 {
  for(s=dest->tag+1; iswspace(*s); s++)
   ;
  s=strdup(s);
  free(dest->tag);
  dest->tag=s;
  if(s==NULL)
   error(m_lowmem);
 }
 if(*dest->tag=='\0')
  return(NULL);
 for(l=1; !iswspace(dest->tag[l])&&dest->tag[l]!=0&&dest->tag[l]!='='; l++)
  ;
 dest->len=l;
 dest->hits=0;
 if(r==NULL)
  dest->locality=NULL;
 else
 {
  dest->locality=malloc(r-str+1);
  memcpy(dest->locality, str, r-str);
  dest->locality[r-str]='\0';
 }
 return(dest);
}

/* Parses a keyword file */

static void parse_kwfile(char *name)
{
 FILE *stream;
 static char *line_buf;

 stream=file_open(name, "rb");
 line_buf=(char *)malloc_msg(MAX_LINE);
 while(fgets(line_buf, MAX_LINE, stream)!=NULL)
 {
  stripcr(line_buf);
  parse_token(line_buf);
 }
 free(line_buf);
 fclose(stream);
}

/* Parses something that may comprise a file inclusion token, a keyword or
   some other entry */

static void parse_token(char *str)
{
 if(str[0]=='@')
  parse_kwfile(str+1);
 else if(str[0]=='%')
  puts(str+1);
 else
 {
  if(total_keywords>=MAX_KEYWORDS)
   error("Keyword limit exceeded, aborting\n");
  if(parse_keyword(&keywords[total_keywords], str))
   total_keywords++;
 }
}

/* Release all keywords */

static void dispose_of_keywords()
{
 int i;

 for(i=0; i<total_keywords; i++)
 {
  if(keywords[i].locality!=NULL)
   free(keywords[i].locality);
  if(keywords[i].tag!=NULL)
   free(keywords[i].tag);
  if(keywords[i].replacement!=NULL)
   free(keywords[i].replacement);
 }
 total_keywords=0;
}

/* Processes the specified file */

static void process_file(char *name)
{
 char *bak=NULL, *fnew=NULL;
 char *fname;
 int fallback=0;
 FILE *stream;
 struct fstr *pfstr, *cp, *p;

 fname=name;
 if(!no_backup)
 {
  bak=malloc_msg(strlen(name)+4);
  fnew=malloc_msg(strlen(name)+4);
  strcpy(bak, name);
  strcpy(fnew, name);
  substext(bak, "bak");
  substext(fnew, "new");
  /* Rely on the BAK file if nothing else is available */
  if(access(name, 0)&&!access(bak, 0))
  {
   puts("Fallback to the BAK file - no master copy available\n");
   fname=bak;
   fallback=1;
  }
 }
 stream=file_open(fname, "rb");
 pfstr=fillitup(stream);
 fclose(stream);
 fix_image(pfstr);
 /* Return home */
 stream=file_open(no_backup?name:fnew, "w");
 for(p=pfstr; p!=NULL; p=cp)
 {
  fputs(p->str, stream);
  cp=p->next;
  fputs("\n", stream);
  free(p->str);
  free(p);
 }
 fclose(stream);
 /* Now shuffle the files if operating in a backup mode */
 if(!no_backup)
 {
  unlink(bak);
  rename(name, bak);
  rename(fnew, name);
  unlink(bak);
 }
}

/* Main routine */

int main(int argc, char **argv)
{
 int i;
 char *fspec=NULL;

 /* Pass 1: Gather command-line options */
 for(i=1; i<argc; i++)
 {
  if(isoption(argv[i]))
  {
   switch(argv[i][1])
   {
    case 'n':
     no_backup=1;
     break;
    case 'p':
     persistent_keywords=1;
     break;
   }
  }
 }
 /* Pass 2: process tag replacements */
 for(i=1; i<argc; i++)
 {
  if(isoption(argv[i]))
   ;
  else if(argv[i][0]!='@'&&argv[i][0]!='%'&&argv[i][0]!=';'&&
          strchr(argv[i], '=')==NULL)
  {
   fspec=argv[i];
   process_file(fspec);
   if(!persistent_keywords)
    dispose_of_keywords();
  }
  else
   parse_token(argv[i]);
 }
 if(fspec==NULL)
 {
  puts("INI updater v 1.10 on " __DATE__ ", " __TIME__ " with " PROD_COMPILER "\n"
       "\n"
       "Command-line: INI <options> <tags> <filename.ext> [<tags> <filename.ext> ...]\n"
       "where:\n"
       "<options>\n"
       "            -n tells to bypass the backup file creation\n"
       "            -p tells to preserve the gathered tags across multiple files\n"
       "<tags>\n"
       "   @<filename> tells to recurse into filename for tags\n"
       "       %%<text> tells to display text on standard output\n"
       "   <opt>=<val> sets an option, where\n"
       "               <opt> is [<group>/]<option>\n"
       "        ;<opt> resets an option, commenting it out\n");
  return(1);
 }
 if(persistent_keywords)
  dispose_of_keywords();
 return(0);
}
