@echo off
:: 7ztool - 7z wrapper

:: Copyright (c) 2013-2021 Steven Levine and Associates, Inc.
:: All rights reserved.

:: This program is free software licensed under the terms of the GNU
:: General Public License.  The GPL Software License can be found in
:: gnugpl2.txt or at http://www.gnu.org/licenses/licenses.html#GPL

:: 2013-06-23 SHL Baseline
:: 2018-07-12 SHL 7z does not understand drive letters - deal with it
:: 2018-12-23 SHL Support -l -t
:: 2019-04-08 SHL Deal with drive letters in ZipTo7Z
:: 2019-04-08 SHL Sync with templates
:: 2020-05-08 SHL Ensure sure 7z l called with unix-ish path for ALTOPs
:: 2020-05-23 SHL Avoid path rewriter in case working from tmp
:: 2020-09-10 SHL Offer to select if 7z omitted
:: 2021-10-27 SHL Correct -x logic
:: 2021-12-07 SHL Sync with templates
:: 2021-12-07 SHL Make setpri optional
:: 2021-12-07 SHL Add some whitespace to output

if "%@eval[0]" == "0" goto is4xxx
  echo Must run in 4OS2/4DOS session
  :: 4os2 /c %0 %1 %2 %3 %4 %5 %6 %7 %8 %9
  :: delay 2
  pause
  goto eof
:is4xxx

loadbtm on
on errormsg pause
setlocal
on break @goto Halted

if "%_DOS%" != "OS2" ( echo Must run in 4OS2 session %+ beep %+ cancel )

:Main

  gosub ScanArgs

  iff not defined NOSETPRI then
    set X=%@search[setpri.exe]
    iff not defined X then
      set NOSETPRI=1
      echo Cannot access setpri.exe - disabling setpri usage
    endiff
  endiff

  set EXT=%@ext[%FN]
  iff not defined EXT then
    gosub Compress
  elseiff %EXT eq zip then
    gosub ZipTo7Z
  elseiff %EXT eq 7z then
    gosub Do7ZOp
  else
    gosub Compress
  endiff

  gosub Cleanup

  quit

  :: end main

::=== Compress() Compress %FN to .7z in current directory ===

:Compress
  set X=%@filename[%FN].7z
  gosub QuoteXForShell
  set ZF=%X
  :: Work from %FN directory compress to current directory
  set DIR=%@path[%FN]
  iff defined DIR then
    set X=%@full[%FN]
    gosub QuoteXForShell
    set FN=%X
    set DIR=%@path[%FN]
    set DIR=%@replace[\\,,%DIR\]
    set X=%_CWD\%ZF
    gosub QuoteXForShell
    set ZF=%X
  endiff

  echo.
  echo Compressing %FN to %ZF

  do 1
    iff exist %ZF then
      iff defined BATCH then
        set R=0
      else
        echo.
        call AskYN Update %ZF with %FN
        set R=%?
      endiff
      if %R ge 2 cancel
      if %R ne 0 leave
    endiff
    :: Work from source directory
    iff defined DIR then
      pushd
      cdd %DIR
    endiff

    :: Build command line

    set X=%ZF
    if not defined NOSETPRI gosub QuoteXForSetPri
    set Y=%X

    set X=%@filename[%FN]
    gosub QuoteXForShell
    if not defined NOSETPRI gosub QuoteXForSetPri

    iff 0 eq 1 then do
      :: Enable for debug
      echo.
      echo 7z file is %Y
      echo source file(s) are %X
    endiff

    set CMD=7z u %Y %X
    if not defined NOSETPRI set CMD=setpri -10 o %CMD
    echo.
    echo on
    %CMD
    @echo off
    if %? ne 0 leave
    :: Zap libc EAS
    eautil %ZF nul /s
    iff defined DIR then
      cd \
      popd
    endiff
    set TCH=
    for %X in ( %FN ) ( set Y=/d%@filedate[%X] /t%@filetime[%X] %+ if "%Y" gt "%TCH" set TCH=%Y )
    if defined TCH touch /q %TCH %ZF

    echo.
    echo on
    7z l %ZF
    @echo.
    dir /m /t %ZF
    @echo.
    7z t %ZF
    @echo off
    if %? ne 0 leave
    if defined KEEP leave
    iff defined BATCH then
      set R=0
    else
      echo.
      call AskYN Delete %FN
      set R=%?
    endiff
    if %R ge 2 cancel
    if %R ne 0 leave
    iff %@attrib[%FN,R,P] == 1 then
      echo.
      beep
      echo Warning: %FN is R/O
      del /pz %FN
      if exist %FN echo Warning: %FN not deleted
    else
      del /q %FN
      if exist %FN echo Cannot delete %FN
    endiff
  enddo
  return
  :: end Compress

::=== Do7ZOp() Extract to current directory or list or test ===

:Do7ZOp
  :: %FN known to be .7z
  set ZF=%FN
  do 1
    iff defined BATCH then
      set R=0
    else
      iff defined ALTOP then
        :: List or test requested
        set X=%@full[%ZF]
        :: Strip drive letter for 7z
        set X=%@substr[%X,2,260]
        gosub QuoteXForShell
        gosub ChangeXToUnixSlashes
        echo.
        echo on
        7z %ALTOP %X
        @echo off
        leave
      endiff
      echo.
      call AskYN Extract contents of %ZF
      set R=%?
    endiff
    if %R ge 2 cancel
    if %R ne 0 leave

    echo.
    echo on
    7z %EXTRACTOP %ZF
    @echo off
    if %? ne 0 leave
    dir /h /m /t
    if defined KEEP leave
    iff defined BATCH then
      set R=0
    else
      echo.
      call AskYN Delete %ZF
      set R=%?
    endiff
    if %R ge 2 cancel
    if %R ne 0 leave
    iff %@attrib[%ZF,R,P] == 1 then
      beep
      echo Warning: %ZF is R/O
      del /pz %ZF
      if exist %ZF echo Warning: %ZF not deleted
    else
      del %ZF
      if exist %ZF echo Cannot delete %ZF
    endiff
  enddo
  return
  :: end Do7ZOp

::=== ScanArgs() Scan command line ===

:ScanArgs
  set EXTRACTOP=
  set ALTOP=
  set BATCH=
  set KEEP=
  set NOSETPRI=
  set FN=
  set WD=

  do I = 1 to %#
    set A=%[%I]
    set X=%@substr[%A,0,1]
    iff "%X" == "/" .or. "%X" == "-" then
      :: Got switch
      set X=%@substr[%A,1,260]
      if "%X" == "h" .or. "%X" == "?" goto UsageHelp
      iff "%X" == "b" then
        set BATCH=-b
        iterate
      endiff
      iff "%X" == "e" then
        set EXTRACTOP=e
        iterate
      endiff
      iff "%X" == "k" then
        set KEEP=1
        iterate
      endiff
      iff "%X" == "l" then
        set ALTOP=l
        iterate
      endiff
      iff "%X" == "n" then
        set NOSETPRI=l
        iterate
      endiff
      iff "%X" == "x" then
        set EXTRACTOP=x
        iterate
      endiff
      iff "%X" == "t" then
        set ALTOP=t
        iterate
      endiff
      echo Switch %A unexpected
      goto UsageError
      iterate
    endiff
    :: Got argument
    if defined FN ( echo Only one filespec allowed %+ goto UsageError )
    if isdir %A ( echo %A must be a file %+ goto UsageError )
    if not exist %A ( echo %A must exist %+ goto UsageError )
    if %@index[%A,*] != -1 .or. %@index[%A,?] != -1 ( echo %A may not contain wildcards %+ goto UsageError )
    set FN=%A
  enddo

  iff not defined FN then
    iff not defined BATCH .or. defined KEEP then
      iff exist *.7z then
        select %0 %$ ( *.7z )
        quit
      endiff
      iff exist *.zip .and. not defined ALTOPT then
        select %0 %$ ( *.7z )
        quit
      endiff
    endiff
  endiff

  if not defined EXTRACTOP set EXTRACTOP=e

  if not defined FN ( echo Filespec required %+ goto UsageError )

  iff defined ALTOP then
    if defined BATCH ( echo -b not allowed with -l or -t %+ goto UsageError )
  endiff

  iff "%@substr[%FN,1,1]" eq ":" then
    :: Strip drive letter to keep 7z happy
    set X=%@substr[%FN,2,260]
    set Y=%X
    gosub QuoteXForShell
    iff not exist %X then
      echo 7z does not support drive letters
      echo Cannot access %Y
      echo Must run from drive %@substr[%FN,0,2]
      beep
      cancel
    endiff
    set FN=%Y
  endiff

  return

  :: end ScanArgs

::=== ZipTo7Z() Convert zip file %FN to 7z ===

:ZipTo7Z
  :: FN known to exist
  :: FN may be quoted
  do 1
    :: SF is 7z file name
    set X=%@path[%FN]%@name[%FN].7z
    gosub QuoteXForShell
    set SF=%X

    iff exist %SF then
      iff defined BATCH then
        set R=0
      else
        echo.
        echo %SF exists
        echo.
        call AskYN Update contents
        set R=%?
      endiff
      if %R ge 2 cancel
      if %R ne 0 leave
    endiff
    iff defined BATCH then
      set R=0
    else
      echo.
      dir /k /m /t %FN
      echo.
      unzip -t %FN
      echo.
      unzip -v %FN
      echo.
      iff defined KEEP then
        echo Add %FN contents
      else
        echo Move %FN contents
      endiff
      call AskYN to %SF
      set R=%?
    endiff
    if %R ge 2 cancel
    if %R ne 0 leave

    :: Create work directory on current drive
    :: 7z does not do drive letters
    :: FIXME to check free space
    set WD=%@path[%FN]
    iff defined WD then
      set WD=%@path[%FN]
      set WD=%@replace[\\,\,%WD\]
    endiff
    set WD=%[WD]%@name[%0].%_PID

    :: Need full paths for FN and SF
    :: Quote if needed
    :: @full strips quotes
    set X=%@full[%FN]
    gosub QuoteXForShell
    set FN=%X

    set X=%@path[%FN]%@name[%FN].7z
    :: Strip drive letter for 7z
    set X=%@substr[%X,2,260]
    gosub QuoteXForShell
    set SF=%X
    gosub ChangeXToUnixSlashes
    set SF7=%X

    iff isdir %WD .and. exist %WD\* then
      iff defined BATCH then
        set R=0
      else
        dir /h /m %WD
        echo.
        call AskYN destroy %WD contents
        set R=%?
      endiff
      if %R ge 2 cancel
      if %R ne 0 leave
      del /eyz %WD
    elseiff not isdir %WD then
      mkdir %WD
    endiff

    pushd
    echo.
    cdd %WD
    echo Work directory is %_CWD

    :: Avoid kLIBC pathrewriter
    set LIBC_HOOK_DLLS=

    echo.
    unzip -jo %FN
    if %? ne 0 leave
    dir /h /m
    set TCH=/d%@filedate[%FN] /t%@filetime[%FN]
    set A=%@attrib[%FN,A,P]
    echo.

    :: Build 7z command line

    set X=%SF7
    if not defined NOSETPRI gosub QuoteXForSetPri
    set CMD=7z u %X *
    if not defined NOSETPRI set CMD=setpri -10 o %CMD

    echo.
    echo on
    timer
    %CMD
    @echo off
    if %? ne 0 leave
    timer

    popd

    dir /k /m %SF

    :: Zap libc EAS
    eautil %SF nul /s
    :: Sync timestamp
    touch /q %TCH %SF
    if %A == 0 attrib /q -a %SF

    :: Verify
    echo.
    echo on
    7z t %SF7
    @echo off
    if %? ne 0 leave

    :: 7z l %SF7
    :: if %? ne 0 leave

    set X=%@filename[%FN]
    gosub QuoteXForShell
    dir /m /t %SF;%X

    :: FIXME to delete subdirectories maybe
    del /qexyz %WD
    set WD=

    if defined KEEP leave

    iff defined BATCH then
      set R=0
    else
      echo.
      call AskYN Delete %FN
      set R=%?
    endiff

    if %R ge 2 cancel
    if %R ne 0 leave
    iff %@attrib[%FN,R,P] == 1 then
      beep
      echo Warning: %FN is R/O
      del /pz %FN
      if exist %FN echo Warning: %FN not deleted
    else
      del %FN
      if exist %FN echo Cannot delete %FN
    endiff

  enddo
  return
  :: end ZipTo7Z

::=== Cancel() Handle cancel ===

:Cancel
  @echo off
  gosub Cleanup
  cancel 255
  :: end Cancel

::=== Cleanup() Clean up ===

:Cleanup
  if defined WD ( if isdir %WD del /qexyz %WD %+ set WD= )
  :: Ensure off
  timer on >nul
  timer >nul
  return
  :: end Cleanup

::=== UsageError() Report usage error ===

:UsageError
  beep
  echo Usage: %@lower[%@name[%0]] `[-b] [-e] [-h] [-k] [-l] [-t] [-x] [-?] [filespec]`
  cancel
  :: end UsageError

::=== UsageHelp() Display usage help ===

:UsageHelp
  echo.
  echo 7z wrapper
  echo.
  echo Usage: %@lower[%@name[%0]] `[-b] [-e] [-h] [-k] [-l] [-n] [-t] [-x] [-?] [filespec]`
  echo.
  echo `  -b        Batch mode`
  echo `  -e        Extract without paths (default)`
  echo `  -h -?     Display this message`
  echo `  -k        Keep original`
  echo `  -l        List contents of original file`
  echo `  -n        Run at normal priority - do not use setpri`
  echo `  -t        Test contents of original file`
  echo `  -x        Extract with paths`

  echo.
  echo `  filespec  File to process, wildcards not supported`
  echo `            If .zip convert zip to 7z`
  echo `            If .7z extract`
  echo `            If other compress to .7z`
  echo `            If omitted offer to select .7z`
  cancel
  :: end UsageHelp

::=============================================================================
::=== SkelCmdFunc standards - Delete unused - Move modified above this mark ===
::=============================================================================

::=== ChangeXToUnixSlashes() Convert %X to unix slashes ===

:ChangeXToUnixSlashes
  :: Force DOS slashes
  set X=%@replace[\,/,%X]
  return
  :: end ChangeXToUnixSlashes

::=== Halted() Handle break ===

:Halted
  @echo off
  echo Halted by user
  goto Cancel
  :: end Halted

::=== QuoteXForSetPri() Quote %X for setpri ===

:QuoteXForSetPri
  :: backslash escape double quotes in %X to protect from setpri
  iff defined X then
    set X=%@replace[%=",\%=",%X]
  endiff
  return
  :: end QuoteXForSetPri

::=== QuoteXForShell() Quote %X for shell ===

:QuoteXForShell
  :: Quote if has whitespace unless already quoted
  iff %@ascii[%X] != %@ascii["] then
    if %@index[%X, ] != -1 set X="%X"
  endiff
  return
  :: end QuoteXForShell

:eof
