/* apachectl - control httpd 2.x

   FIXME to detect if monitor is running - use MONITOR file with pid
   FIXME to suppress pause if monitor not running
   FIXME to suppress unpause if monitor not running
   FIXME to set pidfile location on command line if non-standard

   Return 0 if OK else error code

   Copyright (c) 2009-2023 Steven Levine and Associates, Inc.
   All rights reserved.

   This program is free software licensed under the terms of the GNU
   General Public License.  The GPL Software License can be found in
   gnugpl2.txt or at http://www.gnu.org/licenses/licenses.html#GPL

   2006-05-05 SHL Baseline
   2021-04-16 SHL Drop -f support - use startf
   2021-05-14 SHL Clean up CheckPhpIniPhpVersion
   2021-10-02 SHL Support running without php
   2021-10-10 SHL Support php8
   2021-10-10 SHL Sync with templates - MakePath
   2022-02-25 SHL DoMonitor - honor stop file while paused
   2022-04-05 SHL Sync with templates - MakePath
   2022-04-05 SHL DoMonitor - timestamp console messages, use updated LogWriteVTSC
   2022-05-12 SHL SetLibPath - drop modules logic - no longer needed as of 2.4.53 pr#4
   2022-05-13 SHL SetLibPath - drop php7 logic - no longer needed as of 7.4.28 pr#12
   2022-07-08 SHL SetLibPath - set BEGINLIBPATH only if needed by httpd build
   2022-09-19 SHL FindAppDir - support c:\apps\apache24 for dnacih
   2022-11-14 SHL Support testv for TestVHost
   2022-11-28 SHL Sync with templates - AskYNQ
   2022-11-28 SHL FindAppDir - More 2.4 and drive C: support
   2022-11-28 SHL Use reworked IsPidRunning and IsPidRunningEx
   2022-12-03 SHL Allow monitor to restart if same pid
   2023-01-16 SHL Ensure pid points to httpd instance
*/

/* use 'setlocal' until we know how/when call setlocal corrupts environment */
'@setlocal 2>nul'
envSaved = RC = 0

signal on Error
signal on Failure name Error
signal on Halt
signal on NotReady name Error
signal on NoValue name Error
signal on Syntax name Error

gVersion = '0.1 2021-10-10'

Globals = 'gAppDir gAppExe gAppName gAppTitle gAppVer gArgList. gBatch gCfgFile gCmdName',
	  'gDbgLvl gEditor gEnv gErr gErrCondition gFileName',
	  'gKiller gLibPathSet gLogDir gLogFile gLogWrites gPhpVersion gPid gPidFile',
	  'gTmpDir gVerbose gVersion'

call Initialize

gAppTitle = 'Apache server'
gAppName = 'httpd'
gAppExe = 'httpd.exe'

Main:

  parse arg cmdLine
  call ScanArgs cmdLine
  drop cmdLine

  if 0 & trace() <> '?A' then trace '?A' ; nop

  call CheckRequired

  /* Scan args may have changed current directory */
  call FindAppDir

  /* Guess httpd version - FIXME to be smarter */
  i = lastpos('\', gAppDir)
  s = substr(gAppDir, i + 1)
  if pos('24', s) > 0 then
    gAppVer = 2.4
  else if pos('22', s) > 0 then
    gAppVer = 2.2
  else
    gAppVer = 2.x

  s = ToLower(GetBaseName(gAppDir))
  do 1
    gAppVer = 2.4
    if s == 'apache24' then leave
    gAppVer = 2.2
    if s == 'apache22' then leave
    if s == 'apache2' then leave
    call Die 'Cannot determine apache version for' s
  end


  /* Run from application bin directory */
  if \ envSaved then do
    oldCwd = directory()
    oldCwd2 = directory(left(gAppDir, 2))
  end

  call directory MakePath(gAppDir, 'bin')

  if \ IsExeInPath(gAppName) then
    call Die gAppTitle 'not found in PATH'

  call SetPidFile

  if IsPidRunning() then do
    procname = GetProcessForPid(gPid)
    if pos('HTTPD', translate(procName)) > 0 then do
      /* pid file must be stale - pid in use by some process other than httpd */
      say gFileFile 'PID' gPid 'in use by' procName '- deleting PID file'
      call SysFileDelete(gPidFile)
    end
  end

  gErr = 0

  do iArg = 1 to gArgList.0
    curArg = gArgList.iArg
    call DoArg curArg
  end /* iArg */

  if \ envSaved then do
    if oldCwd \== oldCwd2 then
      call directory oldCwd2
    call directory oldCwd
  end

  exit gErr

/* end main */

/*=== DoArg(action) Process action request; return rc ===*/

DoArg: procedure expose (Globals)
  parse arg curArg
  select
  when curArg == 'start' then
    call DoStart
  when curArg == 'startf' then
    call DoStartf
  when curArg == 'stop' then
    call DoStop
  when curArg == 'pause' then
    call DoPause
  when curArg == 'unpause' then
    call DoUnpause
  when curArg == 'status' then
    call DoStatus
  when curArg == 'config' then
    call DoConfig
  when curArg == 'graceful' then
    call DoGraceful
  when curArg == 'kill' then
    call DoKill
  when curArg == 'restart' then
    call DoRestart
  when curArg == 'monitor' then
    call DoMonitor
  when curArg == 'test' then
    call DoTest
  when curArg == 'testv' then
    call DoTestVHost
  when curArg == 'lynx' then
    call DoLynx
  when curArg == 'lynx2' then
    call DoLynx2
  when curArg == 'help' then
    call ScanArgsHelp
  otherwise
    call ScanArgsUsage 'Request' curArg 'unexpected'
  end
  return

/* end DoArg */

/*=== DoConfig() Config ===*/

DoConfig: procedure expose (Globals)

  call FindCfgFile
  call FindEditor
  say 'Starting' gAppTitle 'configurator'
  gEditor gCfgFile
  say 'Restart' gAppTitle 'to apply changes'
  return

/* end DoConfig */

/*=== DoGraceful() Graceful restart ===*/

DoGraceful: procedure expose (Globals)
  say 'httpd does not yet support graceful restart'
  return

  if \ IsPidRunningEx() then
    say gAppTitle 'is not running'
  else do
    call RunKiller '-USR1'
    if \ IsPidRunningEx(1) then do
      say gAppTitle 'did not restart'
      gErr = 1
    end
    else
      say gAppTitle 'has restarted'
  end
  return

/* end DoGraceful */

/*=== DoKill() Kill ===*/

DoKill: procedure expose (Globals)
  if \ IsPidRunningEx() then
    say gAppTitle 'is not running'
  else do
    call RunKiller '-KILL'
    if IsPidRunningEx(0, gPid) then do
      say gAppTitle 'will not die'
      gErr = 1
    end
    else do
      say gAppTitle 'has stopped'
      /* Stop monitor too */
      if IsMonitorRunning() then
	call MakeStopFileIf
    end
  end
  return

/* end DoKill */

/*=== DoMonitor(arg) Process argument ===*/

DoMonitor: procedure expose (Globals)

  /* Guess log directory */
  do 1
    d = 'd:\logs'			/* dnacih */
    if IsDir(d) then leave
    d = GetEnv('LOGFILES')
    if d \= '' then do
      if IsDir(d) then leave
    end
    call Die 'Cannot determine log file directory'
  end
  gLogDir = d

  /* Ensure monitor not already running */
  if IsMonitorRunning('stopped') then
    return				/* running as some other pid */

  '@title Apache Server and Monitor'
  '@window "Apache Server and Monitor"'

  /* Create monitor pid file */
  monitorFile = MakePath(gAppDir, 'MONITOR')
  call SysFileDelete monitorFile
  info = Sys2QueryProcess(0, 'P')
  parse var info p pp pt tail
  call lineout monitorFile, p
  call stream monitorFile, 'C', 'CLOSE'

  /* Stop httpd when this file created; restart when deleted */
  pauseFile = MakePath(gAppDir, 'PAUSE')
  /* Stop httpd and self when this file created */
  stopFile = MakePath(gAppDir, 'STOP')

  /* Ensure stale pause file deleted */
  if IsFile(pauseFile) then do
    call SysFileDelete pauseFile
    say MakeTimeStamp() 'PAUSE file deleted from' gAppDir
  end

  /* Ensure stale stop file deleted */
  if IsFile(stopFile) then do
    call SysFileDelete stopFile
    say MakeTimeStamp() 'STOP file deleted from' gAppDir
  end

  do forever

    if IsFile(stopFile) then do
      call LogWriteVTSC 'STOP file detected -' gCmdName 'monitor exiting'
      call SysFileDelete stopFile
      leave
    end

    if IsPidRunningEx() then do
      call LogWriteVTSC 'httpd started by some other process -' gCmdName 'monitor exiting'
      leave
    end

    call LogWriteVTSC 'Starting/restarting httpd'
    call DoStartf
    call LogWriteVTSC 'httpd stopped unexpectedly with RC' RC

    /* Wait if paused and not stopped */
    do forever
      if IsFile(stopFile) then leave
      say MakeTimeStamp() 'Waiting 10 seconds before trying to restart Apache'
      call SysSleep 10
      if \ IsFile(pauseFile) then leave
      say
      say MakeTimeStamp() pauseFile 'exists - Apache restart delayed'
      say MakeTimeStamp() 'Use apachectl unpause to allow Apache restart'
    end /* forever */

  end /* forever */

  /* Ensure stale monitor file deleted */
  call SysFileDelete monitorFile

  return

/* end DoMonitor */

/*=== DoPause(arg) Create PAUSE file ===*/

DoPause: procedure expose (Globals)

  /* Ensure monitor running */
  if \ IsMonitorRunning('running') then
    return

  pauseFile = MakePath(gAppDir, 'PAUSE')
  if \ IsFile(pauseFile) then do
    call lineout pauseFile, 'Pause file created at' date() time()
    call lineout pauseFile
    say 'PAUSE file created in' gAppDir
  end
  call DoStop 'nostopfile'	/* Only stop httpd */
  return

/* end DoPause */

/*=== DoRestart() Restart ===*/

DoRestart: procedure expose (Globals)
  /* 27 Feb 09 SHL FIXME to use HUP */
  call DoStop 'nostopfile'		/* Only stop httpd */
  if gErr = 0 then
    call DoStart
  else
    say gAppTitle 'stop failed'
  return

/* end DoRestart */

/*=== DoStart() Start server minimized ===*/

DoStart: procedure expose (Globals)

  if IsPidRunningEx() then
    say gAppTitle 'already running'
  else do
    pauseFile = MakePath(gAppDir, 'PAUSE')
    if IsFile(pauseFile) then do
      call SysFileDelete pauseFile
      say 'PAUSE file deleted from' gAppDir
    end
    if gVerbose then
      say 'Starting' gAppTitle 'from' directory()
    call SetLibPath
    /* FIXME to delete pid file etc. */
    signal off Error
    'start "' || gAppTitle gAppVer || '" /min /c' gAppName '-d..'
    signal on Error
    if RC \= 0 & RC \= 457 then
      signal Error
    if IsPidRunningEx(1) then
      say gAppTitle 'started and running as Pid' gPid'('d2x(gPid)')'
    else do
      say 'Cannot start' gAppTitle
      gErr = 1
    end
    call SetFocus
  end

  return

/* end DoStart */

/*=== DoStartf() Start server in foreground ===*/

DoStartf: procedure expose (Globals)

  if IsPidRunningEx() then
    say gAppTitle 'already running'
  else do
    pauseFile = MakePath(gAppDir, 'PAUSE')
    if IsFile(pauseFile) then do
      call SysFileDelete pauseFile
      say 'PAUSE file deleted from' gAppDir
    end
    if gVerbose then
      say 'Starting' gAppTitle 'from' directory()
    call SetLibPath
    signal off Error
    gAppName '-d..'
    signal on Error
    if RC \= 0 then do
      say gAppTitle 'exited with rc ' RC
      gErr = 1
    end
  end

  return

/* end DoStartf */

/*=== DoStatus() Set gErr 0 if running else set to 1 ===*/

DoStatus: procedure expose (Globals)

  if \ IsPidRunningEx() then do
    say gAppTitle 'is not running.  Will run from' gAppDir
    gErr = 1
  end
  else do
    if \ gVerbose then do
      /* FIXME maybe */
      if gPid == '' then
	say gAppTitle 'is running from' gAppDir 'but' gPidFile 'does not yet exist'
      else
	say gAppTitle 'is running from' gAppDir 'as pid' gPid'('d2x(gPid)')'
    end
  end
  return

/* end DoStatus */

/*=== DoStop([nostopfile]) Stop httpd and monitor maybe ===*/

/**
 * @param nostopfile if not omitted requests httpd stop only
 */

DoStop: procedure expose (Globals)
  parse arg nostopfile
  if \ IsPidRunningEx() then do
    say gAppTitle 'is not running'
    /* Stop monitor too */
    if IsMonitorRunning() then
      call MakeStopFileIf nostopfile
  end
  else do
    call MakeStopFileIf nostopfile
    call RunKiller '-TERM'
    if IsPidRunningEx(0) then do
      say gAppTitle 'did not stop'
      gErr = 1
    end
    else do
      say gAppTitle 'has stopped'
    end
  end
  return

/* end DoStop */

/*=== DoTest() Test ===*/

DoTest: procedure expose (Globals)
  call FindHttpConfPhpVersion
  call CheckPhpIniPhpVersion
  if gVerbose then
    say 'Testing' gAppTitle 'from' directory()
  call SetLibPath
  signal off Error
  gAppName '-d' gAppDir '-t'
  signal on Error
  return

/* end DoTest */

/*=== DoTestVHost() Test ===*/

DoTestVHost: procedure expose (Globals)
  call FindHttpConfPhpVersion
  call CheckPhpIniPhpVersion
  if gVerbose then
    say 'Testing' gAppTitle 'from' directory()
  call SetLibPath
  signal off Error
  gAppName '-d' gAppDir '-S' '| cat'		/* EOLs need fixing */
  signal on Error
  return

/* end DoTestVHost */

/*=== DoUnpause(arg) Delete pause file ===*/

DoUnpause: procedure expose (Globals)

  /* Ensure monitor running */
  if \ IsMonitorRunning('running') then
    return

  f = MakePath(gAppDir, 'PAUSE')
  if \ IsFile(f) then
    say 'Cannot access' f 'PAUSE file'
  else do
    call SysFileDelete f
    say 'PAUSE file deleted from' gAppDir
    do tries = 1 to 6
      running = IsPidRunningEx()
      if running then
	leave
      if tries = 1 then
	say 'Waiting for monitor to restart apache'
      call SysSleep 2
    end /* do tries */
    if running then
      say 'Monitor restarted apache'
    else do
      say 'Monitor did not restart apache - forcing restart'
      call DoStart
    end
  end
  return

/* end DoUnpause */

/*=== DoLynx() Lynx ===*/

DoLynx: procedure expose (Globals)
  say 'lynx is not yet supported'
  LYNX = 'lynx -dump'
  STATUSURL = 'http://localhost:80/server-status'
  /* $LYNX $STATUSURL | awk ' /process$/ { print; exit } { print } ' */
  return

/* end DoLynx */

/*=== DoLynx2() Lynx2 ===*/

DoLynx2: procedure expose (Globals)
  say 'lynx is not yet supported'
  LYNX = 'lynx'
  STATUSURL = 'http://localhost:80/server-status'
  /* $LYNX $STATUSURL | awk ' /process$/ { print; exit } { print } ' */
  return

/* end DoLynx2 */

/*=== Local support functions =======================================*/

/*=== CheckPhpIniPhpVersion() Check php.ini version matches httpd.conf php version ===*/

CheckPhpIniPhpVersion: procedure expose (Globals)

  /* Assume extension_dir of the form
       extension_dir = "d:/internet/php5/modules/"
     where subdir will be php5, php7, php8 etc.
     If extension_dir matches assume OK
     If not, tell user to install correct php.ini
     User needs to create php5.ini, php7.ini, php8 etc. to support this logic
  */

  call FindHttpConfPhpVersion
  parentDir = GetPath(gAppDir)

  if gPhpVersion \== 'none' then do

    phpVersionId = 'php' || gPhpVersion
    extensionDir = MakePath(parentDir, phpVersionId, 'modules')
    extensionDir = translate(extensionDir, '/','\')	/* To unix */
    etcDir = GetEnv('ETC')
    phpIniFile = MakePath(etcDir, 'php.ini')

    if \ IsFile(phpIniFile) then
      call Die 'Cannot access' phpIniFile

    ok = -1
    call stream phpIniFile, 'C', 'OPEN READ'
    do while lines(phpIniFile) > 0
      line = strip(linein(phpIniFile))
      if line == '' then iterate
      if left(line, 1) == ';' then iterate
      parse var line key '=' value
      key = strip(key)
      if translate(key) \== 'EXTENSION_DIR' then
	iterate
      iniExtensionDir = strip(value)
      iniExtensionDir = strip(iniExtensionDir, 'B', '"')	/* In case quoted */
      iniExtensionDir = strip(iniExtensionDir, 'T', '/')	/* In case have trailing slash */
      iniExtensionDir = translate(iniExtensionDir, '/','\')	/* To unix */
      ok = translate(iniExtensionDir) == translate(extensionDir)
      call VerboseMsg 'php.ini extension_dir is' extensionDir
    end /* while */
    call stream phpIniFile, 'C', 'CLOSE'

    if ok = -1 then
      call Die 'Cannot locate extension_dir in php.ini'

    if \ ok then do
      altPhpIniFile = MakePath(etcDir, phpVersionId, '.ini')
      if \ IsFile(altPhpIniFile) then do
	call Die 'httpd.conf requires' phpVersionId,,
		 'php.ini extension dir is' iniExtensionDir,,
		 'cannot access' altPhpIniFile
      end
      else do
	say
	say 'httpd.conf configured for' phpVersionId
	say 'php.ini extension_dir set to' iniExtensionDir
	say altPhpIniFile 'exists'
	say
	'dir /kmt' phpIniFile altPhpIniFile
	call AskYNQ 'Replace php.ini with' GetFileName(altPhpIniFile)
	if RESULT \= 0 then exit
	say
	signal off Error
	'copy /p' altPhpIniFile phpIniFile
	signal on Error
	say
	'dir /kmt' phpIniFile altPhpIniFile
	say
	say 'Restarting...'
	'call' gCmdName 'test'
	exit
      end
    end /* if ok */

  end /* if php enabled */

  return

/* end CheckPhpIniPhpVersion */

/*=== CheckRequired() Check required scripts available ===*/

CheckRequired: procedure expose (Globals)

  script = 'AddPath.cmd'
  s = SysSearchPath('PATH', script)
  if s == '' then
    call Die 'Cannot access' script 'in PATH'
  return

/* end CheckRequired */

/*=== FindAppDir() Find gAppExe and set gAppDir to parent directory or die ===*/

FindAppDir: procedure expose (Globals)

  if symbol('gAppDir') \== 'VAR' then do
    do 1
      runningAppDir = GetRunningAppDir()

      /* Look for gAppExe in current directory */
      gAppDir = directory()
      f = MakePath(gAppDir, gAppExe)
      if IsFile(f) then do
	/* Assume gAppExe in bin directory and parent directory is gAppDir */
	i = lastpos('\', gAppDir)
	if i > 0 then
	  gAppDir = left(gAppDir, i - 1)
      end

      /* Look for gAppExe in bin subdirectory */
      f = MakePath(gAppDir, 'bin', gAppExe)
      if IsFile(f) then do
	if runningAppDir \== '' & translate(runningAppDir) \== translate(gAppDir) then
	  call Die 'Cannot set AppDir to' gAppDir 'with running httpd process in' runningAppDir
	leave
      end

      /* If have running httpd instance, use it set gAppDir */
      if runningAppDir \== '' then do
	gAppDir = runningAppDir
	leave
      end

      /* Look in well known slainc directories */
      gAppDir = 'd:\Internet\Apache24'	/* slainc 2.4 */
      f = MakePath(gAppDir, 'bin', gAppExe)
      if IsFile(f) then leave
      /* FIXME to be gone */
      gAppDir = 'd:\Internet\Apache22'	/* slainc 2.2 */
      f = MakePath(gAppDir, 'bin', gAppExe)
      if IsFile(f) then leave

      /* Look in well known dnacih directories */
      gAppDir = 'D:\Apps\apache24'	/* dnacih 2.4 */
      f = MakePath(gAppDir, 'bin', gAppExe)
      if IsFile(f) then leave
      gAppDir = 'C:\Apps\apache24'	/* dnacih 2.4 */
      f = MakePath(gAppDir, 'bin', gAppExe)
      if IsFile(f) then leave
      /* FIXME to be gone */
      gAppDir = 'D:\Apps\apache2'	/* dnacih 2.2 */
      f = MakePath(gAppDir, 'bin', gAppExe)
      if IsFile(f) then leave
      gAppDir = 'C:\Apps\apache2'	/* dnacih 2.2 */
      f = MakePath(gAppDir, 'bin', gAppExe)
      if IsFile(f) then leave

      s = GetEnv('HOSTNAME')
      call Die 'Can not determine apache httpd base directory for host' s
    end /* do */
  end /* if */
  return

/* end FindAppDir */

/*=== FindCfgFile() Find httpd.conf ===*/

FindCfgFile: procedure expose (Globals)

  do while symbol('gCfgFile') \== 'VAR'
    /* Assume running from bin directory */
    gCfgFile = '..\conf\httpd.conf'
    if IsFile(gCfgFile) then leave
    call Die 'Can not access' gCfgFile
  end /* do */
  return

/* end FindCfgFile */

/*=== FindHttpConfPhpVersion() Set pPhpVersion from httpd.conf ==*/

FindHttpConfPhpVersion: procedure expose (Globals)

  if symbol('gPhpVersion') == 'LIT' then do
    call FindCfgFile
    call stream gCfgFile, 'C', 'OPEN READ'
    do while lines(gCfgFile) > 0
      /* Look for
	 LoadModule php5_module modules/modphp5.dll
	 LoadModule php7_module modules/modphp7.dll
      */
      line = strip(linein(gCfgFile))
      parse var line key mod dll
      if translate(key) \== 'LOADMODULE' then
	iterate
      select
      when mod == 'php_module' then do
	gPhpVersion = 8
	leave
      end
      when mod == 'php7_module' then do
	gPhpVersion = 7
	leave
      end
      when mod == 'php5_module' then do
	gPhpVersion = 5
	leave
      end
      otherwise
	nop
      end
    end /* while */
    call stream gCfgFile, 'C', 'CLOSE'

    /* FIXME */
    if symbol('gPhpVersion') == 'LIT' then do
      say 'php version set to none'
      gPhpVersion = 'none'
    end
    else
      call VerboseMsg 'httpd.conf configured for php version' gPhpVersion

  end /* if gPhpVersion */

  return

/* end FindHttpConfPhpVersion */

/*=== GetRunningAppDir() Return app dir for running gAppExe empty string ===*/

GetRunningAppDir: procedure expose (Globals)

  cnt = Sys2QueryProcessList('procs')
  if cnt == '' then
    call Die 'Sys2QueryProcessList failed with SYS2ERR' SYS2ERR

  minPid = ''
  appExePath = ''

  do procnum = 1 to procs.0
    procInfo = procs.procnum
    procInfo = ToLower(procInfo)
    parse var procInfo pid parent type priority cputime fullname
    iName = lastpos('\', fullname) + 1
    i = pos(gAppExe, fullname, iName)
    /* Select smallest pid */
    if i > 0 then do
      if minPid == '' then do
	minPid = pid
	appExePath = strip(fullname)
      end
      else if pid < minPid then do
	minPid = pid
      end
    end
  end /* procnum */

  if appExePath == '' then
    appDir = appExePath
  else do
    /* Assume app exe running in bin subdirectory - strip bin and gAppExe */
    i = lastpos('\', appExePath)
    appDir = left(appExePath, i - 1)	/* Strip gAppExe */
    i = lastpos('\', appDir)
    appDir = left(appDir, i - 1)	/* Strip \bin */
  end

  return appDir

/* end GetRunningAppDir */

/*=== IsMonitorRunning([expected]) Return true if monitor running ===*/

/**
 * @param expected is optional expected running state - running or stopped
 * If expected state not omitted and does not match actual state complain
 * for caller
 */

IsMonitorRunning: procedure expose (Globals)

  parse arg expected

  monitorFile = MakePath(gAppDir, 'MONITOR')
  monitorPid = ReadPidFromFile(monitorFile, 'query')
  if monitorPid \== '' then do
    running = IsPidRunning(monitorPid)
    if running then do
      info = Sys2QueryProcess(0, 'P')
      parse var info ourPid pp pt tail
      running = monitorPid \= ourPid
    end
  end
  else
    running = 0

  wanted = expected == 'running'
  if expected \== '' & running \= wanted then do
    if running then
      say 'Monitor is already running as pid' monitorPid
    else
      say 'Monitor is not running'
  end

  return running

/* end IsMonitorRunning */

/*=== Initialize() Initialize globals ===*/

Initialize: procedure expose (Globals)
  call SetCmdName
  call LoadRexxUtil
  call LoadRxUtilEx
  gEnv = 'OS2ENVIRONMENT'
  return

/* end Initialize */

/*=== MakeStopFileIf(nostopfile) Make stop file unless suppressed ===*/

/**
 * @param nostopfile suppresses stop file create unless arg omitted
 */

MakeStopFileIf: procedure expose (Globals)

  parse arg nostopfile
  if nostopfile = '' then do
    /* Create stop file */
    stopFile = MakePath(gAppDir, 'STOP')
    if \ IsFile(stopFile) then do
      call lineout stopFile, 'Stop file created at' date() time()
      call lineout stopFile
      say 'STOP file created in' gAppDir
    end
  end
  return

/* end MakeStopFileIf */

/*=== SetLibPath(arg) Set up BEGINLIBPATH for php5 DLLs and httpd modules ===*/

SetLibPath: procedure expose (Globals)

  if symbol('gLibPathSet') == 'LIT' then do

    /* Check httpd build date to know if BEGINLIBPATH needed
       Builds newer than 5/10/2022 support httpd.conf BeginLibPath and
	 2.4.53 pr#4
	 05-10-22  21:57       2,112,765    124  httpd.dll
       do not need BEGINLIBPATH setup
       FIXME to be gone when no longer running older httpd builds
    */

    wildCard = 'httpd.dll'
    /* T: YY/MM/DD/HH/MM Size ADHRS Name */
    call SysFileTree wildCard, 'fileList', 'FT'
    if RESULT \= 0 then
      call Die 'SysFileTree' wildCard 'failed'

    if fileList.0 \= 1 then
      call Die 'SysFileTree' wildCard 'failed to find' wildCard
    parse var fileList.1 fileDateTime fileBytes fileAttrib fileName

    needBeginLibPath = fileDateTime < '22/05/10/00/00'

    gLibPathSet = ''			/* Say checked */

    /* Point at php DLLs if php loaded */
    call FindHttpConfPhpVersion
    select
    when gPhpVersion = 8 then
      phpdir = 'php8'
    when gPhpVersion = 7 then
      phpdir = 'php7'
    when gPhpVersion = 5 then
      phpdir = 'php5'
    otherwise
      phpdir = ''
    end

    if phpdir \== '' & needBeginLibPath then do
      s = MakePath(GetPath(ChopDirSlash(gAppDir)), phpdir)
      if \ IsDir(s) then
	call Die 'Cannot access' s 'directory'
      call AddPath 'BEGINLIBPATH' s
      gLibPathSet = s
    end

    if needBeginLibPath then do
      /* Modules required for httpd modules that reference other modules (i.e. dav_fs -> dav) */
      s = MakePath(gAppDir, 'modules')
      if \ IsDir(s) then
	call Die 'Cannot access' s 'directory'
      call AddPath 'BEGINLIBPATH' s
      if gLibPathSet \= '' then
      if gLibPathSet == '' then
	gLibPathSet = s
      else
	gLibPathSet = s || ';' || gLibPathSet
    end

  end /* if LIT */
  return

/* end SetLibPath */

/*=== SetPidFile() Set gPidFile ===*/

SetPidFile: procedure expose (Globals)

  d = MakePath(gAppDir, 'logs')
  if \ IsDir(d) then
    call Die 'SetPidFile cannot access' d 'directory'
  gPidFile = MakePath(d , 'httpd.pid')
  return

/* end SetPidFile */

/*=== ScanArgsInit() ScanArgs initialization exit routine ===*/

ScanArgsInit: procedure expose (Globals) cmdTail swCtl keepQuoted

  if cmdTail == '' then
    call ScanArgsHelp

  /* Preset defaults */
  gVerbose = 0				/* Verbose messages */
  gArgList.0 = 0			/* Reset arg count */

  return

/* end ScanArgsInit */

/*=== ScanArgsSwitch() ScanArgs switch option exit routine ===*/

ScanArgsSwitch: procedure expose (Globals) curSw curSwArg

  select
  when curSw == 'f' then
    call ScanArgsUsage '-f is obsolete - use startf'
  when curSw == 'h' | curSw == '?' then
    call ScanArgsHelp
  when curSw == 'v' then
    gVerbose = 1
  when curSw == 'V' then do
    say gCmdName gVersion
    exit
  end
  otherwise
    call ScanArgsUsage 'switch '''curSw''' unexpected'
  end /* select */

  return

/* end ScanArgsSwitch */

/*=== ScanArgsArg() ScanArgs argument option exit routine ===*/

ScanArgsArg: procedure expose (Globals) curArg
  do 1
    /* Directory specs must contain \ or / or .
       App dir is parent of bin dir
    */
    if verify(curArg,'\/.', 'M') > 0 then
      if IsDir(curArg) then do
	do 1
	  dir = curArg
	  f = MakePath(dir, 'bin', gAppExe)
	  if IsFile(f) then
	    leave
	  f = MakePath(dir, gAppExe)
	  if IsFile(f) then do
	    if dir == '.' then
	      dir = directory()
	    /* gAppDir is parent of gAppExe directory */
	    i = lastpos('\', dir)
	    if i > 0 then
	      dir = left(dir, i - 1)
	    leave
	  end
	  say 'Cannot access httpd.exe or bin\httpd.exe from' curArg
	  call ScanArgsUsage 'Cannot map' curArg 'to httpd app directory'
	end
	call directory dir		/* Set working directory */
	leave
    end
    i = gArgList.0 + 1
    gArgList.i = curArg
    gArgList.0 = i
  end
  return

/* end ScanArgsArg */

/*=== ScanArgsTerm() ScanArgs scan end exit routine ===*/

ScanArgsTerm: procedure expose (Globals)
  if gArgList.0 = 0 then
    call ScanArgsUsage 'action required (i.e. start, stop)'
  return

/* end ScanArgsTerm */

/*=== ScanArgsHelp() Display ScanArgs usage help exit routine ===*/

ScanArgsHelp:
  say
  say 'Control httpd.'
  say
  say 'Usage:' gCmdName '[-h] [-v] [-V] [-?] [directory] action...'
  say
  say '  -h -?      Display this message'
  say '  -v         Enable verbose output'
  say '  -V         Display version'
  say
  say '  action     start startf stop restart kill graceful status config'
  say '             monitor pause, unpause, test, help'
  say '               start - start minimized'
  say '               startf - start in foreground'
  say '               restart - restart minimized'
  say '  directory  apache application directory - name must include \ or / (i.e. .\.)'
  exit 255

/* end ScanArgsHelp */

/*=== ScanArgsUsage(message) Report Scanargs usage error exit routine ===*/

ScanArgsUsage:
  parse arg msg
  say
  if msg \== '' then
    say msg
  say 'Usage:' gCmdName '[-h] [-v] [-V] [-?] [directory] action...'
  exit 255

/* end ScanArgsUsage */

/*==============================================================================*/
/*=== SkelRexxFunc standards - Delete unused - Move modified above this mark ===*/
/*==============================================================================*/

/*=== AskYNQ([prompt][, noskip[, nofocus]])) returns 0=Yes, 1=No, 2=Quit, skips line unless noskip ===*/

AskYNQ: procedure
  parse arg msg, noskip, nofocus

  /* Take focus with 4OS2 or fail if cannot match window title */
  /* If script designed for CMD too, use nofocus arg to avoid error noise */
  signal off Error
  /* Map 1st left bracket to wild card - [pid] seems to confuse activate */
  if nofocus = '' | nofocus \= 1 then
    '@if defined _WINTITLE activate "%@replace[[,*,%_WINTITLE]"'
  signal on Error

  /* Skip line unless suppressed by noskip arg - any non-zero value requests noskip */
  if noskip = '' | noskip = 0 then
    call lineout 'STDERR', ''

  if msg == '' then
    msg = 'Continue'
  call charout 'STDERR', msg '(y/n/q) ? '
  do forever
    key = translate(SysGetKey('NOECHO'))
    if key == 'Y' | key == 'N' then do
      call lineout 'STDERR', key
      if key == 'Y' then
	ynq = 0
      else
	ynq = 1
      leave
    end
    if key == 'Q' | c2x(key) == '1B' then do
      call lineout 'STDERR', ''
      ynq = 2
      leave
    end
  end /* forever */
  return ynq

/* end AskYNQ */

/*=== ChopDirSlash(directory) Chop trailing \ from directory name unless root ===*/

ChopDirSlash: procedure
  parse arg dir
  if right(dir, 1) == '\' & right(dir, 2) \== ':\' & dir \== '\' then
    dir = substr(dir, 1, length(dir) - 1)
  return dir

/* end ChopDirSlash */

/*=== CompareFiles(file1, file2) Compare files, return 1 if matched ===*/

CompareFiles: procedure expose (Globals)

  parse arg file1, file2

  /* May need to run under Classic REXX if called too many times */
  /* Assume files exist */

  bytes1 = stream(file1, 'C', 'QUERY SIZE')
  bytes2 = stream(file2, 'C', 'QUERY SIZE')

  matched = bytes1 == bytes2

  if matched then do

    call stream file1, 'C', 'OPEN READ'
    call stream file2, 'C', 'OPEN READ'

    offset1 = 1
    toRead = 1000			/* Use 1KB buffer for 1st read */
    do while offset1 <= bytes1
      if offset1 + toRead > bytes1 then
	toRead = bytes1 - offset1 + 1	/* Read to EOF */
      drop buf1				/* Try to avoid out of resources */
      drop buf2				/* Try to avoid out of resources */
      buf1 = charin(file1, , toRead)
      buf2 = charin(file2, , toRead)
      if buf1 \== buf2 then do
	matched = 0
	leave
      end
      offset1 = offset1 + toRead
      if offset1 = 1 + toRead then
	toRead = 10000000		/* Switch to 10MB buffer after 1st read */
    end /* while */

    call stream file1, 'C', 'CLOSE'
    call stream file2, 'C', 'CLOSE'

  end /* if */

  return matched

/* end CompareFiles */

/*=== FindEditor() Find non-GUI editor and set gEditor or Die ===*/

FindEditor: procedure expose (Globals)
  /* Requires GetEnv */
  /* Requires IsExeInPath */
  /* Uses gEditor from Globals */
  /* Uses EDITOR from environment */
  do while symbol('gEditor') \== 'VAR'
    gEditor = GetEnv('EDITOR')
    if gEditor \== '' then leave
    gEditor = 'vim.exe'			/* Force ext in case have vim alias */
    if IsExeInPath(gEditor) then leave
    gEditor = '4os2 /c vimx'
    if IsExeInPath('vimx.cmd') then leave
    gEditor = 'tedit'
    if IsExeInPath(gEditor) then leave
    call Die 'EDITOR not defined and cannot guess editor to use'
  end /* while */
  return

/* end FindEditor */

/*=== GetBaseName(pathName) Return file base name stripping drive path and extension ===*/

GetBaseName: procedure
  parse arg s
  /* return path name with drive, directory and last extension stripped */
  s = filespec('N', s)
  i = lastpos('.', s)
  if i > 1 then
    s = left(s, i - 1)			/* Chop extension */
  return s
/* end GetBaseName */

/*=== GetEnv(var) Return value for environment variable or empty string ===*/

GetEnv: procedure expose (Globals)
  parse arg var
  if var = '' then
    call Die 'GetEnv requires an argument'
  return value(var,, gEnv)

/* end GetEnv */

/*=== GetFileName(pathName) Return file name stripping drive and directory ===*/

GetFileName: procedure
  parse arg s
  /* returns file name with with drive and directory stripped */
  s = filespec('N', s)
  return s
/* end GetFileName */

/*=== GetPath(pathName) Return file drive and directory stripping file name and stripping backslash unless root ===*/

GetPath: procedure
  parse arg s
  /* Support dir\ and dir\file */
  i = lastpos('\', s)
  select
  when i >= 4 then
    s = left(s, i - 1)			/* Chop trailing backslash and file name */
  when substr(s, 2, 2) == ':\' then
    s = left(s, 3)			/* Chop file name keeping drive and root backslash */
  when left(s, 1) == '\' then
    s = left(s, 1)			/* Chop file name keeping root backslash */
  otherwise
    s = ''				/* No path */
  end
  return s
/* end GetPath */

/*=== GetProcessForPid(pid[, radix]) Return fully qualified process name for pid or empty string ===*/

/**
 * pid is decimal pid, unless radix, content or 0x prefix indicates hex
 * radix forces hex regardless of pid value, any non-blank value will do
 */

GetProcessForPid: procedure expose (Globals)
  /* Requires LoadRxUtilEx */
  parse arg pid, radix

  if pid = '' then
    call Die 'PID required'

  select
  when radix \== '' then
    radix = 'H'				/* Assume user knows */
  when  datatype(pid, 'X') then
    radix = 'H'
  when  datatype(pid, 'W') then
    radix = 'P'
  when translate(left(pid, 2)) == '0X' & datatype(substr(pid, 3), 'X') then do
    pid = substr(pid, 3)	/* GetProcessForPid cannot handle 0x prefix */
    radix = 'H'
  end
  otherwise
    call Die pid 'must be a decimal or hex PID number'
  end

  /* Get pid parent-pid process-type priority cpu-time executable-name */
  info = Sys2QueryProcess(pid, radix)

  if info = '' then
    procName = ''
  else do
    parse var info . . . . . procName
    procName = strip(procName)
  end

  return procName

/* end GetProcessForPid */

/*=== IsDir(dirName[, full]) return true if directory is valid, accessible directory ===*/

IsDir: procedure
  /* If arg(2) not omitted, return full directory name or empty string */
  parse arg dir, full
  newdir = ''

  do 1
    if dir == '' then do
      cwd = ''				/* No restore needed */
      leave
    end
    dir = translate(dir, '\', '/')	/* Convert to OS/2 slashes */
    s = strip(dir, 'T', '\')		/* Chop trailing slashes unless root */
    if s \== '' & right(s, 1) \== ":" then
      dir = s				/* Chop */
    drv = filespec('D', dir)
    cwd = directory()			/* Remember */
    /* If have drive letter and requested directory on some other drive */
    if drv \== '' & translate(drv) \== translate(left(cwd, 2)) then do
      /* Avoid slow failures and unwanted directory changes */
      drvs = SysDriveMap('A:')
      if pos(translate(drv), drvs) = 0 then
	leave				/* Unknown drive */
      if SysDriveInfo(drv) == '' then
	leave				/* Drive not ready */
      cwd2 = directory(drv)		/* Remember current directory on other drive */
      newdir = directory(dir)		/* Try to change and get full path name */
      call directory cwd2		/* Restore current directory on other drive */
      leave
    end

    /* If no drive letter or same drive and not UNC name */
    if left(dir, 2) \== '\\' then do
      newdir = directory(dir)		/* Try to change and get full path name */
      leave
    end

    /* UNC name - hopefully server is accessible or this will be slow
       Accept
	 \\server
	 \\server\
	 \\server\dir\
	 \\server\dir
     */
    cwd = ''				/* No restore needed */
    wc = dir
    if right(wc, 1) \== '\' then
      wc = wc || '\'
    i = lastpos('\', wc)
    if substr(wc, 3, 1) == '\' then
      leave				/* Malformed UNC - no server name */
    if pos('*', wc) > 0 | pos('?', wc) > 0 then
      leave				/* No wildcards allowed */
    call SysFileTree wc, 'files', 'O'
    if files.0 > 0 then do
      s = files.1			/* Exists and is not empty */
      i = lastpos('\', s)
      newdir = left(s, i - 1)		/* Extract directory name from full path name */
      leave
    end
    /* Try wildcarded directory name */
    wc = strip(wc, 'T', '\')
    i = lastpos('\', wc)
    base = substr(wc, i + 1)
    if base == '' then
      leave				/* Should have matched above */
    wc = substr(wc, 1, i) || '*' || base || '*'
    call SysFileTree wc, 'files', 'DO'
    do fileNum = 1 to files.0
      /* Find directory name is list */
      s = files.fileNum
      i = lastpos('\', s)
      s2 = substr(s, i + 1)
      if translate(base) == translate(s2) then do
	newdir = left(s, i - 1)
	leave
      end
    end /* i */
  end /* 1 */

  if cwd \== '' then
    call directory cwd			/* Restore original directory and drive */

  if full \== '' then
    ret = newdir			/* Return full directory name or empty string */
  else
    ret = newdir \== ''			/* Return true if valid and accessible */
  return ret

/* end IsDir */

/*=== IsExeInPath(exe, wantPath) return TRUE if executable is in PATH or full path ===*/

/**
 * @param exe is executable to find, .exe assumed if no extension specified
 * @param wantPath optionally requests full path return
 * @returns TRUE if executable found in PATH, FALSE if not
 * @returns full path or emtpy string if wantPath not empty
 */

IsExeInPath: procedure
  parse arg exe, wantPath
  if exe \ == '' then do
    i = lastpos('.', exe)
    j = lastpos('\', exe)
    if i = 0 | i < j then
      exe = exe || '.exe'		/* No extension */
    exe = SysSearchPath('PATH', exe)
  end
  if wantPath == '' then
    exe = exe \== ''			/* Want bool result */

  return exe				/* Return bool or full path */

/* end IsExeInPath */

/*=== IsFile(file) return true if arg is file and file exists ===*/

IsFile: procedure expose (Globals)
  parse arg file
  if file == '' then
    yes = 0
  else do
    /* '.' and '..' returns files in '.' or '..' - so avoid false positives */
    call SysFileTree file, 'files', 'F'
    if RESULT \= 0 then
      call Die 'SysFileTree' file 'failed'
    /* Assume caller knows if arg contains wildcards */
    yes = file \== '.' & file \== '..' & files.0 \= 0
  end
  return yes

/* end IsFile */

/*=== IsPidRunning(pidToChk) Returns TRUE if process is running ===*/

/**
 * Return 1 if process identified by pidToChk or gPidFile or gPid is running othewise return 0
 * @param pidToChk is optional pid to check
 * If pitToChk provided, default gPidFIle and gPid logic is overridden
 * If pidToChk omitted and gPidFile defined, read gPid from gPidFile
 * If pidToChk omitted and gPidFile not defined and gPid defined, check gPid
 * If gPidFile and gPid not defined, returns FALSE
 */

IsPidRunning: procedure expose (Globals)
  /* Requires LoadRxUtilEx Sys2QueryProcess */
  /* Requires ReadPidFromFile */
  /* Uses gAppName if defined */
  /* Uses gVerbose if defined */
  /* Uses gPidFile if defined and pidToChk omitted */
  /* Sets gPid if gPidFile used */
  /* Uses gPid if defined and pidToChk omitted */

  parse arg pidToChk

  do 1
    running = 0
    if pidToChk == '' then do
      if symbol('gPidFile') == 'VAR' then do
	pidToChk = ReadPidFromFile(gPidFile, 'query')
	if pidToChk == '' then
	  leave				/* Return not running */
	gPid = pidToChk			/* Got valid pid */
      end
      else if symbol('gPid') \== 'VAR' then
	call Die 'Expected gPid to be defined'
      pidToChk = gPid
    end /* do 1 */

    if \ datatype(pidToChk, 'W') then
      call Die 'pid' pidToChk 'must be non-zero decimal number'

    info = Sys2QueryProcess(pidToChk, 'P')
    if info == '' then
      leave				/* Return not running */

    /* info = pid parent type priority cputime fullname */
    /* If pid from file and app name known check pid file stale */
    if symbol('pid') == 'VAR' & symbol('gAppName') == 'VAR' then do
      /* No need to parse since fullname contains only slashes */
      iSlash = lastpos('\', info)
      if translate(gAppName) || .'EXE' \== substr(info, iSlash + 1) then
	leave				/* Pid file must be stale */
    end
    running = 1
    if symbol('gVerbose') \== 'VAR' then
      gVerbose = 0
    if gVerbose then do
      say 'pid         ppid ptype pri  cpu time executable name'
      parse var info p pp pt exe
      say left(p, 4) left('('d2x(pid)')', 6) left(pp, 4) left(pt, 5) strip(exe)
    end
  end /* do */
  return running

/* end IsPidRunning */

/*=== IsPidRunningEx([waitFor][,pidToChk]) Return TRUE if gPid running, optionally waits for expected state ===*/

/**
 * Return 1 if process identified by pidToChk or gPidFile or gPid is running othewise return 0
 * Optionally waits up to 10 seconds for process state to change
 * @param waitFor is state to wait for (0 = stopped, 1 = running, '' = don't wait)
 * @param pidToChk is optional pid to check which overrides default behavior
 * @returns 1 if process running else 0
 * @see IsPidRunning for pidToChk gPidFile and gPid usage
 */

IsPidRunningEx: procedure expose (Globals)
  /* Requires LoadRxUtilEx for Sys2QueryProcess */
  /* Requires IsPidRunning */
  /* Uses gVerbose if defined */

  parse arg waitFor, pidToChk

  do c = 1 to 10
    running = IsPidRunning(pidToChk)
    if waitFor == '' | waitFor == running then leave
    call SysSleep 1
  end

  if symbol('gVerbose') \== 'VAR' then
    gVerbose = 0
  if gVerbose & running then do
    if pidToChk == '' then
      pidToChk = gPid
    info = Sys2QueryProcess(pidToChk, 'P')
    say 'pid         ppid ptype pri  cpu time executable name'
    parse var info p pp pt tail
    say left(p, 4) left('('d2x(p)')', 6) left(pp, 4) left(pt, 5) tail
  end

  return running

/* end IsPidRunningEx */

/*=== LoadRxUtilEx() Load Alex's RxUtilEx functions ===*/

LoadRxUtilEx:
  if RxFuncQuery('Sys2LoadFuncs') then do
    call RxFuncAdd 'Sys2LoadFuncs', 'RXUTILEX', 'Sys2LoadFuncs'
    if RESULT then
      call Die 'Cannot load Sys2LoadFuncs'
    call Sys2LoadFuncs
  end
  return

/* end LoadRxUtilEx */

/*=== LogOpen() Open log file for append ===*/

LogOpen: procedure expose (Globals)
  /* Requires LogSetName unless gLogFile defined */
  /* Sets gLogFile if not defined */
  /* Overrides gLogFile if open fails */
  if symbol('gLogFile') \== 'VAR' then
    call LogSetName
  /* Assume closed */
  call stream gLogFile, 'C', 'OPEN WRITE'
  if stream(gLogFile) \== 'READY' then do
    gLogFile = '\' || gCmdName || '.log'	/* Try root */
    call stream gLogFile, 'C', 'OPEN WRITE'
  end
  return

/* end LogOpen */

/*=== LogSetDir() Set gLogDir and provide trailing backslash if needed ===*/

/**
 * Set gLogDir if gLogDir not defined
 * Tries %LOGFILES gTmpDir %TMP
 * Falls back to current directory and returns null string
 */

LogSetDir: procedure expose (Globals)
  if symbol('gLogDir') \== 'VAR' then do
    /* Try gLogDir %LOGFILES gTmpDir %TMP */
    do 1
      /* Try %LOGFILES */
      gLogDir = value('LOGFILES',, gEnv)
      if gLogDir \== '' then leave
      /* Try gTmpDir */
      if symbol('gTmpDir') == 'VAR' then do
	gLogDir = gTmpDir
	leave
      end
      /* Try %TMP - return empty string if TMP not defined */
      gLogDir = value('TMP',, gEnv)
    end
  end
  return

/* end LogSetDir */

/*=== LogSetName() Set log file name ===*/

/**
 * Sets gLogFile if not defined
 * Sets gLogDir if not defined
 */

LogSetName: procedure expose (Globals)
  /* Requires LogSetDir unless gLogDir defined */
  /* Requires gCmdName */
  if symbol('gLogFile') \== 'VAR' then do
    if symbol('gLogDir') \== 'VAR' then
      call LogSetDir
    /* Ensure trailing backslash unless using current directory */
    dir = gLogDir
    if dir \== '' & right(dir, 1) \== ':' & right(dir, 1) \== '\' then
      dir = dir || '\'			/* Ensure trailing backslash */
    gLogFile = dir || gCmdName'.log'
  end
  return

/* end LogSetName */

/*=== LogWriteVTSC(message,...) Write multi-line timestamped message to STDOUT and log file and close log ===*/

LogWriteVTSC: procedure expose (Globals)
  /* Requires LogOpen */
  /* Requires MakeTimestamp */
  if symbol('gLogFile') \== 'VAR' then
    call LogOpen
  ts = MakeTimestamp()
  do i = 1 to arg()
    say ts arg(i)
    call lineout gLogFile, ts arg(i)
    if symbol('gLogWrites') == 'VAR' then
      gLogWrites = gLogWrites + 1
  end
  call stream gLogFile, 'C', 'CLOSE'
  return

/* end LogWriteVTSC */

/*=== MakePath(pathparts,...) Make path name from parts ===*/

MakePath: procedure

  /* All parts optional - code guesses what caller means.
     If last arg begins with a dot and is not .. and does not
     contain a slash, it is assumed to be a file extension.
     To avoid this behavior, pass empty arg as last arg.
     Empty args are ignored.
     Automatically converts unix slashes to dos slashes.
     If 1st arg is drive letter, it must have trailing colon.
   */

  argCnt = arg()

  path = ''

  do argNum = 1 to argCnt
    s = arg(argNum)
    s = translate(s, '\', '/')		/* Ensure DOS */
    if s == '' & argNum = argCnt then
      iterate				/* Ignore nul last arg */
    if argNum = 1 then
      path = s
    else do
      lead = left(s, 1)
      tail = right(path, 1)
      if tail == ':' & argNum = 2 then
	path = path || s		/* Append path part to drive spec */
      else if lead == '.' & argNum = argCnt & s \== '..' & pos('\', s) = 0  then
	path = path || s		/* Assume extension unless .. or contains \ */
      else if tail == '\' & lead == '\' then
	path = path || substr(s, 2)	/* Drop extra backslash */
      else if path \== '' & tail \== '\' & lead \== '\' then
	path = path || '\' || s		/* Ensure have backslash */
      else
	path = path || s
    end
  end /* for */

  return path

/* end MakePath */

/*=== MakeTimestamp() Convert current date/time to sorted, delimited timestamp - yyyy/mm/dd-hh:mm:ss ===*/

MakeTimestamp: procedure
  /* Return yyyy/mm/dd-hh:mm:ss */
  return translate('ABCD/EF/GH',date('S'),'ABCDEFGH')'-'time()

/* end MakeTimestamp */

/*=== ReadLine1FromFile(file) Read first line from file, return line or empty sting ===*/

ReadLine1FromFile: procedure expose (Globals)
  parse arg fileName
  if fileName = '' then
    call Die 'File name argument required'
  line = ''
  /* OK for file to not exist */
  if stream(fileName, 'C', 'QUERY EXISTS') \== '' then do
    call stream fileName, 'C', 'OPEN READ'
    if lines(fileName) > 0 then
      line = strip(linein(fileName))
    call stream fileName, 'C', 'CLOSE'
  end
  return line

/* end ReadLine1FromFile */

/*=== ReadPidFromFile(fileName[, query]) read decimal pid from file and return pid ===*/

/**
 * Read pid from file and return pid
 * Dies if pid file not found and not query mode
 * Dies if pid file content corrupted
 * @param fileName is pid file
 * @param query is optional query mode request - OK for pid file to not exist
 * @returns decimal pid or empty string if pid can not be read
 */

ReadPidFromFile: procedure expose (Globals)

  /* requires ReadLine1FromFile */
  /* requires IsFile */
  parse arg fileName, query
  if \ IsFile(fileName) then do
    if query \== '' then
      pid = ''				/* Return empty string if pid file not found */
    else
      call Die 'Cannot access pid file' fileName
  end
  else do
    pid = ReadLine1FromFile(fileName)
    if \ datatype(pid, 'W') then
      call Die 'Pid file' fileName 'contains corrupted value' pid
  end

  return pid

/* end ReadPidFromFile */

/*=== RunKiller(signal) Run process killer, default signal is kill ===*/

/**
 * Sends signal to named process
 * Sends -kill signal unless overridden
 */

RunKiller: procedure expose (Globals)

  /* Requires IsExeInPath */
  /* Requires gPid */
  /* Requires gKiller */
  /* Sets gKiller if not defined */

  parse arg sig

  if sig == '' then
    sig = '-KILL'

  if symbol('gPid') \== 'VAR' then
    call Die 'gPid not defined'

  if \ datatype(gPid, 'W') | gPid < 2 then
    call Die 'Pid "' || gPid || '" is not a valid pid'

  if symbol('gKiller') \== 'VAR' then do
    do 1
      gKiller = 'apache_kill'
      if IsExeInPath(gKiller) then leave
      gKiller = 'emxkill'
      if IsExeInPath(gKiller) then leave
      call Die 'Can not select killer'
    end
  end
  select
  when gKiller == 'apache_kill' then nop
  when gKiller == 'emxkill' then nop
  otherwise
    /* Convert signal name to number */
    sigTbl = '-HUP 1 -KILL -9 -TERM 15 -USR1 16'
    do i = 1 to words(sigTbl) by 2
      if translate(sig) == word(sigTbl, i) then do
	sig = word(sigTbl, i + 1)
	leave
      end
    end
  end
  cmd = gKiller sig gPid
  say cmd || '('d2x(gPid)')'
  '@' || cmd

  return

/* end RunKiller */

/*=== SetFocus() Set focus to our window ===*/

SetFocus: procedure

  /* Take focus if running 4OS2 or fail silently if cannot match window title */
  signal off Error
  '@if "%_DOS%" == "OS2" activate "%@replace[[,*,%_WINTITLE]"'	/* Take focus if running 4OS2 */
  signal on Error
  return

/* end SetFocus */

/*=== ToLower(s) Convert to lower case ===*/

ToLower: procedure
  parse arg s
  return translate(s, xrange('a', 'z'), xrange('A', 'Z'))

/* end ToLower */

/*=== VerboseMsg([level, ]message,...) Write multi-line message to STDERR if verbose ===*/

VerboseMsg: procedure expose (Globals)
  /* Requires gVerbose */
  level = arg(1)
  if datatype(level, 'W') then
    start = 2
  else do
    level = 1
    start = 1
  end
   if level <= gVerbose then do
    do i = start to arg()
      call lineout 'STDERR', arg(i)
    end
  end
  return

/* end VerboseMsg */

/*==========================================================================*/
/*=== SkelRexx standards - Delete unused - Move modified above this mark ===*/
/*==========================================================================*/

/*=== Die([message,...]) Write multi-line message to STDERR and die ===*/

Die:
  call lineout 'STDERR', ''
  do i = 1 to arg()
    call lineout 'STDERR', arg(i)
  end
  call lineout 'STDERR', gCmdName 'aborting at line' SIGL || '.'
  call beep 200, 300
  call SysSleep 2
  exit 254

/* end Die */

/*=== Error() Set gErrCondition; report to STDOUT; trace and exit or return if called ===*/

Error:
  say
  parse source . . cmd
  gErrCondition = condition('C')
  say gErrCondition 'signaled at line' SIGL 'of' cmd || '.'
  if condition('D') \== '' then
    say 'REXX reason =' condition('D') || '.'
  if gErrCondition == 'SYNTAX' & symbol('RC') == 'VAR' then
    say 'REXX error =' RC '-' errortext(RC) || '.'
  else if symbol('RC') == 'VAR' then
    say 'RC =' RC || '.'
  say 'Source =' sourceline(SIGL)

  if condition('I') \== 'CALL' | gErrCondition == 'NOVALUE' | gErrCondition == 'SYNTAX' then do
    trace '?A'
    say 'Enter REXX commands to debug failure.  Press enter to exit script.'
    nop
    if symbol('RC') \== 'VAR' then
      RC = 255
    exit RC
  end

  return

/* end Error */

/*=== Halt() Report HALT condition to STDOUT and exit ===*/

Halt:
  say
  parse source . . cmd
  say condition('C') 'signaled at' cmd 'line' SIGL || '.'
  say 'Source =' sourceline(SIGL)
  say 'Sleeping for 2 seconds...'
  call SysSleep 2
  exit 253

/* end Halt */

/*=== LoadRexxUtil() Load RexxUtil functions ===*/

LoadRexxUtil:
  if RxFuncQuery('SysLoadFuncs') then do
    call RxFuncAdd 'SysLoadFuncs', 'REXXUTIL', 'SysLoadFuncs'
    if RESULT then
      call Die 'Cannot load SysLoadFuncs.'
    call SysLoadFuncs
  end
  return

/* end LoadRexxUtil */

/*=== ScanArgs(cmdLine) Scan command line ===*/

ScanArgs: procedure expose (Globals)

  /* Calls user exits to process arguments and switches */

  parse arg cmdTail
  cmdTail = strip(cmdTail)

  call ScanArgsInit

  /* Ensure optional settings initialized */
  if symbol('SWCTL') \== 'VAR' then
    swCtl = ''				/* Switches that take args, append ? if optional */
  if symbol('KEEPQUOTED') \== 'VAR' then
    keepQuoted = 0			/* Set to 1 to keep arguments quoted */

  /* Scan */
  curArg = ''				/* Current arg string */
  curSwList = ''			/* Current switch list */
  /* curSwArg = '' */			/* Current switch argument, if needed */
  noMoreSw = 0				/* End of switches */

  do while cmdTail \== '' | curArg \== '' | curSwList \== ''

    /* If arg buffer empty, refill */
    if curArg == '' then do
      qChar = left(cmdTail, 1)		/* Remember quote */
      if \ verify(qChar,'''"', 'M') then
	parse var cmdTail curArg cmdTail	/* Not quoted */
      else do
	/* Arg is quoted */
	curArg = ''
	do forever
	  /* Parse dropping quotes */
	  parse var cmdTail (qChar)quotedPart(qChar) cmdTail
	  curArg = curArg || quotedPart
	  /* Check for escaped quote within quoted string (i.e. "" or '') */
	  if left(cmdTail, 1) \== qChar then do
	    cmdTail = strip(cmdTail)	/* Strip leading whitespace */
	    leave			/* Done with this quoted arg */
	  end
	  curArg = curArg || qChar	/* Append quote */
	  if keepQuoted then
	    curArg = curArg || qChar	/* Append escaped quote */
	  parse var cmdTail (qChar) cmdTail	/* Strip quote */
	end /* do forever */
	if keepQuoted then
	  curArg = qChar || curArg || qChar	/* requote */
      end /* if quoted */
    end /* if curArg empty */

    /* If switch buffer empty, refill */
    if curSwList == '' & \ noMoreSw then do
      if left(curArg, 1) == '-' & curArg \== '-' then do
	if curArg == '--' then
	  noMoreSw = 1
	else
	  curSwList = substr(curArg, 2)	/* Remember switch string */
	curArg = ''			/* Mark empty */
	iterate				/* Refill arg buffer */
      end /* if switch */
    end /* if curSwList empty */

    /* If switch in progress */
    if curSwList \== '' then do
      curSw = left(curSwList, 1)	/* Next switch */
      curSwList = substr(curSwList, 2)	/* Drop from pending */
      /* Check switch allows argument, avoid matching ? */
      if pos(curSw, translate(swCtl,,'?')) \= 0 then do
	if curSwList \== '' then do
	  curSwArg = curSwList		/* Use rest of switch string for switch argument */
	  curSwList = ''
	end
	else if curArg \== '' & left(curArg, 1) \== '-' then do
	  curSwArg = curArg		/* Arg string is switch argument */
	  curArg = ''			/* Mark arg string empty */
	end
	else if pos(curSw'?', swCtl) = 0 then
	  call ScanArgsUsage 'Switch "-' || curSw || '" requires an argument'
	else
	  curSwArg = ''			/* Optional arg omitted */
      end

      call ScanArgsSwitch		/* Passing curSw and curSwArg */
      drop curSwArg			/* Must be used by now */
    end /* if switch */

    /* If arg */
    else if curArg \== '' then do
      noMoreSw = 1
      call ScanArgsArg			/* Passing curArg */
      curArg = ''
    end

  end /* while not done */

  call ScanArgsTerm

  return

/* end ScanArgs */

/*=== SetCmdName() Set gCmdName to short script name ===*/

SetCmdName: procedure expose (Globals)
  parse source . . cmd
  cmd = filespec('N', cmd)		/* Chop path */
  c = lastpos('.', cmd)
  if c > 1 then
    cmd = left(cmd, c - 1)		/* Chop extension */
  gCmdName = translate(cmd, xrange('a', 'z'), xrange('A', 'Z'))	/* Lowercase */
  return

/* end SetCmdName */

/* eof */
