/*****************************************************************************/
/*
 *  exq_rpt.c - exceptq report generator
 *  $Id: exq_rpt.c,v 1.11 2023/02/24 05:18:41 Steven Exp $
 *
 *  Parts of this file are
 *    Copyright (c) 2000-2020 Steven Levine and Associates, Inc.
 *    Copyright (c) 2010-2011 Richard L Walsh
 *
 * 2012-08-24 SHL Add some SMP support logic
 * 2012-11-18 SHL Avoid internal exceptions when exceptq reentered on SMP systems
 * 2013-03-28 SHL Enhance HLL mode formatting to show function offsets like xqs
 * 2013-04-01 SHL Show dll date/time/size to preserve SHL's sanity
 * 2013-05-08 SHL Support "3" option - assume 32-bit stack
 * 2013-05-09 SHL Format ExceptionInfo data whenever available
 * 2013-05-14 SHL Report signal exceptions when "Z" option enabled
 * 2013-09-20 SHL Decode floating point context when availabl
 * 2014-02-07 SHL Support EXCEPTQDIR
 * 2015-08-22 SHL Choose Labels on the Stack start address better
 * 2016-07-05 SHL Show process file info to preserve SHL's sanity
 * 2017-03-06 SHL Correct PrintFPSW IE formatting typo
 * 2018-07-08 SHL Report errno if .trp file open failed
 * 2018-07-08 SHL Show hostname
 * 2020-04-12 SHL Correct distorm decode address range logic
 * 2020-04-12 SHL Report OS2KRNL size and timestamp
 * 2020-04-12 SHL Report DLLs loaded in high-memory
 * 2020-05-16 SHL More distorm debugging - still can get illegal accesses
 * 2020-05-17 SHL Avoid XCPT_PROCESS_TERMINATE memory access failures
 * 2020-05-18 SHL Use QueryMem where DosQueryMem might give false positive
 * 2020-05-19 SHL Report possible DLL load failure
 */
/*****************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <ctype.h>

#define INCL_DOS
#define INCL_DOSERRORS
#include <os2.h>

#include "exq.h"
#include "distorm.h"

// #define DISTORM_DEBUG 1      // 2012-11-10 SHL Enable for distorm exception debugging

/*****************************************************************************/
/** 32-bit stuff **/

#define FLATCS ((USHORT)0x5B)
#define FLATDS ((USHORT)0x53)

/* flags used by PrintMemory() */
#define XQM_FMT8      0x08              /* aka bytes per line */
#define XQM_FMT16     0x10
#define XQM_FMT32     0x20
#define XQM_DWORDS    0x100
#define XQM_WORDS     0x200
#define XQM_BYTES     0x400
#define XQM_CHARS     0x800
#define XQM_FMTMASK   (XQM_FMT8 | XQM_FMT16 | XQM_FMT32)

/* struct used by disassemble routines to reduce stack usage */
typedef struct {
  USHORT    seg16;
  USHORT    addr16;
  ULONG     addr32;
  ULONG     bytesBefore;
  ULONG     bytesAfter;
  ULONG     codeBefore;
  ULONG     codeAfter;
} XQ_DECODE;

/* distorm interface */
typedef _DecodeResult (_System DISTORM_DECODE32)(
                      _OffsetType, const unsigned char*, int, _DecodeType,
                      _DecodedInst*, unsigned int, unsigned int*);

/* distorm insists on decoding a minimum of 15 instructions */
#define DISTORM_CNT   16

APIRET QueryMem(PVOID pb, PULONG pcb, PULONG pFlag);

/*****************************************************************************/
/** 16-bit stuff **/

/* dis386 interface */
#pragma pack(1)
typedef struct
{
    SHORT  ilen;        /* Instruction length */
    LONG   rref;        /* Value of any IP relative storage reference */
    USHORT sel;         /* Selector of any CS:eIP storage reference. */
    LONG   poff;        /* eIP value of any CS:eIP storage reference. */
    CHAR   longoper;    /* YES/NO value. Is instr in 32 bit operand mode? * */
    CHAR   longaddr;    /* YES/NO value. Is instr in 32 bit address mode? * */
    CHAR   buf[40];     /* String holding disassembled instruction * */
} *_Seg16 RETURN_FROM_DISASM;
#pragma pack()

typedef RETURN_FROM_DISASM (CDECL16 DISASM)(char* _Seg16 pSource,
                                            USHORT usIPvalue, USHORT usSegSize);

/** external function prototype **/
APIRET16 APIENTRY16 DOS16SIZESEG(USHORT Seg, ULONG* _Seg16 Size);

/*****************************************************************************/

/** global variables **/

FILE *          hTrap;
char            szModName[CCHMAXPATH];  // Set by exq_cv or exq_dbg
char            szBuffer[1024];
char            bigBuf[2048];           // Used for formatting

/** local variables **/

static int      cTrapCount;
static PTIB     ptib;
static PPIB     ppib;
static ULONG    ulStartTime;
static char *   pStackTop;
static char     szFileName[CCHMAXPATH];

// Used to avoid trapping when handling XCPT_PROCESS_TERMINATE
static BOOL     fIsProcessTerminateInDll;       // 2020-05-17 SHL
static HMODULE  hModTerminateDll;               // 2020-05-17 SHL

static XQ_DECODE          xd;
static RETURN_FROM_DISASM pDis386Data;
static _DecodedInst       diResults[DISTORM_CNT];

/*****************************************************************************/

/** formatting strings **/

static char *  pszTopLine =
  "\n______________________________________________________________________\n\n";

static char *  pszBotLine =
    "______________________________________________________________________\n\n";

static char *  pszFmtLine =
    " %s\n";

static char *  pszStackHdr =
    "   %3s     Address    Module     Obj:Offset    Nearest Public Symbol\n"
    " --------  ---------  --------  -------------  -----------------------\n";

static char *  pszRegEAX =
    " EAX : %08lX   EBX  : %08lX   ECX : %08lX   EDX  : %08lX\n";
static char *  pszRegESI =
    " ESI : %08lX   EDI  : %08lX\n";
static char *  pszRegESP =
    " ESP : %08lX   EBP  : %08lX   EIP : %08lX   EFLG : %08lX\n";
static char *  pszRegCS =
    " CS  : %04hX       CSLIM: %08lX   SS  : %04hX       SSLIM: %08lX\n";
static char *  pszRegDS =
    " DS  : %04hX       ES   : %04hX       FS  : %04hX       GS   : %04hX\n";

static char *  pszDupLine =
    " %08lX : %ld lines not printed duplicate the line above\n";

static char *  pszStackInfoHdr32 =
    "   Size       Base        ESP         Max         Top\n";
static char *  pszStackInfoHdr16 =
    "        Size       Base         SS:SP         Max          Top\n";
static char *  pszStackInfoNA =
    " %08lX   %08lX ->   N/A    -> %08lX -> %08lX\n";
static char *  pszStackInfo32 =
    " %08lX   %08lX -> %08lX -> %08lX -> %08lX\n";
static char *  pszStackInfo16 =
    " 16 : %08lX   %04hX:%04hX -> %04hX:%04hX -> %04hX:%04hX -> %04hX:%04hX\n";
static char *  pszStackInfo1632 =
    " 32 : %08lX   %08lX  -> %08lX  -> %08lX  -> %08lX\n";

static char *pszUnknownAddress = "unknown address";

static PSZ apszFPTagNames[] = {
  "ok",
  "zero",
  "spcl",
  "empty"
};

/*****************************************************************************/

#if 0                                   // For reference
#define XCPT_ACCESS_VIOLATION           0xC0000005
#define XCPT_IN_PAGE_ERROR              0xC0000006
#define XCPT_ILLEGAL_INSTRUCTION        0xC000001C
#define XCPT_INVALID_LOCK_SEQUENCE      0xC000001D
#define XCPT_NONCONTINUABLE_EXCEPTION   0xC0000024
#define XCPT_INVALID_DISPOSITION        0xC0000025
#define XCPT_BAD_STACK                  0xC0000027
#define XCPT_INVALID_UNWIND_TARGET      0xC0000028
#define XCPT_ARRAY_BOUNDS_EXCEEDED      0xC0000093
#define XCPT_INTEGER_DIVIDE_BY_ZERO     0xC000009B
#define XCPT_INTEGER_OVERFLOW           0xC000009C
#define XCPT_PRIVILEGED_INSTRUCTION     0xC000009D
#define XCPT_BREAKPOINT                 0xC000009F
#define XCPT_SINGLE_STEP                0xC00000A0
#define XCPT_FLOAT_DENORMAL_OPERAND     0xC0000094
#define XCPT_FLOAT_DIVIDE_BY_ZERO       0xC0000095
#define XCPT_FLOAT_INEXACT_RESULT       0xC0000096
#define XCPT_FLOAT_INVALID_OPERATION    0xC0000097
#define XCPT_FLOAT_OVERFLOW             0xC0000098
#define XCPT_FLOAT_STACK_CHECK          0xC0000099
#define XCPT_FLOAT_UNDERFLOW            0xC000009A
#define XCPT_B1NPX_ERRATA_02            0xC0010004
#endif

typedef struct {
  ULONG   ul;
  char *  psz;
} ULPSZ;

ULPSZ   xcpt[] =
  {
    {XCPT_ACCESS_VIOLATION,         "Access Violation"},
    {EXCEPTQ_DEBUG_EXCEPTION,       "Exceptq Debug Request"},
    {XCPT_IN_PAGE_ERROR,            "In Page Error"},
    {XCPT_ILLEGAL_INSTRUCTION,      "Illegal Instruction"},
    {XCPT_INVALID_LOCK_SEQUENCE,    "Invalid Lock Sequence"},
    {XCPT_NONCONTINUABLE_EXCEPTION, "Noncontinuable Exception"},
    {XCPT_INVALID_DISPOSITION,      "Invalid Disposition"},
    {XCPT_BAD_STACK,                "Bad Stack"},
    {XCPT_INVALID_UNWIND_TARGET,    "Invalid Unwind Target"},
    {XCPT_ARRAY_BOUNDS_EXCEEDED,    "Array Bounds Exceeded"},
    {XCPT_INTEGER_DIVIDE_BY_ZERO,   "Integer Divide By Zero"},
    {XCPT_INTEGER_OVERFLOW,         "Integer Overflow"},
    {XCPT_PRIVILEGED_INSTRUCTION,   "Privileged Instruction"},
    {XCPT_BREAKPOINT,               "Breakpoint"},
    {XCPT_FLOAT_DENORMAL_OPERAND,   "Float Denormal Operand"},
    {XCPT_FLOAT_DIVIDE_BY_ZERO,     "Float Divide By Zero"},
    {XCPT_FLOAT_INEXACT_RESULT,     "Float Inexact Result"},
    {XCPT_FLOAT_INVALID_OPERATION,  "Float Invalid Operation"},
    {XCPT_FLOAT_OVERFLOW,           "Float Overflow"},
    {XCPT_FLOAT_STACK_CHECK,        "Float Stack Check"},
    {XCPT_FLOAT_UNDERFLOW,          "Float Underflow"},
    {XCPT_B1NPX_ERRATA_02,          "B1NPX Errata 02"},
    // 2013-05-09 SHL
    {XCPT_PROCESS_TERMINATE,        "Process Termination Notice"},
    {XCPT_ASYNC_PROCESS_TERMINATE,  "Asynchronous Process Termination Notice"},
    {XCPT_SIGNAL,                   "Signal Received Notice"},
    {0x00000000,                    "Unknown Exception"}
  };

/*****************************************************************************/

/** local function prototypes **/

void    PrintReportHdr(void);
void    PrintReportFooter(void);
void    PrintException(EXCEPTIONREPORTRECORD* pExRepRec,
                       CONTEXTRECORD* pCtxRec);
BOOL    PrintTrapInfo(EXCEPTIONREPORTRECORD* pExRepRec,
                      CONTEXTRECORD* pCtxRec);
void    PrintCause(EXCEPTIONREPORTRECORD* pExRepRec);
void    PrintDebugInfo(EXCEPTIONREPORTRECORD* pExRepRec);       // 2013-05-09 SHL
void    PrintExceptionInfo(EXCEPTIONREPORTRECORD* pExRepRec);   // 2013-05-09 SHL

void    PrintDisassembly(CONTEXTRECORD* pCtxRec);
BOOL    SetDecodeLimits(CONTEXTRECORD* pCtxRec);
BOOL    PrintDis386Disassembly(PFN pfn);
BOOL    PrintDistormDisassembly(PFN pfn);
PFN     LoadDisassemblerDll(char* pszMod, ULONG ulOrd, char* pszProc);

void    PrintRegisters(CONTEXTRECORD* pCtxRec);
void    GetMemoryAttributes(ULONG ulAddress, char* pBuf);
void    Print16BitRegInfo(CONTEXTRECORD* pCtxRec);

void    PrintStackInfo(CONTEXTRECORD* pCtxRec);
void    PrintCallStack(EXCEPTIONREPORTRECORD* pExRepRec,
                       CONTEXTRECORD* pCtxRec);
char*   GetValidStackTop(void);
void    PrintLabelsOnStack(CONTEXTRECORD* pCtxRec);
void    PrintStackDump(CONTEXTRECORD* pCtxRec);

void    PrintRegisterPointers(CONTEXTRECORD* pCtxRec);
void    PrintMemoryAt(char* pMem, PSZ pszDesc, ULONG flags);
void    PrintMemory(char* pMem, ULONG cbMem, ULONG flags);
void    PrintMemoryLine(char* pSrc, ULONG cbSrc, ULONG flags);
void    PrintMemoryHdr(ULONG flags);

void    PrintDlls(void);
void    WalkStack(void* pvStackBottom, void* pvStackTop,
                  ULONG ulEBP, USHORT usSS, ULONG ulEIP, USHORT usCS);
void    TimedBeep(void);

size_t commafmt(char *pszBuf, UINT cBufSize, ULONG ulNumber);

// 2012-11-18 SHL SMP safety
extern UINT atomic_cmpxchg(UINT ulCode, volatile UINT* pulLock);

/*****************************************************************************/
/* Print exception report                                                    */
/*****************************************************************************/

void    ReportException(EXCEPTIONREPORTRECORD* pExRepRec,
                        EXCEPTIONREGISTRATIONRECORD* pExRegRec,
                        CONTEXTRECORD* pCtxRec,
                        BOOL isDebug)
{
  PPIB ppib2;
  PTIB ptib2;
  ULONG ulNest;
  APIRET apiret;
  UINT owner;
  CHAR szBuf[80];
  PSZ psz;                              // 2014-02-07 SHL

  static volatile UINT ulExceptqOwner;  // 2012-11-18 SHL SMP safety

  /* Get local copy */
  apiret = DosGetInfoBlocks(&ptib2, &ppib2);
  if (apiret) {
    fprintf(stderr, "** Exceptq DosGetInfoBlocks failed with error %d\n", apiret);
    return;                             // Oh heck
  }

  /* Try to own exceptq - we have lots of static variables */
  owner = atomic_cmpxchg(ptib2->tib_ptib2->tib2_ultid, &ulExceptqOwner);
  if (owner != 0 && owner != ptib2->tib_ptib2->tib2_ultid) {
    if (pCtxRec->ContextFlags & CONTEXT_CONTROL) {
      sprintf(szBuf, "cs:eip %04hX:%08lX",
              LOUSHORT(pCtxRec->ctx_SegCs), pCtxRec->ctx_RegEip);
    }
    fprintf(stderr, "** Secondary exception %08lX occurred in TID %ld at %s **\n",
            pExRepRec->ExceptionNum,
            ptib2->tib_ptib2->tib2_ultid,
            pCtxRec->ContextFlags & CONTEXT_CONTROL ? szBuf : pszUnknownAddress,
            LOUSHORT(pCtxRec->ctx_SegCs), pCtxRec->ctx_RegEip);
    fprintf(stderr, "** Exceptq report suppressed - exceptq in use by TID %ld **\n",
            owner);
    return;
  }

  /* Exceptq not in use by some other thread */
  ppib = ppib2;                         /* Make available globally */
  ptib = ptib2;                         /* Make available globally */

  /* Perform initialization if cTrapCount is zero (i.e. the app trapped). */
  if (!cTrapCount) {
    DosEnterMustComplete(&ulNest);
    cTrapCount++;

    /* Exit if the option to disable exceptq reporting is set. */
    InitOptions(NULL, NULL);
    if (!fReport) {
      DosUnsetExceptionHandler(pExRegRec);
      cTrapCount--;
      ulExceptqOwner = 0;
      DosExitMustComplete(&ulNest);
      return;
    }

    if (isDebug && !fDebug) {
      /* Was EXCEPTQ_DEBUG_EXCEPTION and not enabled - ignore */
      cTrapCount--;
      ulExceptqOwner = 0;
      DosExitMustComplete(&ulNest);
      return;
    }

    DosError(FERR_DISABLEEXCEPTION | FERR_DISABLEHARDERR);
    DosQuerySysInfo(QSV_MS_COUNT, QSV_MS_COUNT,
                    &ulStartTime, sizeof(ulStartTime));
    TimedBeep();

    // 2014-02-07 SHL Support EXCEPTQDIR
    if (DosScanEnv("EXCEPTQDIR", &psz))
      psz = szFileName;                 // Assume variable not set
    else {
      strcpy(szFileName, psz);          // Assume given directory name
      psz = szFileName;
      if (*psz) {
        psz += strlen(psz) - 1;         // Point at last char
        if (*psz != '\\' && *psz != ':')
          *++psz = '\\';                // Ensure trailing backslash
        psz++;
      }
    }

    // Append generated file name
    sprintf(psz, "%04lX_%02lX.TRP",
            ppib->pib_ulpid, ptib->tib_ptib2->tib2_ultid);
    fprintf(stderr, "Creating %s\n", szFileName);

    hTrap = fopen(szFileName, "a");
    if (!hTrap) {
      // 2012-08-02 SHL Fallback to stderr
      // 2018-07-08 SHL Show errno
      fprintf(stderr, "Open %s for append failed with errno %d (OS/2 error %u) - writing to stderr\n",
              szFileName, errno, _doserrno);
      hTrap = stderr;
    }
    setbuf(hTrap, NULL);
  }

  else {
    /* We're here because exceptq or distorm trapped */
    UINT state;
    cTrapCount++;
    /* Prevent exceptq from attempting to handle any subsequent traps for this thread. */
    DosUnsetExceptionHandler(pExRegRec);

    /* Prevent debug exceptions from trying to continue execution. */
    if (isDebug)
      pExRepRec->fHandlerFlags |= EH_NONCONTINUABLE;

    /* Try to get output somewhere visible */
    if (!hTrap) {
      hTrap = stderr;
      setbuf(hTrap, NULL);
    }

    // 2012-08-24 SHL rework messaging
    if (pCtxRec->ContextFlags & CONTEXT_CONTROL) {
      ULONG ulObjNum;
      ULONG ulOffset;
      HMODULE hMod;
      sprintf(szBuf, "cs:eip %04hX:%08lX",
              LOUSHORT(pCtxRec->ctx_SegCs), pCtxRec->ctx_RegEip);
      apiret = DosQueryModFromEIP(&hMod, &ulObjNum, CCHMAXPATH,
                                  szModName, &ulOffset,
                                  (ULONG)pExRepRec->ExceptionAddress);
      if (!apiret) {
        PSZ psz = strrchr(szModName, '\\');
        PSZ psz2;
        if (!psz)
          psz = szModName;
        else {
          psz++;
          psz2 = strrchr(psz, '.');
          if (psz2)
            *psz = 0;
        }
        sprintf(szBuf + strlen(szBuf), " (%04X:%08lX) in %s",
                ulObjNum + 1,
                ulOffset,
                psz);
      }
    }

    // If file open, send output to both stderr and file otherwise just stderr
    for (state = hTrap == stderr ? 1 : 0; state < 2; state++) {
      FILE *hOut = state == 0 ? stderr : hTrap;
      fputs(pszTopLine, hOut);
      fprintf(hOut, " ** Exceptq trapped at %s **\n",
              pCtxRec->ContextFlags & CONTEXT_CONTROL ? szBuf : pszUnknownAddress);
      if (fIsProcessTerminateInDll)
        fprintf(hOut, " ** Check POPUPLOG.OS2 or STDERR for DLL load error **\n");
    } // for
    fputs(pszBotLine, hTrap);
    // Try to generate report for both exceptions
    // Probably will crash and burn on return to first exceptq instance
  } // if reentered

  /* Print Report */
  PrintReportHdr();
  PrintException(pExRepRec, pCtxRec);
  PrintRegisters(pCtxRec);
  PrintStackInfo(pCtxRec);
  PrintCallStack(pExRepRec, pCtxRec);
  PrintLabelsOnStack(pCtxRec);
  PrintStackDump(pCtxRec);
  PrintRegisterPointers(pCtxRec);
  PrintDlls();
  PrintReportFooter();

  /* Shut Down */
  CloseSymbolFiles();
  if (hTrap != stdout && hTrap != stderr && cTrapCount == 1) {
    fclose(hTrap);
    if (cTrapCount == 1)
      hTrap = NULL; // 2012-08-24 SHL in case invoked again
  }

  if (fBeep)
    DosBeep(440, 60);

  /* Unset the handler for fatal exceptions but not for debug exceptions. */
  // FIXME to be gone maybe - some exception handlers try to continue execution
  if (!isDebug)
    DosUnsetExceptionHandler(pExRegRec);

  /* Allow other threads to use exceptq in the unlikely event that
     high level exception handler decides to bypass application exit
  */
  cTrapCount--;
  ulExceptqOwner = 0;
  DosExitMustComplete(&ulNest);

  return;
}

/*****************************************************************************/
/*  The 'Exception Report' & 'End of exception report' sections              */
/*****************************************************************************/

void    PrintReportHdr(void)
{
  PSZ       psz;
  DATETIME  dt;
  ULONG     aul[2];

  TimedBeep();

  /* section header with date & time */
  fputs(pszTopLine, hTrap);
  fflush(hTrap);

  DosGetDateTime(&dt);
  fprintf(hTrap, " Exception Report - created %04hd/%02hd/%02hd %02hd:%02hd:%02hd\n",
          (USHORT)dt.year, (USHORT)dt.month, (USHORT)dt.day,
          (USHORT)dt.hours, (USHORT)dt.minutes, (USHORT)dt.seconds);
  fputs(pszBotLine, hTrap);

  /* app info string - skip if not present */
  if (fReportInfo) {
    fprintf(hTrap, pszFmtLine, szReportInfo);
    fputs("\n", hTrap);
  }

  /* Try to show hostname */
  psz = getenv("HOSTNAME");
  fprintf(hTrap, " Hostname:         %s\n", psz ? psz : "unknown");

  /* version - always print something */
  if (DosQuerySysInfo(QSV_VERSION_MAJOR, QSV_VERSION_MINOR,
                      aul, sizeof(aul))) {
    aul[0] = 0;
    aul[1] = 0;
  }
  fprintf(hTrap, " OS2/eCS Version:  %ld.%ld\n", aul[0] / 10, aul[1]);

  /* nbr of processors - skip if not supported */
  if (!DosQuerySysInfo(QSV_NUMPROCESSORS, QSV_NUMPROCESSORS,
                      &aul[0], sizeof(aul[0])))
    fprintf(hTrap, " # of Processors:  %ld\n", aul[0]);

  /* physical memory - always print something */
  if (DosQuerySysInfo(QSV_TOTPHYSMEM, QSV_TOTPHYSMEM,
                      &aul[0], sizeof(aul[0])))
    aul[0] = 0;
  fprintf(hTrap, " Physical Memory:  %ld mb\n", aul[0] / 0x100000);

  /* virtual address limit - skip if not supported */
  if (!DosQuerySysInfo(QSV_VIRTUALADDRESSLIMIT, QSV_VIRTUALADDRESSLIMIT,
                      &aul[0], sizeof(aul[0])))
    fprintf(hTrap, " Virt Addr Limit:  %ld mb\n", aul[0]);

  /* Exceptq version & build date */
  fprintf(hTrap, " Exceptq Version:  %s (%s)\n",
          EXCEPTQ_VERSION, __DATE__ " " __TIME__);

  return;
}

/*****************************************************************************/

void    PrintReportFooter(void)
{
  ULONG    ulEnd;

  DosQuerySysInfo(QSV_MS_COUNT, QSV_MS_COUNT, &ulEnd, sizeof(ulEnd));

  fputs(pszTopLine, hTrap);
  fprintf(hTrap, " End of Exception Report - report took %ld ms to generate\n",
          ulEnd - ulStartTime);
  fputs(pszBotLine, hTrap);

  return;
}

/*****************************************************************************/
/*  The 'Exception' section                                                  */
/*****************************************************************************/

void    PrintException(EXCEPTIONREPORTRECORD* pExRepRec,
                       CONTEXTRECORD* pCtxRec)
{
  ULPSZ *   pxcpt = xcpt;
  APIRET rc;
  FILESTATUS3 fs3;
  CHAR szFileBytes[16];

  if (DosQueryModuleName(ppib->pib_hmte, CCHMAXPATH, szFileName))
    strcpy(szFileName, "N/A");

  // Decode exception
  while (pxcpt->ul && pxcpt->ul != pExRepRec->ExceptionNum)
    pxcpt++;

  fputs(pszTopLine, hTrap);
  fprintf(hTrap, " Exception %08lX - %s\n",
          pExRepRec->ExceptionNum, pxcpt->psz);
  fputs(pszBotLine, hTrap);

  // 2016-07-05 SHL Show date/time/size of failing process
  memset(&fs3,0,sizeof(fs3));
  rc = DosQueryPathInfo(szFileName, FIL_STANDARD, &fs3, sizeof(fs3));
  if (rc)
    fprintf(hTrap, " Process:  %s\n", szFileName);
  else {
    commafmt(szFileBytes, sizeof(szFileBytes), fs3.cbFile);
    fprintf(hTrap, " Process:  %s (%02u/%02u/%02u %02u:%02u:%02u %s)\n",
            szFileName,
            fs3.fdateLastWrite.month,
            fs3.fdateLastWrite.day,
            fs3.fdateLastWrite.year + 1980,
            fs3.ftimeLastWrite.hours,
            fs3.ftimeLastWrite.minutes,
            fs3.ftimeLastWrite.twosecs,
            szFileBytes);
  }

  fprintf(hTrap, " PID:      %02lX (%lu)\n",
          ppib->pib_ulpid, ppib->pib_ulpid);
  fprintf(hTrap, " TID:      %02lX (%lu)\n",
          ptib->tib_ptib2->tib2_ultid, ptib->tib_ptib2->tib2_ultid);

  if (verbosity > XQ_CONCISE)
    fprintf(hTrap, " Slot:     %02lX (%lu)\n",
            ptib->tib_ordinal, ptib->tib_ordinal);

  fprintf(hTrap, " Priority: %lX\n",
          ptib->tib_ptib2->tib2_ulpri);
  fputs("\n", hTrap);

  if (PrintTrapInfo(pExRepRec, pCtxRec))
    PrintDisassembly(pCtxRec);

  return;
}

/**
 * Format number with commas
 * adapted from fm/2
 */

size_t commafmt(char *pszBuf,           // Output buffer
                UINT cBufSize,          // Buffer size, including nul
                ULONG ulNumber)         // Number to convert
{
  UINT cChars = 1;              // Number of characters generated (excluding nul)
  UINT cDigits = 1;             // For commas

  char *pch = pszBuf + cBufSize - 1;

  if (cBufSize < 2)
    goto ABORT;

  *pch-- = 0;                           // Stuff terminator
  --cBufSize;

  for (; cChars <= cBufSize; ++cChars, ++cDigits) {
    *pch-- = (CHAR)(ulNumber % 10 + '0');
    ulNumber /= 10;
    if (!ulNumber)
      break;
    if (cDigits % 3 == 0) {
      *pch-- = ',';
      ++cChars;
    }
    if (cChars >= cBufSize)
      goto ABORT;
  } // for

  strcpy(pszBuf, ++pch);                // Left align

  return cChars;

ABORT:
  *pszBuf = 0;
  return 0;
}

/*****************************************************************************/

BOOL    PrintTrapInfo(EXCEPTIONREPORTRECORD* pExRepRec,
                      CONTEXTRECORD* pCtxRec)
{
  APIRET    rc;
  ULONG     ulObjNum;
  ULONG     ulOffset;
  HMODULE   hMod;
  FILESTATUS3 fs3;
  char szFileBytes[16];

  TimedBeep();

  /* If the module can't be identified, print some dummy lines then exit. */
  if (DosQueryModFromEIP(&hMod, &ulObjNum, CCHMAXPATH,
                         szModName, &ulOffset,
                         (ULONG)pExRepRec->ExceptionAddress)) {
    if (verbosity > XQ_CONCISE)
      fputs(" Module:   N/A\n", hTrap);
    fputs(" Filename: N/A\n", hTrap);

    fprintf(hTrap, " Cause:    Invalid execution address %08lX\n",
            (ULONG)pExRepRec->ExceptionAddress);

    return FALSE;                       // Give up
  }

  ulObjNum++;

  if (verbosity > XQ_CONCISE)
    fprintf(hTrap, " Module:   %s\n", szModName);

  rc = DosQueryModuleName(hMod, CCHMAXPATH, szFileName);

  // 2013-04-01 SHL show date/time/size of failing module

  if (rc) {
    fprintf(hTrap, " Filename: %s\n", szModName);
    *szFileName = 0;                    // Use module name
  }
  else {
    memset(&fs3,0,sizeof(fs3));
    rc = DosQueryPathInfo(szFileName, FIL_STANDARD, &fs3, sizeof(fs3));
    if (rc) {
      fprintf(hTrap, " Filename: %s\n", szFileName);
      *szFileName = 0;                  // Use module name
    }
    else {
      commafmt(szFileBytes, sizeof(szFileBytes), fs3.cbFile);
      fprintf(hTrap, " Filename: %s (%02u/%02u/%02u %02u:%02u:%02u %s)\n",
              szFileName,
              fs3.fdateLastWrite.month,
              fs3.fdateLastWrite.day,
              fs3.fdateLastWrite.year + 1980,
              fs3.ftimeLastWrite.hours,
              fs3.ftimeLastWrite.minutes,
              fs3.ftimeLastWrite.twosecs,
              szFileBytes);
    }
  }

  // 2020-05-17 SHL special case
  if (pExRepRec->ExceptionNum == XCPT_PROCESS_TERMINATE) {
    /* When XCPT_PROCESS_TERMINATE is reported and EIP points to a DLL
       there are cases where the kernel claims that the
       memory pointed to by EIP is accessible, but it is not, so we must
       avoid attempting to access shared read/exec memory for this DLL.
       This most likely occurs because the DLL load has failed due
       to a missing ordinal.
       We cannot use a local exception handler to verify access because the
       exception will be marked as EH_NONCONTINUABLE.
       We tried to DosLoadModule to regain access and this did not work.
       We tried to DosLoadModule to check if the DLL load would fail and it did not.
       The best we can do is avoid accessing this memory and aim the
       user in the right direction.
    */
    PSZ pszDot = strrchr(szFileName, '.');
    if (pszDot && strcmp(pszDot, ".DLL") == 0) {
      PSZ pszSlash = strrchr(szFileName, '\\');
      if (pszSlash && strcmp(pszSlash + 1, "DOSCALL1.DLL") != 0) {
        fIsProcessTerminateInDll = TRUE;
        hModTerminateDll = hMod;
#       ifdef DISTORM_DEBUG // 2020-05-16 SHL distorm exception debugging
        fprintf(hTrap, " * PrintTrapInfo: fIsProcessTerminateInDll %u hModTerminateDll %x (%u)\n",
                fIsProcessTerminateInDll, hModTerminateDll, __LINE__);
#       endif
      } // if !DOSCALL1
    } // if pszDot

  } // if TERMINATE

  /* If the control registers are available, use them;
   * Otherwise, print the failing address.
   */
  if (~pCtxRec->ContextFlags & CONTEXT_CONTROL)
    fprintf(hTrap, " Address:  %08lX (%04lX:%08lX)\n",
            (ULONG)pExRepRec->ExceptionAddress, ulObjNum, ulOffset);
  else
  if (LOUSHORT(pCtxRec->ctx_SegCs) == FLATCS || pCtxRec->ctx_RegEip > 0x00010000)
    fprintf(hTrap, " Address:  %04hX:%08lX (%04lX:%08lX)\n",
            LOUSHORT(pCtxRec->ctx_SegCs), pCtxRec->ctx_RegEip, ulObjNum, ulOffset);
  else
    fprintf(hTrap, " Address:  %04hX:%04lX (%04lX:%08lX)\n",
            LOUSHORT(pCtxRec->ctx_SegCs), pCtxRec->ctx_RegEip, ulObjNum, ulOffset);

  if (pExRepRec->ExceptionNum == XCPT_ACCESS_VIOLATION)
    PrintCause(pExRepRec);
  else
  if (pExRepRec->ExceptionNum == EXCEPTQ_DEBUG_EXCEPTION)
    PrintDebugInfo(pExRepRec);
  else
    PrintExceptionInfo(pExRepRec);

  return (~pCtxRec->ContextFlags & CONTEXT_CONTROL) ? FALSE : TRUE;
}

/*****************************************************************************/

void    PrintCause(EXCEPTIONREPORTRECORD* pExRepRec)
{
  *szBuffer = 0;

  switch (pExRepRec->ExceptionInfo[0]) {
    case XCPT_READ_ACCESS:
      sprintf(bigBuf, "Attempted to read from %08lX",
              pExRepRec->ExceptionInfo[1]);
      GetMemoryAttributes(pExRepRec->ExceptionInfo[1], szBuffer);
      break;
    case XCPT_WRITE_ACCESS:
      sprintf(bigBuf, "Attempted to write to %08lX",
              pExRepRec->ExceptionInfo[1]);
      GetMemoryAttributes(pExRepRec->ExceptionInfo[1], szBuffer);
      break;
    case XCPT_EXECUTE_ACCESS:
      sprintf(bigBuf, "Attempted to execute at %08lX",
              pExRepRec->ExceptionInfo[1]);
      GetMemoryAttributes(pExRepRec->ExceptionInfo[1], szBuffer);
      break;
    case XCPT_SPACE_ACCESS:
      /* It looks like this is off by one... - FIXME to know why */
      sprintf(bigBuf, "Attempted to access beyond selector %08lX limit",
              pExRepRec->ExceptionInfo[1] ?
              pExRepRec->ExceptionInfo[1] + 1 : 0);
      break;
    case XCPT_LIMIT_ACCESS:
      strcpy(bigBuf, "Limit access fault");
      break;
    case XCPT_UNKNOWN_ACCESS:
      strcpy(bigBuf, "Unknown access fault");
      break;
    default:
      strcpy(bigBuf, "Other unknown access fault");
  }

  fprintf(hTrap, " Cause:    %s\n", bigBuf);
  if (*szBuffer)
    fprintf(hTrap, "           (%s)\n", szBuffer);

  return;
}

/*****************************************************************************/

void    PrintDebugInfo(EXCEPTIONREPORTRECORD* pExRepRec)
{
  int   ctr;

  fprintf(hTrap, " Cause:    Program requested an Exceptq debug report\n");

  if (!pExRepRec->cParameters ||
      pExRepRec->cParameters > EXCEPTION_MAXIMUM_PARAMETERS)
    return;

  fputs(" Debug:    ", hTrap);

  /* If ExceptionInfo[0] non-zero assume given format string and args
     otherwise just print ExceptionInfo 1..N in decimal and hex
  */
  if (pExRepRec->ExceptionInfo[0])
    vfprintf(hTrap, (char*)pExRepRec->ExceptionInfo[0], (char*)&pExRepRec->ExceptionInfo[1]);
  else {
    for (ctr = 1; ctr < pExRepRec->cParameters; ctr++) {
      fprintf(hTrap, "[%d]= %08lx  ", ctr, pExRepRec->ExceptionInfo[ctr]);
    }
  }

  fputs("\n", hTrap);

  return;
}

/*****************************************************************************/

// 2013-05-09 SHL Added

void    PrintExceptionInfo(EXCEPTIONREPORTRECORD* pExRepRec)
{
  int   ctr;

  if (!pExRepRec->cParameters ||
      pExRepRec->cParameters > EXCEPTION_MAXIMUM_PARAMETERS)
    return;

  fputs(" Exception Info: ", hTrap);

  for (ctr = 0; ctr < pExRepRec->cParameters; ctr++) {
    fprintf(hTrap, "[%d]= %08lx (%lu)  ", ctr, pExRepRec->ExceptionInfo[ctr], pExRepRec->ExceptionInfo[ctr]);
  }

  fputs("\n", hTrap);

  return;
}

/*****************************************************************************/

void    PrintDisassembly(CONTEXTRECORD* pCtxRec)
{
  PFN     pfn;

  if (verbosity < XQ_TERSE)
    return;

  TimedBeep();

  pfn = LoadDisassemblerDll("DISTORM", 0, "distorm_decode32");
  if (pfn) {
    if (!SetDecodeLimits(pCtxRec) ||
        !PrintDistormDisassembly(pfn))
      fputs(" Code:     failing instruction can not be disassembled\n", hTrap);
      if (fIsProcessTerminateInDll)
        fputs(" Code:     check POPUPLOG.OS2 or STDERR for DLL load error\n", hTrap);
    return;
  }

  pfn = LoadDisassemblerDll("DIS386", 1, 0);
  if (pfn) {
    if (!SetDecodeLimits(pCtxRec) ||
        !PrintDis386Disassembly(pfn))
      fputs(" Code:     failing instruction can not be disassembled\n", hTrap);
    return;
  }

  fputs(" Code:     disassembler dll not available (distorm or dis386)\n", hTrap);

  return;
}

/*****************************************************************************/

BOOL    SetDecodeLimits(CONTEXTRECORD* pCtxRec)
{
  void *    pv32;
  ULONG     ulAttr;
  ULONG     ulLimit;
  APIRET    apiret;

  TimedBeep();

  memset(&xd, 0, sizeof(XQ_DECODE));

  /* Set the number of instructions to display before & after the one
   * that trapped, plus the number of bytes to decode before & after.
  */
  if (verbosity < XQ_CONCISE) {
    xd.bytesBefore = 0;
    xd.codeBefore = 0;
    xd.codeAfter = 0;
  }
  else {
    xd.bytesBefore = 80;
    xd.codeBefore = 4;
    xd.codeAfter = 3;
  }
  xd.bytesAfter = 64;

  /* 32-bit code */
  if (LOUSHORT(pCtxRec->ctx_SegCs) == FLATCS || pCtxRec->ctx_RegEip > 0x00010000) {
    xd.addr32 = pCtxRec->ctx_RegEip;
    pv32 = (void*)xd.addr32;
  }
  /* 16-bit code - use an explicit thunk to get a flat address
     Assume in tiled memory
  */
  else {
    xd.seg16 = LOUSHORT(pCtxRec->ctx_SegCs);
    xd.addr16 = (USHORT)pCtxRec->ctx_RegEip;
    xd.addr32 = ((LOUSHORT(pCtxRec->ctx_SegCs) & ~7) << 13) | pCtxRec->ctx_RegEip;
    pv32 = (void*)xd.addr32;
  }

# ifdef DISTORM_DEBUG // 2020-05-16 SHL distorm exception debugging
  fprintf(hTrap, " * SetDecodeLimits in: pv32 %x xd.bytesAfter %u (%u)\n",
          pv32, xd.bytesAfter, __LINE__);
# endif

  /* confirm that the trapping address & the X bytes after it are accessible */
  // 2020-05-18 SHL Use extended DosQueryMem
  apiret = QueryMem(pv32, &xd.bytesAfter, &ulAttr);
  if (apiret != NO_ERROR || (ulAttr & (PAG_READ | PAG_EXECUTE)) != (PAG_READ | PAG_EXECUTE))
  {
#   ifdef DISTORM_DEBUG // 2020-05-16 SHL distorm exception debugging
    fprintf(hTrap, " * SetDecodeLimits out: apiret %u ulAttr 0x%x (%u)\n", apiret, ulAttr, __LINE__);
#   endif
    return FALSE;
  }

# ifdef DISTORM_DEBUG // 2020-05-16 SHL distorm exception debugging
  fprintf(hTrap, " * SetDecodeLimits out: xd.bytesAfter %u ulAttr 0x%x (%u)\n", xd.bytesAfter, ulAttr, __LINE__);
# endif

  /* If the address is less than 80 bytes from the start of the segment,
   * reduce bytes-before so we'll start at offset 0
   * because the prior segment might not be accessible.
  */
  // 2020-04-12 SHL Correct inverted mask logic - was accidentally disabled too
  if ((ulAttr & PAG_BASE) && (xd.addr32 & 0xfff) < xd.bytesBefore)
    xd.bytesBefore = xd.addr32 & 0xfff;

  /* For 16-bit code, ensure that the EIP lies within the segment,
   * and reduce the bytes-after figure if it exceeds the seg's limit.
  */
  if (xd.seg16) {
    if (DOS16SIZESEG(xd.seg16, &ulLimit) || xd.addr16 >= ulLimit)
      return FALSE;

    if (xd.addr16 + xd.bytesAfter > ulLimit)
      xd.bytesAfter = ulLimit - xd.addr16;
  }

  return TRUE;
}

/*****************************************************************************/

BOOL    PrintDis386Disassembly(PFN pfn)
{
  void *  pv32;

  TimedBeep();

  if (xd.seg16) {
    pDis386Data = ((DISASM*)pfn)(MAKE16P(xd.seg16, xd.addr16),
                                 xd.addr16, FALSE);
  }
  else {
    if (DosAllocMem(&pv32, xd.bytesAfter, PAG_READ | PAG_WRITE | PAG_COMMIT))
      return FALSE;

    memcpy(pv32, (void*)xd.addr32, xd.bytesAfter);
    pDis386Data = ((DISASM*)pfn)(pv32, (USHORT)pv32, TRUE);
    DosFreeMem(pv32);
  }

  fprintf(hTrap, " Code:     %s\n", pDis386Data->buf);

  return TRUE;
}

/*****************************************************************************/

BOOL    PrintDistormDisassembly(PFN pfn)
{
  ULONG     decode;
  ULONG     used;
  ULONG     ndx;
  ULONG     stop;
  ULONG     mneLth;
  ULONG     opLth;
  _DecodeType   dt;
  _DecodeResult dres;

  TimedBeep();

  /* Decode until the trapping address appears as the last instruction decoded,
   * or until the decode address is the trapping address.  With each iteration,
   * reduce the bytes decoded so the high-end limit is never exceeded.
   */

  dt = xd.seg16 ? Decode16Bits : Decode32Bits;
  xd.bytesBefore++;
  decode = xd.addr32 - xd.bytesBefore;

# ifdef DISTORM_DEBUG // 2012-11-10 SHL distorm exception debugging
  fprintf(hTrap, " * PrintDistormDisassembly in: xd.addr32 %x dt %u (%u)\n", xd.addr32, dt, __LINE__);
# endif

  do {
    xd.bytesBefore--;
    decode++;

    // Try to decode
    dres = ((DISTORM_DECODE32*)pfn)(decode, (char*)decode,
                                    xd.bytesBefore + xd.bytesAfter, dt,
                                    diResults, DISTORM_CNT, (UINT*)&used);
    if (dres == DECRES_INPUTERR) {
#     ifdef DISTORM_DEBUG // 2020-05-16 SHL distorm exception debugging
      fprintf(hTrap, " * PrintDistormDisassembly out: dres %u (%u)\n", dres, __LINE__);
#     endif
      return FALSE;
    }

#   if 0 // ifdef DISTORM_DEBUG // 2012-11-10 SHL distorm exception debugging
    fprintf(hTrap, " * PrintDistormDisassembly out: dres %u used %u (%u)\n", dres, used, __LINE__);
    if (used > 0) {
      // Show last
      fprintf(hTrap, " * PrintDistormDisassembly out: %x: %s %s %s\n",
             diResults[used - 1].offset,
             diResults[used - 1].mnemonic.p,
             diResults[used - 1].operands.p,
             diResults[used - 1].instructionHex.p);
    }

#   if 0 // FIXME disabled for now - FIXME to be gone?
    // Show all
    for (ndx = 0; ndx < used; ndx++) {
      fprintf(hTrap, " * %x: %s %s %s\n",
             diResults[ndx].offset,
             diResults[ndx].mnemonic.p,
             diResults[ndx].operands.p,
             diResults[ndx].instructionHex.p);
    }
#   endif // 0
#   endif // DISTORM_DEBUG

    if (!used)
      continue;                         // Try again

    /* If we started close to the beginning of the segment, the disassembler
     * may have decoded beyond the desired instruction.  If so, look for the
     * desired address and make it look like the last instruction decoded.
     * If it isn't found, then the decoder is out of sync, so we'll continue.
     */
    if (diResults[used-1].offset > xd.addr32) {
      for (ndx = 0; ndx < used; ndx++) {
        if (diResults[ndx].offset == xd.addr32) {
          used = ndx + 1;
          break;
        }
      }
    }

    /* If the last instruction decoded is the one we want, exit the loop. */
    if (diResults[used-1].offset == xd.addr32) {
#     ifdef DISTORM_DEBUG // 2012-11-10 SHL distorm exception debugging
      fprintf(hTrap, " * PrintDistormDisassembly matched: used %u (%u)\n", used, __LINE__);
#     endif
      break;
    }

  } while (decode < xd.addr32);

  if (!used)
    return FALSE;

  /* Prepare for the final decode.  Go back codeBefore entries
   * (if there are that many) and use its offset as the starting
   * address.  Calculate the number of bytes-before to decode.
   */
  if (xd.codeBefore >= used) {
    xd.codeBefore = used - 1;
# ifdef DISTORM_DEBUG // 2012-11-10 SHL distorm exception debugging
    fprintf(hTrap, " * PrintDistormDisassembly final: xd.codeBefore %u (%u)\n", xd.codeBefore, __LINE__);
# endif
  }

  decode = diResults[used - 1 - xd.codeBefore].offset;
  xd.bytesBefore = diResults[used - 1].offset - decode;

  TimedBeep();

# ifdef DISTORM_DEBUG // 2012-11-10 SHL distorm exception debugging
  // Show distorm inputs
  fprintf(hTrap, " * PrintDistormDisassembly in: decode %x xd.bytesBefore + xd.bytesAfter %u (%u)\n",
          decode, xd.bytesBefore + xd.bytesAfter, __LINE__);
# endif

  dres = ((DISTORM_DECODE32*)pfn)(decode, (char*)decode,
                                  xd.bytesBefore + xd.bytesAfter, dt,
                                  diResults, DISTORM_CNT, (UINT*)&used);
  if ( dres == DECRES_INPUTERR || !used) {
#   ifdef DISTORM_DEBUG // 2020-05-16 SHL distorm exception debugging
    fprintf(hTrap, " * PrintDistormDisassembly final: dres %u used %u (%u)\n", dres, used, __LINE__);
#   endif
    return FALSE;
  }

# ifdef DISTORM_DEBUG // 2012-11-10 SHL distorm debugging
  // Show results
  fprintf(hTrap, " * PrintDistormDisassembly final: dres %u used %u (%u)\n", dres, used, __LINE__);
# if 0 // FIXME disabled for now - FIXME to be gone?
  // Show all
  for (ndx = 0; ndx < used; ndx++) {
    fprintf(hTrap, " * %x: %s %s %s\n",
           diResults[ndx].offset,
           diResults[ndx].mnemonic.p,
           diResults[ndx].operands.p,
           diResults[ndx].instructionHex.p);
  }
# endif
# endif

  /* Set the index where the display should stop to the trap's entry
   * plus codeAfter entries but don't print more info than we have.
   */
  stop = xd.codeBefore + xd.codeAfter + 1;
  if (stop > used)
    stop = used;

  /* Determine the maximum widths of the mnemonic and operand
   * strings so we can format the display into tidy little columns.
   */
  mneLth = 0;
  opLth = 0;
  for (ndx = 0; ndx < stop; ndx++) {
    if (diResults[ndx].mnemonic.length > mneLth)
      mneLth = diResults[ndx].mnemonic.length;
    if (diResults[ndx].operands.length > opLth)
      opLth = diResults[ndx].operands.length;
  }

  fputs(pszTopLine, hTrap);
  fputs(" Failing Instruction\n", hTrap);
  fputs(pszBotLine, hTrap);

  /* Finally!  We have something to print. */
  for (ndx = 0; ndx < stop; ndx++) {
    if (xd.seg16)
      sprintf(szBuffer, " %04hX:%04hX",
              xd.seg16, LOUSHORT(diResults[ndx].offset));
    else
      sprintf(szBuffer, " %08X ", diResults[ndx].offset);

    fprintf(hTrap, "%s%c%-*s %-*s  (%s)\n",
            szBuffer,
            (diResults[ndx].offset == xd.addr32 ? '>' : ' '),
            mneLth, diResults[ndx].mnemonic.p,
            opLth, diResults[ndx].operands.p,
            diResults[ndx].instructionHex.p);
  } // for

  return TRUE;
}

/*****************************************************************************/
/* Load a dll, then retrieve a proc address by name or ordinal.
 * pszMod must be just the module name (preferably uppercased).
 * pszProc must be null (not a null string) if loading by ordinal.
 * ulOrd must be zero if loading by proc name.
*/

PFN     LoadDisassemblerDll(char* pszMod, ULONG ulOrd, char* pszProc)
{
  ULONG     seg;
  ULONG     offs;
  HMODULE   hMod;
  PFN       pfn = 0;
  char *    ptr;

  /* See if the dll can be found along the path.
   * If not, see if it's in the same directory as exceptq.dll.
   */
  if (DosLoadModule(szBuffer, CCHMAXPATH, pszMod, &hMod)) {

    if (DosQueryModFromEIP(&hMod, &seg, CCHMAXPATH, szBuffer,
                           &offs, (ULONG)&LoadDisassemblerDll) ||
        DosQueryModuleName(hMod, CCHMAXPATH, szBuffer) ||
        (ptr = strrchr(szBuffer, '\\')) == 0)
      return 0;

    strcpy(ptr + 1, pszMod);
    strcat(szBuffer, ".DLL");
    if (DosLoadModule(&szBuffer[CCHMAXPATH], CCHMAXPATH, szBuffer, &hMod))
      return 0;
  }

  /* If the proc address can't be retrieved, unload the dll. */
  if (DosQueryProcAddr(hMod, ulOrd, pszProc, &pfn))
    DosFreeModule(hMod);

  return pfn;
}

/*****************************************************************************/
/*  The 'Registers' section                                                  */
/*****************************************************************************/

void    PrintRegisters(CONTEXTRECORD* pCtxRec)
{
  APIRET16  rc16;
  ULONG     ulSize;
  ULONG     ulCSSize;

  TimedBeep();

  fputs(pszTopLine, hTrap);
  fputs(" Registers\n", hTrap);
  fputs(pszBotLine, hTrap);

  if (pCtxRec->ContextFlags & CONTEXT_INTEGER) {
    fprintf(hTrap, pszRegEAX,
            pCtxRec->ctx_RegEax, pCtxRec->ctx_RegEbx,
            pCtxRec->ctx_RegEcx, pCtxRec->ctx_RegEdx);

    fprintf(hTrap, pszRegESI,
            pCtxRec->ctx_RegEsi, pCtxRec->ctx_RegEdi);
  }

  if (pCtxRec->ContextFlags & CONTEXT_CONTROL) {
    rc16 = DOS16SIZESEG(LOUSHORT(pCtxRec->ctx_SegCs), &ulCSSize);
    if (rc16) {
      if (LOUSHORT(pCtxRec->ctx_SegCs) == FLATCS)
        ulCSSize = 0xFFFFFFFFL;
      else
        ulCSSize = 0;
    }

    rc16 = DOS16SIZESEG(LOUSHORT(pCtxRec->ctx_SegSs), &ulSize);
    if (rc16) {
      if (LOUSHORT(pCtxRec->ctx_SegSs) == FLATDS)
        ulSize = 0xFFFFFFFFL;
      else
        ulSize = 0;
    }

    fprintf(hTrap, pszRegESP,
            pCtxRec->ctx_RegEsp, pCtxRec->ctx_RegEbp,
            pCtxRec->ctx_RegEip, pCtxRec->ctx_EFlags);

    fprintf(hTrap, pszRegCS,
            LOUSHORT(pCtxRec->ctx_SegCs), ulCSSize,
            LOUSHORT(pCtxRec->ctx_SegSs), ulSize);
  }

  if (verbosity >= XQ_CONCISE &&
      pCtxRec->ContextFlags & CONTEXT_CONTROL) {
    fputs("\n", hTrap);

    if (LOUSHORT(pCtxRec->ctx_SegDs) == FLATDS) {
      GetMemoryAttributes(pCtxRec->ctx_RegEax, szBuffer);
      fprintf(hTrap, " EAX : %s\n", szBuffer);
      GetMemoryAttributes(pCtxRec->ctx_RegEbx, szBuffer);
      fprintf(hTrap, " EBX : %s\n", szBuffer);
      GetMemoryAttributes(pCtxRec->ctx_RegEcx, szBuffer);
      fprintf(hTrap, " ECX : %s\n", szBuffer);
      GetMemoryAttributes(pCtxRec->ctx_RegEdx, szBuffer);
      fprintf(hTrap, " EDX : %s\n", szBuffer);
      GetMemoryAttributes(pCtxRec->ctx_RegEsi, szBuffer);
      fprintf(hTrap, " ESI : %s\n", szBuffer);
      GetMemoryAttributes(pCtxRec->ctx_RegEdi, szBuffer);
      fprintf(hTrap, " EDI : %s\n", szBuffer);
    }
    else
      Print16BitRegInfo(pCtxRec);
  }

  // 2013-09-20 SHL Show FP context if available
  if (pCtxRec->ContextFlags & CONTEXT_FLOATING_POINT) {
    // ULONG ctx_env[7];        // 4
    typedef struct {
      USHORT fpcw;              // 4 - control word - env0
      USHORT unused_2;          // 6
      USHORT fpsw;              // 8 - status word - env1
      USHORT unused_6;          // a
      USHORT fptw;              // c - tag word - env2
      USHORT unused_a;          // e
      ULONG ipoffset;           // 10 - FPU IP offset - env3
      USHORT ipsel;             // 14 - FPU IP selector - env4
      USHORT opcode;            // 16 - FPU opcode
      ULONG  opoffset;          // 18 - FPU operand offset - env5
      USHORT  opsel ;           // 1a - FPU operand data selector - env6
      USHORT  unused_1a;        // 1e - unused
    } *PCTX_ENV;

    UINT regnum;
    UINT stkndx;
    UINT top;
    USHORT fptw;

    VOID PrintFPCW(USHORT fpcw);
    VOID PrintFPSW(USHORT fpsw);

    fputs(pszTopLine, hTrap);
    fputs(" Floating Point Context\n", hTrap);
    fputs(pszBotLine, hTrap);

#if 0 // FIXME DEBUG
    for (regnum = 0; regnum < 7; regnum++) {
      fprintf(hTrap, " ctx_env[%u]: %08lX\n",
              regnum,
              pCtxRec->ctx_env[regnum]);
    }
    fputs("\n", hTrap);
#endif

    PrintFPSW(((PCTX_ENV)pCtxRec->ctx_env)->fpsw);
    PrintFPCW(((PCTX_ENV)pCtxRec->ctx_env)->fpcw);
    fptw = ((PCTX_ENV)pCtxRec->ctx_env)->fptw;
    fprintf(hTrap, " Tags    : %04X\n", fptw);

    fprintf(hTrap, " CS:EIP  : %04X:%08X  Opcode : %04X  DS:Offset : %04X:%08X\n",
            ((PCTX_ENV)pCtxRec->ctx_env)->ipsel,
            ((PCTX_ENV)pCtxRec->ctx_env)->ipoffset,
            ((PCTX_ENV)pCtxRec->ctx_env)->opcode,
            ((PCTX_ENV)pCtxRec->ctx_env)->opsel,
            ((PCTX_ENV)pCtxRec->ctx_env)->opoffset);

    fputs("\n", hTrap);
     // FPREG ctx_stack[8];
    top = ((PCTX_ENV)pCtxRec->ctx_env)->fpsw >> 11 & 7; // mask 0x3800
    for (stkndx = 0; stkndx < 8; stkndx++) {
      regnum = top + stkndx;
      fptw = ((PCTX_ENV)pCtxRec->ctx_env)->fptw;

      if (regnum > 7)
        regnum -= 8;
      fprintf(hTrap, " st(%d) fpreg[%u] %-5s : %04X %08lX%08lX\n",
              stkndx,
              regnum,
              apszFPTagNames[fptw >> (regnum * 2) & 3], // Tag(0) is ls 2 bits etc.
              pCtxRec->ctx_stack[regnum].signexp,
              pCtxRec->ctx_stack[regnum].hisig,
              pCtxRec->ctx_stack[regnum].losig);
    }
  }

  return;
}

/*****************************************************************************/
/* Identify memory attributes & ownership */

void    GetMemoryAttributes(ULONG ulAddress, char* pBuf)
{
  ULONG     ul;
  ULONG     ulAttr;
  ULONG     ulObjNum;
  ULONG     ulOffset;
  char *    ptr;

  ul = 1;
  if (DosQueryMem((void*)ulAddress, &ul, &ulAttr)) {
    strcpy(pBuf, "not a valid address");
    return;
  }
  if (ulAttr & PAG_FREE) {
    strcpy(pBuf, "unallocated memory");
    return;
  }

  /* Identify attributes */
  if (~ulAttr & PAG_COMMIT)
    ptr = "uncommitted";
  else
  if (ulAttr & PAG_EXECUTE) {
    if (ulAttr & PAG_READ)
      ptr = "read/exec ";
    else
      ptr = "executable";
  }
  else
  if (ulAttr & PAG_WRITE)
    ptr = "read/write";
  else
    ptr = "read-only ";

  if (ulAddress <  (ULONG)ptib->tib_pstacklimit &&
      ulAddress >= (ULONG)ptib->tib_pstack) {
    sprintf(pBuf, "%s memory on this thread's stack", ptr);
    return;
  }

  /* Round the address down to the base page so we get the correct owner
   * for the segment containing the address.  For low memory, it's the
   * previous 64k boundary;  for high mem, it's the previous 4k boundary.
   */
  if (ulAddress >= 0x20000000)
    ul = ulAddress & 0xFFFFF000;
  else
    ul = ulAddress & 0xFFFF0000;

  if (DosQueryModFromEIP(&ulAttr, &ulObjNum, CCHMAXPATH,
                         szModName, &ulOffset, ul)) {
    sprintf(pBuf, "%s memory - owner unknown", ptr);
  }
  else {
    /* -1 usually(?) means memory allocated by a module */
    if (ulObjNum == (ULONG)-1)
      sprintf(pBuf, "%s memory allocated by %s", ptr, szModName);
    else
      /* This is one of the module's segments. */
      sprintf(pBuf, "%s memory at %04lX:%08lX in %s",
              ptr, ulObjNum + 1, ulOffset + (ulAddress & 0xFFFF), szModName);
  }
  return;
}

/*****************************************************************************/

void    Print16BitRegInfo(CONTEXTRECORD* pCtxRec)
{
  APIRET16  rc16;
  ULONG     ulSize;

  rc16 = DOS16SIZESEG(LOUSHORT(pCtxRec->ctx_SegDs), &ulSize);
  if (!rc16) {
    if ((USHORT)ulSize < (USHORT)pCtxRec->ctx_RegEsi)
      fputs(" DS:SI points outside Data Segment\n", hTrap);
    else
      fputs(" DS:SI is a valid source\n", hTrap);
  }
  else {
    if (LOUSHORT(pCtxRec->ctx_SegDs) == FLATDS)
      fputs(" DS is a 32-bit selector\n", hTrap);
    else
      fputs(" DS is invalid\n", hTrap);
  }

  rc16 = DOS16SIZESEG(LOUSHORT(pCtxRec->ctx_SegEs), &ulSize);
  if (!rc16) {
    if ((USHORT)ulSize < (USHORT)pCtxRec->ctx_RegEdi)
      fputs(" ES:DI points outside Extra Segment\n", hTrap);
    else
      fputs(" ES:DI is a valid destination\n", hTrap);
  }
  else {
    if (LOUSHORT(pCtxRec->ctx_SegEs) == FLATDS)
      fputs(" ES is a 32-bit selector\n", hTrap);
    else
      fputs(" ES is invalid\n", hTrap);
  }

  return;
}

/*****************************************************************************/
/*  The 'Floating Point Context' section                                     */
/*****************************************************************************/

/**
 * Print FP Status word
 */

void    PrintFPSW(USHORT fpsw)
{
  CHAR sz[80];
  PSZ psz = sz;
  UINT u;

  if (fpsw & 0x8000) {
    memcpy(psz, "Busy", 4);
    psz += 4;
  }
  if (fpsw & 0x4000) {
    memcpy(psz, " C3", 3);
    psz += 3;
  }
  u = (fpsw >> 11) & 7;                 // mask 0x3800
  sprintf(psz, " TOP=%u", u);
  psz = psz + strlen(psz);
  if (fpsw & 0x0400) {
    memcpy(psz, " C2", 3);
    psz += 3;
  }
  if (fpsw & 0x0200) {
    memcpy(psz, " C1", 3);
    psz += 3;
  }
  if (fpsw & 0x0100) {
    memcpy(psz, " C0", 3);
    psz += 3;
  }
  if (fpsw & 0x0080) {
    memcpy(psz, " ES", 3);
    psz += 3;
  }
  if (fpsw & 0x0040) {
    memcpy(psz, " SF", 3);
    psz += 3;
  }
  if (fpsw & 0x0020) {
    memcpy(psz, " PE", 3);
    psz += 3;
  }
  if (fpsw & 0x0010) {
    memcpy(psz, " UE", 3);
    psz += 3;
  }
  if (fpsw & 0x0008) {
    memcpy(psz, " OE", 3);
    psz += 3;
  }
  if (fpsw & 0x0004) {
    memcpy(psz, " ZE", 3);
    psz += 3;
  }
  if (fpsw & 0x0002) {
    memcpy(psz, " DE", 3);
    psz += 3;
  }
  if (fpsw & 0x0001) {
    memcpy(psz, " IE", 3);
    psz += 3;
  }

  *psz = 0;
  psz = sz;
  if (*psz == ' ')
    psz++;

  fprintf(hTrap, " Status  : %04X - %s", fpsw, psz);
}

/**
 * Print FP Control word
 */

void    PrintFPCW(USHORT fpcw)
{
  CHAR sz[80];
  PSZ psz = sz;
  UINT u;

  if (fpcw & 0x1000) {
    memcpy(psz, "X", 1);
    psz += 1;
  }
  u = (fpcw >> 10) & 3;                 // mask = 0x800
  sprintf(psz, " RC=%u", u);
  psz = psz + strlen(psz);

  u = fpcw >> 7 & 3;                    // mask = 0x300
  sprintf(psz, " PC=%u", u);
  psz = psz + strlen(psz);

  if (fpcw & 0x0020) {
    memcpy(psz, " PM", 3);
    psz += 3;
  }
  if (fpcw & 0x0010) {
    memcpy(psz, " UM", 3);
    psz += 3;
  }
  if (fpcw & 0x0008) {
    memcpy(psz, " OM", 3);
    psz += 3;
  }
  if (fpcw & 0x0004) {
    memcpy(psz, " ZM", 3);
    psz += 3;
  }
  if (fpcw & 0x0002) {
    memcpy(psz, " DM", 3);
    psz += 3;
  }
  if (fpcw & 0x0001) {
    memcpy(psz, " IM", 3);
    psz += 3;
  }

  *psz = 0;
  psz = sz;
  if (*psz == ' ')
    psz++;

  fprintf(hTrap, "  Control : %04X - %s\n", fpcw, psz);
}

/*****************************************************************************/
/*  The 'Call Stack' section                                                 */
/*****************************************************************************/

void    PrintCallStack(EXCEPTIONREPORTRECORD* pExRepRec,
                       CONTEXTRECORD* pCtxRec)
{
  ULONG     ulEIP;
  ULONG     ulEBP;
  USHORT    usCS;
  USHORT    usSS;

  TimedBeep();

  if (pCtxRec->ContextFlags & CONTEXT_CONTROL) {
    ulEIP = pCtxRec->ctx_RegEip;
    usCS = LOUSHORT(pCtxRec->ctx_SegCs);
    ulEBP = pCtxRec->ctx_RegEbp;
    usSS = LOUSHORT(pCtxRec->ctx_SegSs);
  }
  else {
    /* FIXME to guess better */
    ulEIP = (ULONG)pExRepRec->ExceptionAddress,
    usCS = 0;
    ulEBP = (ULONG)ptib->tib_pstack;
    usSS = 0;
  }

  fputs(pszTopLine, hTrap);
  fputs(" Call Stack\n", hTrap);
  fputs(pszBotLine, hTrap);

  /* FIXME to bypass if insufficient context */
  WalkStack(ptib->tib_pstack, ptib->tib_pstacklimit,
            ulEBP, usSS, ulEIP, usCS);

  return;
}

/*****************************************************************************/
/*  The 'Stack Info' section                                                 */
/*****************************************************************************/

void    PrintStackInfo(CONTEXTRECORD* pCtxRec)
{
  TimedBeep();

  pStackTop = GetValidStackTop();

  fputs(pszTopLine, hTrap);
  fprintf(hTrap, " Stack Info for Thread %02lX\n", ptib->tib_ptib2->tib2_ultid);
  fputs(pszBotLine, hTrap);

  if (~pCtxRec->ContextFlags & CONTEXT_CONTROL) {
    fputs(pszStackInfoHdr32, hTrap);
    fprintf(hTrap, pszStackInfoNA,
            (ULONG)ptib->tib_pstacklimit - (ULONG)ptib->tib_pstack,
            ptib->tib_pstacklimit, pStackTop, ptib->tib_pstack);

    pStackTop = 0;
    fprintf(hTrap, "\n Stack can not be accessed: control context is not available\n");
  }
  else {
    if (LOUSHORT(pCtxRec->ctx_SegSs) == FLATDS) {
      fputs(pszStackInfoHdr32, hTrap);
      fprintf(hTrap, pszStackInfo32,
              (ULONG)ptib->tib_pstacklimit - (ULONG)ptib->tib_pstack,
              ptib->tib_pstacklimit, pCtxRec->ctx_RegEsp,
              pStackTop, ptib->tib_pstack);
    }
    else {
      #define XQ_F2S(addr, sel)  (USHORT)(((((ULONG)(addr)) >> 13) & ~7) | (((ULONG)(sel)) & 7))

      fputs(pszStackInfoHdr16, hTrap);
      fprintf(hTrap, pszStackInfo16,
              (ULONG)ptib->tib_pstacklimit - (ULONG)ptib->tib_pstack,
              XQ_F2S(ptib->tib_pstacklimit, pCtxRec->ctx_SegSs),
              LOUSHORT(ptib->tib_pstacklimit),
              LOUSHORT(pCtxRec->ctx_SegSs),
              LOUSHORT(pCtxRec->ctx_RegEsp),
              XQ_F2S(pStackTop, pCtxRec->ctx_SegSs),
              LOUSHORT(pStackTop),
              XQ_F2S(ptib->tib_pstack, pCtxRec->ctx_SegSs),
              LOUSHORT(ptib->tib_pstack));
      fprintf(hTrap, pszStackInfo1632,
              (ULONG)ptib->tib_pstacklimit - (ULONG)ptib->tib_pstack,
              ptib->tib_pstacklimit,
              ((LOUSHORT(pCtxRec->ctx_SegSs) & ~7) << 13) | LOUSHORT(pCtxRec->ctx_RegEsp),
              pStackTop, ptib->tib_pstack);
    }

    if (pStackTop > (char*)ptib->tib_pstacklimit) {
      pStackTop = 0;
      fprintf(hTrap, "\n Stack can not be accessed: stack ptr appears to be invalid\n");
    }
  }

  return;
}

/*****************************************************************************/
/* Starting at the base of the stack (i.e. the highest address) minus 4kb,
 * look for the page with the lowest address whose memory is committed.
 * Somewhere in that page is the furthest the stack ever grew.  If anyone
 * were interested, we could search for the lowest non-zero dword and
 * reasonably claim that it was the historical top of the stack.
 */

char*   GetValidStackTop(void)
{
  APIRET    rc;
  ULONG     ulSize;
  ULONG     ulState;
  char *    pStackPtr;

  TimedBeep();

  /* Find accessible stack range */
  for (pStackPtr = ((char*)ptib->tib_pstacklimit - 0x1000);
       pStackPtr >= (char*)ptib->tib_pstack;
       pStackPtr -= 0x1000) {
    ulSize = 0x1000;
    rc = DosQueryMemState(pStackPtr, &ulSize, &ulState);
    if (rc || ~ulState & PAG_PRESENT) {
      break;
    }
  }
  pStackPtr += 0x1000;

  return pStackPtr;
}

/*****************************************************************************/
/*  The 'Labels on the Stack' section                                        */
/*****************************************************************************/

void    PrintLabelsOnStack(CONTEXTRECORD* pCtxRec)
{
  APIRET    rc;
  ULONG     ul;
  ULONG     ulSize;
  ULONG     ulAttr;
  ULONG     ulObjNum;
  ULONG     ulOffset;
  HMODULE   hMod;
  PCHAR     pStackPtr;

  TimedBeep();

  if (verbosity < XQ_TERSE || !pStackTop)
    return;

  fputs(pszTopLine, hTrap);
  fputs(" Labels on the Stack\n", hTrap);
  fputs(pszBotLine, hTrap);

  fprintf(hTrap, pszStackHdr, "ESP");

  /* Optimize start location to smaller of ESP or EBP if
     the register looks like a valid stack pointer
     2015-08-22 SHL Give priority to ESP in case code not generating stack frames
   */
  pStackPtr = NULL;
  if ((pCtxRec->ctx_RegEsp & 0x3) == 0 &&
      pCtxRec->ctx_RegEsp > (ULONG)pStackTop &&
      (PVOID)pCtxRec->ctx_RegEsp < ptib->tib_pstacklimit) {
    pStackPtr = (PCHAR)pCtxRec->ctx_RegEsp;
  }
  if ((pCtxRec->ctx_RegEbp & 0x3) == 0 &&
      pCtxRec->ctx_RegEbp > (ULONG)pStackTop &&
      (void*)pCtxRec->ctx_RegEbp < ptib->tib_pstacklimit &&
      pCtxRec->ctx_RegEbp < (ULONG)pStackPtr) {
    pStackPtr = (PCHAR)pCtxRec->ctx_RegEbp;
  }
  if (pStackPtr == NULL)
    pStackPtr =  pStackTop;             // Default to smallest accessable address

  for (; pStackPtr < (char*)ptib->tib_pstacklimit; pStackPtr += 4) {
    TimedBeep();
    ul = *(PULONG)pStackPtr;
    if (ul < 0x10000L)
      continue;

    ulSize = 4;
    rc = DosQueryMem((void*)ul, &ulSize, &ulAttr);
    if (rc || ~ulAttr & PAG_COMMIT || ~ulAttr & PAG_EXECUTE)
      continue;

    rc = DosQueryModFromEIP(&hMod, &ulObjNum, CCHMAXPATH,
                            szModName, &ulOffset, ul);
    if (rc || ulOffset == 0xffffffff)
      continue;

    fprintf(hTrap, " %08lX  %08lX   %-08s  %04lX:%08lX ",
            pStackPtr, ul, szModName, ulObjNum + 1, ulOffset);

    /* If the filename can be identified, try to get the symbol's name using
     * embedded debug info or a DBG file; if that fails, try for a .sym file.
     */
    if (DosQueryModuleName(hMod, CCHMAXPATH, szFileName))
      fputc('\n', hTrap);
    else if (PrintLineNum(szFileName, ulObjNum, ulOffset, FALSE)) {
      // Try .xqs or .sym file
      PrintSymbolFromFile(szFileName, hMod, ulObjNum, ulOffset);
    }

#if 0 /* TESTING - enable to test trap in handler logic */
    if (cTrapCount == 1)
      *(CHAR*)0 = cTrapCount;
#endif
  }

  return;
}

/*****************************************************************************/
/*  The 'Stack Contents' section                                             */
/*****************************************************************************/

void    PrintStackDump(CONTEXTRECORD* pCtxRec)
{
  ULONG     byteCnt;
  ULONG     flags;
  char *    pStackPtr;

  TimedBeep();

  if (verbosity < XQ_TERSE || !pStackTop)
    return;

  pStackPtr = pStackTop;

  /* Print stack dump.  For terse, print from roughly 64 bytes above ESP to
   * 256 bytes below;  for concise, print from roughly 256 bytes above ESP
   * to the bottom;  for verbose & v-verbose, print the entire valid stack.
  */
  if (verbosity == XQ_TERSE) {
    if (((pCtxRec->ctx_RegEsp & ~0x0f) - 0x40) > (ULONG)pStackPtr)
      pStackPtr = (char*)((pCtxRec->ctx_RegEsp & ~0x0f) - 0x40);
  }
  else
  if (verbosity == XQ_CONCISE) {
    if (((pCtxRec->ctx_RegEsp & ~0x0f) - 0x100) > (ULONG)pStackPtr)
      pStackPtr = (char*)((pCtxRec->ctx_RegEsp & ~0x0f) - 0x100);
  }

  byteCnt = (char*)ptib->tib_pstacklimit - pStackPtr;

  // 2013-05-08 SHL If running from Thunk stack, TIB values are not valid
  if (pStackPtr < (char*)ptib->tib_pstack ||
      pStackPtr > (char*)ptib->tib_pstacklimit) {
    ULONG     ul = byteCnt;
    ULONG     ulAttr;
    /* Calculate size of Thunk stack
       If query fails, use values from TIB which will probably cause exception
       Oh well
     */
    if (!DosQueryMem((void*)pStackPtr, &ul, &ulAttr) && ulAttr & PAG_READ) {
#     if 0 // 2013-05-08 SHL FIXME DEBUG
      fprintf(hTrap, "\n ** Using DOSCALL1 Thunk Stack with base at 0x%lx **", pStackPtr + ul);
#     endif
      byteCnt = ul;
    }
  }

  /* begin header */
  fputs(pszTopLine, hTrap);

  if (verbosity == XQ_TERSE) {
    if (byteCnt > 0x140)
      byteCnt = 0x140;
    fprintf(hTrap, " Stack Contents from ESP-%lX to ESP+%lX  (ESP = %08lX)\n",
            (char*)pCtxRec->ctx_RegEsp - pStackPtr,
            (pStackPtr + byteCnt) - (char*)pCtxRec->ctx_RegEsp,
            pCtxRec->ctx_RegEsp);
  }
  else
  if (verbosity == XQ_CONCISE)
    fprintf(hTrap, " Stack Contents from ESP-%lX to Stack Base  (ESP = %08lX)\n",
            (char*)pCtxRec->ctx_RegEsp - pStackPtr, pCtxRec->ctx_RegEsp);
  else
    fprintf(hTrap, " Stack Contents from Historic Top to Stack Base  (ESP = %08lX)\n",
            pCtxRec->ctx_RegEsp);

  fputs(pszBotLine, hTrap);
  /* end header */

  // 2013-05-08 SHL Support fAssume32BitStack
  flags = XQM_CHARS | XQM_FMT16 |
          (LOUSHORT(pCtxRec->ctx_SegSs) == FLATDS || fAssume32BitStack ?
            XQM_DWORDS : XQM_WORDS);

  PrintMemoryHdr(flags);
  PrintMemory(pStackPtr, byteCnt, flags);

  return;
}

/*****************************************************************************/
/*  The 'Memory addressed by ...' sections                                   */
/*****************************************************************************/

void    PrintRegisterPointers(CONTEXTRECORD* pCtxRec)
{
  ULONG   flags;

  if (verbosity < XQ_TERSE)
    return;

  /* If we can determine whether this is 16- or 32-bit data,
   * show words or dwords;  otherwise, just show bytes.
  */
  if (pCtxRec->ContextFlags & CONTEXT_CONTROL)
    flags = XQM_BYTES | XQM_CHARS | XQM_FMT8 |
            ((LOUSHORT(pCtxRec->ctx_SegDs) == FLATDS) ? XQM_DWORDS : XQM_WORDS);
  else
    flags = XQM_BYTES | XQM_CHARS | XQM_FMT16;

  PrintMemoryAt((char*)pCtxRec->ctx_RegEax, "EAX", flags);
  PrintMemoryAt((char*)pCtxRec->ctx_RegEbx, "EBX", flags);
  PrintMemoryAt((char*)pCtxRec->ctx_RegEcx, "ECX", flags);
  PrintMemoryAt((char*)pCtxRec->ctx_RegEdx, "EDX", flags);
  PrintMemoryAt((char*)pCtxRec->ctx_RegEsi, "ESI", flags);
  PrintMemoryAt((char*)pCtxRec->ctx_RegEdi, "EDI", flags);

  return;
}

/*****************************************************************************/
/* Print memory block if address is valid */

void    PrintMemoryAt(char* pMem, PSZ pszDesc, ULONG flags)
{
  ULONG ulSize;
  ULONG ulAttr;
  APIRET apiret;

  TimedBeep();

  if (verbosity == XQ_TERSE)
    ulSize = 64;
  else
  if (verbosity == XQ_VERBOSE)
    ulSize = 512;
  else
  if (verbosity == XQ_VERYVERBOSE)
    ulSize = 1024;
  else
    ulSize = 256;

  /* Use special case memory access checks in case
     shared read/exec memory in terminating DLL inaccessible
     2020-05-17 SHL
  */
  apiret = QueryMem(pMem, &ulSize, &ulAttr);
  if (apiret != NO_ERROR)
    return;
  if (~ulAttr & PAG_COMMIT)
    return;
  // Delay if shared read/exec probably inaccessible
  if (~ulAttr & PAG_READ) {
    // If PAG_READ turned off in QueryMem report below
    if (!fIsProcessTerminateInDll || ~ulAttr & PAG_SHARED || ~ulAttr & PAG_EXECUTE)
      return;
  }

  fputs(pszTopLine, hTrap);
  fprintf(hTrap, " Memory addressed by %s (%08lX) for %ld bytes\n",
          pszDesc, pMem, ulSize);
  fputs(pszBotLine, hTrap);

  if (fIsProcessTerminateInDll && ulAttr & PAG_SHARED && ulAttr & PAG_EXECUTE) {
      // szModName set by QueryMem
      fprintf(hTrap, " Shared read/exec memory in %s is inaccessible at this time\n",
              szModName);
      return;
  }

  PrintMemoryHdr(flags);
  PrintMemory(pMem, ulSize, flags);

  return;
}

/*****************************************************************************/

void    PrintMemory(char* pMem, ULONG cbMem, ULONG flags)
{
  ULONG   cbLine = flags & XQM_FMTMASK;
  ULONG   cnt = 0;
  char *  pLast = 0;

  for (; cbMem >= cbLine; cbMem -= cbLine, pMem += cbLine) {
    /* Suppress duplicate lines. */
    if (pLast && !memcmp(pMem, pLast, cbLine)) {
      cnt++;
      pLast = pMem;
      continue;
    }
    /* If there's more than one dup line, print the message;  if
     * there's only one, don't suppress it.  Instead, print it now -
     * the current line will get printed on the next iteration.
    */
    if (cnt > 1)
      fprintf(hTrap, pszDupLine, pLast, cnt);
    if (cnt == 1)
      pMem = pLast;
    else
      pLast = pMem;
    cnt = 0;

    /* Print a full line. */
    PrintMemoryLine(pMem, cbLine, flags);
  }

  /* If the final lines were dups, print the msg now. */
  if (cnt)
    fprintf(hTrap, pszDupLine, pLast, cnt);

  /* If cbMem wasn't a multiple of cbLine, print the remaining bytes. */
  if (cbMem)
    PrintMemoryLine(pMem, cbMem, flags);

  return;
}

/*****************************************************************************/

void    PrintMemoryLine(char* pSrc, ULONG cbSrc, ULONG flags)
{
  ULONG   cbLine = flags & XQM_FMTMASK;
  ULONG   ctr;
  char *  pDst;

  pDst = szBuffer;
  pDst += sprintf(pDst, " %08lX ", pSrc);

  if (flags & (XQM_WORDS | XQM_DWORDS)) {
    pDst = strcpy(pDst, ": ") + 2;
    if (flags & XQM_DWORDS) {
      for (ctr = 0; ctr < cbSrc/4; ctr++)
        pDst += sprintf(pDst, "%08lX ", ((ULONG*)pSrc)[ctr]);

      if (ctr < cbLine/4)
        pDst += sprintf(pDst, "%*s", 9*((cbLine/4)-ctr), "");
    }
    else {
      for (ctr = 0; ctr < cbSrc/2; ctr++)
        pDst += sprintf(pDst, "%04hX ", ((USHORT*)pSrc)[ctr]);

      if (ctr < cbLine/2)
        pDst += sprintf(pDst, "%*s", 5*((cbLine/2)-ctr), "");
    }
  }

  if (flags & XQM_BYTES) {
    pDst = strcpy(pDst, ": ") + 2;
    for (ctr = 0; ctr < cbSrc; ctr++)
      pDst += sprintf(pDst, "%02hX ", pSrc[ctr]);

    if (cbSrc < cbLine)
      pDst += sprintf(pDst, "%*s", 3*(cbLine-cbSrc), "");
  }

  if (flags & XQM_CHARS) {
    pDst = strcpy(pDst, ": ") + 2;
    for (ctr = 0; ctr < cbSrc; ctr++)
      *pDst++ = (isprint(pSrc[ctr]) && pSrc[ctr] >= 0x20) ? pSrc[ctr] : '.';
  }

  strcpy(pDst, "\n");
  fputs(szBuffer, hTrap);

  return;
}

/*****************************************************************************/

void    PrintMemoryHdr(ULONG flags)
{
  ULONG   cbLine = flags & XQM_FMTMASK;
  ULONG   width;
  char *  pDst;

  pDst = szBuffer;
  pDst = strcpy(pDst, " --addr--") + 9;

  if (flags & (XQM_WORDS | XQM_DWORDS)) {
    pDst = strcpy(pDst, "   ") + 3;
    if (flags & XQM_DWORDS) {
      width = (2 * cbLine) + ((cbLine / 4) - 1) - 6;
      pDst = (char*)memset(pDst, '-', width/2) + width/2;
      pDst = strcpy(pDst, "dwords") + 6;
    }
    else {
      width = (2 * cbLine) + ((cbLine / 2) - 1) - 5;
      pDst = (char*)memset(pDst, '-', width/2) + width/2;
      pDst = strcpy(pDst, "words") + 5;
    }
    width -= width/2;
    pDst = (char*)memset(pDst, '-', width) + width;
  }

  if (flags & XQM_BYTES) {
    pDst = strcpy(pDst, "   ") + 3;
    width = (2 * cbLine) + (cbLine - 1) - 5;
    pDst = (char*)memset(pDst, '-', width/2) + width/2;
    pDst = strcpy(pDst, "bytes") + 5;
    width -= width/2;
    pDst = (char*)memset(pDst, '-', width) + width;
  }

  if (flags & XQM_CHARS) {
    pDst = strcpy(pDst, "   ") + 3;
    width = cbLine - 5;
    pDst = (char*)memset(pDst, '-', width/2) + width/2;
    pDst = strcpy(pDst, "chars") + 5;
    width -= width/2;
    pDst = (char*)memset(pDst, '-', width) + width;
  }

  strcpy(pDst, "\n");
  fputs(szBuffer, hTrap);

  return;
}

/*****************************************************************************/
/*  The 'DLLs accessible from this process' section                          */
/*****************************************************************************/

typedef struct {
  HMODULE hMod;
  ULONG ulModAddress;
  ULONG ulModBytes;
} tModInfo;

#define XQ_DEFAULT_LOWER_SHARED_BASE_ADDR       ((PCHAR)0x4000000)      // 64MB
#define XQ_LOWER_SHARED_MAX_ADDR                ((PCHAR)0x1FFFFFFF)     // 512MB -1
#define XQ_DEFAULT_UPPER_SHARED_BASE_ADDR       ((PCHAR)0x20000000)     // 512MB
#define XQ_CODE_ATTR   (PAG_EXECUTE | PAG_BASE | PAG_SHARED)
#define XQ_PMODINFO_MAX    ((tModInfo*)(bigBuf + sizeof(bigBuf)))

void    PrintDlls(void)
{
// #define DLL_DEBUG 1      // 2012-11-10 SHL Enable for PrintDLLs debugging
  APIRET    rc;
  ULONG     ulBootDrive;                // A: = 1
  ULONG     ulMaxPrMem;
  ULONG     ulMaxHPrMem;
  ULONG     ulVirtualAddressLimit;
  PVOID     pvBase;
  ULONG     ulSize;
  ULONG     ulAttr;
  ULONG     ulObjNum;
  ULONG     ulOffset;
  HMODULE   hMod;
  tModInfo *pModInfo;
  char *    ptr;
  FILESTATUS3 fs3;
  char szFileBytes[16];

  static PCSZ szOs2KrnlPath = ":\\OS2KRNL";
  static PCSZ szDosCallsMod = "DOSCALLS";

  if (verbosity < XQ_TERSE)
    return;

  /* Scan the lower shared arena and accumulate list of loaded DLLs
     Start scan after lower private/shared arena boundary
     Stop scan at end of lower shared arena
  */
  memset(bigBuf, 0, sizeof(bigBuf));

  rc = DosQuerySysInfo(QSV_MAXPRMEM, QSV_MAXPRMEM, &ulMaxPrMem, sizeof(ulMaxPrMem));
  if (rc)
    pvBase = XQ_DEFAULT_LOWER_SHARED_BASE_ADDR; // Fall back silently
  else
    pvBase = (PVOID)(ulMaxPrMem + 0x10000); // Account for inaccessible 1st 64KB
  ulSize = XQ_LOWER_SHARED_MAX_ADDR - (PCHAR)pvBase; // Calc max object size

  // Scan for accessible DLLs
  do {
    TimedBeep();
    rc = DosQueryMem(pvBase, &ulSize, &ulAttr);
    if (rc) {
      if (rc == ERROR_INVALID_ADDRESS || rc == ERROR_NO_OBJECT) {
        pvBase = (PCHAR)pvBase + 0x10000; // Point at next object
        ulSize = XQ_LOWER_SHARED_MAX_ADDR - (PCHAR)pvBase; // Calc max object size
        continue;
      }
      fprintf(hTrap, " ** PrintDlls:  DosQueryMem failed with error %u (%u)\n", rc, __LINE__);
      return;
    }

    if ((ulAttr & XQ_CODE_ATTR) == XQ_CODE_ATTR) {
      rc = DosQueryModFromEIP(&hMod, &ulObjNum, CCHMAXPATH,
                              szModName, &ulOffset, (ULONG)pvBase);
      if (!rc) {
        // Add to list if new
        for (pModInfo = (tModInfo*)bigBuf;
             pModInfo < XQ_PMODINFO_MAX && pModInfo->hMod != hMod;
             pModInfo++) {
          if (!pModInfo->hMod) {
            // Free slot
            pModInfo->hMod = hMod;
            pModInfo->ulModAddress = (ULONG)pvBase;
            pModInfo->ulModBytes = ulSize;
            break;
          }
        } // for
      } // if accessible module
    } // if code

    /* Round object size up to multiple of 64KB */
    ulSize += 0x0FFFF;
    ulSize &= ~0xFFFF;
    pvBase = (PCHAR)pvBase + ulSize;    // Point at next object to check
    ulSize = XQ_LOWER_SHARED_MAX_ADDR - (PCHAR)pvBase; // Calc max object size

  } while (pvBase < (void*)XQ_LOWER_SHARED_MAX_ADDR);

  /* Scan above 512MB upper shared arena and accumulate list of loaded DLLs
     Start scan at 512MB + QSV_MAXHPRMEM
     Stop scan at QSV_VIRTUALADDRESSLIMIT
     Scan nothing for kernels which do not support upper memory
     2020-04-12 SHL Added
  */
  pvBase = XQ_DEFAULT_UPPER_SHARED_BASE_ADDR;
  rc = DosQuerySysInfo(QSV_MAXHPRMEM, QSV_MAXHPRMEM,
                       &ulMaxHPrMem, sizeof(ulMaxHPrMem));
  if (!rc) {
    rc = DosQuerySysInfo(QSV_VIRTUALADDRESSLIMIT, QSV_VIRTUALADDRESSLIMIT,
                         &ulVirtualAddressLimit, sizeof(ulVirtualAddressLimit));
    ulVirtualAddressLimit *= 0x100000;  // 1MB
  }
#ifdef DLL_DEBUG
  fprintf(hTrap, "\nPrintDlls: ulMaxHPrMem 0x%x ulVirtualAddressLimit 0x%x rc %d", ulMaxHPrMem, ulVirtualAddressLimit, rc);
#endif

  if (!rc) {
    pvBase = XQ_DEFAULT_UPPER_SHARED_BASE_ADDR + ulMaxHPrMem;
    ulSize = ulVirtualAddressLimit - (ULONG)pvBase; // Calc max object size
  }
#ifdef DLL_DEBUG
  fprintf(hTrap, "\nPrintDlls: scanning upper memory from 0x%x for 0x%x bytes rc %d", pvBase, ulSize, rc);
#endif
  if (rc ||
      (PCHAR)pvBase < XQ_DEFAULT_UPPER_SHARED_BASE_ADDR ||
      (ULONG)pvBase > ulVirtualAddressLimit)
  {
    pvBase = XQ_DEFAULT_UPPER_SHARED_BASE_ADDR; // Be careful
    ulSize = 0;                         // Assume no upper memory
  }

#ifdef DLL_DEBUG
  fprintf(hTrap, "\nPrintDlls: scanning upper memory from 0x%x for 0x%x bytes rc %d", pvBase, ulSize, rc);
#endif

  // Scan upper shared arena for accessible DLLs
  while (ulSize > 0) {

    TimedBeep();

    rc = DosQueryMem(pvBase, &ulSize, &ulAttr);
    if (rc) {
      if (rc == ERROR_INVALID_ADDRESS || rc == ERROR_NO_OBJECT) {
        // Point at next possible object and calc max possible object size
        pvBase = (PCHAR)pvBase + 0x10000; // 64KB
        ulSize = (PCHAR)XQ_LOWER_SHARED_MAX_ADDR - (PCHAR)pvBase;
        continue;
      }
      fprintf(hTrap, " ** PrintDlls:  DosQueryMem failed with error %u\n", rc, __LINE__);
      return;
    }

#ifdef DLL_DEBUG
    fprintf(hTrap, "\nPrintDlls: found memory at 0x%x ulSize 0x%x ulAttr 0x%x rc %d",
            pvBase, ulSize, ulAttr, rc);
#endif
    // Upper shared does not set PAG_BASE (0x10000) FIXME
    if ((ulAttr & ((PAG_EXECUTE | PAG_SHARED))) == (PAG_EXECUTE | PAG_SHARED)) {
#ifdef DLL_DEBUG
      fprintf(hTrap, "\nPrintDlls: found executable (0x2004) ulAttr 0x%x", ulAttr);
#endif
      rc = DosQueryModFromEIP(&hMod, &ulObjNum, CCHMAXPATH,
                              szModName, &ulOffset, (ULONG)pvBase);
#ifdef DLL_DEBUG
      if (rc) {
        fprintf(hTrap, "\nPrintDlls: DosQueryModFromEIP(0x%x) reported error %d",
                ulOffset, rc);
      }
#endif
      if (!rc) {
#ifdef DLL_DEBUG
        fprintf(hTrap, "\nPrintDlls: found module 0x%x \"%s\" ulOffset 0x%x rc %d",
                hMod, szModName, ulOffset, rc);
#endif
        // Add module to list if new
        for (pModInfo = (tModInfo*)bigBuf;
             pModInfo < XQ_PMODINFO_MAX && pModInfo->hMod != hMod;
             pModInfo++) {
          if (!pModInfo->hMod) {
            // Free slot
            pModInfo->hMod = hMod;
            pModInfo->ulModAddress = (ULONG)pvBase;
            pModInfo->ulModBytes = ulSize;
            break;
          }
        } // for
        if (pModInfo >= XQ_PMODINFO_MAX)
          break;                        // Too many to list all
      } // if module
    } // if code

    /* Round reported object size to 64KB multiple
       Kernel allocates address space in 64KB minumum
    */
    ulSize += 0x0FFFF;
    ulSize &= ~0xFFFF;
    // Point at next possible object and calc max possible object size
    pvBase = (PVOID)((ULONG)pvBase + ulSize);
    if ((ULONG)pvBase > ulVirtualAddressLimit)
      break;                            // Should never occur, but why assume
    ulSize = ulVirtualAddressLimit - (ULONG)pvBase;

  } // while scanning upper memory

  fputs(pszTopLine, hTrap);
  fputs(" DLLs accessible from this process\n", hTrap);
  fputs(pszBotLine, hTrap);

  // 2020-04-12 SHL Show kernel too
  rc = DosQuerySysInfo(QSV_BOOT_DRIVE, QSV_BOOT_DRIVE, &ulBootDrive, sizeof(ulBootDrive));
  if (rc)
    ulBootDrive = 3;                    // Oh well
  szFileName[0] = ulBootDrive + ('A' - 1);
  strcpy(szFileName + 1, szOs2KrnlPath);
  memset(&fs3,0,sizeof(fs3));
  rc = DosQueryPathInfo(szFileName, FIL_STANDARD, &fs3, sizeof(fs3));
  if (rc)
    fprintf(hTrap, " %-8s  %s (%u)\n", szDosCallsMod, szFileName, rc);
  else {
    commafmt(szFileBytes, sizeof(szFileBytes), fs3.cbFile);
    fprintf(hTrap, " %-8s                   %02u/%02u/%02u %02u:%02u:%02u %10s %s\n",
            szDosCallsMod,
            fs3.fdateLastWrite.month,
            fs3.fdateLastWrite.day,
            fs3.fdateLastWrite.year + 1980,
            fs3.ftimeLastWrite.hours,
            fs3.ftimeLastWrite.minutes,
            fs3.ftimeLastWrite.twosecs,
            szFileBytes,
            szFileName);
  }

  // Report
  for (pModInfo = (tModInfo*)bigBuf;
       pModInfo < XQ_PMODINFO_MAX && pModInfo->hMod;
       pModInfo++)
  {
    TimedBeep();

    if (DosQueryModuleName(pModInfo->hMod, CCHMAXPATH, szFileName))
      strcpy(szFileName, "unknown");

    ptr = strrchr(szFileName, '\\');
    ptr = ptr ? ptr + 1 : szFileName;
    strcpy(szModName, ptr);             // Assume module name same as file name without path
    ptr = strchr(szModName, '.');
    if (ptr)
      *ptr = 0;                         // Strip extension

    // 2013-04-01 SHL show date/time/size of modules
    // 2020-04-23 SHL show module address and size
    memset(&fs3,0,sizeof(fs3));
    rc = DosQueryPathInfo(szFileName, FIL_STANDARD, &fs3, sizeof(fs3));
    if (rc)
      fprintf(hTrap, " %-8s  %s (%u)\n", szModName, szFileName, rc);
    else {
      commafmt(szFileBytes, sizeof(szFileBytes), fs3.cbFile);
      fprintf(hTrap, " %-8s %08x %08x %02u/%02u/%02u %02u:%02u:%02u %10s %s\n",
              szModName,
              pModInfo->ulModAddress,
              pModInfo->ulModBytes,
              fs3.fdateLastWrite.month,
              fs3.fdateLastWrite.day,
              fs3.fdateLastWrite.year + 1980,
              fs3.ftimeLastWrite.hours,
              fs3.ftimeLastWrite.minutes,
              fs3.ftimeLastWrite.twosecs,
              szFileBytes,
              szFileName);
    }
  } // for

  if (pModInfo >= XQ_PMODINFO_MAX)
    fputs("\n Warning: capacity exceeded.  Additional DLLs may not have been listed.\n",
          hTrap);

  return;
}

/*****************************************************************************/
/*  Utilities:  WalkStack, TimedBeep                                         */
/*****************************************************************************/
/* Walk stack and print function addresses and local variables.
 * Better New WalkStack From John Currier.
 */

// #define WALKSTACK_DEBUG 1      // Enable for WalkStack debugging

void    WalkStack(void* pvStackBottom, void* pvStackTop,
                  ULONG ulEBP, USHORT usSS, ULONG ulEIP, USHORT usCS)
{
  APIRET    rc;
  HMODULE   hMod;
  ULONG     ul32BitAddr;
  ULONG     ulLastEbp;
  ULONG     ulSize;
  ULONG     ulAttr;
  ULONG     ulObjNum;
  ULONG     ulOffset;
  ULONG     ulOverrideEBP = 0;          // For EDC3216 interface
  BOOL      isPass1;
  BOOL      is32BitStk;
  BOOL      isFarCall = 0;              // For EDC3216 interface
  BOOL      wasFarCall;
  BOOL      printLocals;
  BOOL      ebpOk;

  /* Note: we can't handle stacks bigger than 64K for now... */

  fprintf(hTrap, pszStackHdr, "EBP");

  /* Use passed address 1st time thru */
  isPass1 = TRUE;
  is32BitStk = usSS == FLATDS;          // 2020-05-25 SHL

  for (;;) {
    TimedBeep();

    ulSize = 12;    // Sufficient to hold ebp, cs:eip
    if (is32BitStk)
      rc = DosQueryMem((void*)(ulEBP), &ulSize, &ulAttr);
    else {
      ul32BitAddr = (ULONG)(usSS & ~7) << 13 | ulEBP;
      rc = DosQueryMem((void*)(ul32BitAddr), &ulSize, &ulAttr);
    }

    // Ensure OK and committed
    ebpOk = rc == NO_ERROR && ulAttr & PAG_COMMIT && ulSize >= 12;

    /* If pass1, use passed initial values */
    if (!isPass1) {
      // Not pass1 - get values from stack
      if (!ebpOk) {
        if (is32BitStk)
          fprintf(hTrap, " Invalid EBP %08X\n", ulEBP);
        else
          fprintf(hTrap, " Invalid SS:BP: %04X:%04X\n", usSS, (USHORT)ulEBP);
        break;                          // Quit now
      }

      // Get return address
      if (is32BitStk) {
        ulEIP = *((PULONG)(ulEBP + 4));
#       if 0    // FIXME to be gone
        /* If there's a "return address" of FLATDS following
         * EBP on the stack we have to adjust EBP by 44 bytes to get
         * at the real return address.  This has something to do with
         * calling 32-bit code via a 16-bit interface
         * This offset applies to VAC runtime and might vary for others.
         * FIXME to be gone - moved to 16-bit code
         */
        if (ulEIP == FLATDS) {
          ulEBP += 44;                  // Assume addressible
          ulEIP = *(PULONG)(ulEBP + 4);
        }
#     endif // FIXME to be gone
      }
      else {
        /* 16-bit */
        /* If last return was to far call and this looks like near call
           Check if returning to 32-bit code via VAC EDC3216 interface
           Stack will contain 32-bit EBP followed by 32-bit SS
        */
        wasFarCall = isFarCall;
        isFarCall = *((PUSHORT)ul32BitAddr) & 1;        // Bit 1 indicates called function is far
        if (wasFarCall &&
            !isFarCall &&
            *((PULONG)(ul32BitAddr + 4)) == FLATDS)
        {
          // Returning to 32-bit code via EDC3216 interface
          is32BitStk = TRUE;
          ulEBP = *(PULONG)ul32BitAddr;         // Point at EDC3216 local vars
          ulEIP = *(PULONG)(ulEBP + 24);        // Point at return to EDC3216
          ulOverrideEBP = ulEBP + 36;
          usSS = FLATDS;
          usCS = FLATCS;
#         ifdef WALKSTACK_DEBUG
          fprintf(hTrap, " * WalkStack edc3216: SS %x EBP %x EIP %x (%u)\n", usSS, ulEBP, ulEIP, __LINE__);
#         endif
        }
        else {
          ulEIP = *((PUSHORT)(ul32BitAddr + 2));
          if (isFarCall)
            usCS = *(PUSHORT)(ul32BitAddr + 4);
          /* If CS is now FLATDS, we are returning to 32 bit code from 16-bit code
           * FIXME to check ulEIP == 0x150B? SHL
           * FIXME to detect near return to 32-bit code SHL
           * FIXME to be gone maybe
           */
          if (usCS == FLATDS) {
            /* Returning to 32-bit code */
             /* Probably VAC specific - FIXME to be sure SHL */
            ulEBP = ul32BitAddr + 20;
            usSS = FLATDS;
            ulEIP = *(PULONG)(ulEBP + 4);
            usCS = FLATCS;
            is32BitStk = TRUE;
          }
        }
      } // if 16-bit
    } // if !Pass1

#   ifdef WALKSTACK_DEBUG
    fprintf(hTrap, " * WalkStack: SS %x EBP %x (%u)\n", usSS, ulEBP, __LINE__);
#   endif

    if (!is32BitStk) {
      /* If the return address points to the stack then it's
       * really just a pointer to the return address (UGH!).
       */
      if (usCS == usSS) {
        ul32BitAddr = (ULONG)(usSS & ~7) << 13 | ulEIP;
        ulSize = 4;    /* sufficient to hold cs:ip */
        rc = DosQueryMem((void*)(ul32BitAddr), &ulSize, &ulAttr);
        ebpOk = rc == NO_ERROR && ulAttr & PAG_COMMIT && ulSize >= 4;
        if (!ebpOk) {
          fprintf(hTrap, " Invalid SS:SP: %02X:%02X\n", usSS, (USHORT)ulEIP);
          break;                          // Quit now
        }
        ulEIP = *(PUSHORT)(ul32BitAddr);
        usCS = *(PUSHORT)(ul32BitAddr + 2);
      }

      /* End of the stack so these are both shifted by 2 bytes
       * FIXME to understand SHL
       */
      // 2013-10-15 SHL avoid exception - ulEBP is 16-bit
      ul32BitAddr = (ULONG)(usSS & ~7) << 13 | ulEBP;
      ulSize = 12;    /* sufficient to hold ebp, cs:eip */
      rc = DosQueryMem((void*)(ul32BitAddr), &ulSize, &ulAttr);
      ebpOk = rc == NO_ERROR && ulAttr & PAG_COMMIT && ulSize >= 8;
      if (!ebpOk) {
        fprintf(hTrap, " Invalid SS:SP: %04X:%04X\n", usSS, (USHORT)ulEIP);
        break;                          // Quit now
      }
      if (ulEIP == 0 && *(PUSHORT)ul32BitAddr == 0) {
        ulEIP = *(PUSHORT)(ul32BitAddr + 4);
        usCS = *(PUSHORT)(ul32BitAddr + 6);
      }
    } // if 16-bit

    /* For far calls in 16-bit programs have on the stack:
     *   BP:IP:CS
     *   where CS may be thunked
     *
     *     in byte order               swapped
     *    BP        IP   CS          BP   CS   IP
     *   4677      53B5 F7D0        7746 D0F7 B553
     *
     * for near calls 32bit programs have:
     *   EBP:EIP
     * and you'd have something like this (with SP added) (not
     * accurate values)
     *
     *       in byte order           swapped
     *      EBP       EIP         EBP       EIP
     *   4677 2900 53B5 F7D0   0029 7746 D0F7 B553
     *
     * So the basic difference is that 32bit programs have a 32bit
     * EBP and we can attempt to determine whether we have a 32bit
     * EBP by checking to see if its 'selector' is the same as SP.
     * Note that this technique limits us to checking stacks < 64K.
     *
     * Soooo, if IP (which maps into the same USHORT as the swapped
     * stack page in EBP) doesn't point to the stack (i.e. it could
     * be a 16bit IP) then see if CS is valid (as is or thunked).
     *
     * Note that there's the possibility of a 16bit return address
     * that has an offset that's the same as SP so we'll think it's
     * a 32bit return address and won't be able to successfully resolve
     * its details.
     */

    if (!is32BitStk) {
      if (ulEIP != usSS) {
        if (DOS16SIZESEG((USHORT)usCS, &ulSize) == NO_ERROR)
          ; /* ulRetAddr = MAKEULONG(ulEIP, usCS);  2007-07-27 SHL */
        else if (DOS16SIZESEG((usCS << 3) | 7, &ulSize) == NO_ERROR) {
          /* usCS = (usCS << 3) | 7; */
          /* ulRetAddr = (ULONG)(USHORT * _Seg16)MAKEULONG(ulEIP, usCS); */
        }
        else {
          is32BitStk = TRUE;
          /* FIXME to be sure this is right */
          ulEIP = (ULONG)usCS << 16 | ulEIP;
          usCS = FLATCS;                /* FIXME? */
        }
      }
      else {
        /* FIXME to get adjusted EIP? */
        is32BitStk = TRUE;
        usCS = FLATCS;                  /* FIXME? */
      }
    }

    if (isPass1)
      fputs(" Trap  -> ", hTrap);
    else if (is32BitStk)
      fprintf(hTrap, " %08X ", ulEBP);
    else
      fprintf(hTrap, " %04X:%04X", usSS, (USHORT)ulEBP);

    if (is32BitStk)
      fprintf(hTrap, " %08X ", ulEIP);
    else
      fprintf(hTrap, " %04X:%04X", usCS, (USHORT)ulEIP);

    if (is32BitStk)
      ul32BitAddr = ulEIP;
    else
      ul32BitAddr = (ULONG)(usCS & ~7) << 13 | ulEIP;

    // 2023-02-23 SHL Do not stop stack walk for bad cs:eip

    ulSize = 10;                        // ulSize must be non-zero
    rc = DosQueryMem((void*)ul32BitAddr, &ulSize, &ulAttr);
    if (rc || ~ulAttr & PAG_COMMIT) {
      fprintf(hTrap, "  Invalid address: %08X\n", ul32BitAddr);
      if (!rc)
        rc = ERROR_INVALID_ADDRESS;     // Prevent access attempts
    }

    printLocals = FALSE;

    if (!rc)
      rc = DosQueryModFromEIP(&hMod, &ulObjNum, sizeof(szModName),
                              szModName, &ulOffset, (ULONG)ul32BitAddr);
    if (rc || ulObjNum == -1)
      fputs("  *Unknown*\n", hTrap);
    else {
      fprintf(hTrap, "  %-08s  %04X:%08X ", szModName, ulObjNum + 1, ulOffset);

      /* If the filename can be identified, try to get the symbol's name using
       * embedded debug info or a DBG file; if that fails, try for a sym file.
       */
      if (DosQueryModuleName(hMod, sizeof(szFileName), szFileName))
        fputc('\n', hTrap);             // Finish off line
      else if (!PrintLineNum(szFileName, ulObjNum, ulOffset, TRUE))
        printLocals = TRUE;             // Got line# info - try locals
      else
        PrintSymbolFromFile(szFileName, hMod, ulObjNum, ulOffset);
    }

    /* Double-space the call-stack listing. */
    fputc('\n', hTrap);

    if (is32BitStk) {
      /* If EBP points to the FLATDS rather than something that looks
       * like a pointer, we are probably looking at a thunk sequence.
       * For VAC this is 0x44 bytes in size  - 2007-07-19 SHL
       * FIXME to be gone - moved to 16-bit code, never worked quite right anyway
       */
      // 2014-03-01 SHL Avoid exception
      if (ebpOk) {
        if (*(PULONG)ulEBP == FLATDS)
          ulEBP += 0x44;                // Assume addressible
        rc = DosQueryMem((void*)ulEBP, &ulSize, &ulAttr);
        if (rc || ~ulAttr & PAG_COMMIT) {
          fprintf(hTrap, "  Cannot access thunk stack at address: %08X\n", ulEBP);
          break;                /* avoid death */
        }
        if (*(PULONG)ulEBP == 0)
          break;                        /* End of call stack */
      }
    }
    else {
      // 16-bit
      ul32BitAddr = (ULONG)(usSS & ~7) << 13 | ulEBP;
      ulSize = 4;    /* sufficient to hold bp and end marker */
      rc = DosQueryMem((void*)(ul32BitAddr), &ulSize, &ulAttr);
      ebpOk = rc == NO_ERROR && ulAttr & PAG_COMMIT && ulSize >= 4;
      if (!ebpOk) {
          fprintf(hTrap, " Cannot access stack at SS:BP: %02X:%02X\n", usSS, (USHORT)ulEBP);
          break;                /* avoid death */
      }
      /* End of call stack? */
      if (*(PUSHORT)ul32BitAddr == 0 && *(PUSHORT)(ul32BitAddr + 2) == 0)
        break;
    } // if 16-bit

    if (isPass1) {
      isPass1 = FALSE;
      if (ebpOk && is32BitStk && printLocals) {
        if (PrintLocalVariables(ulEBP))
          fputc('\n', hTrap);
      }
    }
    else {
      // !Pass1 - get caller's frame pointer
      ulLastEbp = ulEBP;
      ulSize = 4;
      if (is32BitStk) {
        // If return was from EDC3216 inferface, EBP is already set
        if (!ulOverrideEBP)
          ulEBP = *(PULONG)ulEBP;
        else {
          ulEBP = ulOverrideEBP;
          ulOverrideEBP = 0;
        }
        rc = DosQueryMem((void*)ulEBP, &ulSize, &ulAttr);
      }
      else {
        ulLastEbp &= 0xffff;
        ulEBP = *(PUSHORT)ul32BitAddr;
        ulEBP &= ~1;
        ul32BitAddr = (ULONG)(usSS & ~7) << 13 | ulEBP;
        rc = DosQueryMem((void*)ul32BitAddr, &ulSize, &ulAttr);
      }

      // OK if not committed here - will check before attempting to read
      ebpOk = rc == NO_ERROR && ulSize >= 4;

      if (ebpOk && is32BitStk && printLocals) {
        if (PrintLocalVariables(ulEBP))
          fputc('\n', hTrap);
      }

      if (ulEBP < ulLastEbp) {
        fputs(" Lost Stack chain - new EBP below previous\n", hTrap);
        break;
      }
    } // !Pass1

    if (!ebpOk) {
      if (is32BitStk)
        fprintf(hTrap, " Lost Stack chain - invalid EBP %08X\n", ulEBP);
      else
        fprintf(hTrap, " Lost Stack chain - invalid SS:BP %04X:%04X\n", usSS, ulEBP);
      break;
    }

  } /* forever */

  return;
}

/*****************************************************************************/

/* Emit a signature beep sequence at the start, and then a
 * non-threatening low-toned beep every 1.5 seconds thereafter.
 */

void    TimedBeep(void)
{
  static ULONG  ulNextBeep = 0;

  ULONG     now;

  if (!fBeep)
    return;

  DosQuerySysInfo(QSV_MS_COUNT, QSV_MS_COUNT, &now, sizeof(now));
  if (now > ulNextBeep) {
    if (!ulNextBeep) {
      DosBeep(220, 32);
      DosBeep(880, 32);
      DosBeep(440, 32);
    }
    else
      DosBeep(220, 32);

    ulNextBeep = now + 1500;
  }

  return;
}

/*****************************************************************************/

/* DosQueryMem wrapper to handle special case where XCPT_PROCESS_TERMINATE
   occurs when read/exec memory is not really committed yet
   We cannot use a local exception handler to check for access,
   we use some heuristics
*/

APIRET QueryMem(PVOID pb, PULONG pcb, PULONG pFlag)
{
  APIRET apiret;
  HMODULE hMod;
  ULONG ulObjNum;
  ULONG ulOffset;

  // Check access
  apiret = DosQueryMem(pb, pcb, pFlag);

# ifdef DISTORM_DEBUG // 2020-05-16 SHL distorm exception debugging
  if (fIsProcessTerminateInDll) {
    fprintf(hTrap, " * QueryMem out: apiret %u pb %x *pcb %u *pFlag 0x%x (%u)\n",
            apiret, pb, *pcb, *pFlag, __LINE__);
  }
# endif

  // Determine if we need extended access checking
# define PAG_FLAGS (PAG_SHARED | PAG_COMMIT | PAG_EXECUTE | PAG_READ)
  if (apiret == NO_ERROR &&
     (*pFlag & PAG_FLAGS) == PAG_FLAGS &&
     fIsProcessTerminateInDll)
  {
    // Check if address is in DLL pointed to by exception EIP
    apiret = DosQueryModFromEIP(&hMod, &ulObjNum, CCHMAXPATH,
                                szModName, &ulOffset,
                                (ULONG)pb);
    if (apiret != NO_ERROR) {
      fprintf(hTrap, " * QueryMem: DosQueryModFromEIP(%x) failed with error %u (%u)\n",
              pb, apiret, __LINE__);
    }
    // If address is same DLL as and exception EIP, we probably cannot read, so turn off PAG_READ
    else if (hMod == hModTerminateDll)
      *pFlag &= ~PAG_READ;

  } // if need
# undef PAG_FLAGS

# ifdef DISTORM_DEBUG // 2020-05-16 SHL distorm exception debugging
  fprintf(hTrap, " * QueryMem final: apiret %u *pFlag 0x%x (%u)\n", apiret, *pFlag, __LINE__);
# endif

  return apiret;
}

/*****************************************************************************/
