(********************************************************************************************
** PROGRAM     : mmind
** VERSION     : 1.0.0
** DESCRIPTION : Simple Master Mind Game
** AUTHOR      : Stuart King
** COPYRIGHT   : Copyright (c) Irie Tools, 2002. All Rights Reserved.
** NOTES       :
**    This sample program is distributed with Irie Pascal, and was written to provide
** an example of how to write programs with Irie Pascal. To make best use of this sample
** you should have a basic understanding of Pascal.
********************************************************************************************)
program mmind;
const
	MIN_LETTERS = 3;
	MAX_LETTERS = 8;
type
	NumLettersType = MIN_LETTERS..MAX_LETTERS;
	SequenceType = string[MAX_LETTERS];
	positive = 0..maxint;
var
	NumLetters : NumLettersType;
	sequence, mask : SequenceType;
	guess : string;
	iBlack : 0..MAX_LETTERS;
	iWhite : 0..MAX_LETTERS;
	iTries : 0..maxint;
	blnDone, blnValidGuess : boolean;

	//PURPOSE: Makes zero or more copies of an input string
	//PARAMERTER(s):
	//    1. StrIn - The input string
	//    2. iNum - The Number of copies to make
	function copies(strIn : string; iNum : positive) : string;
	var
		strOut : string;
		i : integer;
	begin (* copies *)
		strOut := '';
		for i := 1 to iNum do
			strOut := strOut + strIn;
		copies := strOut
	end; (* copies *)

	//This function checks that the string "strIn" contains only
	//characters in "strOnly".
	function verify(strIn, strOnly : string) : boolean;
	var
		blnOnly : boolean;
		i : 1..maxint;
	begin (* verify *)
		blnOnly := true;
		for i := 1 to length(strIn) do
			if pos(strIn[i], strOnly) = 0 then
				blnOnly := false;
		verify := blnOnly
	end; (* verify *)

	//PURPOSE: Gets the number of letters in the sequence from the user.
	procedure GetNumLettersFromUser;
	var
		iNum, iErr : integer;
		strInput : string;
	begin (* GetNumLettersFromUser *)
		write('How many letters (A-F) are in the sequence? (', MIN_LETTERS:1,'-', MAX_LETTERS:1, '): ');
		readln(strInput);
		val(strInput, iNum, iErr);
		if (iErr<>0) or (iNum<MIN_LETTERS) or (iNum>MAX_LETTERS) then
			begin
				writeln('You must enter a number between ', MIN_LETTERS:1, ' and ', MAX_LETTERS:1);
				halt
			end;
		NumLetters := iNum;
	end; (* GetNumLettersFromUser *)

	//PURPOSE: Gets the number of letters in the sequence from the program parameters
	procedure GetNumLettersFromParam;
	var
		iNum, iErr : integer;
	begin (* GetNumLettersFromParam *)
		val(paramstr(1), iNum, iErr);
		if (iErr<>0) or (iNum<MIN_LETTERS) or (iNum>MAX_LETTERS) then
			begin
				writeln('The number of letters in the sequence must be between ', MIN_LETTERS:1, ' and ', MAX_LETTERS);
				halt
			end;
	end; (* GetNumLettersFromParam *)

	//PURPOSE: Generate MasterMind sequence
	//GLOBAL(s):
	//    1. NumLetters - The number of letter in the sequence
	procedure GenerateSequence;
	type
		ValidLetter = 'A'..'F';
	var
		i : 1..MAX_LETTERS;

		//PURPOSE: Generates a single letter in the sequence
		//RETURNS:
		//   The generated letter.
		function GenerateLetter : ValidLetter;
		var
			c : ValidLetter;
		begin (* GenerateLetter *)
			c := chr(ord('A')+random(6));
			GenerateLetter := c
		end; (* GenerateLetter *)

	begin (* GenerateSequence *)
		randomize;
		sequence := '';
		for i := 1 to NumLetters do
			sequence := sequence + GenerateLetter
	end; (* GenerateSequence *)

	//PURPOSE: Searches for Black (i.e. correct letter in correct position) matches
	//         Between the user's guess and the sequence.
	//GLOBAL(s):
	//    1. NumLetters - Number of letters in the sequence (and the guess).
	//    2. iBlack - Number of black matches
	//    3. guess - The user's guess
	//    4. sequence - The sequence
	//    5. mask - The mask starts off as an exact copy of the sequence, however
	//           during matching the matched characters are overwritten with
	//           a character known to be not in the guess. This is done to prevent
	//           the matched characters from being matched more than once. And since
	//           the original sequence needs to be preserved for the next guess
	//           then the matching is done with this copy instead of the original.
	procedure SearchForBlackMatches;
	var
		i : 0..maxint;
	begin (* SearchForBlackMatches *)
		iBlack := 0;
		mask := sequence;
		for i := 1 to NumLetters do
			if guess[i] = mask[i] then
				begin
					iBlack := iBlack + 1;
					mask[i] := 'X';
					guess[i] := 'Y';
				end;
	end; (* SearchForBlackMatches *)

	//PURPOSE: Searches for White (i.e. correct letter in wrong position) matches
	//         Between the user's guess and the sequence.
	//GLOBAL(s):
	//    1. NumLetters - Number of letters in the sequence (and the guess).
	//    2. iWhite - Number of white matches
	//    3. guess - The user's guess
	//    4. sequence - The sequence
	//    5. mask - The mask starts off as an exact copy of the sequence, however
	//           during matching the matched characters are overwritten with
	//           a character known to be not in the guess. This is done to prevent
	//           the matched characters from being matched more than once. And since
	//           the original sequence needs to be preserved for the next guess
	//           then the matching is done with this copy instead of the original.
	//NOTES:
	//   This procedure should be called after SearchForBlackMatches which copies
	// the sequence into the mask and overwrites the black matches.
	procedure SearchForWhiteMatches;
	var
		i, iPos : 0..maxint;
	begin (* SearchForWhiteMatches *)
		iWhite := 0;
		for i := 1 to NumLetters do
			begin
				iPos := pos(guess[i], mask);
				if iPos <> 0 then
					begin
						iWhite := iWhite + 1;
						mask[iPos] := 'X';
						guess[i] := 'Y';
					end;
			end
	end; (* SearchForWhiteMatches *)

	//PURPOSE: Reports the resuls of the matches.
	procedure ReportMatches;
	begin (* ReportMatches *)
		writeln(iWhite:1, ' WHITE   ', iBlack:1, ' BLACK      ', iTries);
	end; (* ReportMatches *)

begin
	if paramcount = 0 then
		GetNumLettersFromUser
	else
		GetNumLettersFromParam;

	GenerateSequence;
	iBlack := 0;
	iWhite := 0;
	iTries := 0;
	blnDone := FALSE;

	writeln('****** W E L C O M E    T O    M A S T E R M I N D *******');
	repeat
		blnValidGuess := true;
		write('What is my sequence ? ');
		readln(guess);
		guess := uppercase(trim(guess));
		if (guess='QUIT') or (Guess='STOP') then
			begin
				blnValidGuess := false;
				blnDone := true;
			end;
		if blnValidGuess and (not verify(guess, 'ABCDEF.')) then
			begin
				writeln(copies('?', NumLetters));
				blnValidGuess := false;
			end;
		if blnValidGuess and (not blnDone) then
			begin
				if length(guess) <> NumLetters then
					begin
						writeln(copies('?', NumLetters));
						blnValidGuess := false;
					end;
			end;
		if blnValidGuess and (not blnDone) then
			begin
				iTries := iTries + 1;
				SearchForBlackMatches;
				SearchForWhiteMatches;
				ReportMatches
			end;
		if iBlack=NumLetters then
			begin
				writeln('**** C O N G R A T U L A T I O N S ****');
				blnDone := true
			end;
	until blnDone;
end.

