module: format
author: chiles@cs.cmu.edu
synopsis: This file implements a simple mechanism for formatting output.
copyright: See below.
rcs-header: $Header: /afs/cs.cmu.edu/project/gwydion/hackers/nkramer/mindy/mindy-1.3/libraries/format/RCS/format.dylan,v 1.4 95/03/12 16:52:55 nkramer Exp $

//======================================================================
//
// Copyright (c) 1994  Carnegie Mellon University
// All rights reserved.
// 
// Use and copying of this software and preparation of derivative
// works based on this software are permitted, including commercial
// use, provided that the following conditions are observed:
// 
// 1. This copyright notice must be retained in full on any copies
//    and on appropriate parts of any derivative works.
// 2. Documentation (paper or online) accompanying any system that
//    incorporates this software, or any part of it, must acknowledge
//    the contribution of the Gwydion Project at Carnegie Mellon
//    University.
// 
// This software is made available "as is".  Neither the authors nor
// Carnegie Mellon University make any warranty about the software,
// its performance, or its conformity to any specification.
// 
// Bug reports, questions, comments, and suggestions should be sent by
// E-mail to the Internet address "gwydion-bugs@cs.cmu.edu".
//
//======================================================================
//



/// format-to-string.
///

/// format-to-string -- Exported.
///
define generic format-to-string (control-string :: <string>, #rest args)
    => result :: <string>;

define method format-to-string (control-string :: <byte-string>, #rest args)
    => result :: <byte-string>;
  let s = make(<byte-string-output-stream>);
  apply(format, s, control-string, args);
  s.string-output-stream-string;
end method;



/// Print-message.
///

/// print-message -- Exported.
///
define generic print-message (object :: <object>, stream :: <stream>)
    => ();


define method print-message (object :: type-or(<string>, <character>),
			     stream :: <stream>)
    => ();
  write(object, stream);
end method;

define method print-message (object :: <condition>, stream :: <stream>)
    => ();
  report-condition(object, stream);
end method;

define method print-message (object :: <symbol>, stream :: <stream>)
    => ();
  write(as(<string>, object), stream);
end method;



/// Format.
///
define constant $dispatch-char = '%';

define constant char-classes
    = make(<vector>, size: 256, fill: #f);
///
for (i from as(<byte>, '0') below (as(<byte>, '9') + 1))
  char-classes[i] := #"digit";
end;
char-classes[as(<byte>, '-')] := #"digit";


define generic format (stream :: <stream>, control-string :: <string>,
		       #rest args)
    => ();

define method format (stream :: <stream>, control-string :: <byte-string>,
		      #rest args)
    => ();
  let control-len :: <fixed-integer> = control-string.size;
  block (exit)
    let start :: <fixed-integer> = 0;
    let arg-i :: <fixed-integer> = 0;
    // Ensure all output is contiguous at stream's destination.
    lock-stream(stream);
    while (start < control-len)
      // Skip to dispatch char.
      for (i = start then (i + 1),
	   until ((i = control-len) | (control-string[i] = $dispatch-char)))
      finally
	write(control-string, stream, start: start, end: i);
	if (i = control-len)
	  exit();
	else
	  start := i + 1;
	end;
      end for;
      // Parse for field within which to pad output.
      let (field, field-spec-end)
	= if (char-classes[as(<byte>, control-string[start])] = #"digit")
	    parse-integer(control-string, start);
	  end;
      if (field)
	// Capture output in string and compute padding.
	let s = make(<byte-string-output-stream>);
	if (do-dispatch(control-string[field-spec-end], s,
			element(args, arg-i, default: #f)))
	  arg-i := arg-i + 1;
	end;
	let output = s.string-output-stream-string;
	let output-len :: <fixed-integer> = output.size;
	let padding :: <fixed-integer> = (abs(field) - output-len);
	case
	  (padding < 0) =>
	    write(output, stream);
	  (field > 0) =>
	    write(make(<byte-string>, size: padding, fill: ' '), stream);
	    write(output, stream);
	  otherwise =>
	    write(output, stream);
	    write(make(<byte-string>, size: padding, fill: ' '), stream);
	end;
	start := field-spec-end + 1;  // Add one to skip dispatch char.
      else
	if (do-dispatch(control-string[start], stream,
			element(args, arg-i, default: #f)))
	  arg-i := arg-i + 1;
	end;
	start := start + 1;  // Add one to skip dispatch char.
      end;
    end while;
  cleanup
    unlock-stream(stream);
  end;
end method;
    
/// do-dispatch -- Internal.
///
/// This function dispatches on char, which should be a format directive.
/// The return value indicates whether to consume one format argument;
/// otherwise, consume none.
///
define method do-dispatch (char :: <byte-character>, stream :: <stream>, arg)
    => consumed-arg? :: <boolean>;
  select (char by \=)
    ('s'), ('S'), ('c'), ('C') =>
      print-message(arg, stream);
      #t;
    ('=') =>
      print(arg, stream);
      #t;
    ('d'), ('D') =>
      format-integer(arg, 10, stream);
      #t;
    ('b'), ('B') =>
      format-integer(arg, 2, stream);
      #t;
    ('o'), ('O') =>
      format-integer(arg, 8, stream);
      #t;
    ('x'), ('X') =>
      format-integer(arg, 16, stream);
      #t;
    ('m'), ('M') =>
      apply(arg, list(stream));
      #t;
    ('%') =>
      write('%', stream);
      #f;
    otherwise =>
      error("Unknown format dispatch character, %c", char);
  end;
end method;

/// parse-integer -- Internal.
///
/// This function reads an integer from input starting at index.  Index must
/// be at the first digit or a leading negative sign.  This function reads
/// decimal representation, and it stops at the first character that is not
/// a decimal degit.  It returns the integer parsed and the index
/// immediately following the last decimal digit.
///
define method parse-integer (input :: <byte-string>, index :: <fixed-integer>)
    => (result :: false-or(<integer>), index :: <fixed-integer>);
  let result :: <integer> = 0;
  let negative? = if (input[index] = '-')
		    index := index + 1;
		  end;
  for (i :: <fixed-integer> = index then (i + 1),
       len :: <fixed-integer> = input.size then len,
       ascii-zero :: <byte> = as(<byte>, '0') then ascii-zero,
       until ((i = len) |
	      (~ (char-classes[as(<byte>, input[i])] == #"digit"))))
    result := ((result * 10) + (as(<byte>, input[i]) - ascii-zero));
  finally
    if (result = 0)
      values(#f, index);
    else
      values(if (negative?) (- result) else result end, i);
    end;
  end;
end method;


define constant $digits = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ";

/// format-integer -- internal
///
define method format-integer (arg :: <integer>,
			      radix :: limited(<integer>, min: 2, max: 36),
			      stream :: <stream>)
    => ();
  // Collect the digit print representations.
  local method repeat (arg, digits)
	  let (quotient, remainder) = floor/(arg, radix);
	  let digits = pair($digits[as(<fixed-integer>, remainder)], digits);
	  if (zero?(quotient))
	    for (digit in digits)
	      write(digit, stream);
	    end;
	  else
	    repeat(quotient, digits);
	  end;
	end;
  if (negative?(arg))
    write('-', stream);
    repeat(-arg, #());
  else
    repeat(arg, #());
  end;
end;

define method format-integer (arg == $minimum-fixed-integer,
			      radix :: limited(<integer>, min: 2, max: 36),
			      stream :: <stream>)
 => ();
  format-integer(as(<extended-integer>, arg), radix, stream);
end;
 