implementation module deltaTimer

import StdEnv, misc
import oskernel, timerDevice

TicksPerSecond :== 60

::  CurrentTime :== (!Int, !Int, !Int)
/*                      (hours  (0-23),
		  minutes    (0-59),
		  seconds    (0-59)) */

::  CurrentDate :== (!Int, !Int, !Int, !Int)
/*                      (year,
		  month       (1-12),
		  day             (1-31),
		  day of week (1-7, Sunday = 1, Saturday = 7)) */

::  DeltaTimerHandle * s * io :== (TimerHandle s io) ->  TimerHandle s io  // for local use

//  Opening and Closing timers:
OpenTimer:: !(TimerDef s (IOState s)) !(IOState s) -> IOState s
OpenTimer timer_def io_state
		= let!
			timer = NewTimer timer_def
		  in IOStateSetDevice io_state` (TimerSystemState [timer : timers])
		  where
		  (timers,io_state`) = IOStateGetTimerDevice io_state

NewTimer:: !(TimerDef s (IOState s)) -> TimerHandle s (IOState s)
NewTimer (Timer id select interval funcs)
		= let!
			handle = OSTimerStart id ostime
		  in newtimer
		  where
		  newtimer = (handle, id, ostime, funcs)
		  ostime   = (1000 * interval) / TicksPerSecond

CloseTimer:: !TimerId !(IOState s) -> IOState s
CloseTimer id io_state
		= let!
			timers` = TimerHandlesCloseTimer id timers
		  in IOStateSetDevice io_state` (TimerSystemState timers`)
		  where
		  (timers, io_state`) = IOStateGetTimerDevice io_state

TimerHandlesCloseTimer:: !TimerId !(TimerHandles s (IOState s)) -> TimerHandles s (IOState s)
TimerHandlesCloseTimer id [timer=:(handle,id2,ostime,fun) : timers]
	| id <> id2 
		= let!
			strict1 = TimerHandlesCloseTimer id timers
		  in [timer : strict1]
		= Evaluate_2 timers (OSTimerStop handle)
TimerHandlesCloseTimer id timers
		= timers


//  Enabling and Disabling of TimerDevices:
EnableTimer:: !TimerId !(IOState s) -> IOState s
EnableTimer id io_state
		= let!
			timers` = TimerHandlesSetTimer id (DeltaSetAbilityTimer Able) timers
		  in
		  IOStateSetDevice io_state` (TimerSystemState timers`)
		  where
		  (timers,io_state`) = IOStateGetTimerDevice io_state


DisableTimer:: !TimerId !(IOState s) -> IOState s
DisableTimer id io_state
		= let!
			timers` = TimerHandlesSetTimer id (DeltaSetAbilityTimer Unable) timers
		  in IOStateSetDevice io_state` (TimerSystemState timers`)
		  where
		  (timers,io_state`) = IOStateGetTimerDevice io_state


//  Changing the TimerFunction and TimerInterval:
ChangeTimerFunction:: !TimerId !(TimerFunction s (IOState s)) !(IOState s) -> IOState s
ChangeTimerFunction id f io_state
		= let!
			timers` = TimerHandlesSetTimer id (DeltaSetTimerFunction f) timers
		  in IOStateSetDevice io_state` (TimerSystemState timers`)
		  where
		  (timers,io_state`) = IOStateGetTimerDevice io_state


SetTimerInterval:: !TimerId !TimerInterval !(IOState s) -> IOState s
SetTimerInterval id interval_time io_state
		= let!
			timers` = TimerHandlesSetTimer id (DeltaSetTimerInterval interval_time) timers
		  in IOStateSetDevice io_state` (TimerSystemState timers`)
		  where
		  (timers,io_state`) = IOStateGetTimerDevice io_state


IOStateGetTimerDevice:: !(IOState s) -> (!TimerHandles s (IOState s), !IOState s)
IOStateGetTimerDevice io_state
		= (GetTimerHandles timers, io_state`)
		  where
		  (timers,io_state`) = IOStateGetDevice io_state TimerDevice


TimerHandlesSetTimer:: !TimerId !(DeltaTimerHandle s (IOState s)) !(TimerHandles s (IOState s)) -> TimerHandles s (IOState s)
TimerHandlesSetTimer id dt [timer=:(handle,id2,ostime,fun) : timers]
	| id == id2
		= let!
			strict1 = dt timer
		  in [strict1 : timers]
		= let!
			strict2 = TimerHandlesSetTimer id dt timers
		  in [timer : strict2]
TimerHandlesSetTimer id dt []
		= []


DeltaSetAbilityTimer:: !SelectState !(TimerHandle s (IOState s)) -> TimerHandle s (IOState s)
DeltaSetAbilityTimer Able (-1,id,ostime,fun)
		= let!
			handle = OSTimerStart id ostime
		  in (handle,id,ostime,fun)
DeltaSetAbilityTimer Unable (handle,id,ostime,fun)
		= let!
			handle` = OSTimerStop handle
		  in (handle`,id,ostime,fun)
DeltaSetAbilityTimer able (handle,id,ostime,fun)
		= (handle,id,ostime,fun)


DeltaSetTimerFunction:: !(TimerFunction s (IOState s)) !(TimerHandle s (IOState s)) -> TimerHandle s (IOState s)
DeltaSetTimerFunction newfun (handle,id,ostime,fun)
		= (handle, id, ostime, newfun)


DeltaSetTimerInterval:: !TimerInterval !(TimerHandle s (IOState s)) -> TimerHandle s (IOState s)
DeltaSetTimerInterval newinterval (-1,id,ostime,fun)
		= (-1,id,ostime`,fun)
		  where
		  ostime`  = (1000 * newinterval) / TicksPerSecond
DeltaSetTimerInterval newinterval (handle,id,ostime,fun)
		= let!
			handle`  = OSTimerStop handle
			handle`` = Evaluate_2 (OSTimerStart id ostime`) handle`
		  in (handle``,id,ostime`,fun)
		  where
		  ostime`  = (1000 * newinterval) / TicksPerSecond


GetTimerHandles:: !(DeviceSystemState * s) ->  [TimerHandle *s (IOState *s)]
GetTimerHandles (TimerSystemState timers)
		= timers

//  Suspend the interaction for a number of ticks.
Wait:: !TimerInterval x -> x
Wait nrticks x 
		= Evaluate_2 x (OSTimeSleep ostime)
		  where
		  ostime = (nrticks * 1000) / TicksPerSecond


UWait:: !TimerInterval * x -> *x
UWait nrticks ux 
		= Evaluate_2 ux (OSTimeSleep ostime)
		  where
		  ostime = (nrticks * 1000) / TicksPerSecond


//  Getting the blinking time:
GetTimerBlinkInterval:: !(IOState s) -> (!TimerInterval, !IOState s)
GetTimerBlinkInterval io_state
		= (OSTimeGetBlinkInterval 0, io_state)


//  Get current time and date:
GetCurrentTime:: !(IOState s) -> (!CurrentTime, !IOState s)
GetCurrentTime io_state
		= ((hours, minutes, seconds), io_state)
		  where
		  (_,_,_, hours, minutes, seconds,_) = OSTimeGetCurrentTime 0


GetCurrentDate:: !(IOState s) -> (!CurrentDate, !IOState s)
GetCurrentDate io_state
		= ((year, month, day, dayofweek), io_state)
		  where
		  (year, month, day,_,_,_, dayofweek) = OSTimeGetCurrentTime 0
