//===============================================================
// vwinprtr.cpp - Windows Printer class
//
// Copyright (C) 1995,1996,1997,1998  Bruce E. Wampler
//
// This file is part of the V C++ GUI Framework, and is covered
// under the terms of the GNU Library General Public License,
// Version 2. This library has NO WARRANTY. See the source file
// vapp.cxx for more complete information about license terms.
//===============================================================
#include <v/vos2.h>           // for OS/2 stuff
#include <v/vapp.h>
#include <v/vwinprtr.h>
#include <v/vthislst.h>         // For this list

  // call back for the print setup dialog
  MRESULT EXPENTRY PrintDlgProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2);

  static vThisList _thisList;           // hide this in this file

  int vWinPrinter::_instances = 0;     // reference counter for copy / assignment
//================>>> vWinPrinter::vWinPrinter <<<========================
  vWinPrinter::vWinPrinter()
  {
    SysDebug(Constructor,"vWinPrinter::vWinPrinter() constructor\n")

    _name = "VPrintJob";     // a default name
    _paperType = vPaperDefault;
    _copies = 1;
    _toFile = 0;
    _width = 0;
    _height = 0;
    _portrait = 1;
    _useColor = 0;
    _printhDC = 0;
    ++_instances;               // bump reference counter

    // allocate a block of memory to hold printer information
    // of size 4kB
    DosAllocMem((PPVOID)&_pMem, 0x8000, PAG_READ | PAG_WRITE);
    // allow block to be suballocated into smaller chunks as needed
    // to conserve RAM.  The size of the pool is increased to 32kB
    DosSubSetMem(_pMem, DOSSUB_INIT | DOSSUB_SPARSE_OBJ, 0x8000);
    // set default printer properties
    _PrintRange = 0;
    _Collate = 0;
    _Priority = 50;
    _pPRQCurrent = 0;
    _SizePRQCurrent = 0;
    _pDDCurrent = 0;
    _SizeDDCurrent = 0;
    _pPRQArray = 0;
    _SizePRQArray = 0;
    _pHCInfo = 0;
    _SizeHC = 0;
    _hInfoDevC = 0;
    _hInfoPS = 0;
    _hPrintDevC = 0;
    _printhDC = 0;
    _jobID = 0;

    // Query the default printer
    char Buf[MAX_BUFF];

    _SizePRQCurrent = PrfQueryProfileString(HINI_PROFILE, "PM_SPOOLER", "QUEUE",
      0, Buf, MAX_BUFF);
    // if got default printer ok, then build PRQINFO3 structure
    // describing the printer in detail
    if (_SizePRQCurrent)
    {
      // chop off the ';' at end
      PSZ pStr = strchr(Buf, ';');
      *pStr = 0;
      // how much space do we need?
      SplQueryQueue(0, Buf, 3, 0, 0, &_SizePRQCurrent);
      // make enough room for the printer info
      if (_SizePRQCurrent && !(DosSubAlloc(_pMem, (PPVOID) &_pPRQCurrent, _SizePRQCurrent)))
      {
        SplQueryQueue(0, Buf, 3, _pPRQCurrent, _SizePRQCurrent, &_SizePRQCurrent);
      }
      // If the queue description is a NULL string then use the
      //the queue name so the dialog won't have a blank entry
      if (*_pPRQCurrent->pszComment)
        strcpy(_QueueDesc, _pPRQCurrent->pszComment);
      else
        strcpy(_QueueDesc, _pPRQCurrent->pszName);
      // update queue name
      strcpy(_QueueName, _pPRQCurrent->pszName);
      // load the default printer properties
      GetJobProps(_pPRQCurrent, DPDM_QUERYJOBPROP);
    }



  }
//================>>> vWinPrinter::vWinPrinter <<<========================
  vWinPrinter::vWinPrinter(const vWinPrinter& pr)
  {
    // copy constructor
    _name = pr._name;
    _width = pr._width;
    _height = pr._height;
    _portrait = pr._portrait;
    _useColor = pr._useColor;
    _paperType = pr._paperType;
    _copies = pr._copies;
    _toFile = pr._toFile;
    _printhDC = pr._printhDC;

    _PageSizel.cx = pr._PageSizel.cx;
    _PageSizel.cy = pr._PageSizel.cy;
    _PhyPagePels.cx = pr._PhyPagePels.cx;
    _PhyPagePels.cy = pr._PhyPagePels.cy;
    _PrintOffsetSizel.cx = pr._PrintOffsetSizel.cx;
    _PrintOffsetSizel.cy = pr._PrintOffsetSizel.cy;
    _PrintRange = pr._PrintRange;
    _Collate = pr._Collate;
    _Priority = pr._Priority ;
    _pPRQCurrent = pr._pPRQCurrent;
    _SizePRQCurrent = pr._SizePRQCurrent;
    _pDDCurrent = pr._pDDCurrent;
    _SizeDDCurrent = pr._SizeDDCurrent;
    _pPRQArray = pr._pPRQArray;
    _SizePRQArray = pr._SizePRQArray;
    _pHCInfo = pr._pHCInfo;
    _SizeHC = pr._SizeHC;
    _hInfoDevC = pr._hInfoDevC;
    _hInfoPS = pr._hInfoPS;
    _hPrintDevC = pr._hPrintDevC;
    _printhDC = pr._printhDC;
    _jobID = pr._jobID;

    ++_instances;               // bump reference counter
  }
//================>>> vWinPrinter::= <<<========================
  vWinPrinter& vWinPrinter::operator =(const vWinPrinter& pr)
  {
    if (this == &pr)    // to self?
        return *this;
//    if (_name)
//        delete [] _name;        // free space before overwrite
    _name = pr._name;
    _width = pr._width;
    _height = pr._height;
    _portrait = pr._portrait;
    _useColor = pr._useColor;
    _paperType = pr._paperType;
    _copies = pr._copies;
    _toFile = pr._toFile;
    _printhDC = pr._printhDC;

    _PageSizel.cx = pr._PageSizel.cx;
    _PageSizel.cy = pr._PageSizel.cy;
    _PhyPagePels.cx = pr._PhyPagePels.cx;
    _PhyPagePels.cy = pr._PhyPagePels.cy;
    _PrintOffsetSizel.cx = pr._PrintOffsetSizel.cx;
    _PrintOffsetSizel.cy = pr._PrintOffsetSizel.cy;
    _PrintRange = pr._PrintRange;
    _Collate = pr._Collate;
    _Priority = pr._Priority ;
    _pPRQCurrent = pr._pPRQCurrent;
    _SizePRQCurrent = pr._SizePRQCurrent;
    _pDDCurrent = pr._pDDCurrent;
    _SizeDDCurrent = pr._SizeDDCurrent;
    _pPRQArray = pr._pPRQArray;
    _SizePRQArray = pr._SizePRQArray;
    _pHCInfo = pr._pHCInfo;
    _SizeHC = pr._SizeHC;
    _hInfoDevC = pr._hInfoDevC;
    _hInfoPS = pr._hInfoPS;
    _hPrintDevC = pr._hPrintDevC;
    _printhDC = pr._printhDC;
    _jobID = pr._jobID;

    return *this;
  }
//================>>> vWinPrinter::~vWinPrinter <<<========================
  vWinPrinter::~vWinPrinter()
  {
    // We are using a reference count to track instances of printers.
    // We want to allow the user to pass copies of the single printer
    // around, but there should only be one name and one stream;
    SysDebug1(Destructor,"vWinPrinter::~vWinPrinter() destructor (instance=%u)\n",
      _instances)
    --_instances;               // not reference any more
    if (_instances > 0)
        return;                 // don't do anything else
//    if (_name)
//        delete [] _name;

    // deallocate resources for printer info
    if (_SizePRQArray != 0)
    {
      DosSubFreeMem(_pMem, (PPVOID)_pPRQArray, _SizePRQArray);
      _SizePRQArray =0;
    }
    if (_SizePRQCurrent != 0)
    {
      DosSubFreeMem(_pMem, (PPVOID)_pPRQCurrent, _SizePRQCurrent);
      _SizePRQCurrent = 0;
    }

    if (_SizeDDCurrent != 0)
    {
      DosSubFreeMem(_pMem, (PPVOID)_pDDCurrent, _SizeDDCurrent);
      _SizeDDCurrent = 0;
    }
    // Free existing info DCs and PSs
    if (_hInfoPS)
    {
      GpiAssociate(_hInfoPS, NULLHANDLE);
      GpiDestroyPS(_hInfoPS);
      _hInfoPS = 0;
      DevCloseDC(_hInfoDevC);
      _hInfoDevC = 0;
    }

    if (_printhDC)
    {
      GpiAssociate(_printhDC, NULLHANDLE);
      GpiDestroyPS(_printhDC);
      _printhDC = 0;
      DevCloseDC(_hPrintDevC);
      _hPrintDevC = 0;
    }

    if (_pHCInfo)
    {
      DosSubFree(_pMem, _pHCInfo, (_SizeHC * sizeof(HCINFO)) );
      _pHCInfo = 0;
      _SizeHC = 0;
    }
  }
//================>>> vWinPrinter::GetPaperName <<<========================
  char* vWinPrinter::GetPaperName()
  {
    if (_paperType == vPaperLetter)     // we have (8.5x11) added...
        return "Letter";
    else
        return "Unknown";
  }
//================>>> vWinPrinter::Setup <<<========================
  int vWinPrinter::Setup(char* fn)
  {
    // fn is an override default name
    if (fn)
      _name = fn;

    //destroy the PS before running setup again
    if (_printhDC != 0)
        return 0;

    // this allows user to set printer and printer parms.
    // the global printer parms are then updated before returning
//    int result = WinDlgBox(HWND_DESKTOP, _myHwnd,
//      (PFNWP)&PrintDlgProc, 0L, vID_PRINT_DLG, NULL);
    int result = WinDlgBox(HWND_DESKTOP, HWND_DESKTOP,
      (PFNWP)&PrintDlgProc, 0L, vID_PRINT_DLG, (PVOID) this);

    if (result == DID_OK)
    {
/*
      pixw = ::GetDeviceCaps(_printhDC, HORZRES);
        pixh = ::GetDeviceCaps(_printhDC, CAPS_HEIGHT);
        _width = ::GetDeviceCaps(_printhDC, CAPS_WIDTH) * 3;
        GpiQueryPS (,) & PS_UNITS;
        GpiSetDefaultViewMatrix (,,,);
        GpiQueryPageViewport (,);
        GpiSetPageViewport (,);
*/
      return DID_OK;
    }
    return 0;    // user cancelled print job
  }



/*-----------------------------------------------------------------

   This function will enumerate the installed print queues and add
   their description to hWndCBox.  The function returns a pointer
   to an array of PPRQINFO3 structures if successful. The number of
   entries in the array and the total size of the allocated memory
   is returned in the _numPRQ and SizePRQArray parameters respectfully.
   The index of the current application default queue is returned in
   the _ixCurrent parameter.

-------------------------------------------------------------------*/
  VOID vWinPrinter::EnumPrintQueues(HWND hWnd)
  {
    ULONG     Returned, numPRQ;
    int       i;
    PSZ       pName;
    HWND hWndCBox = WinWindowFromID(hWnd, vID_PRINTER);

    // get number/size of available printers if first time
    // delete any previous data first
    if (_SizePRQArray != 0)
    {
      DosSubFreeMem(_pMem, (PPVOID)_pPRQArray, _SizePRQArray);
    }
    SplEnumQueue(0, 3, 0, 0L, (PULONG)&Returned, (PULONG)&numPRQ, (PULONG)&_SizePRQArray, 0);

    // suballocate required RAM for printer info and point to it with pPRQArray
    if (!DosSubAllocMem(_pMem, (PPVOID)&_pPRQArray, _SizePRQArray))
    {
      // get printer data
      SplEnumQueue( 0, 3, _pPRQArray, _SizePRQArray, &Returned, &numPRQ, &_SizePRQArray, 0);
      // for performance, we disable the combobox while its list is updated
      WinEnableWindowUpdate(hWndCBox,FALSE);
      WinSendMsg (hWndCBox, LM_DELETEALL,0,0);
      for (i = 0; i < numPRQ; i++)
      {
        // If the queue description is a NULL string then use the
        //the queue name so the dialog won't have a blank entry
        if (*_pPRQArray[i].pszComment)
          pName = _pPRQArray[i].pszComment;
        else
          pName = _pPRQArray[i].pszName;
        // we check for embedded carriage returns and replace with a space
        // if we find any to make the listbox entry more esthetic
        char *ch;
        while ( (ch = strpbrk(pName, "\r\n")) != 0)
          *ch = ' ';

        WinSendMsg (hWndCBox, LM_INSERTITEM, (MPARAM)LIT_END, (MPARAM)pName);
        // check if it is default queue
        if (strcmp(_QueueDesc, pName) == 0)
        {
          // This is the current application default
          _ixCurrent = i;
        }
      }
      WinEnableWindowUpdate(hWndCBox, TRUE);
    }
  }


/*-----------------------------------------------------------------*\

   This function will query or post job properties for the queue
   identified by the pPRQ parameter. Option should be set
   to DPDM_QUERYJOBPROP or DPDM_POSTJOBPROP.
   Space for the corresponding job properties is allocated as needed
   and _pDDCurrent is set to point to it.

\*-----------------------------------------------------------------*/
  VOID vWinPrinter::GetJobProps(PPRQINFO3 pPRQ, ULONG Option)
  {
    int   i;
    PSZ   pTemp;
    CHAR  DriverName[MAX_DRIVERNAME];
    CHAR  DeviceName[MAX_DEVICENAME];

    // get printer driver and device names from queue info
    if ((i = strcspn(pPRQ->pszDriverName,".")) != 0)
    {
      strncpy((PSZ)DriverName, (PSZ)pPRQ->pszDriverName, i);
      DriverName[i] = '\0';
      strcpy((PSZ)DeviceName, (PSZ)&pPRQ->pszDriverName[i + 1]);
    }
    else  // some drivers do not have a device name
    {
      strcpy((PSZ)DriverName, pPRQ->pszDriverName);
      *DeviceName = '\0';
    }
    // we only want the first printer if queue is pooled
    // so chop off everything after first comma
    pTemp = (PSZ)strchr(pPRQ->pszPrinters, ',');
    if ( pTemp )
    {
      *pTemp = '\0' ;
    }

    if (_SizeDDCurrent == 0)  // first time or printer changed
    {
      // Allocate memory for the driver data by first finding out how
      // much space to allocate
      _SizeDDCurrent = DevPostDeviceModes(theApp->AppHab(), 0, DriverName, DeviceName,
        pPRQ->pszPrinters, Option);
      DosSubAllocMem(_pMem, (PPVOID) &_pDDCurrent, _SizeDDCurrent);
      // fill in a few blanks as well
      _pDDCurrent->cb = _SizeDDCurrent;  // this is unnecesary
      _pDDCurrent->lVersion = 0;
      strcpy(_pDDCurrent->szDeviceName, DeviceName);
    }

    // get the info
    // previous changes to the job props will be retained
    // from last time if the printer wasn't changed
    DevPostDeviceModes(theApp->AppHab(), _pDDCurrent, DriverName, DeviceName,
      pPRQ->pszPrinters, Option);
  }


/*-----------------------------------------------------------------*\

   This function will intialize the capabilities based
   on the queue pointed to by _pPRQCurrent. If a previous INFO HDC exists
   the HDC it is destroyed and any form info is freed.
   A new info HDC is created and the the metrics data is
   refreshed.

\*-----------------------------------------------------------------*/
  BOOL vWinPrinter::CreateInfoDC(void)
  {
    PPRDINFO3    pPrd3;
    ULONG        SizePrd3;
    DEVOPENSTRUC DevOpenData;
    LONG         i;
    CHAR         DriverName[MAX_DRIVERNAME];
    CHAR         DeviceName[MAX_DEVICENAME];
    SIZEL        sizel;
    POINTL       PtlRes;

    // Free existing info DCs and PSs
    if (_hInfoPS)
    {
      GpiAssociate(_hInfoPS, NULLHANDLE);
      GpiDestroyPS(_hInfoPS);
      _hInfoPS = 0;
      DevCloseDC(_hInfoDevC);
      _hInfoDevC = 0;
    }
    if (_pHCInfo)
    {
      DosSubFree(_pMem, _pHCInfo, (_SizeHC * sizeof(HCINFO)) );
      _pHCInfo = 0;
      _SizeHC = 0;
    }

    // splits "driver.device" into separate driver and device strings
    i = strcspn(_pPRQCurrent->pszDriverName,".");
    strncpy( DriverName, _pPRQCurrent->pszDriverName, i);
    DriverName[i] = '\0';
    strcpy(DeviceName, &_pPRQCurrent->pszDriverName[i + 1]);

    // allocate space for printer device info data
    SPLERR rc = SplQueryDevice(0, _pPRQCurrent->pszPrinters, 3, 0, 0, &SizePrd3);
//printf("CreateInfoDC: SplQueryDevice rc = %u \n", rc);
    // we expect to get NERR_BufTooSmall if all is well (since return buffer is NULL)
    if (rc != NERR_BufTooSmall)
    {
      return(FALSE);  // shouldn't happen!
    }

    if (DosSubAllocMem(_pMem, (PPVOID)&pPrd3, SizePrd3))
      return (FALSE);

    // load printer device data and disassemble
    if (SplQueryDevice(0, _pPRQCurrent->pszPrinters, 3,
      (PVOID)pPrd3, SizePrd3, &SizePrd3) == 0)
    {
//printf("CreateInfoDC: pszLogAddress = %s \n", pPrd3->pszLogAddr);
//printf("              DriverName = %s \n", DriverName);
      DevOpenData.pszLogAddress = pPrd3->pszLogAddr;
      DevOpenData.pszDriverName = DriverName;
      DevOpenData.pdriv         = _pDDCurrent;
      DevOpenData.pszDataType   = "PM_Q_STD";

      // open the info DC and PS to get page metrics
      _hInfoDevC = DevOpenDC (theApp->AppHab(), OD_INFO, "*", 4,
        (PDEVOPENDATA) &DevOpenData, 0);
//printf("CreateInfoDC: _hInfoDevC = %x \n", _hInfoDevC);

      if (_hInfoDevC == 0)
        return(FALSE);      // couldn't create OD_INFO
      // otherwise, plow ahead
      else
      {
        sizel.cx = 0;
        sizel.cy = 0;
        // use PU_LOENGLISH to get approx 72dpi res
        _hInfoPS = GpiCreatePS (theApp->AppHab(), _hInfoDevC, &sizel,
          GPIA_ASSOC | PU_LOENGLISH);
        if(_hInfoPS == 0)
        {
          // couldn't create PS
          DevCloseDC(_hInfoDevC);
          _hInfoDevC = 0;
          return(FALSE);
        }
        else
        {
          // created PS, now get page metrics
          GpiQueryPS(_hInfoPS, (PSIZEL)&_PageSizel);  // get page size in 0.1 inch units
//printf("CreateInfoDC: _hInfoPS = %x \n", _hInfoPS);
//printf("              _PageSizel = (%u, %u) \n", _PageSizel.cx, _PageSizel.cy);
          // we want the width/height in PU_LOENGLISH units (100 dpi)
          _width = _PageSizel.cx;
          _height = _PageSizel.cy;

          // now get forms data, but first find out how big structure is
          _SizeHC = DevQueryHardcopyCaps(_hInfoDevC, 0, 0, 0);
          if ( (_SizeHC > 0) &&
            (DosSubAlloc(_pMem, (PPVOID)&_pHCInfo, _SizeHC * sizeof(HCINFO)) == 0) )
          {
            DevQueryHardcopyCaps(_hInfoDevC, 0, _SizeHC, _pHCInfo);
            for (i = 0; i < _SizeHC; i++)
            {
              if (_pHCInfo[i].flAttributes & HCAPS_CURRENT)
              {
                // Calculate the physical page size
                DevQueryCaps (_hInfoDevC, CAPS_HORIZONTAL_RESOLUTION, 1L,
                     (PLONG) &PtlRes.x);
                DevQueryCaps (_hInfoDevC, CAPS_VERTICAL_RESOLUTION, 1L,
                    (PLONG) &PtlRes.y);
                // Convert from millimeters to pixels  (dpi units)
                _PhyPagePels.cx = (PtlRes.x * _pHCInfo[i].cx) / 1000;
                _PhyPagePels.cy = (PtlRes.y * _pHCInfo[i].cy) / 1000;
//printf("              _PhyPagePels = (%u, %u) \n", _PhyPagePels.cx, _PhyPagePels.cy);

                // now do the same for margin offsets (dpi units)
                _PrintOffsetSizel.cx = (PtlRes.x * _pHCInfo[i].xLeftClip) / 1000;
                _PrintOffsetSizel.cy = (PtlRes.y * _pHCInfo[i].yBottomClip) / 1000;
//printf("CreateInfoDC: width=%u, _height=%u \n", _width, _height);
//printf("              _PrintOffset = (%u, %u) \n", _PrintOffsetSizel.cx, _PrintOffsetSizel.cy);
                break;
              }
            } // for
          }
          else
          {
            // DevQueryHardcopyCaps failed
            _pHCInfo = 0;
            _SizeHC = 0;
          }
        }
      }
    }
    // cleanup
    DosSubFree(_pMem, pPrd3, SizePrd3);
    return((BOOL) _hInfoPS);
  }

/*-----------------------------------------------------------------*\
   This function will create an OD_QUEUED device context based on the
   contents of the pDDCurrent parameter. If successful a presentation space
   The presentation space is then reset from indexed to RGB mode.
   If a PS already exists, it will be destroyed first.
\*-----------------------------------------------------------------*/
  HPS vWinPrinter::CreateHDC(VOID)
  {
    DEVOPENSTRUC  DOP;
    CHAR          QueueProcParams[32];
    CHAR          SpoolerParams[32];
    CHAR          DriverName[MAX_DRIVERNAME];
    int           i;
    SIZEL         size;

    // destroy old PS and DevC if they exist
    if (_printhDC)
    {
      GpiAssociate(_printhDC, NULLHANDLE);
      GpiDestroyPS(_printhDC);
      _printhDC = 0;
      DevCloseDC(_hPrintDevC);
      _hPrintDevC = 0;
    }

    // clear the DOP struct
    memset((PVOID)&DOP, 0, sizeof(DOP));

    // Set priority in spooler params
    sprintf((PSZ)SpoolerParams,(PSZ)"PRTY=%d", _Priority);

    // Set number of copies in the queue processor params
    // and turn page fit transform off
    sprintf((PSZ)QueueProcParams,(PSZ)"COP=%d XFM=0", _copies);

    // splits "driver.device" into separate driver and device strings
    i = strcspn(_pPRQCurrent->pszDriverName,".");
    strncpy((PSZ)DriverName, (PSZ)_pPRQCurrent->pszDriverName, i);
    DriverName[i] = '\0';

    DOP.pszLogAddress      = _QueueName;
//printf("CreateHDC: _QueueName = %s \n", _QueueName);
    DOP.pszDriverName      = DriverName;
    DOP.pdriv              = _pDDCurrent;
    DOP.pszDataType        = "PM_Q_STD";
    DOP.pszComment         = (PSZ) _name;
    DOP.pszQueueProcName = "";
    DOP.pszQueueProcParams = (PSZ)QueueProcParams;
    DOP.pszSpoolerParams   = (PSZ)SpoolerParams;
    DOP.pszNetworkParams = "";


    _hPrintDevC = DevOpenDC(theApp->AppHab(), OD_QUEUED, "*", 9,
      (PDEVOPENDATA) &DOP, 0);

    if (_hPrintDevC == 0)
    {
      ERRORID err;
      err = WinGetLastError(theApp->AppHab());
      SysDebug1(Build,"vWinPrinter::CreateHDC() failed to create _hPrintDevC (err=%x) \n",
        err)
      return 0;
    }

    // DevC created okay, now create the PS
    size.cx =0;
    size.cy =0;
//    _printhDC = GpiCreatePS(theApp->AppHab(), _hPrintDevC, (PSIZEL) &_PageSizel,
//      GPIA_ASSOC | PU_LOENGLISH);
    _printhDC = GpiCreatePS(theApp->AppHab(), _hPrintDevC, (PSIZEL) &size,
      GPIA_ASSOC | PU_PELS);
    if (_printhDC == 0)
    {
      ERRORID err;
      err = WinGetLastError(theApp->AppHab());
      SysDebug1(Build,"vWinPrinter::CreateHDC() failed to create _hPrintDC (err=%x) \n",
        err)
      DevCloseDC(_hPrintDevC);
      _hPrintDevC = 0;
      return 0;
    }

    // PS created okay, now set the canvas to RGB mode
    GpiCreateLogColorTable (_printhDC, 0L, LCOLF_RGB, 0, 0, NULL);
    // set model space transform to put origin at upper left corner
    // since OS/2 puts the origin in the bottom left by default
    MATRIXLF TransformMatrix;
    POINTL translate, origin;
    FIXED scale[2];
    translate.x = 0;
    translate.y = -_height+1;
    GpiTranslate(_printhDC, &TransformMatrix, TRANSFORM_REPLACE, &translate);
    origin.x = 0; origin.y = 0;

    // invert in the y direction
    scale[0] = MAKEFIXED( 1,0);   // x scale
    scale[1] = MAKEFIXED(-1,0);   // y scale
    GpiScale(_printhDC, &TransformMatrix, TRANSFORM_ADD, scale, &origin);

    // we want to scale the PS so that we get 1 pel = 0.1 inch (100 dpi)
    // I tried fractional scaling, but some print drivers get all weird and
    // don't print single pel lines and so on... sigh!

//    double magx = 1.0*_PhyPagePels.cx /_PageSizel.cx;
//    double magy = 1.0*_PhyPagePels.cy / _PageSizel.cy;
//printf("CreateHDC: magx = %lf magy = %lf \n", magx, magy);
//    scale[0] = DBLTOFX( magx );   // x scale
//    scale[1] = DBLTOFX( magy );   // y scale
    scale[0] = MAKEFIXED(_PhyPagePels.cx /_PageSizel.cx ,0);   // x scale
    scale[1] = MAKEFIXED(_PhyPagePels.cx /_PageSizel.cx, 0);   // y scale
//printf("CreateHDC: scale[0] = %x scale[1] = %x \n", scale[0], scale[1]);
    GpiScale(_printhDC, &TransformMatrix, TRANSFORM_ADD, scale, &origin);

    GpiSetModelTransformMatrix(_printhDC, 9, &TransformMatrix, TRANSFORM_REPLACE);

    return( _printhDC);
}

/*-----------------------------------------------------------------*\
   This function will destroyed the print PS and DevC if they exist.
\*-----------------------------------------------------------------*/
  void vWinPrinter::DestroyHDC(VOID)
  {
    // destroy old OD_QUEUED PS and DevC if they exist
    if (_printhDC)
    {
      GpiAssociate(_printhDC, NULLHANDLE);
      GpiDestroyPS(_printhDC);
      _printhDC = 0;
      DevCloseDC(_hPrintDevC);
      _hPrintDevC = 0;
    }
  }


//====================>>> vDialog::DynamicDlgProc <<<=======================
  int vWinPrinter::DynamicDlgProc(HWND hwnd, UINT msg,
			    MPARAM mp1, MPARAM mp2)
  {
  RECTL   Rectl;
  LONG    i;
  LONG    Selection;
  char    Temp[MAX_BUFF], *pStr;

  switch(msg)
  {
    case WM_INITDLG:
    {
      // update the dialog status area for the selected printer
      if (_pPRQCurrent->fsStatus == PRQ3_PAUSED)
        sprintf(Temp, "Not Ready,  Queue Paused");
      else if (_pPRQCurrent->fsStatus == PRQ3_PENDING)
        sprintf(Temp, "Not Ready,  Queue Pending Deletion");
      else
        sprintf(Temp, "Ready,  %u Job(s) in Queue", _pPRQCurrent->cJobs);
      WinSetWindowText(WinWindowFromID(hwnd, vID_STATUS), Temp);
      // Queue Type
      i = strcspn(_pPRQCurrent->pszDriverName,".");
      strcpy((PSZ)Temp, (PSZ)&_pPRQCurrent->pszDriverName[i + 1]);
      WinSetWindowText(WinWindowFromID(hwnd, vID_TYPE), Temp);
      // Where
//      WinSetWindowText(WinWindowFromID(hwnd, vID_WHERE), _pPRQCurrent->pszName);
      WinSetWindowText(WinWindowFromID(hwnd, vID_WHERE), _QueueName);
      // initialize remaining controls
      switch(_PrintRange)
      {
        case 0:
          WinCheckButton(hwnd, vID_PRINTALL, 1);
          break;
        case 1:
          WinCheckButton(hwnd, vID_PRINTPAGES, 1);
          break;
        case 2:
          WinCheckButton(hwnd, vID_PRINTSEL, 1);
          break;
      }
      WinCheckButton(hwnd, vID_COLLATE, _Collate);
      WinSendMsg(WinWindowFromID(hwnd, vID_COPIES), SPBM_SETLIMITS,
        (MPARAM) 999, (MPARAM) 1);
      WinSendMsg(WinWindowFromID(hwnd, vID_COPIES),
        SPBM_SETCURRENTVALUE, MPFROMLONG(_copies), (MPARAM) NULL);

      // load combobox with printer names, punt if trouble
      EnumPrintQueues(hwnd);
      // Select the current (default) entry in the list box
      WinSendMsg (WinWindowFromID(hwnd, vID_PRINTER), LM_SELECTITEM, MPFROMSHORT(_ixCurrent), MPFROMSHORT(TRUE));
    }
    break;

  case WM_CONTROL:
  {
    switch (SHORT1FROMMP(mp1))
    {
      case vID_PRINTER:
      {
        switch (SHORT2FROMMP(mp1))
        {
          // selected printer from combobox
          case CBN_ENTER:
//          case CBN_LBSELECT:
          {
            // get new printer selection
            Selection = (ULONG)WinSendMsg(WinWindowFromID(hwnd, vID_PRINTER),
              LM_QUERYSELECTION, MPFROMSHORT(LIT_FIRST),0);
            if (Selection != LIT_NONE)
            {
              // get QueueDesc for selection
              if (*_pPRQArray[Selection].pszComment)
              {
                pStr = _pPRQArray[Selection].pszComment;
              }
              else
              {
                pStr = _pPRQArray[Selection].pszName;
              }

              // if it is not the current queue, then we must update info
              if (strcmp(_QueueDesc, pStr) != 0)
              {
                // update current queue name
                strcpy(_QueueDesc, pStr);
                // delete the old queue info and replace with new stuff
                if (_SizePRQCurrent != 0)
                {
                  DosSubFreeMem(_pMem, (PPVOID)_pPRQCurrent, _SizePRQCurrent);
                }
                // how much space do we need?
                SplQueryQueue(0, _pPRQArray[Selection].pszName, 3, 0, 0, &_SizePRQCurrent);
                // make enough room for the printer info
                if (_SizePRQCurrent && !(DosSubAlloc(_pMem, (PPVOID) &_pPRQCurrent, _SizePRQCurrent)))
                {
                  SplQueryQueue(0, _pPRQArray[Selection].pszName, 3, _pPRQCurrent,
                    _SizePRQCurrent, &_SizePRQCurrent);
                }
                // update queue name
                strcpy(_QueueName, _pPRQCurrent->pszName);
                // JobProps will reallocate mem for DD if its null
                // De-allocate memory for the driver data
                DosSubFreeMem(_pMem, _pDDCurrent, _SizeDDCurrent);
                _pDDCurrent = 0;
                _SizeDDCurrent = 0;
                GetJobProps(_pPRQCurrent, DPDM_QUERYJOBPROP);
              }
            }
            // update the dialog status area for the selected printer
            if (_pPRQCurrent->fsStatus == PRQ3_PAUSED)
              sprintf(Temp, "Not Ready,  Queue Paused");
            else if (_pPRQCurrent->fsStatus == PRQ3_PENDING)
              sprintf(Temp, "Not Ready,  Queue Pending Deletion");
            else
              sprintf(Temp, "Ready,  %u Jobs in Queue", _pPRQCurrent->cJobs);
            WinSetWindowText(WinWindowFromID(hwnd, vID_STATUS), Temp);
            // Queue Type
            i = strcspn(_pPRQCurrent->pszDriverName,".");
            strcpy((PSZ)Temp, (PSZ)&_pPRQCurrent->pszDriverName[i + 1]);
            WinSetWindowText(WinWindowFromID(hwnd, vID_TYPE), Temp);
            // Where
//            WinSetWindowText(WinWindowFromID(hwnd, vID_WHERE), _pPRQCurrent->pszName);
            WinSetWindowText(WinWindowFromID(hwnd, vID_WHERE), _QueueName);
            return 0;
          }
        }
      }
    }
  }
  case WM_COMMAND:
  {
    switch (SHORT1FROMMP(mp1))
    {
      // pushed the properties button
      case vID_JOBPROPS:
      {
        GetJobProps(_pPRQCurrent, DPDM_POSTJOBPROP);
        return 0;
      }

      case DID_OK:
      {
        // delete _pPRQArray structure
        if (_SizePRQArray != 0)
        {
          DosSubFreeMem(_pMem, (PPVOID)_pPRQArray, _SizePRQArray);
          _SizePRQArray = 0;
        }
        // update remaining control vars
        WinSendDlgItemMsg( hwnd, vID_COPIES, SPBM_QUERYVALUE,
          &_copies, MPFROM2SHORT(0, SPBQ_UPDATEIFVALID));
        _Collate = WinQueryButtonCheckstate(hwnd, vID_COLLATE);
        if ( WinQueryButtonCheckstate(hwnd, vID_PRINTALL) )
          _PrintRange = 0;
        else if ( WinQueryButtonCheckstate(hwnd, vID_PRINTPAGES) )
          _PrintRange = 1;
        else if ( WinQueryButtonCheckstate(hwnd, vID_PRINTSEL) )
          _PrintRange = 2;

        // create INFO DC to update physical capabiliies of current printer
        CreateInfoDC();
        // now return 1 to the calling program
        break;
      }

      case DID_CANCEL:
      {
        // delete _pPRQArray structure
        if (_SizePRQArray != 0)
        {
          DosSubFreeMem(_pMem, (PPVOID)_pPRQArray, _SizePRQArray);
          _SizePRQArray = 0;
        }
        // create INFO DC to update physical capabiliies of current printer
        CreateInfoDC();
        // abort printing if pressed cancel
        // now return 2 to the calling program
        break;
      }
    }
  }
  default:
    break;
  }
  return (int) WinDefDlgProc(hwnd, msg, mp1, mp2);
}   /* End of DynamicDlgProc   */


//====================>>> DlgProc <<<=======================
MRESULT EXPENTRY PrintDlgProc(HWND hDlg, ULONG uMsg,MPARAM mp1, MPARAM mp2)
{
  vWinPrinter* useThis;
  if (uMsg == WM_INITDLG) // first time!
    {
      useThis = (vWinPrinter*)mp2;

      _thisList.Add((ThisId)hDlg, (void*)useThis);
    }
  else
    useThis = (vWinPrinter*)_thisList.GetThis((ThisId)hDlg);
  if (!useThis)
    return WinDefDlgProc(hDlg, uMsg, mp1, mp2);

  return (MRESULT) useThis->DynamicDlgProc(hDlg, uMsg, mp1, mp2);
}

