/**********************************************************************
*   REXXDATE.C                                                        *
*                                                                     *
*   This program extends the REXX language by providing               *
*   REXX external functions for working with calendar dates           *
*                                                                     *
*   These functions are:                                              *
*       RxDate              -- Convert a given date to a number or    *
*                              return today's number if no arg given  *
*                                                                     *
*   To compile:    MAKE REXXDATE                                      *
*                                                                     *
**********************************************************************/
/* Include files */

#define  INCL_ERRORS
#define  INCL_REXXSAA
#define  _DLL
#define  _MT
#include <os2.h>
#include <rexxsaa.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include "julcal.h"

const int kRexxBasedateOffset = 1721426;

/*********************************************************************/
/*  Declare all exported functions as REXX functions.                */
/*********************************************************************/

RexxFunctionHandler RxDate;

/*********************************************************************/
/*  Various definitions used by various functions.                   */
/*********************************************************************/

#define  MAX            256        /* temporary buffer length        */

/*********************************************************************/
/* Structures used throughout REXXUTIL.C                              */
/*********************************************************************/


/*********************************************************************/
/* Numeric Error Return Strings                                      */
/*********************************************************************/

#define  NO_UTIL_ERROR    "0"          /* No error whatsoever        */

/*********************************************************************/
/* Alpha Numeric Return Strings                                      */
/*********************************************************************/

#define  ERROR_RETSTR   "ERROR:"

/*********************************************************************/
/* Numeric Return calls                                              */
/*********************************************************************/

#define  INVALID_ROUTINE 40            /* Raise Rexx error           */
#define  VALID_ROUTINE    0            /* Successful completion      */

/*********************************************************************/
/* Some useful macros                                                */
/*********************************************************************/

#define BUILDRXSTRING(t, s) { \
  strcpy((t)->strptr,(s));\
  (t)->strlength = strlen((s)); \
}


/*************************************************************************
    Parse the input string, returning the number of seconds that represent
    the date, or -1 on error.
*************************************************************************/
long parseDate(char *s)
    {
    struct tm *pTm;
    char *q, *pm = 0, *pd = 0, *py = 0;
    int i;
    long result;
    int century;

    if (*s == '-')  /* if the string starts with '-' then it's probably an integer */
        return atoi(s) + kRexxBasedateOffset;

    for (pm = s; !isdigit(*pm) && (*pm); pm++); /* Skip over non-digits */
    for (q = pm; isdigit(*q); q++);             /* Skip over digits */
    if (*q)                                     /* Are there more chars left? */
        {
        *q++ = 0;
        for (pd = q; !isdigit(*pd) && (*pd); pd++); /* Skip non-digits */
        for (q = pd; isdigit(*q); q++);             /* Skip digits */
        if (*q)                                     /* more? */
            {
            *q++ = 0;
            for (py = q; !isdigit(*py) && (*py); py++);  /* Skip non-digits */
            for (q = py; isdigit(*q); q++);              /* Skip digits */
            *q = 0;
            }
        }

    if (!pd) /* there was only one number in the first param */
        return atoi(pm) + kRexxBasedateOffset;

    result = time(NULL);
    pTm = localtime(&result);
    century = ((pTm->tm_year + 1900) / 100) * 100;

    if (pm && (*pm))
        pTm->tm_mon  = atoi(pm);
    if (pd && (*pd))
        pTm->tm_mday = atoi(pd);
    if (py && (*py))
        pTm->tm_year = atoi(py);

    /* we must have read yyyy-mm-dd so swap it around */
    if (pTm->tm_mon > 12)
        {
        i = pTm->tm_mon;
        pTm->tm_mon  = pTm->tm_mday;
        pTm->tm_mday = pTm->tm_year;
        pTm->tm_year = i;
        }

    pTm->tm_mon -= 1;

    /* 2-digit years are assumed to be referring to the current century */
    if (pTm->tm_year < 100)
        pTm->tm_year += century;

    pTm->tm_year -= 1900;
    return juldn(pTm);
    }


/*************************************************************************
* Function:  RxDate                                                      *
*                                                                        *
* Syntax:    rc = RxDate(<date>, <format-specifier>)                     *
*                                                                        *
* Params:    date-string (optional): a date in the form of mm/dd/yy or   *
*                    mm/dd or mm-dd-yy or mm-dd or so on.                *
*                    will also accept yyyy-mm-dd or yy-mm-dd if yy > 12  *
*                                                                        *
* Return:    The number of days since 1/1/1970                           *
*************************************************************************/

ULONG RxDate(CHAR *name, ULONG numargs, RXSTRING args[],
                            CHAR *queuename, RXSTRING *retstr)
    {
    UCHAR buf[MAX];
    CHAR *holdarg; /*                                                jrp*/
    long jDate;
    struct tm *pTm;
    time_t t;

    if ((numargs > 0) && RXVALIDSTRING(args[0]))
        { /*                                                         jrp*/
        holdarg = strdup( args[0].strptr ); /*                       jrp*/
        /* should test that we have memory                           jrp*/
        jDate = parseDate(holdarg); /*                               jrp*/
        free(holdarg); /*                                            jrp*/
        } /*                                                         jrp*/
    else
        {
        t = time(NULL);
        pTm = localtime(&t);
        jDate = juldn(pTm);
        }

    if (numargs < 2) /* just return date-number */
        _itoa(jDate - kRexxBasedateOffset, buf, 10);
    else    /* there is a format-specifier string, so return a formatted date instead */
        {
        pTm = julcd(jDate);
        strftime(buf, sizeof(buf), args[1].strptr, pTm);
        }

    BUILDRXSTRING(retstr, buf);
    return VALID_ROUTINE;
    }
