
/*
 *  Copyright 1995 Microsoft Corporation. All rights reserved.
 *  Developed by Ataman Software, Inc., ftp://rmii.com/pub2/ataman,
 *       info@ataman.com
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Modified by Patrick McPhee to be a loadable library, 1997/08/10
 * Changes copyright 1997,2003 Patrick McPhee
 *
 * Array code based on a contribution by Alan Insley
 * $Header: C:/ptjm/rexx/w32funcs/RCS/w32ole.cpp 1.23 2003/12/19 15:11:11 ptjm Rel $
 */


#define STRICT
#pragma warning(disable: 4201)
#include <windows.h>
#pragma warning(default: 4201)
#include <ole2ver.h>
#include <shlobj.h>
#include <malloc.h>
#include <stdio.h>
#include <ctype.h>

/* functions defined in this file:
 * w32createobject(ProgramId) -> handle or 0
 * w32getobject([FileName], [ProgramId]) -> handle or 0
 * w32releaseobject(handle) -> 0 (success) or 1 (failure)
 * w32callproc(handle, name, typelist, [parm1, ...]) -> 0 or 1
 * w32callfunc(handle, name, typelist, [parm1, ...]) -> value (sets rc to 0 or 1)
 * w32olenext(object [, 'Reset' | number_to_skip]) -> handle or 0
 * w32getsubobj(handle, name, typelist, [parm1, ...]) -> handle or 0
 * w32getproperty(handle, name, typelist, [parm1, ...]) -> value (sets rc)
 * w32putproperty(object, name, typelist, value) -> 0 or 1
 * w32olegetarray(handle, name, stem) -> 0 or 1
 * w32oleputarray(object, name, stem) -> 0 or 1
 * w32olegeterror() -> error text
 * w32olecleanup() -> nil
 */


extern "C" {
#include "w32funcs.h"
}

/* you've got to love this -- there are two distinct definitions of MAKELCID */
static const LCID lcid = LOCALE_USER_DEFAULT,
                  cvtlcid = MAKELCID(MAKELANGID(LANG_ENGLISH, SUBLANG_NEUTRAL), SORT_DEFAULT);

#define MAXSTRING (16*1024)

static void del_objstack(void);

extern "C" void CleanupOLE(void) {
        del_objstack();
	OleUninitialize();
}

static void OLEErr(const char *format, ...);
static void OLEErrClear(void);
static void OLEStdErr(const char *name, HRESULT hr, const char *tag);
static void OLEExcErr(const char *name, EXCEPINFO * excp);
static const char * th(int argno);

static BOOL DoChange(const PRXSTRING name, VARIANT *, unsigned short type);

extern "C" BOOL InitOLE(void)
{
	DWORD dwVersion = OleBuildVersion();

	if (HIWORD(dwVersion) != rmm || LOWORD(dwVersion) < rup)
		return FALSE;

	if (FAILED(OleInitialize(NULL)))
		return FALSE;

	return TRUE;
}

enum CallType_t {ct_func, ct_proc, ct_subobj };

static void push_object(LPDISPATCH obj);
static void pop_object(LPDISPATCH obj);

static APIRET DoInvoke(PRXSTRING iptr, PRXSTRING name, CallType_t calltype,
	unsigned short itype, PRXSTRING typelist, LONG argc, PRXSTRING argv, PRXSTRING result);

rxfunc(w32createobject)
{
   /* val = w32CreateObject(ProgramID) */
   HRESULT hr;
   IUnknown *punk = NULL;
   IDispatch *pdsp = NULL;
   CLSID clsid;
   OLECHAR ocBuf[MAXSTRING];

   checkparam(1, 1) ;

   OLEErrClear();

   if (MultiByteToWideChar(CP_ACP, 0, argv[0].strptr, argv[0].strlength, ocBuf, sizeof ocBuf) == 0) {
      return BADGENERAL;
   }
   ocBuf[argv[0].strlength] = 0;

   hr = CLSIDFromProgID(ocBuf, &clsid);

   if (FAILED(hr)) {
      hr = CLSIDFromString(ocBuf, &clsid);
   }

   if (FAILED(hr)) {
      OLEStdErr("CreateObject", hr, "CLSIDFromProgID and CLSIDFromString failed");
   }
   else {
      /* see if there's one running already */
      hr = GetActiveObject(clsid, NULL, &punk);

      /* No? then let's try creating one */
      if (FAILED(hr))
         hr = CoCreateInstance(clsid, NULL, CLSCTX_SERVER, IID_IUnknown, (LPVOID *)&punk);

      if (FAILED(hr)) {
         OLEStdErr("CreateObject", hr, "CoCreateInstance failed");
      }

      else {
         hr = punk->QueryInterface(IID_IDispatch, (LPVOID *)&pdsp);
         if (FAILED(hr)) {
            OLEStdErr("CreateObject", hr, "QueryInterface failed");
         }
      }

   }

   if (punk) {
      punk->Release();
   }

   if (!pdsp) {
      result_zero();
   }
   else {
      result->strlength = sizeof(pdsp);
      memcpy(result->strptr, &pdsp, sizeof(pdsp));
      push_object(pdsp);
   }

   return 0;
}


rxfunc(w32getobject)
{
   /* val = w32GetObject([FileName], [ProgramID]) */
   HRESULT hr;
   IUnknown *punk = NULL;
   IDispatch *pdsp = NULL;
   IMoniker *pmon = NULL;
   IPersistFile *ppf = NULL;
   LPBC pbc = NULL;
   CLSID clsid;
   ULONG cEaten;
   OLECHAR ocBuf[MAXSTRING];
   char * file;
   enum gettype_t { gt_fileonly, gt_apponly, gt_both } gettype;

   checkparam(1,2);

   if (argc == 1) {
      rxstrdup(file, argv[0]);
      gettype = gt_fileonly;
   }
   else if (!argv[0].strptr) {
      gettype = gt_apponly;
   }
   else {
      rxstrdup(file, argv[0]);
      gettype = gt_both;
   }

   OLEErrClear();

   /* how you get the application depends on what you have to
    * work with. I've combined the apponly and both cases, because
    * there's some similarity which points up a problem in its
    * dissimilarity. Currently, if you don't have a file, you
    * can't create a new object. There should be a New call
    * if there is no active object.  */
   if (gettype == gt_apponly || gettype == gt_both) {
      if (MultiByteToWideChar(CP_ACP, 0, argv[1].strptr,
          argv[1].strlength, ocBuf, sizeof ocBuf) == 0) {
         return BADGENERAL;
      }
      ocBuf[argv[1].strlength] = 0;

      hr = CLSIDFromProgID(ocBuf, &clsid);
      if (FAILED(hr)) {
         OLEStdErr("GetObject", hr, "CLSIDFromProgID failed");
      }

      else {
         if (gettype == gt_apponly) {
            hr = GetActiveObject(clsid, NULL, &punk);
            if (FAILED(hr)) {
               OLEStdErr("GetObject", hr, "GetActiveObject failed");
            }
         }
         else {
            hr = CoCreateInstance(clsid, NULL, CLSCTX_SERVER, IID_IUnknown, (LPVOID *)&punk);
            if (FAILED(hr)) {
               OLEStdErr("GetObject", hr, "CoCreateInstance failed");
            }
            else {
               hr = punk->QueryInterface(IID_IPersistFile, (LPVOID *)&ppf);
               if (FAILED(hr)) {
                  OLEStdErr("GetObject", hr, "QueryInterface (IPersistFile) failed");
               }
               else {

                  /* make punk point to the persist file so we can get the
                   * IDispatch consistently */
                  punk->Release();
                  punk = (IUnknown *)ppf;

                  if (MultiByteToWideChar(CP_ACP, 0, argv[0].strptr,
                      argv[0].strlength, ocBuf, sizeof ocBuf) == 0) {
                     OLEErr("Error converting filename: '%s' to UNICODE", file);
                  }
                  else {
                     ocBuf[argv[0].strlength] = 0;
                     hr = ppf->Load(ocBuf, 0);
                     if (FAILED(hr)) {
                        OLEStdErr("GetObject", hr, file);
                     }
                  }
               }
            }

         }
         if (!FAILED(hr)) {
            hr = punk->QueryInterface(IID_IDispatch, (LPVOID *)&pdsp);
            if (FAILED(hr)) {
               OLEStdErr("GetObject", hr, "QueryInterface failed");
            }
         }
      }
   }
   else if (gettype == gt_fileonly) {
      if (MultiByteToWideChar(CP_ACP, 0, argv[0].strptr, argv[0].strlength, ocBuf, sizeof ocBuf) == 0) {
         return BADGENERAL;
      }
      ocBuf[argv[0].strlength] = 0;

      hr = CreateBindCtx(0, &pbc);
      if (FAILED(hr)) {
         OLEStdErr("GetObject", hr, "CreateBindCtx failed");
      }

      else {
         hr = MkParseDisplayName(pbc, ocBuf, &cEaten, &pmon);
         if (FAILED(hr)) {
            OLEStdErr("GetObject", hr, "MkParseDisplayName failed");
         }

         hr = BindMoniker(pmon, 0, IID_IDispatch, (LPVOID *)&pdsp);
         if (FAILED(hr)) {
            OLEStdErr("GetObject", hr, "BindMoniker failed");
         }
      }
   }

   /* don't release ppf, since punk is pointing to it if it was
    * ever got in the first place */
   if (punk) {
           punk->Release();
   }

   if (pmon) {
           pmon->Release();
   }

   if (pbc) {
           pbc->Release();
   }

   if (pdsp) {
      push_object(pdsp);
      result->strlength = sizeof(pdsp);
      memcpy(result->strptr, &pdsp, sizeof(pdsp));
   }
   else {
      result_zero();
   }

   return 0;
}


rxfunc(w32releaseobject)
{
   register int i;

   /* call w32ReleaseObject object */
   IDispatch *pdsp = NULL;

   checkparam(1, -1) ;

   for (i = 0; i < argc; i++) {
      if (argv[i].strlength != sizeof(pdsp))
         continue;
      memcpy(&pdsp, argv[i].strptr, sizeof(pdsp));
      pop_object(pdsp);
   }

   result_zero();

   return 0;
}

rxfunc(w32callproc)
{
	/* call w32CallProc object, name, typelist, ... */

	checkparam(2, -1) ;

	return DoInvoke(argv, argv+1, ct_proc,
		DISPATCH_METHOD, argc>2 && argv[2].strlength ? argv+2:NULL, argc-3, argv+3, result);
}

rxfunc(w32callfunc)
{
	/* val = w32CallFunc(object, name, typelist, ...) */

	checkparam(2, -1) ;

	return DoInvoke(argv, argv+1, ct_func,
		DISPATCH_METHOD, argc>2 && argv[2].strlength ? argv+2:NULL, argc-3, argv+3, result);
}

/* retrieve the next instance of a collection. This could be useful for
 * someone wanting to step through all instances without worrying about count
 * and item, and it's essential for collections which don't provide count and
 * item. */
rxfunc(w32olenext)
{
   /* val = w32OleNext(object [, 'Reset' | number_to_skip]) */
   int skipval = 0;
   LPUNKNOWN punk = NULL;
   static LPENUMVARIANT pev = NULL;
   static LPDISPATCH olddsp = NULL;
   LPDISPATCH pdsp = NULL;
   HRESULT hr;
   VARIANT vRet;
   unsigned long count;
   DWORD dwLen;
   EXCEPINFO ex;

   result_zero();

   checkparam(1, 2);

   OLEErrClear();


   if (argc == 2) {
      char * skips;

      if (argv[1].strlength && toupper(argv[1].strptr[0]) == 'R') {
         /* reset */
         skipval = -1;
      }
      else {
         rxstrdup(skips, argv[1]);
         skipval = atoi(skips);
      }
   }

   /* get IUnknown */
   if (argv[0].strlength == sizeof(pdsp))
      memcpy(&pdsp, argv[0].strptr, argv[0].strlength);
   
   if (pdsp && pdsp != olddsp) {
      DISPPARAMS dspp;

      if (pev) {
         pev->Release();
         pev = NULL;
      }

      VariantInit(&vRet);

      memset(&dspp, 0, sizeof(dspp));

      hr = pdsp->Invoke(DISPID_NEWENUM, IID_NULL, lcid, DISPATCH_METHOD|DISPATCH_PROPERTYGET, &dspp,
                        &vRet, &ex, NULL);

      if (!FAILED(hr) && vRet.vt == VT_UNKNOWN) {
         punk = vRet.pdispVal;
         olddsp = pdsp;
      }
      else if (GetScode(hr) == DISP_E_EXCEPTION) {
         if (ex.pfnDeferredFillIn) {
            (*ex.pfnDeferredFillIn)(&ex);
         }
         OLEExcErr("OLENext", &ex);
      }
      else {
         OLEStdErr("OLENext", hr, "fudge fudge fudge!");
      }
   }

   if (punk) {
      /* QueryInterface IEnumVariant */
      hr = punk->QueryInterface(IID_IEnumVARIANT, (LPVOID *)&pev);
      if (FAILED(hr)) {
         OLEStdErr("OLENext", hr, "QueryInterface failed");
      }

      punk->Release();
   }

   if (pev) {
      if (skipval == -1) {
         pev->Reset();
      }
      else if (skipval) {
         pev->Skip(skipval);
      }

      hr = pev->Next(1, &vRet, &count);

      if (FAILED(hr) || count == 0) {
         result_zero();
         pev->Release();

         olddsp = NULL;

         pev = NULL;
      }

      else {
         if (vRet.vt == VT_DISPATCH) {
            result->strlength = sizeof(vRet.pdispVal);
            memcpy(result->strptr, &vRet.pdispVal, sizeof(vRet.pdispVal));
            push_object(vRet.pdispVal);

            /* VariantClear will release this */
            vRet.pdispVal->AddRef();
         } else {
            if (vRet.vt != VT_BSTR) {
               hr = VariantChangeTypeEx(&vRet, &vRet, cvtlcid, 0, VT_BSTR);
               if (FAILED(hr)) {
                  OLEStdErr("OLENext", hr, "Failure converting return value to string");
                  result->strlength = 0;
                  return 0;
               }
            }

            dwLen = WideCharToMultiByte(CP_ACP, 0, vRet.bstrVal, -1, NULL, 0, NULL, NULL);
            rxresize(result, dwLen+1);
            WideCharToMultiByte(CP_ACP, 0, vRet.bstrVal, -1, result->strptr, result->strlength, NULL, NULL);
            if (dwLen)
               result->strlength = dwLen - 1;
            else
               result->strlength = 0;
         }

         VariantClear(&vRet);
      }
   }

   return 0;
}


rxfunc(w32getsubobj)
{
	/* val = w32GetSubObj(object, name, typelist, ...) */

	checkparam(2, -1) ;

	return DoInvoke(argv, argv+1, ct_subobj,
		DISPATCH_METHOD|DISPATCH_PROPERTYGET, argc>2 && argv[2].strlength ? argv+2:NULL, argc-3, argv+3, result);
}

rxfunc(w32getproperty)
{
	/* val = w32GetProperty(object, name[, typelist, ...]) */

	checkparam(2, -1) ;

	return DoInvoke(argv, argv+1, ct_func,
		DISPATCH_PROPERTYGET, argc>2 && argv[2].strlength ? argv+2:NULL, argc-3, argv+3, result);
}

rxfunc(w32putproperty)
{
	/* call w32PutProperty(object, name, typelist, value) */

	checkparam(4,4);

	return DoInvoke(argv, argv+1, ct_proc,
		DISPATCH_PROPERTYPUT, argv+2, 1, argv+3, result);
}

rxfunc(w32olegetarray)
{
	/* rc = w32OLEGetArray(object, name, stem) */
	/* same as rc = w32GetProperty(object, name, 'a', stem) */
   RXSTRING typelist;
   
   typelist.strlength = 1;
   typelist.strptr = "a";

	checkparam(3,3);

	return DoInvoke(argv, argv+1, ct_func,
		DISPATCH_PROPERTYGET, &typelist, 1, argv+2, result);
}

rxfunc(w32oleputarray)
{
	/* call w32OLEPutArray(object, name, stem) */
	/* same as call w32PutProperty(object, name, 'A', stem) */
   RXSTRING typelist;
   
   typelist.strlength = 1;
   typelist.strptr = "A";

	checkparam(3,3);

	return DoInvoke(argv, argv+1, ct_proc,
		DISPATCH_PROPERTYPUT, &typelist, 1, argv+2, result);
}

/* retrieve the id number of a function or method so we don't have to call
 * getids of names repeatedly */
static const char prefix[] = "@:";
rxfunc(w32olegetid)
{
	/* call w32OLEGetId(object, name, ...) */
   IDispatch * pdsp;
   OLECHAR ocBuf[MAXSTRING], *namePtr = ocBuf;
   HRESULT hr;
   DISPID dispid;
   register int i;

   checkparam(2,-1);

   OLEErrClear();
   result->strlength = 0;

   /* deal with the pointer to the IDispatch object */
   if (argv[0].strlength == sizeof(pdsp)) {
      memcpy(&pdsp, argv[0].strptr, sizeof(pdsp));
   }
   else {
      pdsp = NULL;
   }

   if (!pdsp) {
      return 0;
   }

   /* despite the name, GetIDsOfNames can handle only one name at a time
    * (unless the other names are arguments to the method in name 0)
    */

   result->strlength = 0;

   for (i = 1; i < argc; i++) {
      if (MultiByteToWideChar(CP_ACP, 0, argv[i].strptr, argv[i].strlength, ocBuf, sizeof(ocBuf)) == 0) {
	  return BADGENERAL;
	}
        ocBuf[argv[i].strlength] = 0;

        hr = pdsp->GetIDsOfNames(IID_NULL, &namePtr, 1, lcid, &dispid);
        if (FAILED(hr)) {
           OLEStdErr("w32OLEGetId", hr, "GetIDsOfNames failed");
           rc_one();
           return 0;
        }
      
        memcpy(result->strptr+result->strlength, prefix, sizeof(prefix)-1);
        memcpy(result->strptr+result->strlength+sizeof(prefix)-1, &dispid, sizeof(dispid));
        result->strptr[result->strlength+sizeof(prefix)-1+sizeof(dispid)] = ' ';

        /* don't subtract 1 from sizeof(prefix), since we want one for the space */
        result->strlength += sizeof(prefix)+sizeof(dispid);
   }

   /* get rid of the final space */
   result->strlength--;

   rc_zero();
   return 0;
}

/* set a rexx stem to match an OLE safe array
 * an array can have any number of dimensions
 * and each dimensions can have whatever bounds the developer likes.
 * it's always accessed as a big block, though.
 * the contributed code mapped this into stems which go from 1 to n.
 * the easiest thing would be to accept only 1-dimensional arrays.
 */
static int arrayToStem(PRXSTRING stemname, VARIANTARG *array)
{
   chararray * ca = new_chararray();
   char val[MAXSTRING], stembuf[1024];
   int d, dims, vlen;
   int *dimsizes, *ind;
   int i;
   int items;
   long lowerBound, upperBound;
   char *pvData;
   int  eltSize;
   VARIANTARG *vArg;
   VARIANTARG vValue;
   RXSTRING stem, ostem;

   if (!stemname || !stemname->strptr || !stemname->strlength) {
      OLEErr("Array result: no argument with type `a'.");
      return 0;
   }

   /* the array is a dims-dimensional array. In Rexx, we turn this into
    * stem.i1.i2...i[dims]. We need to set product(dim(i1), dim(i2),..., dim(i[dims-1]))
    * stems of dim(i[dims]) elements */
   dims = SafeArrayGetDim(array->parray);  /* often a 1 */
   if (dims < 1) {
      OLEErr("%.*s: dimension of array is %d.", (int)stemname->strlength,
             stemname->strptr, dims);
      return 0;
   }

   /* make a modifiable copy of the stem name in case this is a
    * multi-dimensional array */
   ostem.strptr = stembuf;
   memcpy(ostem.strptr, stemname->strptr, stemname->strlength);
   if (stemname->strptr[stemname->strlength - 1] == '.')
      ostem.strlength = stemname->strlength;
   else {
      ostem.strptr[stemname->strlength] = '.';
      ostem.strlength = stemname->strlength + 1;
   }

   dimsizes = (int *)alloca(dims * sizeof(dimsizes[0]));
   ind = (int *)alloca(dims * sizeof(ind[0]));
   items = 1;
   for (d = 0; d < dims; d++) {
      SafeArrayGetLBound(array->parray, d+1, &lowerBound);
      SafeArrayGetUBound(array->parray, d+1, &upperBound);
      dimsizes[d] = upperBound - lowerBound + 1;
      items *= dimsizes[d];
      ind[d] = 1;
   }

   eltSize = SafeArrayGetElemsize(array->parray);
   SafeArrayAccessData(array->parray, (void **)&pvData);

   /* build the name stem.i1.i2...i[dims-1]. We will create a chararray
    * with each dimsizes[dims-1] elements and set it using this name  */

   stem = ostem;
   for (d = 0; d < (dims-1); d++) {
      stem.strlength += sprintf(stem.strptr+stem.strlength, "%d.", ind[d]);
   }

   for (i = 0; i < items; i++, pvData += eltSize) {
      vArg = (VARIANTARG *)pvData;
      if ( vArg && (vArg->vt & VT_ARRAY) ) {
         setavar(&stem, "array", 5);            /* set stem.#.#.# to "array" */
         if (!arrayToStem(&stem, vArg))
            return 0;
      } else if ( (vArg->vt != VT_EMPTY) && (vArg->vt != VT_ERROR) ) {
         VariantInit(&vValue);
         if (vArg->vt == VT_BSTR)
            vlen = WideCharToMultiByte(CP_ACP, 0, vArg->bstrVal, -1, val, sizeof(val), NULL, NULL);
         else if (vArg->vt == VT_DISPATCH) {
            memcpy(val, &vArg->pdispVal, sizeof(vArg->pdispVal));
            vlen = sizeof(vArg->pdispVal);
         }
         else {
            HRESULT hr = VariantChangeTypeEx(&vValue, vArg, cvtlcid, 0, VT_BSTR);

            if (SUCCEEDED(hr)) {
               vlen = WideCharToMultiByte(CP_ACP, 0, vValue.bstrVal, -1, val, sizeof(val), NULL, NULL);
            }
            else {
               OLEStdErr("Array", hr, "type conversion failed");
               return 0;
            }
         }
         cha_addstr(ca, val, vlen);
         VariantClear(&vValue);
      }

      /* if we've reached the end of the innermost dimension, set the stem
       * and increment the ind arrays appropriately */
      if (!((i+1) % dimsizes[dims-1])) {
         setastem(&stem, ca);
         delete_chararray(ca);
         ca = new_chararray();

         /* rebuild stem */
         if (dims > 1) {
            ind[dims-2]++;

            for (d = dims - 2; d > 1; d--) {
               if (ind[d] > dimsizes[d]) {
                  ind[d-1]++;
                  ind[d] = 1;
               }
            }


            stem = ostem;
            for (d = 0; d < (dims-1); d++) {
               stem.strlength += sprintf(stem.strptr+stem.strlength, "%d.", ind[d]);
            }
         }
      }
   }
   SafeArrayUnaccessData(array->parray);
   delete_chararray(ca);

   /* set stem.0, stem.0.0, ... */
   stem = ostem;
   stem.strlength--;
   for (d = 0; d < dims; d++) {
      memcpy(stem.strptr+stem.strlength, ".0", 2);
      stem.strlength += 2;

      vlen = sprintf(val, "%d", dimsizes[d]);
      setavar(&stem, val, vlen);
   }

   return 1;
}

/* Given the name of a rexx stem, add all the elements to an array */ 
static int stemToArray(VARIANTARG *array, PRXSTRING stemname)
{
   chararray * ca = new_chararray();

   getastem(stemname, ca);

   delete_chararray(ca);

   return 1;
}



/* given a string of the form a1.a2.a3.a4...a{n} and an LPDISPATCH, retrieve
 * a1 and resolve it against the LPDISPATCH. If there are any more a{i}s,
 * get the LPDISPATCH associated with a1 and start over. At the end, put
 * the final dispid in pdispid and the final name in theName, which the
 * caller guarantees will be big enough. Each component can be of the
 * form @:bbbb, in which case bbbb is the dispid in binary format. */

static LPDISPATCH dnameToDispID(LPDISPATCH pdsp, PRXSTRING name, char * theName, DISPID * pdispid)
{
   OLECHAR ocBuf[MAXSTRING];
   OLECHAR *tb = ocBuf;
   char * dot;
   RXSTRING newname;
   HRESULT hr;

   if ((name->strlength == (sizeof(prefix)+sizeof(*pdispid)-1) ||
        (name->strlength > (sizeof(prefix)+sizeof(*pdispid)-1) &&
         name->strptr[sizeof(prefix)+sizeof(*pdispid)-1] == '.') ) &&
       !memcmp(name->strptr, prefix, sizeof(prefix)-1) ) {
      memcpy(pdispid, name->strptr+sizeof(prefix)-1, sizeof(*pdispid));
      newname.strlength = name->strlength - (sizeof(prefix)+sizeof(*pdispid)-1);
      newname.strptr = name->strptr + (sizeof(prefix)+sizeof(*pdispid)-1);
      *theName = 0;
   }
   else {
      if (dot = (char *)memchr(name->strptr,'.', name->strlength)) {
           if (MultiByteToWideChar(CP_ACP, 0, name->strptr, dot - name->strptr, ocBuf, sizeof ocBuf) == 0) {
              pdsp->Release();
              return NULL;
           }
           ocBuf[dot - name->strptr] = 0;

           newname.strptr = dot;
           newname.strlength = name->strlength - (dot - name->strptr);
           memcpy(theName, name->strptr, dot - name->strptr);
           theName[dot - name->strptr] = 0;
      }
      else {
         if (MultiByteToWideChar(CP_ACP, 0, name->strptr, name->strlength, ocBuf, sizeof ocBuf) == 0) {
            pdsp->Release();
            return NULL;
         }
         ocBuf[name->strlength] = 0;
         newname.strlength = 0;
         memcpy(theName, name->strptr, name->strlength);
         theName[name->strlength] = 0;
      }

      hr = pdsp->GetIDsOfNames(IID_NULL, &tb, 1, lcid, pdispid);

      if (FAILED(hr)) {
         OLEStdErr(theName, hr, "GetIDsOfNames failed");
         pdsp->Release();
         return NULL;
      }
   }

   /* At this point, pdispid is set to the dispid for the first element,
    * newname.strlength is set to the number of bytes left in name after the
    * first element, and newname.strptr is set to the start of those bytes,
    * if any. If there is anything there, we need to get the intermediate
    * IDispatch and try again */

   if (newname.strlength) {
      DISPPARAMS dspp;
      VARIANTARG vRet;
      EXCEPINFO ex;
      unsigned int bt;

      VariantInit(&vRet);
      memset(&dspp, 0, sizeof(dspp));

      hr = pdsp->Invoke(*pdispid, IID_NULL, lcid, DISPATCH_METHOD|DISPATCH_PROPERTYGET, &dspp,
                        &vRet, &ex, &bt);

      pdsp->Release();

      if (FAILED(hr)) {
         if (GetScode(hr) == DISP_E_UNKNOWNNAME) {
            OLEErr("%s: unknown name", theName);
         } else if (GetScode(hr) == DISP_E_MEMBERNOTFOUND) {
            OLEErr("%s: member not found", theName);
         } else if (GetScode(hr) == DISP_E_EXCEPTION) {
            if (ex.pfnDeferredFillIn) {
               (*ex.pfnDeferredFillIn)(&ex);
            }

            OLEExcErr(theName, &ex);
         } else {
            OLEStdErr(theName, hr, "Invoke failed");
         }
         pdsp = NULL;
      }

      else if (vRet.vt != VT_DISPATCH) {
         OLEErr("%s: does not return a subobject", theName);
         pdsp = NULL;
      }
      else if (vRet.pdispVal == NULL) {
         OLEErr("%s: returns NULL object", theName);
         pdsp = NULL;
      }
      else {
         /* skip the dot */
         newname.strptr++;
         newname.strlength--;

         pdsp = dnameToDispID(vRet.pdispVal, &newname, theName, pdispid);
      }
   }

   return pdsp;
}


static APIRET DoInvoke(PRXSTRING iptr, PRXSTRING name, CallType_t calltype,
	unsigned short itype, PRXSTRING typelist, LONG argc, PRXSTRING argv, PRXSTRING result)
{
	IDispatch *pdsp;
	HRESULT hr;
	DISPID dispid;
	DISPPARAMS dspp;
	EXCEPINFO ex;
	unsigned int bt;
	VARIANTARG vRet;
	VARIANTARG *vArgs;
	long l;
	char *tl = NULL;
        RXSTRING arrayRet = {0, 0};
	OLECHAR ocBuf[MAXSTRING];
	DISPID did;
	char * theName;

        /* in case we fail at some point... */
        result->strlength = 0;

        OLEErrClear();

	/* deal with the pointer to the IDispatch object */
	if (iptr->strlength == sizeof(pdsp)) {
           memcpy(&pdsp, iptr->strptr, sizeof(pdsp));
        }
        else {
           pdsp = NULL;
        }

        if (!pdsp) {
           OLEErr("DispID is '%s'||'00000000'x", prefix);
           goto errexit;
        }

        /* addref so I don't have to distinguish between the original pdsp
         * and a temporary one in the event of an x.y.z name. This means I
         * have to be very careful to release whenever I return, which
         * unfortunately is all over the place */
        pdsp->AddRef();

        if ((name->strlength == sizeof(prefix) - 1 + sizeof(dispid)) &&
            name->strptr[0] == prefix[0] && name->strptr[1] == prefix[1]) {
           theName = "dspidfn";
           memcpy(&dispid, name->strptr+sizeof(prefix)-1, sizeof(dispid));
        }
        else {
           /* now make the name a null-terminated string */
           rxstrdup(theName, *name);

           /* this always releases pdsp if necessary */
           pdsp = dnameToDispID(pdsp, name, theName, &dispid);

           if (!pdsp) {
              goto errexit;
           }
        }


        /* make the typelist a null-terminated string */
        if (typelist) {
           rxstrdup(tl, *typelist);

           /* `a' is a pseudo-argument. Really, it's a stem name
            * which should be used to hold the (array) result of
            * a function call. To avoid checking for it all the time,
            * I accept it only as the first argument. */ 
           if (tl[0] == 'a') {
              arrayRet = argv[0];
              argc--;
              if (argc) {
                 memmove(argv, argv+1, sizeof(*argv)*argc);
                 memmove(tl, tl+1, sizeof(*tl)*argc);
              }
           }
        }


        if (calltype != ct_proc)
           VariantInit(&vRet);

        if (argc > 0)
           vArgs = (VARIANTARG *)alloca(sizeof(*vArgs)*argc);
        else
           vArgs = NULL;

	for (l=0; l<argc; l++) {
		VariantInit(&vArgs[l]);
	}

	for (l = argc-1; l>=0; l--) {
		if (!argv[l].strptr) {
			vArgs[l].vt = VT_ERROR;
			vArgs[l].scode = DISP_E_PARAMNOTFOUND;
		} else {

		  if (MultiByteToWideChar(CP_ACP, 0, argv[argc-l-1].strptr, argv[argc-l-1].strlength,
				 ocBuf, sizeof ocBuf) == 0) {
                                 int argno = argc - l;

                                 OLEErr("%s: Error converting %d%s argument.", theName, argno, th(argno));
				 goto errexit;
		  }
                  ocBuf[argv[argc-l-1].strlength] = 0;
			vArgs[l].vt = VT_BSTR;
			vArgs[l].bstrVal = SysAllocStringLen(ocBuf, argv[argc-l-1].strlength);

			if (tl && *tl) {
				switch (*tl) {
					case 'A': {
                                           if (!stemToArray(&vArgs[l], &argv[argc-l-1]))
                                              goto errexit;
                                        }
                                        break;

					case 'b':
					  if (!DoChange(&argv[argc-l-1], &vArgs[l], VT_BOOL)) {
							goto errexit;
					  }
						break;
					case 'c':
					  if (!DoChange(&argv[argc-l-1], &vArgs[l], VT_CY)) {
							goto errexit;
					  }
						break;
					case 'd':
					  if (!DoChange(&argv[argc-l-1], &vArgs[l], VT_DATE)) {
							goto errexit;
					  }
						break;
					case 'i':
					  if (!DoChange(&argv[argc-l-1], &vArgs[l], VT_I2)) {
							goto errexit;
					  }
						break;
					case 'I':
						if (!DoChange(&argv[argc-l-1], &vArgs[l], VT_I4)) {
								goto errexit;
						}
						break;
					case 'o':
						VariantClear(&vArgs[l]);
						vArgs[l].vt = VT_DISPATCH;
                                                if (argv[argc-l-1].strlength != sizeof(LPDISPATCH)) {
                                                   OLEStdErr(theName, 0, "Invalid object pointer");
                                                   goto errexit;
                                                }
                                                memcpy(&vArgs[l].pdispVal, argv[argc-l-1].strptr, sizeof(LPDISPATCH));
                                                vArgs[l].pdispVal->AddRef();
						break;
					case 'r':
					  if (!DoChange(&argv[argc-l-1], &vArgs[l], VT_R4)) {
							goto errexit;
					  }
						break;
					case 'R':
					  if (!DoChange(&argv[argc-l-1], &vArgs[l], VT_R8)) {
							goto errexit;
					  }
						break;
					case 's':
						break;
					default:
						OLEErr("%s: Bad type at %d%s position of type list", theName, argc-l, th(argc-l));
						goto errexit;
				}
			}
		}

		if (tl && *tl) {
			tl++;
		}
	}

	dspp.rgvarg = vArgs;
	dspp.cArgs = argc > 0 ? argc : 0;
	if (itype == DISPATCH_PROPERTYPUT) {
		did = DISPID_PROPERTYPUT;
		dspp.rgdispidNamedArgs = &did;
		dspp.cNamedArgs = 1;
	} else {
		dspp.rgdispidNamedArgs = NULL;
		dspp.cNamedArgs = 0;
	}


	hr = pdsp->Invoke(dispid, IID_NULL, lcid, itype, &dspp,
		calltype != ct_proc?&vRet:NULL, &ex, &bt);

	for (l=0; l<(int)argc; l++) {
		VariantClear(&vArgs[l]);
	}

	if (FAILED(hr)) {
		if (GetScode(hr) == DISP_E_TYPEMISMATCH) {
			OLEErr("%s: Argument %d is of an incorrect type", theName, (int)bt);
		} else if (GetScode(hr) == DISP_E_UNKNOWNNAME) {
			OLEErr("%s: unknown name", theName);
		} else if (GetScode(hr) == DISP_E_MEMBERNOTFOUND) {
			OLEErr("%s: member not found", theName);
		} else if (GetScode(hr) == DISP_E_EXCEPTION) {
			if (ex.pfnDeferredFillIn) {
				(*ex.pfnDeferredFillIn)(&ex);
			}
			OLEExcErr(theName, &ex);
		} else {
			OLEStdErr(theName, hr, "Invoke failed");
		}
	  goto errexit;
	}

	if (calltype != ct_proc) {
		DWORD dwLen;

		if (calltype == ct_subobj) {
			if (vRet.vt != VT_DISPATCH) {
				OLEErr("%s: does not return a subobject", theName);
				goto errexit;
			}
		}

		if (vRet.vt == VT_DISPATCH) {
                   if (vRet.pdispVal == NULL) {
                      OLEErr("%s: returns NULL object", theName);
                      result->strlength = 1;
                      result->strptr[0] = '0';
                   }
                   else {
                      result->strlength = sizeof(vRet.pdispVal);
                      memcpy(result->strptr, &vRet.pdispVal, sizeof(vRet.pdispVal));
                      push_object(vRet.pdispVal);
                      vRet.vt = VT_I4; /* avoid the Release() */
                   }
                } else if (vRet.vt & VT_ARRAY) {
                   if (!arrayToStem(&arrayRet, &vRet)) {
                      goto errexit;
                   }
		} else {
                   if (vRet.vt != VT_BSTR) {
                      hr = VariantChangeTypeEx(&vRet, &vRet, cvtlcid, 0, VT_BSTR);
                      if (FAILED(hr)) {
                         OLEStdErr(theName, hr, "Failure converting return value to string");
                         goto errexit;
                      }
                   }


                   dwLen = WideCharToMultiByte(CP_ACP, 0, vRet.bstrVal, -1, NULL, 0, NULL, NULL);

                   rxresize(result, dwLen+1);
                   if (dwLen)
                      result->strlength = dwLen - 1;
                   else {
                      result->strlength = 0;
                   }
                   if (dwLen > 0 &&
                       WideCharToMultiByte(CP_ACP, 0, vRet.bstrVal, -1,
                       result->strptr, dwLen+1, NULL, NULL) == 0) {
                      OLEErr("%s: Error converting result.", theName);
                      goto errexit;
                   }
                }

                VariantClear(&vRet);
        }


	rc_zero();
        pdsp->Release();

	return 0;

errexit:
	rc_one();

        if (pdsp)
           pdsp->Release();

	return 0;
}


static BOOL DoChange(const PRXSTRING sname, VARIANT *bstr, unsigned short type)
{
	 HRESULT hr;

	 hr = VariantChangeTypeEx(bstr, bstr, cvtlcid, 0, type);
	 if (FAILED(hr)) {
		  char * name;
                  rxstrdup(name, *sname);
		  OLEStdErr(name, hr, "Bad data conversion");
		  return FALSE;
	 }

	 return TRUE;
}

#include <stdarg.h>

static const char * th(int argno)
{
   const char * th = "th";

   switch (argno % 10) {
      case 1: if ((argno % 100) != 11) th = "st"; break;
      case 2: if ((argno % 100) != 12) th = "nd"; break;
      case 3: if ((argno % 100) != 13) th = "rd"; break;
   }

   return th;
}

static char OLEErrBuf[8192];
static int  OLEErrLen = 0;

static void OLEErrClear()
{
   OLEErrLen = 0;
   OLEErrBuf[0] = 0;
}


static void OLEErr(const char *format, ...) {
	va_list ap;

        if (OLEErrLen)
           OLEErrBuf[OLEErrLen++] = '\n';

	va_start(ap, format);
	OLEErrLen += vsprintf(OLEErrBuf+OLEErrLen, format, ap);
	va_end(ap);
}


static void OLEStdErr(const char *name, HRESULT hr, const char *tag) {
   register int len;
   char buf[1024];

   len = FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM, NULL, hr,
                       lcid, buf, sizeof(buf), NULL);
   buf[len] = 0;

   OLEErr("OLE Error in %s: %s. SCODE=%x, text = %s", name, tag, hr, buf);
}


static void OLEExcErr(const char *name, EXCEPINFO * excp)
{
   OLEErr("%s: %ls: %ls", name, excp->bstrSource, excp->bstrDescription);
   if (excp->bstrSource);
      SysFreeString(excp->bstrSource);
   if (excp->bstrDescription);
      SysFreeString(excp->bstrDescription);
   if (excp->bstrHelpFile);
      SysFreeString(excp->bstrHelpFile);
}



rxfunc(w32olegeterror)
{
   rxresize(result, OLEErrLen);
   memcpy(result->strptr, OLEErrBuf, OLEErrLen);
   return 0;
}

/* stack of IDispatch interfaces, to remove the requirement that the programmer
 * keep track of them */
static struct obj_stack {
   LPDISPATCH obj;
   struct obj_stack * next;

   obj_stack(LPDISPATCH lpd, struct obj_stack * pos) { obj = lpd; next = pos; }
   ~obj_stack();
} * os = NULL;

static void push_object(LPDISPATCH obj)
{
   os = new obj_stack(obj, os);
}

static void pop_object(LPDISPATCH lpd)
{
   obj_stack * pobj, *ppobj;

   for (ppobj = NULL, pobj = os; pobj && lpd != pobj->obj; ppobj = pobj, pobj = pobj->next)
      ;

   if (pobj && lpd == pobj->obj) {

      if (ppobj)
         ppobj ->next = pobj->next;
      else
         os = pobj->next;
      pobj->next = NULL;
      delete pobj;
   }
}


obj_stack::~obj_stack()
{
   if (next)
      delete next;
   if (obj)
      obj->Release();
}


/* clear out any open object handles. This is invoked at exit time */
static void del_objstack(void)
{
   if (os)
      delete os;

   os = NULL;
}

extern "C" {
int createLink(const char *fullpath, const char * display, const char *path,
                      const char * dir, const char *args, int icon,
                      const char * iconpath, int hotkey)
{
   IShellLink * pShellLink;
   IPersistFile *pPersistFile;
   HRESULT hResult;
   int rc = 0;

   hResult = CoCreateInstance(CLSID_ShellLink, NULL, CLSCTX_INPROC_SERVER, IID_IShellLink, (void **)&pShellLink);

   if (!SUCCEEDED(hResult)) {
      rc = -GetLastError();
   }

   else {
      hResult = pShellLink->QueryInterface(IID_IPersistFile, (void **)&pPersistFile);

      if (!SUCCEEDED(hResult)) {
         rc = -GetLastError();
      }
   }

   if (SUCCEEDED(hResult)) {
      WCHAR wszShortcutPath[_MAX_PATH+1];
      pShellLink->SetPath (path);
      if (args)
         pShellLink->SetArguments (args);
      if (dir)
         pShellLink->SetWorkingDirectory (dir);
      pShellLink->SetDescription (display);
		
      if (hotkey) {
         pShellLink->SetHotkey(hotkey);
      }

      if ((icon > -1) || iconpath) {
         if (!iconpath)
            iconpath = path;
         if (icon < 0)
            icon = 0;
         pShellLink->SetIconLocation(iconpath, icon);
      }

      memset(wszShortcutPath, 0, sizeof(wszShortcutPath));

      MultiByteToWideChar(CP_ACP,	
                          0, 
                          fullpath, 
                          strlen(fullpath), 
                          wszShortcutPath, 
                          _MAX_PATH);

      hResult = pPersistFile->Save (wszShortcutPath, FALSE);

      if (!SUCCEEDED(hResult))
         rc = -4;

      pPersistFile->Release();
   }

   if (pShellLink)
      pShellLink->Release();

   return rc;
}
int adjustLink(const char *fullpath, const char * display, const char *path,
                      const char * dir, const char *args, int icon,
                      const char * iconpath, int hotkey)
{
   IShellLink * pShellLink;
   IPersistFile *pPersistFile;
   HRESULT hResult;
   int rc = 0;

   hResult = CoCreateInstance(CLSID_ShellLink, NULL, CLSCTX_INPROC_SERVER, IID_IShellLink, (void **)&pShellLink);

   if (!SUCCEEDED(hResult)) {
      rc = -GetLastError();
   }

   else {
      hResult = pShellLink->QueryInterface(IID_IPersistFile, (void **)&pPersistFile);

      if (!SUCCEEDED(hResult)) {
         rc = -GetLastError();
      }
   }

   if (SUCCEEDED(hResult)) {
      WCHAR wszShortcutPath[_MAX_PATH+1];

      memset(wszShortcutPath, 0, sizeof(wszShortcutPath));

      MultiByteToWideChar(CP_ACP,	
                          0, 
                          fullpath, 
                          strlen(fullpath), 
                          wszShortcutPath, 
                          _MAX_PATH);

      hResult = pPersistFile->Load(wszShortcutPath, 0);

      if (!SUCCEEDED(hResult))
         rc = -4;

      else {
         if (path)
            pShellLink->SetPath (path);
         if (args)
            pShellLink->SetArguments (args);
         if (dir)
            pShellLink->SetWorkingDirectory (dir);
         if (display)
            pShellLink->SetDescription (display);

         if (hotkey) {
            pShellLink->SetHotkey(hotkey);
         }

         if ((icon > -1) || iconpath) {
            char curloc[1024];
            int curicon;

            pShellLink->GetIconLocation(curloc, sizeof(curloc), &curicon);

            if (!iconpath)
               iconpath = curloc;
            if (icon < 0)
               icon = curicon;

            pShellLink->SetIconLocation(iconpath, icon);
         }

         hResult = pPersistFile->Save (wszShortcutPath, TRUE);
         pPersistFile->Release();
      }

      if (pShellLink)
         pShellLink->Release();
   }

   return rc;
}
   
rxfunc(w32olecleanup)
{
   del_objstack();
   return 0;
}

}
