/* shell functions:
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is w32funcs.
 *
 * The Initial Developer of the Original Code is Patrick TJ McPhee.
 * Portions created by Patrick McPhee are Copyright 1999, 2002
 * Patrick TJ McPhee. All Rights Reserved.
 *
 * Contributors:
 *  Mark Hessling provided w32execute and w32executestem
 *
 * $Header: C:/ptjm/rexx/w32funcs/RCS/shell.c 1.12 2003/12/03 04:11:01 ptjm Rel $
 */

/* functions defined in this file:
 * w32menuadditem([menu], [submenu], display, path, [workingdir], [arguments], [icon number], [icon file], [hot key])
 * w32menuremoveitem([menu], [submenu], display)
 * w32menumoveitem([menu], [submenu], oldname, newname)
 * w32menumove([menu], submenu, [newmenu], [newsubmenu])
 * w32execute(file)
 * w32executestem(file., result.)
 * w32dlgopenfile(file.[, deffile][, filter.][, directory][, title][, flags])
 * w32dlgsavefile(file.[, deffile][, filter.][, directory][, title][, flags])
 * w32dlgchoosecolor([red, green, blue])
 * w32dlgchoosecolour([red, green, blue])
 * all functions return 0 for success, or an error code.
 */ 
 
#pragma warning(disable: 4115 4201 4214 4514)
#include <windows.h>
/* don't re-enable 4514 */
#pragma warning(default: 4115 4201 4214)
#include <malloc.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <direct.h>
#include <ctype.h>
#include "w32funcs.h"

/* defined in w32ole.cpp */
int createLink(const char *fullpath, const char * display, const char *path,
                      const char * dir, const char *args, int icon,
                      char * iconpath, int hotkey);
int adjustLink(const char *fullpath, const char * display, const char *path,
                      const char * dir, const char *args, int icon,
                      char * iconpath, int hotkey);


static int getmenupath(const char * menu, char * fullpath, unsigned long * fplen)
{
   HKEY key = 0;
   static const char regpath[] = "SOFTWARE\\Microsoft\\Windows\\CurrentVersion\\Explorer\\Shell Folders",
                     mystartup[] = "Programs",
                     allstartup[] = "Common Programs",
                     mydesktop[] = "Desktop",
                     alldesktop[] = "Common Desktop";
   
   if (!strcmp(menu, "all")) {
      RegOpenKeyEx(HKEY_LOCAL_MACHINE, regpath, 0, MAXIMUM_ALLOWED, &key);
      RegQueryValueEx(key, allstartup, NULL, NULL, fullpath, fplen);
      RegCloseKey(key);
      /* fplen includes a trailing null */
      (*fplen)--;
   }
   else if (!strcmp(menu, "desktop")) {
      RegOpenKeyEx(HKEY_CURRENT_USER, regpath, 0, MAXIMUM_ALLOWED, &key);
      RegQueryValueEx(key, mydesktop, NULL, NULL, fullpath, fplen);
      RegCloseKey(key);
      (*fplen)--;
   }
   else if (!strcmp(menu, "current")) {
      RegOpenKeyEx(HKEY_CURRENT_USER, regpath, 0, MAXIMUM_ALLOWED, &key);
      RegQueryValueEx(key, mystartup, NULL, NULL, fullpath, fplen);
      RegCloseKey(key);
      (*fplen)--;
   }
   else if (!strcmp(menu, "all desktop")) {
      RegOpenKeyEx(HKEY_LOCAL_MACHINE, regpath, 0, MAXIMUM_ALLOWED, &key);
      RegQueryValueEx(key, alldesktop, NULL, NULL, fullpath, fplen);
      RegCloseKey(key);
      (*fplen)--;
   }
   else {
      strcpy(fullpath, menu);
      *fplen = strlen(menu);
   }

   return 0;
}




rxfunc(w32menuadditem)
{
   char * menu = NULL, * folder, *display, *path, *dir = NULL, *args = NULL,
        * icons, * iconpath = NULL;
   char fullpath[_MAX_PATH];
   int icon = -1;
   unsigned long fplen = sizeof(fullpath);
   int rc;

   checkparam(4,9);

   if (argv[0].strptr)
      rxstrdup(menu, argv[0]);
   else
      menu = "all";

   if (argv[1].strptr)
      rxstrdup(folder, argv[1]);
   else
      folder = "";

   rxstrdup(display, argv[2]);

   rxstrdup(path, argv[3]);


   if (argc > 4 && argv[4].strptr) {
      rxstrdup(dir, argv[4]);
   }

   if (argc > 5 && argv[5].strptr) {
      rxstrdup(args, argv[5]);
   }

   if (argc > 6 && argv[6].strptr) {
      rxstrdup(icons, argv[6]);
      icon = atoi(icons);
   }

   if (argc > 7 && argv[7].strptr) {
      rxstrdup(iconpath, argv[7]);
   }

   /* ignore the hotkey for now */

   /* find the path to the menu */
   if (getmenupath(menu, fullpath, &fplen)) {
      return BADGENERAL;
   }

   if (*folder) {
      menu = fullpath + fplen + 1;
      fplen += sprintf(fullpath+fplen, "\\%s", folder);

      /* make sure the folder's there */
      while (menu = strchr(menu, '\\')) {
         *menu = 0;
         mkdir(fullpath);
         *menu = '\\';
         menu++;
      }
      mkdir(fullpath);
   }

   /* now create the link */
   sprintf(fullpath+fplen, "\\%s.lnk", display);

   /* and call a c++ function which does the OLE stuff */
   rc = createLink(fullpath, display, path, dir, args, icon, iconpath, 0);

   result->strlength = sprintf(result->strptr, "%d", rc);

   return 0;
}

rxfunc(w32menuremoveitem)
{
   char * menu = NULL, * folder, *display;
   char fullpath[_MAX_PATH];
   unsigned long fplen = sizeof(fullpath);
   int rc;

   checkparam(3,3);

   if (argv[0].strptr)
      rxstrdup(menu, argv[0]);
   else
      menu = "all";

   if (argv[1].strptr)
      rxstrdup(folder, argv[1]);
   else
      folder = "";

   rxstrdup(display, argv[2]);

   if (getmenupath(menu, fullpath, &fplen)) {
      return BADGENERAL;
   }

   if (*folder) {
      menu = fullpath + fplen + 1;
      fplen += sprintf(fullpath+fplen, "\\%s", folder);
   }
   else {
      menu = NULL;
   }

   /* now delete the link */
   sprintf(fullpath+fplen, "\\%s.lnk", display);

   rc = remove(fullpath);

   /* and all those pesky sub-directories */
   if (menu)
      for (display = strrchr(menu, '\\'); display; display = strrchr(menu, '\\')) {
         *display = 0;
         rmdir(fullpath);
      }

   result->strlength = sprintf(result->strptr, "%d", rc);   

   return 0;
}

rxfunc(w32menumoveitem)
{
   char * menu = NULL, * folder, *display, *newname;
   char fullpath[_MAX_PATH], mullpath[_MAX_PATH];
   unsigned long fplen = sizeof(fullpath);
   int rc;

   checkparam(4,4);

   if (argv[0].strptr)
      rxstrdup(menu, argv[0]);
   else
      menu = "all";

   if (argv[1].strptr)
      rxstrdup(folder, argv[1]);
   else
      folder = "";

   rxstrdup(display, argv[2]);
   rxstrdup(newname, argv[3]);

   if (getmenupath(menu, fullpath, &fplen)) {
      return BADGENERAL;
   }

   if (*folder) {
      fplen += sprintf(fullpath+fplen, "\\%s", folder);
   }
   memcpy(mullpath, fullpath, fplen);

   /* now rename the link */
   sprintf(fullpath+fplen, "\\%s.lnk", display);
   sprintf(mullpath+fplen, "\\%s.lnk", newname);
   
   rc = rename(fullpath, mullpath);

   result->strlength = sprintf(result->strptr, "%d", rc);   

   return 0;
}

rxfunc(w32menumove)
{
   char * menu = NULL, * folder, *display, *newname;
   char fullpath[_MAX_PATH], mullpath[_MAX_PATH];
   unsigned long fplen = sizeof(fullpath);
   int rc;

   checkparam(4,4);

   if (argv[0].strptr)
      rxstrdup(menu, argv[0]);
   else
      menu = "all";

   if (argv[1].strptr)
      rxstrdup(folder, argv[1]);
   else
      folder = "";

   rxstrdup(display, argv[2]);
   rxstrdup(newname, argv[3]);

   if (getmenupath(menu, fullpath, &fplen)) {
      return BADGENERAL;
   }

   if (*folder) {
      fplen += sprintf(fullpath+fplen, "\\%s", folder);
   }
   memcpy(mullpath, fullpath, fplen);

   /* now rename the link */
   sprintf(fullpath+fplen, "\\%s.lnk", display);
   sprintf(mullpath+fplen, "\\%s.lnk", newname);

   rc = rename(fullpath, mullpath);

   result->strlength = sprintf(result->strptr, "%d", rc);

   return 0;
}

rxfunc(w32menuedititem)
{
   char * menu = "all", * folder = "", *display, *path = NULL, *dir = NULL, *args = NULL,
        * icons, * iconpath = NULL;
   char fullpath[_MAX_PATH];
   int icon = -1;
   unsigned long fplen = sizeof(fullpath);
   int rc;

   checkparam(3,9);

   if (argv[0].strptr)
      rxstrdup(menu, argv[0]);

   if (argv[1].strptr)
      rxstrdup(folder, argv[1]);
   else
      folder = "";

   if (argv[2].strptr)
      rxstrdup(display, argv[2]);
   else
      return BADARGS;

   if (argc > 3 && argv[3].strptr)
      rxstrdup(path, argv[3]);

   if (argc > 4 && argv[4].strptr) {
      rxstrdup(dir, argv[4]);
   }

   if (argc > 5 && argv[5].strptr) {
      rxstrdup(args, argv[5]);
   }

   if (argc > 6 && argv[6].strptr) {
      rxstrdup(icons, argv[6]);
      icon = atoi(icons);
   }

   if (argc > 7 && argv[7].strptr) {
      rxstrdup(iconpath, argv[7]);
   }

   /* ignore the hotkey for now */

   /* find the path to the menu */
   if (getmenupath(menu, fullpath, &fplen)) {
      return BADGENERAL;
   }

   if (*folder) {
      fplen += sprintf(fullpath+fplen, "\\%s", folder);
   }

   /* now create the link */
   sprintf(fullpath+fplen, "\\%s.lnk", display);

   /* and call a c++ function which does the OLE stuff */
   rc = adjustLink(fullpath, display, path, dir, args, icon, iconpath, 0);

   result->strlength = sprintf(result->strptr, "%d", rc);

   return 0;
}

/*
 * w32execute( file )
 */
rxfunc(w32execute)
{
   SHELLEXECUTEINFO sei;
   int rc = 0; /* another good one; 0 means fail, 1 means success */
   char prog[256];

   checkparam(1,1);

   ZeroMemory(&sei,sizeof(sei));

   sei.cbSize = sizeof(sei);
   sei.fMask = SEE_MASK_FLAG_DDEWAIT;
   sei.nShow = SW_SHOWNORMAL;
   sei.lpVerb = "open";

   rc = (int)FindExecutable( argv[0].strptr, NULL, prog );
   if ( rc > 32 ) /* found the executable */
   {
      sei.lpFile = prog;
      sei.lpParameters = argv[0].strptr;
   }
   else
   {
      sei.lpFile = argv[0].strptr;
      sei.lpParameters = NULL;
   }
   rc = ShellExecuteEx( &sei );

   result->strlength = sprintf(result->strptr, "%d", rc);

   return 0;
}


/*
 * w32executestem( file., result. )
 */
rxfunc(w32executestem)
{
   char prog[256];
   chararray * vals, *results;
   SHELLEXECUTEINFO sei;
   char * lpOperation = NULL, *lpDirectory = NULL, *tmp;
   int rc, i, len;

   checkparam(2,2);

   vals = new_chararray();
   results = new_chararray();
   getastem( argv, vals );

   /* call the Win32 API for each file */
   for (i=0;i<vals->count;i++)
   {
      ZeroMemory( &sei, sizeof(sei) );
      sei.cbSize = sizeof(sei);
      sei.fMask = SEE_MASK_FLAG_DDEWAIT;
      sei.lpDirectory = NULL;
      sei.nShow = SW_SHOWNORMAL;

      tmp = malloc( 1+vals->array[i].strlength );
      memcpy( tmp,vals->array[i].strptr,vals->array[i].strlength);
      *(tmp+vals->array[i].strlength) = '\0';
      rc = (int)FindExecutable( tmp, NULL, prog );
      if ( rc > 32 )
      {
         sei.lpFile = prog;
         sei.lpParameters = tmp;
      }
      else
      {
         sei.lpFile = tmp;
         sei.lpParameters = NULL;
      }
      rc = ShellExecuteEx( &sei );
      len = sprintf( prog, "%ld", rc );
      cha_addstr( results, prog, len );
      free( tmp );
   }
   setastem( argv+1, results );
   delete_chararray( vals );
   delete_chararray( results );

   rc = 0;
   result->strlength = sprintf(result->strptr, "%d", rc);

   return 0;
}


/* skip non-alphabetics */
static unsigned char * skip_f_delim(unsigned char * const s, const int l)
{
   register int i;
   for (i = 0; i < l && !isalpha(s[i]); i++)
      ;
   return s + i;
}

/* skip alphabetics */
static unsigned char * skip_f_alpha(unsigned char * const s, const int l)
{
   register int i;
   for (i = 0; i < l && isalpha(s[i]); i++)
      ;
   return s + i;
}


/* call either GetOpenFileName or GetSaveFileName with the specified flags
 * and things. See the .tex file for a description of the arguments
 * to the rexx function. */

/* w32dlgopenfile(file [,deffile][, filter][, directory][, title][, flags]) */
rxfunc(w32dlgopenfile)
{
   static const char display[] = ".DISPLAY.", wildcard[] = ".WILDCARD.",
                     readonly[] = ".READONLY", fdefault[] = ".DEFAULT";
   OPENFILENAME ofn;
   char file[1024], file2[1024];
   char filter[4096], *fp, wname[512], dname[512];
   unsigned char * flp, * s;
   int nfilter = 0, rc;
   register int i;
   SHVBLOCK var;
   chararray * names;

   checkparam(1, 6);

   memset(&ofn, 0, sizeof(ofn));
   memset(&var, 0, sizeof(var));

   ofn.lStructSize = sizeof(ofn);
   ofn.lpstrFile = file;
   ofn.nMaxFile = sizeof(file);
   ofn.hwndOwner = GetForegroundWindow();

   /* default file name -- can't use rxstrdup because this is the
    * return buffer */ 
   if (argc >  1 && argv[1].strlength > 0) {
      memcpy(file, argv[1].strptr, argv[1].strlength);
      file[argv[1].strlength] = 0;
   }

   else {
      file[0] = 0;
   }


   /* filters */
   if (argc >  2 && argv[2].strlength > 0) {
      getstemsize(argv+2, &nfilter);

      if (nfilter > 0) {
         /* strip trailing . */
         if (argv[2].strptr[argv[2].strlength-1] == '.') {
            argv[2].strlength--;
         }
         memcpy(dname, argv[2].strptr, argv[2].strlength);
         memcpy(wname, argv[2].strptr, argv[2].strlength);

         memcpy(dname+argv[2].strlength, display, sizeof(display)-1);
         memcpy(wname+argv[2].strlength, wildcard, sizeof(wildcard)-1);

         var.shvcode = RXSHV_FETCH;

         for (i = 1, fp = filter; i <= nfilter; i++) {
            var.shvname.strptr = dname;
            var.shvname.strlength = argv[2].strlength+sizeof(display)-1 +
               sprintf(dname+argv[2].strlength+sizeof(display)-1, "%d", i);

            var.shvvalue.strptr = fp;
            var.shvvalue.strlength = var.shvvaluelen =
               sizeof(filter) - (fp - filter);

            rc = RexxVariablePool(&var);

            fp[var.shvvalue.strlength] = 0;
            fp += var.shvvalue.strlength + 1;


            var.shvname.strptr = wname;
            var.shvname.strlength = argv[2].strlength+sizeof(wildcard)-1 +
               sprintf(wname+argv[2].strlength+sizeof(wildcard)-1, "%d", i);

            var.shvvalue.strptr = fp;
            var.shvvalue.strlength = var.shvvaluelen =
               sizeof(filter) - (fp - filter);
            
            RexxVariablePool(&var);

            fp[var.shvvalue.strlength] = 0;
            fp += var.shvvalue.strlength + 1;
         }

         /* terminate with a zero-length string */
         fp[0] = 0;

         ofn.lpstrFilter = filter;

         memcpy(dname+argv[2].strlength, fdefault, sizeof(fdefault)-1);
         var.shvname.strptr = dname;
         var.shvname.strlength = argv[2].strlength+sizeof(fdefault)-1;
         var.shvvalue.strptr = wname;
         var.shvvalue.strlength = var.shvvaluelen =
            sizeof(wname)-1;

         RexxVariablePool(&var);

         wname[var.shvvalue.strlength] = 0;
         ofn.nFilterIndex = atoi(wname);
         if (!ofn.nFilterIndex || ofn.nFilterIndex > nfilter)
            ofn.nFilterIndex = 1;

      }
   }

   /* initial directory */
   if (argc >  3 && argv[3].strlength > 0) {
      rxstrdup(s, argv[3]);
      ofn.lpstrInitialDir = s;
   }

   /* title */
   if (argc >  4 && argv[4].strlength > 0) {
      rxstrdup(s, argv[4]);
      ofn.lpstrTitle = s;
   }

   /* flags -- always use the `new-style' of dialog box, which allows for
    * long file names */
   ofn.Flags = OFN_EXPLORER;

   if (argc >  5 && argv[5].strlength > 0) {
      /* add additional flags */
      for (flp = skip_f_delim(argv[5].strptr, argv[5].strlength);
           (flp - (unsigned char *)argv[5].strptr) < argv[5].strlength;
           flp = skip_f_delim(flp, argv[5].strlength - (flp - (unsigned char *)argv[5].strptr))) {
         switch (flp[0]) {
            case 'a':
            case 'A':
               ofn.Flags |= OFN_ALLOWMULTISELECT;
               break;
            case 'c':
            case 'C':
               ofn.Flags |= OFN_CREATEPROMPT|OFN_FILEMUSTEXIST;
               break;
            case 'f':
            case 'F':
               ofn.Flags |= OFN_FILEMUSTEXIST;
               break;
            case 'h':
            case 'H':
               ofn.Flags |= OFN_HIDEREADONLY;
               break;
            case 'n':
            case 'N':
               switch (flp[2]) {
                  case 'c':
                  case 'C':
                     ofn.Flags |= OFN_NOCHANGEDIR;
                     break;
                  case 'n':
                  case 'N':
                     ofn.Flags |= OFN_NONETWORKBUTTON;
                     break;
               }
               break;
            case 'o':
            case 'O':
               ofn.Flags |= OFN_OVERWRITEPROMPT;
               break;
            case 'p':
            case 'P':
               ofn.Flags |= OFN_PATHMUSTEXIST;
               break;
            case 'r':
            case 'R':
               ofn.Flags |= OFN_READONLY;
               break;
         }

         flp = skip_f_alpha(flp, argv[5].strlength - (flp - (unsigned char *)argv[5].strptr));
      }
   }

   /* which function to execute depends on the name under which the function
    * was invoked */
   if (fname[6] == 'S')
      rc = GetSaveFileName(&ofn);
   else
      rc = GetOpenFileName(&ofn);

   /* rc != 0 means success, which we return as 0 */
   if (rc) {
      result_zero();

      /* set the default filter */
      if (nfilter) {
         var.shvcode = RXSHV_SET;
         var.shvvalue.strlength = sprintf(wname, "%d", ofn.nFilterIndex);
         RexxVariablePool(&var);
      }

      /* set file name stem */

      names = new_chararray();

      /* if more than one file was selected, the files are returned as a
       * directory, a null, the first file, a null, and so forth. We know
       * this has happened because lpstrFile[nFileOffset-1] is 0. */
      if (ofn.lpstrFile[ofn.nFileOffset-1] != 0) {
         cha_addstr(names, file, strlen(file));
      }
      else {
         memcpy(file2, file, ofn.nFileOffset-1);
         file2[ofn.nFileOffset-1] = '\\';
         for (fp = file+ofn.nFileOffset; *fp; fp += strlen(fp)+1) {
            strcpy(file2+ofn.nFileOffset, fp);
            cha_addstr(names, file2, strlen(file2));
         }
      }

      setastem(argv, names);
      delete_chararray(names);

      /* set read-only member */
      if (argv[0].strptr[argv[0].strlength-1]== '.') {
         argv[0].strlength--;
      }

      memcpy(dname, argv[0].strptr, argv[0].strlength);
      memcpy(dname+argv[0].strlength, readonly, sizeof(readonly)-1);
      var.shvname.strptr = dname;
      var.shvname.strlength = argv[0].strlength+sizeof(readonly)-1;
      var.shvvaluelen = var.shvvalue.strlength = 1;
      var.shvcode = RXSHV_SET;

      if (ofn.Flags & OFN_READONLY) {
         var.shvvalue.strptr = "1";
      }
      else {
         var.shvvalue.strptr = "0";
      }
      RexxVariablePool(&var);
   }
   else {
      rc = CommDlgExtendedError();
      if (rc == 0) {
         result_one();
      }
      else {
         result->strlength = sprintf(result->strptr, "%d", rc);
      }
   }


   return 0;
}

/* w32dlgsavefile(file [,deffile][, filter][, directory][, title][, flags]) */
rxfunc(w32dlgsavefile)
{
   return w32dlgopenfile(fname, argc, argv, pSomething, result);
}

/* w32dlgchoosecolor([red, green, blue]) */
rxfunc(w32dlgchoosecolor)
{
   return w32dlgchoosecolour(fname, argc, argv, pSomething, result);
}

/* w32dlgchoosecolour([red, green, blue]) */
rxfunc(w32dlgchoosecolour)
{
   char * reds, *greens, *blues;
   int red = 0x3f, green = 0x3f, blue = 0x3f, rc;
   static COLORREF custc[16];
   CHOOSECOLOR cc;

   checkparam(0, 3);

   if (argc > 0 && argv[0].strptr) {
      rxstrdup(reds, argv[0]);
      red = atoi(reds);
   }

   if (argc > 1 && argv[1].strptr) {
      rxstrdup(greens, argv[1]);
      green = atoi(greens);
   }

   if (argc > 2 && argv[2].strptr) {
      rxstrdup(blues, argv[2]);
      blue = atoi(blues);
   }

   cc.lStructSize = sizeof(cc);
   cc.hwndOwner = NULL;
   cc.hInstance = NULL;
   cc.rgbResult = RGB(red, green, blue);
   cc.lpCustColors = custc;
   cc.Flags = CC_RGBINIT|CC_FULLOPEN;
   cc.lCustData = 0;
   cc.lpfnHook = NULL;
   cc.lpTemplateName = NULL;

   rc = ChooseColor(&cc);

   if (rc) {
      result->strlength = sprintf(result->strptr, "%d %d %d", GetRValue(cc.rgbResult),
                                  GetGValue(cc.rgbResult), GetBValue(cc.rgbResult));
   }
   else {
      rc = CommDlgExtendedError();
      if (!rc) rc = 1;
      result->strlength = sprintf(result->strptr, "-%d 0 0", rc);
   }



   return 0;
}
