%{
/* patrick's message compiler
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is ptjmc.
 *
 * The Initial Developer of the Original Code is Patrick TJ McPhee.
 * Portions created by Patrick McPhee are Copyright 2003
 * Patrick TJ McPhee. All Rights Reserved.
 *
 * Contributors:
 *
 * $Header: C:/ptjm/rexx/w32funcs/RCS/ptjmc.l 1.3 2003/02/27 18:57:23 ptjm Rel $
 */

#include <stdlib.h>
#include <string.h>
#include <windows.h>

static unsigned int yy_line = 1, nlc = 0;

static void addsev(char * s);
static void addfac(char * s);
static void addmsg(const char * s);
static void addmsgsev(const char * s);
static void addmsgfac(const char * s);
static void addsymname(const char * s);
static void addmsgtext(const char * s, int l);
static void finishmsg(void);


%}

str [a-zA-Z]+
whatever [^a-zA-Z \t\n\f]+
ws [ \t\f]+
nl "\n"
nws [^ \t\n\f]+
%x SEVS FACS MSGH MSGT
%%
"SeverityNames=("	{ BEGIN SEVS; }
<SEVS>")"		{ BEGIN 0; }
<SEVS>[a-zA-Z0-9]+"="[0-9xX]+	{ addsev(yytext); }

"FacilityNames=("	{ BEGIN FACS; }
<FACS>")"		{ BEGIN 0; }
<FACS>[a-zA-Z0-9]+"="[0-9xX]+	{ addfac(yytext); }

"MessageId="[0-9]+	{ BEGIN MSGH; addmsg(yytext+10); }
<MSGH>"Severity="{nws}	{ addmsgsev(yytext+9); }
<MSGH>"Facility="{nws}	{ addmsgfac(yytext+9); }
<MSGH>"SymbolicName="{nws}	{ addsymname(yytext+13); }
<MSGH>"Language="{nws}	{ BEGIN MSGT; nlc = 0; }

<MSGT>{nl}"."		{ finishmsg(); BEGIN 0; }
<MSGT>[^\n]+		{ if (nlc) addmsgtext(yytext, yyleng); }
<MSGT>{nl}		{ if (nlc++) addmsgtext(" ", yyleng); yy_line++; }

";".+			|
<SEVS>":"{nws}		|
<FACS>":"{nws}		|
<MSGH>{ws}		|
<SEVS>{ws}		|
<FACS>{ws}		|
{ws}			{ ; }

<MSGH>{nl}		|
<SEVS>{nl}		|
<FACS>{nl}		|
{nl}			{ yy_line++; }

{str}			|
{whatever}		{ fprintf(stderr, "unexpected string %s on line %d\n", yytext, yy_line); }
%%

#pragma pack(1)
struct msgfmt_T {
   unsigned short id;
   unsigned short facility : 12, reserved : 1, customer : 1, severity : 2;
};
#pragma pack()

struct msgcont {
   union {
      struct msgfmt_T set;
      unsigned int   sort;
   } ind;
   unsigned int len;
   unsigned char * text, *symname;
};


static struct msgcont * msgs = NULL;
static unsigned int msgcount = 0;


struct symbol {
   unsigned char * text;
   unsigned int value;
};

static struct symbol *sevs = NULL, * facs = NULL;
static unsigned int sevcount = 0, faccount = 0;

static void addsev(char * s)
{
   char * cp;

   if (!(sevcount % 10)) {
      sevs = realloc(sevs, sizeof(*sevs) * (sevcount + 10));
   }

   cp = strchr(s, '=');

   if (!cp) {
      fprintf(stderr, "error at line %d. found %s, expected <symbol>=<number>\n",
              yy_line, s);
   }

   *cp = 0;
   sevs[sevcount].text = strdup(s);

   s = cp+1;

   if (s[0] == '0' && (s[1] == 'x' || s[1] == 'X')) {
      sscanf(s+2, "%x", &sevs[sevcount].value);
   }
   else {
      sevs[sevcount].value = atoi(s);
   }

   if (sevs[sevcount].value > 3) {
      fprintf(stderr, "severity out of range at line %d. Found %s, maximum value is 3.\n",
              yy_line, s);
   }

   sevcount++;
}


static void addfac(char * s)
{
   char * cp;

   if (!(faccount % 10)) {
      facs = realloc(facs, sizeof(*facs) * (faccount + 10));
   }

   cp = strchr(s, '=');

   if (!cp) {
      fprintf(stderr, "error at line %d. found %s, expected <symbol>=<number>\n",
              yy_line, s);
   }

   *cp = 0;
   facs[faccount].text = strdup(s);

   s = cp+1;

   if (s[0] == '0' && (s[1] == 'x' || s[1] == 'X')) {
      sscanf(s+2, "%x", &facs[faccount].value);
   }
   else {
      facs[faccount].value = atoi(s);
   }

   faccount++;
}


static void addmsg(const char * s)
{
   if (!(msgcount % 10)) {
      msgs = realloc(msgs, sizeof(*msgs) * (msgcount + 10));
   }

   msgs[msgcount].ind.sort =
   msgs[msgcount].len = 0;
   msgs[msgcount].text =
   msgs[msgcount].symname = NULL;

   if (s[0] == '0' && (s[1] == 'x' || s[1] == 'X')) {
      sscanf(s+2, "%hx", &msgs[msgcount].ind.set.id);
   }
   else {
      msgs[msgcount].ind.set.id = atoi(s);
   }
}



static void addmsgsev(const char * s)
{
   register int i;

   for (i = 0; i < sevcount; i++) {
      if (!stricmp(sevs[i].text, s)) {
         msgs[msgcount].ind.set.severity = sevs[i].value;
         break;
      }
   }

   if (i == sevcount) {
      /* standard severities which don't have to be specified */
      if (!stricmp(s, "Warning")) {
         msgs[msgcount].ind.set.severity = EVENTLOG_WARNING_TYPE;
      }
      else if (!stricmp(s, "Success")) {
         msgs[msgcount].ind.set.severity = EVENTLOG_INFORMATION_TYPE;
      }
      else if (!stricmp(s, "Error")) {
         msgs[msgcount].ind.set.severity = EVENTLOG_ERROR_TYPE;
      }
      else if (!stricmp(s, "None")) {
         msgs[msgcount].ind.set.severity = 0;
      }
      else
         fprintf(stderr, "found unknown severity %s at line %d\n", s, yy_line);
   }
}

static void addmsgfac(const char * s)
{
   register int i;

   for (i = 0; i < faccount; i++) {
      if (!stricmp(facs[i].text, s)) {
         msgs[msgcount].ind.set.facility = facs[i].value;
         break;
      }
   }

   if (i == faccount) {
      fprintf(stderr, "found unknown facility %s at line %d\n", s, yy_line);
   }
}

static int lsymname = 0;

static void addsymname(const char * s)
{
   int l = strlen(s);

   if (l > lsymname)
      lsymname = l;

   msgs[msgcount].symname = strdup(s);
}

static void addmsgtext(const char * s, int l)
{
   msgs[msgcount].text = realloc(msgs[msgcount].text, msgs[msgcount].len + l + 1);
   memcpy(msgs[msgcount].text+msgs[msgcount].len, s, l+1);
   msgs[msgcount].len += l;
}

static void finishmsg(void)
{
   msgcount++;
}


static int msgcmp(const void * l, const void * r)
{
   if (((struct msgcont *)l)->ind.sort == ((struct msgcont *)r)->ind.sort)
      return 0;
   else if (((struct msgcont *)l)->ind.sort > ((struct msgcont *)r)->ind.sort)
      return 1;
   else
      return -1;
}


struct pair_t {
   unsigned int msg[2];
   unsigned int msgoff;
};


/* generate name.h, name.bin, name.rc */
void emitstuff(const char * name)
{
   int prevmsg;
   int pairc, msgoff;
   struct pair_t * pairs;
   char * fname, * cp;
   register int i, j;
   FILE * fp;

   if (msgcount == 0) {
      fprintf(stderr, "no messages in %s\n", name);
      return;
   }

   /* sort the messages */
   qsort(msgs, msgcount, sizeof(*msgs), msgcmp);

   /* figure out how many pairs of messages we need -- messages are indexed
    * in ranges, with a break on each new severity/facility code */
   for (j = pairc = i = 0;
        i < msgcount;
        prevmsg = msgs[i].ind.sort, i++) {

      if (j == 0) {
         j = 1;
      }
      else if ((prevmsg+1) != msgs[i].ind.sort) {
         j = 0;
         pairc++;
         i--;
         continue;
      }
   }
   
   if (j)
      pairc++;

   pairs = malloc(pairc * sizeof(*pairs));

   /* now set up the index  */
   for (msgoff = sizeof(int)+pairc*sizeof(*pairs), j = pairc = i = 0;
        i < msgcount;
        prevmsg = msgs[i].ind.sort, i++) {


      if (j == 0) {
         pairs[pairc].msgoff = msgoff;
         pairs[pairc].msg[0] = msgs[i].ind.sort;
         j = 1;
      }
      else if ((prevmsg+1) != msgs[i].ind.sort) {
         pairs[pairc].msg[1] = prevmsg;
         j = 0;
         pairc++;
         i--;
         continue;
      }

      msgoff += msgs[i].len + sizeof(int);

      if (msgoff % 4)
         while (msgoff % 4)
            msgoff++;
      else
         msgoff += 4;
   }

   if (j) {
      pairs[pairc].msg[1] = prevmsg;
      pairc ++;
   }

   fname = malloc(strlen(name) + 4);
   strcpy(fname, name);
   cp = strrchr(fname, '.');
   if (!cp) {
      cp = strchr(fname, '\0');
   }

   /* .h file -- this has a definition for all the symbolic names. In
    * contrast to microsoft's message compiler, I don't explain the
    * definition of the message in any great depth (but I do include
    * a structure definition), and I don't include the text of the messages
    * (since you can look at the mc file for that). */

   strcpy(cp, ".h");
   fp = fopen(fname, "w");
   if (fp) {
      fprintf(fp, "/* generated from %s by ptjmc */\n\n"
                  "#pragma pack(1)\nstruct msgfmt_T {\n"
                  "  unsigned short id;\n"
                  "  unsigned char facility;\n"
                  "  unsigned char reserved : 1, customer : 1, severity : 2;\n"
                  "};\n#pragma pack()\n\n", name);

      for (i = j = 0; i < msgcount; i++) {
         if (msgs[i].symname) {
            fprintf(fp, "#define %-*s 0x%08x\n", lsymname, msgs[i].symname, msgs[i].ind.sort);
            j++;
         }
      }

      if (!j) {
         fprintf(fp, "/* no messages of SymbolicName entry */\n");
      }

      fclose(fp);
   }
   else {
      perror(fname);
   }

   strcpy(cp, ".rc");
   
   fp = fopen(fname, "w");
   if (fp) {
      strcpy(cp, ".bin");

      fprintf(fp, "/* generated from %s by ptjmc */\n\n"
                  "LANGUAGE 0x9,0x1\n"
                  "1 11 %s\n\n", name, fname);

      fclose(fp);
   }
   else {
      perror(fname);
   }

   strcpy(cp, ".bin");
   fp = fopen(fname, "wb");
   if (fp) {
      fwrite(&pairc, sizeof(pairc), 1, fp);
      fwrite(pairs, sizeof(*pairs), pairc, fp);

      for (i = 0; i < msgcount; i++) {
         msgoff = msgs[i].len + sizeof(int);

         if (msgoff % 4)
            while (msgoff % 4)
               msgoff++;
         else
            msgoff += 4;

         fwrite(&msgoff, sizeof(msgoff), 1, fp);
         fwrite(msgs[i].text, 1, msgs[i].len, fp);
         fwrite("\0\0\0", 1, msgoff - sizeof(int) - msgs[i].len, fp);
      }


      fclose(fp);
   }
   else {
      perror(fname);
   }


   for (i = 0; i < msgcount; i++) {
      free(msgs[i].text);
      if (msgs[i].symname)
         free(msgs[i].symname);
   }

   free(msgs);
   msgs = NULL;
   msgcount = 0;

   for (i = 0; i < sevcount; i++) {
      free(sevs[i].text);
   }

   if (sevs)
      free(sevs);
   sevs = NULL;
   sevcount = 0;

   for (i = 0; i < faccount; i++) {
      free(facs[i].text);
   }

   if (facs)
      free(facs);
   facs = NULL;
   faccount = 0;

   free(pairs);
}


main(int ac, char **av)
{
   if (ac != 2) {
      fprintf(stderr, "usage: %s file\n", av[0]);
      return 1;
   }      

   yyin = fopen(av[1], "r");
   if (!yyin) {
      perror(av[1]);
      return 1;
   }

   yylex();

   fclose(yyin);

   emitstuff(av[1]);

   return 0;
}

static int yywrap()
{
   return 1;
}
