/* clipboard functions:
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is w32funcs.
 *
 * The Initial Developer of the Original Code is Patrick TJ McPhee.
 * Portions created by Patrick McPhee are Copyright 2003
 * Patrick TJ McPhee. All Rights Reserved.
 *
 * Contributors:
 *
 * $Header: C:/ptjm/rexx/w32funcs/RCS/clipboard.c 1.3 2003/02/27 18:36:37 ptjm Rel $
 */

/* functions defined in this file:
 * w32clipopen()
 * w32clipclose()
 * w32clipgetstem(stemname)
 * w32clipget(format)
 * w32clipsetstem(stemname)
 * w32clipset(value[, format])
 * w32clipregisterformat(format)
 * w32clipenumformat(stem)
 * w32clipformatname(formatid)
 * w32clipempty()
 */ 

#include "rxproto.h"
#include <windows.h>

static int ocb(int open)
{
   static int opencount;
   int rc = TRUE;

   if (open && !opencount++) {
      rc = OpenClipboard(NULL);
   }
   else if (!--opencount) {
      rc = CloseClipboard();
   }

   /* try to avoid issues with excess calls to w32clipclose */
   if (opencount < 0)
      opencount = 0;

   return rc;
}


struct formatid_T {
   int idno;
   char * s;
};



static int formatcmp(const char * s, const struct formatid_T * idt)
{
   return strcmp(s, idt->s);
}



/* given a string with a format id, convert it to an integer. The id can be
 * either a number or a name. If the id is a name, it's either well-known
 * to this library, previously registered, or we will register it. */
static int formatid(PRXSTRING id)
{
   char * s;
   int iid;

   rxstrdup(s, id[0]);
   strupr(s);

   iid = atoi(s);

   if (iid != 0)
      return iid;
   
   else {
      static struct formatid_T stdformat[] = {
         CF_BITMAP, "BITMAP",
         CF_DIB, "DIB",
         CF_DIF, "DIF",
         CF_DSPBITMAP, "DSPBITMAP",
         CF_DSPENHMETAFILE, "DSPENHMETAFILE",
         CF_DSPMETAFILEPICT, "DSPMETAFILEPICT",
         CF_DSPTEXT, "DSPTEXT",
         CF_ENHMETAFILE, "ENHMETAFILE",
         CF_HDROP, "HDROP",
         CF_LOCALE, "LOCALE",
         CF_MAX, "MAX",
         CF_METAFILEPICT, "METAFILEPICT",
         CF_OEMTEXT, "OEMTEXT",
         CF_OWNERDISPLAY, "OWNERDISPLAY",
         CF_PALETTE, "PALETTE",
         CF_PENDATA, "PENDATA",
         CF_RIFF, "RIFF",
         CF_SYLK, "SYLK",
         CF_TEXT, "TEXT",
         CF_TIFF, "TIFF",
         CF_UNICODETEXT, "UNICODETEXT",
         CF_WAVE, "WAVE"
      };
      struct formatid_T * res = bsearch(s, stdformat, DIM(stdformat), sizeof(*stdformat), formatcmp);

      if (res)
         iid = res->idno;
     else
        iid = RegisterClipboardFormat(s);
   }

   return iid;
}




/* load current text clipboard into a stem */
rxfunc(w32clipgetstem)
{
   HANDLE clip;
   unsigned char * b, * f, *l;
   chararray * ca;

   checkparam(1,1);

   if (!ocb(1)) {
      result_zero();
      return 0;
   }

   clip = GetClipboardData(CF_TEXT);
   ocb(0);

   if (!clip) {
      result_zero();
      return 0;
   }

   ca = new_chararray();

   b = GlobalLock(clip);
   for (f = b, l = strchr(f, '\r'); f && l; f = l + 2, l = strchr(f, '\r')) {
      cha_adddummy(ca, f, l - f);
   }

   /* there is remaining data without a cr/lf */
   if (f && *f) {
      cha_adddummy(ca, f, strlen(f));
   }



   setastem(argv, ca);
   GlobalUnlock(clip);

   delete_chararray(ca);
   result_one();

   return 0;
}

/* retrieve the clip-board data in the specified format */
rxfunc(w32clipget)
{
   HANDLE clip;
   unsigned char * b;
   int bs, fmt;

   checkparam(0,1);
   
   if (!ocb(1)) {
      result->strlength = 0;
      return 0;
   }

   if (argc == 1)
      fmt = formatid(argv);
   else
      fmt = CF_TEXT;

   clip = GetClipboardData(fmt);
   ocb(0);

   if (!clip) {
      result->strlength = 0;
      return 0;
   }

   b = GlobalLock(clip);
   bs = GlobalSize(clip);

   if (bs > DEFAULTSTRINGSIZE) {
      rxresize(result, bs);
   }

   result->strlength = bs;
   memcpy(result->strptr, b, bs);

   GlobalUnlock(clip);

   return 0;
}

/* set the clip-board data in the specified format */
rxfunc(w32clipsetstem)
{
   HANDLE clip;
   unsigned char * b;
   unsigned int bs;
   register int i;
   chararray * ca;

   checkparam(1, 1);
   
   ca = new_chararray();
   getastem(argv, ca);

   /* calculate the length of the buffer -- we need space for each stem
    * variable, plus two for cr-lf */
   for (bs = i = 0; i < ca->count; i++) {
      bs += ca->array[i].strlength + 2;
   }

   clip = GlobalAlloc(GMEM_MOVEABLE|GMEM_DDESHARE, bs+1);

   if (!clip) {
      result_zero();
      return 0;
   }


   b = GlobalLock(clip);
   for (bs = i = 0; i < ca->count; i++) {
      memcpy(b+bs, ca->array[i].strptr, ca->array[i].strlength);
      memcpy(b+bs+ca->array[i].strlength, "\r\n", 2);
      bs += ca->array[i].strlength + 2;
   }

   b[bs] = 0;
   GlobalUnlock(clip);

   if (!ocb(1)) {
      result_zero();
      return 0;
   }

   b = SetClipboardData(CF_TEXT, clip);

   if (b) {
      result_zero();
   }
   else {
      result->strlength = sprintf(result->strptr, "%d", GetLastError());
   }


   ocb(0);


   return 0;
}

/* set the clip-board data in the specified format */
rxfunc(w32clipset)
{
   HANDLE clip;
   unsigned char * b;
   int fmtid;
   int rc;

   checkparam(1,2);
   
   if (argc > 1) {
      fmtid = formatid(argv+1);
   }
   else {
      fmtid = CF_TEXT;
   }

   clip = GlobalAlloc(GMEM_MOVEABLE|GMEM_DDESHARE, argv[0].strlength+2);

   if (!clip) {
      result_zero();
      return 0;
   }


   /* copy the value into a global. Double null-terminate it because it makes
    * life easier for the caller in cases where null termination is
    * required. */
   b = GlobalLock(clip);
   memcpy(b, argv[0].strptr, argv[0].strlength);
   memcpy(b+argv[0].strlength, "\0", 2);
   GlobalUnlock(clip);

   if (!ocb(1)) {
      result->strlength = 0;
      return 0;
   }

   b = SetClipboardData(fmtid, clip);

   if (b) {
      result_zero();
   }
   else {
      result->strlength = sprintf(result->strptr, "%d", GetLastError());
   }

   ocb(0);

   return 0;
}

rxfunc(w32clipopen)
{
   result->strlength = sprintf(result->strptr, "%d", ocb(1));
   return 0;
}

rxfunc(w32clipclose)
{
   result->strlength = sprintf(result->strptr, "%d", ocb(0));
   return 0;
}

rxfunc(w32clipregisterformat)
{
   char * s;

   checkparam(1, 1);

   rxstrdup(s, argv[0]);

   if (!ocb(1)) {
      result_zero();
      return 0;
   }

   result->strlength = sprintf(result->strptr, "%d", RegisterClipboardFormat(s));

   ocb(0);

   return 0;
}

rxfunc(w32clipenumformat)
{
   chararray * ca;
   unsigned int fmt = 0;
   char buf[11];
   int bufl;

   checkparam(1, 1);

   if (!ocb(1)) {
      result_zero();
      return 0;
   }

   ca = new_chararray();

   while ((fmt = EnumClipboardFormats(fmt)) > 0) {
      bufl = sprintf(buf, "%d", fmt);
      cha_addstr(ca, buf, bufl);
   }

   ocb(0);


   setastem(argv, ca);

   delete_chararray(ca);

   result_one();

   return 0;
}

rxfunc(w32clipformatname)
{
   char * fids;
   int lb = DEFAULTSTRINGSIZE;
   int fmtid;

   checkparam(1, 1);

   if (!ocb(1)) {
      result->strlength = 0;
      return 0;
   }

   fmtid = formatid(argv);

   /* keep retrying if the buffer is too small */
   while ((result->strlength = GetClipboardFormatName(fmtid, result->strptr, lb)) == lb) {
      lb += lb;
      rxresize(result, lb);
   }

   if (result->strlength == 0) {
      register int i;
      static struct formatid_T stdformat[] = {
         CF_BITMAP, "BITMAP",
         CF_DIB, "DIB",
         CF_DIF, "DIF",
         CF_DSPBITMAP, "DSPBITMAP",
         CF_DSPENHMETAFILE, "DSPENHMETAFILE",
         CF_DSPMETAFILEPICT, "DSPMETAFILEPICT",
         CF_DSPTEXT, "DSPTEXT",
         CF_ENHMETAFILE, "ENHMETAFILE",
         CF_HDROP, "HDROP",
         CF_LOCALE, "LOCALE",
         CF_MAX, "MAX",
         CF_METAFILEPICT, "METAFILEPICT",
         CF_OEMTEXT, "OEMTEXT",
         CF_OWNERDISPLAY, "OWNERDISPLAY",
         CF_PALETTE, "PALETTE",
         CF_PENDATA, "PENDATA",
         CF_RIFF, "RIFF",
         CF_SYLK, "SYLK",
         CF_TEXT, "TEXT",
         CF_TIFF, "TIFF",
         CF_UNICODETEXT, "UNICODETEXT",
         CF_WAVE, "WAVE"
      };
      
      for (i = 0; i < DIM(stdformat); i++) {
         if (stdformat[i].idno == fmtid) {
            strcpy(result->strptr, stdformat[i].s);
            result->strlength = strlen(result->strptr);
            break;
         }
      }
   }

   ocb(0);

   return 0;
}

rxfunc(w32clipempty)
{
   checkparam(0, 0);

   if (!ocb(1)) {
      result_zero();
      return 0;
   }

   result->strlength = sprintf(result->strptr, "%d", EmptyClipboard());

   ocb(0);


   return 0;
}

rxfunc(w32cliptestformat)
{
   chararray * ca;
   unsigned int *fmts;
   register int i;

   checkparam(1, 1);

   if (!ocb(1)) {
      result_zero();
      return 0;
   }

   ca = new_chararray();

   getastem(argv, ca);

   fmts = alloca(sizeof(*fmts) * ca->count);

   for (i = 0; i < ca->count; i++) {
      fmts[i] = formatid(ca->array+i);
   }

   result->strlength = sprintf(result->strptr, "%d", GetPriorityClipboardFormat(fmts, ca->count));

   ocb(0);

   return 0;
}

