/**
 * PciAc5.c
 *
 * Copyright (c) 2013-2018 David Azarewicz david@88watts.net
 *
 * The following source code is provided to you solely for the purpose of
 * assisting you in developing your own OS/2 device drivers. You may use
 * this software in your device drivers free of charge.
 */
#include "Dev32lib.h"
#include <bsekee.h>
#include "Dev32rmbase.h"
#include "ioctl.h"
#include "version.h"

/* global variables */
char *BaseAdr;
char cDevName[9] = "PCIAC5$";
int iVerbose;

static DRIVERSTRUCT drs =
{
  NULL, /* We cannot do Flat to Far16 conversion at compile time */
  NULL, /* so we put NULLs in all the Far16 fields and then fill */
  NULL, /* them in at run time                                   */
  DVERMAJOR,
  DVERMINOR,
  DYEAR, DMONTH, DDAY,
  DRF_STATIC,
  DRT_UNDEFINED,
  DRS_UNDEFINED,
  NULL
};

static ADJUNCT adj =
{
  NULL, sizeof(ADJUNCT), ADJ_ADAPTER_NUMBER
};

static ADAPTERSTRUCT ads =
{
  NULL,
  AS_NO16MB_ADDRESS_LIMIT,
  AS_BASE_COMM,
  AS_SUB_OTHER,
  AS_INTF_GENERIC,
  AS_HOSTBUS_PCI,
  AS_BUSWIDTH_32BIT,
  NULL,
  0
};

/**
 * Return information from this driver
 */
void IoctlInfo(AC5INFO *pInfo)
{
  pInfo->Data = 0x11223344;
}

/**
 * Read data from an IO port
 */
USHORT IoctlIoRW(AC5INFO *pInfo)
{
  USHORT usStatus;

  usStatus = 0;

  switch (pInfo->Command)
  {
  case IOCTL_FCN_READ_DATA_0:
    pInfo->Data = ReadDword(BaseAdr+0x00000300);
    break;
  case IOCTL_FCN_READ_DATA_1:
    pInfo->Data = ReadDword(BaseAdr+0x00000700);
    break;
  case IOCTL_FCN_WRITE_DATA_0:
    WriteDword(BaseAdr+0x00000008, pInfo->Data);
    break;
  case IOCTL_FCN_WRITE_DATA_1:
    WriteDword(BaseAdr+0x00000018, pInfo->Data);
    break;
  case IOCTL_FCN_READ_CONFIG_0:
    pInfo->Data = ReadDword(BaseAdr+0x00000100);
    break;
  case IOCTL_FCN_READ_CONFIG_1:
    pInfo->Data = ReadDword(BaseAdr+0x00000500);
    break;
  case IOCTL_FCN_WRITE_CONFIG_0:
    WriteDword(BaseAdr+0x00000004, pInfo->Data);
    break;
  case IOCTL_FCN_WRITE_CONFIG_1:
    WriteDword(BaseAdr+0x00000014, pInfo->Data);
    break;
  default:
    usStatus = RPERR | RPERR_BADCOMMAND;
    break;
  }

  return usStatus;
}

void StrategyIoctl(REQPACKET *prp)
{
  if (prp->ioctl.bCategory != IOCTL_CATEGORY) {
    prp->usStatus |= RPERR | RPERR_BADCOMMAND;
    return;
  }

  if (prp->ioctl.usDataLen != sizeof(AC5INFO)) {
    prp->usStatus |= RPERR | RPERR_BADCOMMAND;
    return;
  }

  switch (prp->ioctl.bFunction) {
  case IOCTL_FUNCTION_INFO:
    IoctlInfo(prp->ioctl.pvData); /* Far16 to Flat conversion is done by the compiler */
    /* The above statement could also be written with a manual pointer conversion as
     * shown below with the exact same result.
     * IoctlInfo(Far16ToFlat(prp->ioctl.pvData));
     */
    break;
  case IOCTL_FUNCTION_IORW:
    prp->usStatus |= IoctlIoRW(prp->ioctl.pvData); /* Far16 to Flat conversion is done by the compiler */
    break;
  default:
    prp->usStatus |= RPERR | RPERR_BADCOMMAND;
    return;
  }
}

int ParseCmdParms(char *pszCmdLine) {
  int iStatus;

  if (!pszCmdLine) return 0;

  iStatus = 0;
  while (*pszCmdLine)
  {
    if (*pszCmdLine++ != '/') continue; /* Ignore anything that doesn't start with '/' */
    /* pszCmdLine now points to first char of argument */

    if (ArgCmp(pszCmdLine, "V"))
    {
      pszCmdLine += 1;
      iVerbose = 1;
      continue;
    }
    iprintf("Unrecognized switch: %s\n", pszCmdLine-1);
    iStatus = 1; /* unrecognized argument */
  }

  return(iStatus);
}

void StrategyInit(REQPACKET *prp)
{
  int rc;
  short Success;
  static PCI_DEVICEINFO PciDevInfo;

  UtSetDriverName(cDevName);
  /* Process command line parameters. */
  ParseCmdParms(prp->init_in.szArgs); /* Far16 to Flat conversion is done by the compiler */

  Success = 0;

  do {
    int iAdapterNumber;
    USHORT PciBusDevFunc;

    /* Create the resource manager entry for this driver. */
    drs.DrvrName = DFILE; /* Flat to Far16 conversion is done by the compiler. */
    /* The above statement could also be written with a manual pointer conversion as
     * shown below with the exact same result.
     * drs.DrvrName = FlatToFar16(DFILE);
     */
    drs.DrvrDescript = "Pci Ac5 I/O Driver"; /* Flat to Far16 conversion is done by the compiler. */
    drs.VendorName = DVENDOR; /* Flat to Far16 conversion is done by the compiler. */
    if (Rm1CreateDriver(&drs)) break; /* fail if we can't. */

    /* Find an adapter we can control. Instead of controlling multiple
     * adapters in one driver, we only support one adapter per driver
     * and require multiple instances of the driver, one for each adapter.
     * We support up to 10 instances.
     */
    rc = 1; /* start with fail status */
    for (iAdapterNumber=0; iAdapterNumber<10; iAdapterNumber++)
    {
      if (iAdapterNumber)
      {
        UtModifyName(cDevName, iAdapterNumber, iAdapterNumber>1);
        UtSetDriverName(cDevName);
      }

      /* look for this specific vendor and device ID */
      PciBusDevFunc = PciFindDevice(0x148a, 0xac05, iAdapterNumber);
      if (PciBusDevFunc == 0xffff) break; /* can't find a device. Stop looking. */

      if (PciGetDeviceInfo(PciBusDevFunc, &PciDevInfo)) break; /* Stop if this fails */

      if (PciDevInfo.bars[0].io != 0) break; /* Stop if not memory address */

      /* Try to acquire the resources for this card. If acquiring a resourcd fails,
       * that means that another driver has already claimed that resource.
       */
      do {
        rc = Rm1AddMem(PciDevInfo.bars[0].start, PciDevInfo.bars[0].size);
        if (rc) break;

        rc = Rm1AddIrq(PciDevInfo.irq, PciDevInfo.ipin, RS_IRQ_SHARED);
        if (rc) break;

        adj.Adapter_Number = iAdapterNumber;
        ads.AdaptDescriptName = "PCIAC5_# OPTO22 IO adapter"; /* Flat to Far16 conversion is done by the compiler */
        ads.pAdjunctList = &adj; /* Flat to Far16 conversion is done by the compiler */
        rc = Rm1CreateAdapter(&ads);
      } while (0);

      if (!rc) break; /* We found an adapter */

      Rm1Destroy(0); /* Undo all the resources we might have added */
    }
    if (rc) break; /* couldn't find an adapter */

    if (iVerbose) iprintf("PCIAC5 found card at %lx irq %d\n", PciDevInfo.bars[0].start, PciDevInfo.irq);
    /* We found an adapter and allocated its resources. Now map the memory.
     * Only map 1 page even though the device reports a bigger size.
     */
    BaseAdr = MapPhysToLin(PciDevInfo.bars[0].start, 0x1000);
    if (!BaseAdr) break;

    Success = 1;
  } while (0);


  if (Success)
  {
    prp->usStatus = RPDONE;
    if (iVerbose) iprintf("PCIAC5 driver %s loaded successfully.\n", cDevName);
  }
  else
  {
    /* We failed to install. Undo everything we did. */
    Rm1Destroy(1);
    prp->usStatus = RPDONE | RPERR;
    if (iVerbose) iprintf("PCIAC5 initialization failed. %s not loaded.\n", cDevName);
  }
}

void StrategyHandler(REQPACKET *prp)
{
  prp->usStatus = RPDONE;

  switch (prp->bCommand) {
  case STRATEGY_INIT:
    StrategyInit(prp);
    break;
  case STRATEGY_OPEN:
     break;
  case STRATEGY_CLOSE:
     break;
  case STRATEGY_GENIOCTL:
    StrategyIoctl(prp);
    break;
  case STRATEGY_DEINSTALL:
    break;
  case STRATEGY_INITCOMPLETE:
    break;
  case STRATEGY_SAVERESTORE:
    break;
  default:
    prp->usStatus = RPDONE | RPERR | RPERR_GENERAL;
    break;
  }
}

