/* abrefr, a program for the calculation of refractive indices and critical
 * angles, Bragg angles, interplanar distances and angles between
 * crystallographic planes
 *
 * Copyright (C) 1995-1999 Petr Mikulik
 *
 * e-mail: mikulik@physics.muni.cz
 * web:    http://www.sci.muni.cz/~mikulik/
 *
*/

/* This is version 1.6, November 1999
 * The latest version of the program can be found on the above-mentioned
 * web page.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
*/

#define TheDate "November 1999"

#include <stdlib.h>
#include <stdio.h>
#include <fstream.h>
#include <iomanip.h>
#include <math.h>
#include <string.h>
#include <ctype.h>
#include "ansicpp.h"
#include "pmmath.h"
#include "pmcomplx.h"
#include "lindef3.h"
#include "mfcht.h"


#define bform(X)     form("%-1.6g",X)
#define tabbform(X)  form("\t%-1.6g",X)

char atomsf[128] = ""; // file with Int. Table

char debug = 0;

Matrix DirLat(3), RecLat(3), MatSS(3);
char DoRecLat = 0, DoPlaneAngles = 0;
char LatticeName[128], LatticeParams[128], ChemFormula[512];


void cin_pos ( double& a )
{
while (1) {
  cin >> a;
  if (!cin) {
    cin.clear();
    cin.ignore(32535,'\n');
    cout <<"ERROR: expecting real number. Try again: ";
    continue; }
  if (a > 0) return;
  cin.ignore(32535,'\n');
  cout <<"ERROR: positive number required. Try again: ";
  }
}


void cin_nonneg ( double& a )
{
while (1) {
  cin >> a;
  if (!cin) {
    cin.clear();
    cin.ignore(32535,'\n');
    cout <<"ERROR: expecting real number. Try again: ";
    continue; }
  if (a >= 0) return;
  cin.ignore(32535,'\n');
  cout <<"ERROR: positive number required. Try again: ";
  }
}


void cin_nonneg ( int& a )
{
while (1) {
  cin >> a;
  if (!cin) {
    cin.clear();
    cin.ignore(32535,'\n');
    cout <<"ERROR: expecting positive integer number. Try again: ";
    continue; }
  if (a >= 0) return;
  cin.ignore(32535,'\n');
  cout <<"ERROR: positive number required. Try again: ";
  }
}


void cin_hkl ( int& h, int& k, int& l )
{
while (1) {
  cin >> h >> k >> l;
  if (cin) return;
  cin.clear();
  cin.ignore(32535,'\n');
  cout <<"ERROR: expecting three integer numbers. Try again: ";
  }
}


int CalcASF ( char *Atom, double& asf, Complex& dc, double Lambda, double& mass )
// returns 0 on success
{
double tmp,tmp2; int i,k;
ifstream inp; inp.open(atomsf);
if (!inp) { cout <<"ERROR: cannot open " <<atomsf <<" !\n"; exit(1); }
char S[129];
do { inp >> S; } while (strcmp(S,"H")); // H is the 1st element
strupr(Atom);
while (strcmp(S,Atom)) { // find that element
  inp.ignore(32535,'\n'); if (inp) inp >> S;
  if (!inp) {
    cout <<"ERROR: properties of your strange atom are unknown.\n";
    return 1;
    // exit(1);
    }
  }
inp >> mass; inp.ignore(32535,'\n');
// now there should be  a1 b1 a2 b2 ... c
tmp = asf = 0;
for (i=1; i<=9; i++)
  { inp >> tmp ; if (odd(i)) asf += tmp; }
asf = int(asf+0.5); // it must be number of electrons, so round it

// now there should be  dispersion corrections for
//    CrKa      FeKa       CuKa      MoKa      AgKa

if (Lambda>(XrayLambda[CrKa1]+XrayLambda[FeKa1])/2)
  k = 0; /* use CrKa disp. corr. */ else
if (Lambda>(XrayLambda[FeKa1]+XrayLambda[CuKa1])/2)
  k = 2; /* use FeKa disp. corr. */ else
if (Lambda>(XrayLambda[CuKa1]+XrayLambda[MoKa1])/2)
  k = 4; /* use CuKa disp. corr. */ else
if (Lambda>(XrayLambda[MoKa1]+XrayLambda[AgKa1])/2)
  k = 6; /* use MoKa disp. corr. */ else
  k = 8; /* use AgKa disp. corr. */;
for (i=k; i--; ) inp >> tmp;
inp >> tmp >> tmp2;
dc = Complex( tmp, tmp2 );
inp.close();
return 0;
}


double Dhkl ( int H, int K, int L )
{
Vector B(3);
B = H*RecLat[1] + K*RecLat[2] + L*RecLat[3]; // Bhkl
return M_TWO_PI / B.length();
}


double BraggAngle ( double Lambda, int H,int K, int L, double& dhkl )
{
if (!H && !K && !L) return 0;
dhkl = Dhkl(H,K,L);
double X = Lambda / (2*dhkl);
if (fabs(X)>1) return 1000*M_PI;
return asin(X);
}


double AngleBetweenPlanes ( int H1, int K1, int L1, int H2, int K2, int L2)
{
double a,b,c;

#if 1  // version with reciprocal space vectors
if (debug==1) {
Vector B1(3), B2(3);
//cout <<" RecLat="<<RecLat<<'\n';
B1 = H1*RecLat[1] + K1*RecLat[2] + L1*RecLat[3]; // Bhkl
B2 = H2*RecLat[1] + K2*RecLat[2] + L2*RecLat[3]; // Bhkl
a = B1*B2;
//cout <<"B1="<<B1<<endl;
//cout <<"B2="<<B2<<endl;
b = B1.norm();
c = B2.norm();
double x = a/sqrt(b*c);
if (debug==1) cout <<"Testing purposes: first x=" <<x <<endl;
}
#endif

#if 1  // version with the matrix of the scalar product
#define M MatSS
a =(H1*M[1][1]+K1*M[1][2]+L1*M[1][3])*H2; /* scalar product Plane1*Plane2 */
a+=(H1*M[2][1]+K1*M[2][2]+L1*M[2][3])*K2;
a+=(H1*M[3][1]+K1*M[3][2]+L1*M[3][3])*L2;
b =(H1*M[1][1]+K1*M[1][2]+L1*M[1][3])*H1; /* scalar product Plane1*Plane1 */
b+=(H1*M[2][1]+K1*M[2][2]+L1*M[2][3])*K1;
b+=(H1*M[3][1]+K1*M[3][2]+L1*M[3][3])*L1;
c =(H2*M[1][1]+K2*M[1][2]+L2*M[1][3])*H2; /* scalar product Plane2*Plane2 */
c+=(H2*M[2][1]+K2*M[2][2]+L2*M[2][3])*K2;
c+=(H2*M[3][1]+K2*M[3][2]+L2*M[3][3])*L2;
#undef M
#endif
a /= sqrt(b*c);
if (debug==1) cout <<"Testing purposes: second x=" <<a <<endl;
return acos(a);
}


double CalcVel ()
{
double a,b,c, alfa,beta,gama, Vel;
char lattice;
do {
  cout <<"Please choose crystalline system: *\n";
  cout <<"  Cubic       Tetragonal  Orthorombic  Hexagonal \n";
  cout <<"  Monoclinic  tRigonal    trIclinic:   ";
  cin >> lattice; lattice = tolower(lattice);
  }
while (!strchr("citmorh",lattice));

strcpy(LatticeName,"Crystal system:\t");
switch (lattice) {
  case 'c': strcat(LatticeName,"Cubic"); break;
  case 't': strcat(LatticeName,"Tetragonal"); break;
  case 'o': strcat(LatticeName,"Orthorombic"); break;
  case 'h': strcat(LatticeName,"Hexagonal"); break;
  case 'm': strcat(LatticeName,"Monoclinic"); break;
  case 'r': strcat(LatticeName,"Trigonal"); break;
  case 'i': strcat(LatticeName,"Triclinic"); break;
  }

// I have to calculate Vel
cout <<"Enter lattice parameters (Angstroms, degrees):\n";
if (lattice=='c')
  cout <<"\tHint - some lattice parameters:\n"
       <<"\tSi: " <<lpSi <<"  GaAs: " <<lpGaAs <<"  GaP:  5.4512  InP: 5.8688\n"
       <<"\tGe: " <<lpGe <<"  AlAs: " <<lpAlAs <<"    InAs: 6.0584\n";
cout <<"a = ";
cin_pos( a );
c=b=a; gama=beta=alfa=90; Vel=0;
if (lattice=='c') Vel = a*a*a;
if (strchr("imo",lattice))
  { cout <<"b = "; cin_pos( b ); }
if (strchr("itmoh",lattice))
  { cout <<"c = "; cin_pos( c ); }
if (lattice=='t') Vel = a*a*c;
if (lattice=='o') Vel = a*b*c;
if (strchr("ir",lattice))
  { cout <<"alfa = "; cin_pos( alfa ); }
gama=beta=alfa;
if (strchr("i",lattice))
  { cout <<"beta = "; cin_pos( beta ); }
if (strchr("mi",lattice))
  { cout <<"gamma = "; cin_pos( gama ); }
if (lattice=='h') gama=120;

sprintf(LatticeParams,
  "a=%g A, b=%g A, c=%g A, alfa=%g deg, beta=%g deg, gama=%g deg",
  a,b,c,alfa,beta,gama);

alfa *= deg2rad; beta *= deg2rad; gama *= deg2rad;

// DirLatVect is the matrix of vectors of the direct lattice
// in the cartesian system

// c/=a; b/=a; a=1; //!!!
double c1 = c*sin(beta), c2 = c1*cos(alfa);
DirLat[1] = Vector( a,            0,            0            );
DirLat[2] = Vector( b*cos(gama),  b*sin(gama),  0            );
DirLat[3] = Vector( c2*cos(gama), c2*sin(gama), c1*sin(alfa) );
neglect_small( DirLat, 1e-6 );

// DirLat=transp3(DirLat);

// now volume:  Vel = (a x b).c
if (!Vel)
  Vel = vectproduct( DirLat[1], DirLat[2] ) * DirLat[3];

if (DoRecLat || DoPlaneAngles) { // reciprocal lattice:
  //cout<<" DirLat =\n"<<DirLat;
  RecLat = invmat3( DirLat );
  //cout<<" RecLat = invmat3( DirLat );\n"<<RecLat;
  RecLat = transp3( RecLat );
  //cout<<" RecLat = transp3( DirLat );\n"<<RecLat;
  RecLat *= M_TWO_PI; // matrix with rec. lat. vectors
  neglect_small(RecLat, 1e-6);
  //cout<<" RecLat = 2PI, neglected\n"<<RecLat;
  }
return Vel;
}


void DoBraggAngles ( double Lambda )
{
double B,D; int H,K,L; char c;
cout <<"Remark: this calculates only the Bragg angle, not the extinction rules.\n";
while (1) {
  cin.clear();
  cout <<"H K L = ";
  cin_hkl( H, K, L );
  B = BraggAngle(Lambda,H,K,L,D);
  cout <<"Bragg angle for wavelength " << Lambda <<" A (energy "
       <<A2M/Lambda <<" eV): ";
  if (B<100) cout <<B*rad2deg <<DEG <<'\n';
    else cout <<"no diffraction\n";
  cout <<"inter-planar distance Dhkl = " <<D <<" A\n";
  cout <<"*** type L to choose another wavelength/energy, X to exit, otherwise continue: ";
  cin.ignore(32765,'\n');
  cin.get(c);
  if (toupper(c)=='X') exit(0);
  if (toupper(c)=='L') {
    while (1) {
      Lambda = ReadWaveLength(1);
      if (Lambda>0) break;
      cout <<"\nERROR: wavelength must be positive.\n";
      }
    }
  }
}


void CalcPlaneAngles()
{
int H1=0,K1=0,L1=1,H2=1,K2=0,L2=0;
Matrix Pom1(3),Recip(3);
double B; char c;
Pom1 = invmat3(DirLat);
Recip = transp3(Pom1); // Pom2 je ted matice B vektoru reciproke mrizky
MatSS = Recip*Pom1;
neglect_small(MatSS, 1e-6);

/*
cout<<"DirLat="<<DirLat<<endl;
cout<<"RecLat="<<RecLat<<endl;
cout<<"Recip="<<Recip<<endl;
cout<<"MatSS="<<MatSS<<endl;
*/

// => scalar product is: (N1,N2)=(H1 K1 L1)*B*Transpozice(B)*Transpozice(H2 K2 L2)
// angle between planes is now [h1 k1 l1]T a [h2 k2 l2]T

int P = 5;

while (1) {
  cin.clear();
  B = AngleBetweenPlanes(H1,K1,L1,H2,K2,L2);
  B *= rad2deg; if (B>90) B=180-B;
  cout <<"\n==================================================="
       <<"\n1 ... change plane (H1 K1 L1) = (" <<H1<<' ' <<K1 <<' ' <<L1 <<")"
       <<"\n2 ... change plane (H2 K2 L2) = (" <<H2<<' ' <<K2 <<' ' <<L2 <<")"
       <<"\n        => angle between them = " <<setprecision(P) <<B <<DEG
       <<"\nP ... set precision: " <<P
       <<"\nX ... exit\n"
       <<"\n  Your choice, please: ";
  cin >> c; c = toupper(c);
  switch (c) {
    case 'X': exit(0); break;
    case 'P': cout <<"Precision: "; cin_nonneg( P ); break;
    case '1': cout <<"New indices: "; cin_hkl( H1, K1, L1); break;
    case '2': cout <<"New indices: "; cin_hkl( H2, K2, L2); break;
    }
  }
}


void License ()
{
cout <<
"\n"
"   Copyright (C) 1995-1999 Petr Mikulik\n\n"
"   This program is free software; you can redistribute it and/or modify\n"
"   it under the terms of the GNU General Public License as published by\n"
"   the Free Software Foundation; either version 2, or (at your option)\n"
"   any later version.\n"
"\n"
"   This program is distributed in the hope that it will be useful,\n"
"   but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
"   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
"   GNU General Public License for more details.\n"
"\n"
"   You should have received a copy of the GNU General Public License\n"
"   along with this program; if not, write to the Free Software\n"
"   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.\n"
"\n";
exit(0);
}


// ************************* MAIN *****************************

int main ( int argc, char *argv[] )
{
cout <<"\n***** Program  abrefr  (version " <<TheDate <<") - (c) " <<Petr <<" *****\n";

if (argc < 2) {
  cout <<" Usage:\n"
       <<"   abrefr -r ... calculate refractive indices\n"
       <<"   abrefr -b ... calculate Bragg angles\n"
       <<"   abrefr -a ... calculate interplanar angles\n"
       <<"   abrefr -L ... display software license\n\n"
       <<"Author's web page: http://www.sci.muni.cz/~mikulik/\n";
  exit(0);
  }

DoRecLat = 0;
if (argc==3 && !strcmp(strlwr(argv[1]),"-d"))
  { debug=1; argc=2; }
if (argc==2) {
  if (!strcmp(strlwr(argv[1]),"-l")) License();
  if (!strcmp(strlwr(argv[1]),"-r")) ;
  else if (!strcmp(strlwr(argv[1]),"-a")) DoPlaneAngles = 1;
  else {
    if (strcmp(strlwr(argv[1]),"-b"))
      { cout <<"ERROR: unknown parameter " <<argv[1] <<endl; exit(1); }
    DoRecLat = 1;
    }
  }
if (argc>2) { cout <<"ERROR: too much params!\n"; exit(1); }

if (!DoRecLat) {
  char *p;
  #ifdef __MSDOS__
    #define SLASH '\\'
  #else
    #define SLASH '/'
  #endif
  #if __BORLANDC__
    p = strrchr(argv[0],SLASH);
    if (p) { *(p+1)=0; strcpy(atomsf,argv[0]); }
  #elif defined(__EMX__)
    _execname(atomsf,80);
    p = strrchr(argv[0],'\\');
    if (p) { *(p+1)=0; strcpy(atomsf,argv[0]); }
  #else
    p = strrchr(argv[0],SLASH);
    if (p) { *(p+1)=0; strcpy(atomsf,argv[0]); }
  #endif
  strcat(atomsf,"atomsf.dat"); // localize atomsf.dat file
  #undef SLASH
  }

double Vel = CalcVel();
cout <<"Volume of the elementary cell = " <<Vel <<" A^3\n\n";

if (DoPlaneAngles) CalcPlaneAngles();

double Lambda = ReadWaveLength(1);
while (Lambda <= 0) {
  cout <<"\nERROR: wavelength must be positive.\n\n";
  Lambda = ReadWaveLength(1);
  }

if (DoRecLat) DoBraggAngles(Lambda);

char Atom[30]; double natoms; // int n;
double Tasf=0, asf, Tmass=0, mass; Complex Tdc=0, dc=0;
ChemFormula[0] = 0;
do {
   cout <<"\nAtom (element)";
   if (Tasf) cout <<" [type . to end]";
   cout <<": "; cin >> Atom;
   if (!strcmp(Atom,"."))
     if (Tasf) break; else continue;
   if ( CalcASF( Atom, asf, dc, Lambda, mass ) ) continue;
   cout <<"     look: atomic scatt. factor = Z = " <<asf
	<<"\n           atomic mass              = " <<mass
	<<"\n           dispersion correction    = " <<dc
	<<"\nNumber of this atom in the cell: ";
   cin_nonneg( natoms );
   if (natoms > 0) {
     Tasf += natoms*asf; Tdc += natoms*dc; Tmass += natoms*mass;
     char ChemTmp[50]; sprintf(ChemTmp,"%s %g ",Atom,natoms);
     strcat(ChemFormula,ChemTmp);
     }
   }
while (1);

ostream *out; ofstream fout; Complex xTdc; double xTasf;

do { // repeat for given wavelengths
for (int k=3; --k; ) {
if (k==2) out=&cout; else
  { fout.open("abrefr.dat",ios::app);
    out = &fout; cout <<"Results saved in file abrefr.dat\n";
    *out <<"\n==========================================================\n"
	 <<"\nFormula:\t" <<ChemFormula <<'\n'
	 <<LatticeName
	 <<"\nLattice parameters:\n  "<<LatticeParams <<'\n';
  }
//*out<<"\n**********************************************************\n";
*out <<"\nWavelength:\t"<<setprecision(7)<<Lambda<<" A"
     <<"\tEnergy: " <<A2M/Lambda <<" eV\n"
     <<"\nElementary cell:"
     <<"\n       sum of atomic scatt. factors  = " <<Tasf
     <<"\n       sum of dispersion corrections = " <<Tdc
     <<"\n       electron density [1/A^3]      = " <<Tasf/Vel
     <<"\n       total relative mass           = " <<Tmass
     <<"\n       mass density [g/cm^3]         = " <<Tmass*1e27*m_unit/Vel
     <<'\n';
xTdc = Tdc;
xTdc += Tasf;
*out <<setprecision(8);
*out <<" => structure factor     = " <<xTdc <<'\n';

out->setf(ios::scientific);
*out <<setprecision(6);
xTdc *= -r_el*1e10*sqr(Lambda) / (M_PI*Vel);
*out <<" => susceptibility 0     = " <<xTdc <<'\n';
xTdc *= -0.5;
*out <<" => 1 - refractive_index = " <<xTdc <<'\n';
xTasf = sqrt( 2*real(xTdc) );
out->setf(ios::fixed);
*out <<setprecision(4);
*out <<" => critical angle       = " <<xTasf*rad2deg <<DEG <<endl;
}
fout.close();

cout <<"\n** New wavelength or energy? Enter zero for exiting from abrefr **\n";
Lambda = ReadWaveLength(1);
} while (Lambda>0);

cout << "\nProgram  abrefr  (version " << TheDate << ") - end.\n";
return 0;
}

// eof abrefr.cpp
