#include "CImageCV.h"

#include <QtGui/QPaintEvent>
#include <QtGui/QMouseEvent>
#include <QtGui/QPainter>
#include <QtGui/QRgb>
#include <QtGui/QPen>
#include <math.h>
#include <QTime>
#include <QtGui/QCursor>
#include <QtGui/QPrinter>
#include <QtGui/QPrintDialog>

/*!
 \brief конструктор для отображения и управления моделью

 \param width_ ширина для инициализации
 \param height_ высота для инициализации
*/
CImageCV::CImageCV(int width_, int height_)
{
    imageModel = new CImageModel(width_, height_);
    resize(imageModel->getImage()->size());
    penWidth = 1;
    actions = zoomFactor = 0;
    brushColor = Qt::white;
    penColor = Qt::black;
    activeTool = NONE;
    loupeFlag = modifiedFlag = undoFlag
            = redoFlag = isSelecting = isDraging = false;
    stored = 50;
    displacePoint = QPoint(0, 0);
    rememberChanges();
    defaultCursor = new QCursor(this->cursor());
}

/*!
 \brief деструктор

*/
CImageCV::~CImageCV()
{
    delete imageModel;
    delete defaultCursor;
}

/*!
 \brief метод для получения текущего размера истории

 \return int размер
*/
int CImageCV::getStored()
{
    return stored;
}

/*!
 \brief метод для установки размера истории

 \param size устанавливаемый размер
*/
void CImageCV::setStored(int size)
{
    while(actionsHistory.size()>size)
    {
        actionsHistory.remove(0);
        actions--;
    }
    if(actionsHistory.size()>size)
        actionsHistory.resize(size);
    stored = size;
}

/*!
 \brief метод возращающий флаг изменилось изображение или нет

 \return bool флаг возрата
*/
bool CImageCV::getModifiedFlag()
{
    return modifiedFlag;
}

/*!
 \brief установка флага изменилось изображение или нет

 \param flag флаг для установка
*/
void CImageCV::setModifiedFlag(bool flag)
{
    modifiedFlag = flag;
}

/*!
 \brief вспомогательный метод для посылка сигнала

*/
void CImageCV::setURButtons()
{
    emit sendURButton(undoFlag, redoFlag);
}

/*!
 \brief метод для сохранения изображений в массив
        для обработки undo/redo

*/
void CImageCV::rememberChanges()
{
    QImage *tempImage = new QImage;
    if(loupeFlag)
        *tempImage = imageModel->getImage()->scaled(imageModel->getImage()->size() / 4);
    else
        *tempImage = imageModel->getImage()->scaled(imageModel->getImage()->size() / pow(1.2, zoomFactor));
    if(actions != actionsHistory.size())
        actionsHistory.resize(actions);
    actionsHistory.push_back(*tempImage);
    delete tempImage;
    actions++;

    if(actionsHistory.size() > stored)
    {
        actionsHistory.remove(0);
        actions--;
    }
    if(actions > 1)
    {
        emit sendURButton(true, false);
        undoFlag = true;
        redoFlag = false;
    }
}

/*!
 \brief метод реализующий отмену действия

*/
void CImageCV::setPreviousAction()
{
    if(actions > 1)
    {
        QImage *tempImage = new QImage;
        clickToInsert();
        actions--;
        *tempImage = actionsHistory[actions - 1];
        if(loupeFlag)
            imageModel->setImage(tempImage->scaled(tempImage->size() * 4));
        else
            imageModel->setImage(tempImage->scaled(tempImage->size() * pow(1.2, zoomFactor)));
        delete tempImage;
    }
    resize(imageModel->getImage()->size());
    update();
    if(actions <= 1)
    {
        emit sendURButton(false, true);
        undoFlag = false;
        redoFlag = true;
    }
    else
    {
        emit sendURButton(true, true);
        undoFlag = redoFlag = true;
    }
}

/*!
 \brief метод реадизующий повтор действия

*/
void CImageCV::setNextAction()
{
    if(actions < actionsHistory.size())
    {
        QImage *tempImage = new QImage;
        clickToInsert();
        actions++;
        *tempImage = actionsHistory[actions - 1];
        if(loupeFlag)
            imageModel->setImage(tempImage->scaled(tempImage->size() * 4));
        else
            imageModel->setImage(tempImage->scaled(tempImage->size() * pow(1.2, zoomFactor)));
        delete tempImage;
        resize(imageModel->getImage()->size());
        update();
    }
    if(actionsHistory.size() > actions)
    {
        emit sendURButton(true, true);
        undoFlag = redoFlag = true;
    }
    else
    {
        emit sendURButton(true, false);
        undoFlag = true;
        redoFlag = false;
    }
}

/*!
 \brief метод для установки цвета пера

 \param color цвет для установки
*/
void CImageCV::setPenColor(const QColor &color)
{
    penColor = color;
}

/*!
 \brief метод для установки цвета заливки

 \param color цвет для установки
*/
void CImageCV::setBrushColor(const QColor &color)
{
    brushColor = color;
}

/*!
 \brief метод для утсановки ширины пера

 \param size значение для установки
*/
void CImageCV::setPenTrickness(const int &size)
{
    penWidth = size;
}

/*!
 \brief метод для установки активного инструмента

 \param tool значение для установки
*/
void CImageCV::setActiveTool(const ETools &tool)
{
    activeTool = tool;
    isDraging = false;
    switch(activeTool)
    {
    case NONE:
        this->setCursor(*defaultCursor);
        displacePoint = QPoint(0, 0);
        break;
    case PEN:
        this->setCursor(QCursor(QPixmap(":/gadgets/icons/gadgets/pencil.png")));
        displacePoint = QPoint(-8, 7);
        break;
    case LASTIC:
        this->setCursor(QCursor(QPixmap(":/gadgets/icons/gadgets/lastic.png")));
        displacePoint = QPoint(-4, 6);
        break;
    case LINE:
        this->setCursor(QCursor(QPixmap(":/cursors/icons/cursors/cross.png")));
        displacePoint = QPoint(-1, -1);
        break;
    case RECTANGLE: case ELLIPSE: case SELECT:
        this->setCursor(QCursor(QPixmap(":/cursors/icons/cursors/cross.png")));
        displacePoint = QPoint(-2, -2);
        break;
    case SPRAY:
        this->setCursor(QCursor(QPixmap(":/gadgets/icons/gadgets/spray.png")));
        displacePoint = QPoint(-20, -12);
        break;
    case PIPET:
        this->setCursor(QCursor(QPixmap(":/gadgets/icons/gadgets/pipette.png")));
        displacePoint = QPoint(-8, 7);
        break;
    case FILL:
        this->setCursor(QCursor(QPixmap(":/gadgets/icons/gadgets/fill.png")));
        displacePoint = QPoint(6, 5);
        break;
    case LOUPE:
        this->setCursor(QCursor(QPixmap(":/gadgets/icons/gadgets/loupe.png")));
        displacePoint = QPoint(0, 0);
        break;
    case HAND:
        this->setCursor(QCursor(QPixmap(":/gadgets/icons/gadgets/hand.png")));
        displacePoint = QPoint(0, 0);
        break;
    }
}

/*!
 \brief метод позволяющий изменять необходимую часть
        изображения вместо перерисовки всего изображения

*/
void CImageCV::backInTime()
{
    QPainter painter(imageModel->getImage());
    int rad(penWidth + 5);
    if(activeTool == ELLIPSE)
        rad += round(abs(startPoint.x() - endPoint.x()) / 3);
    QRect rect(QRect(startPoint, endPoint).normalized().adjusted(-rad, -rad, +rad, +rad));
    painter.drawImage(rect, bufImage, rect);
    update(QRect(startPoint, endPoint).normalized().adjusted(-rad, -rad, +rad, +rad));
    painter.end();
}

/*!
 \brief метод для обработки инструмента карандаш\линия

*/
void CImageCV::drawLine(bool flag)
{
    QPainter painter(imageModel->getImage());
    if(flag)
        painter.setPen(QPen(brushColor, penWidth, Qt::SolidLine, Qt::RoundCap, Qt::RoundJoin));
    else
        painter.setPen(QPen(penColor, penWidth, Qt::SolidLine, Qt::RoundCap, Qt::RoundJoin));
    if(startPoint != endPoint)
        painter.drawLine(startPoint, endPoint);
    if(activeTool == PEN && startPoint == endPoint)
    {
        if(penWidth > 1)
        {
            painter.setPen(QPen(penColor, round(penWidth/2), Qt::SolidLine, Qt::RoundCap, Qt::RoundJoin));
            painter.drawRect(QRect(startPoint, endPoint));
        }
        else
            painter.drawPoint(startPoint);
    }
    modifiedFlag = true;
    int rad(penWidth + round(sqrt((startPoint.x()-endPoint.x())*(startPoint.x()-endPoint.x())
                                  + (startPoint.y()-endPoint.y())*(startPoint.y()-endPoint.y()))));
    update(QRect(startPoint, endPoint).normalized().adjusted(-rad, -rad, +rad, +rad));
    painter.end();
}

/*!
 \brief метод для обработки инструмента пипетка

 \return QColor полученный цвет после отработки
*/
QColor CImageCV::drawPipet()
{
    if(startPoint.x() < 0 || startPoint.y() < 0
            || startPoint.x() > imageModel->getImage()->width()
            || startPoint.y() > imageModel->getImage()->height())
        return penColor;
    QRgb pixsel(imageModel->getImage()->pixel(startPoint));
    QColor getColor;
    getColor.setRgb(pixsel);
    return getColor;
}

/*!
 \brief метод для обработки инструмента прямоугольник

 \param flag индикатор для обозначения: рисовать с заливкой или нет
*/
void CImageCV::drawRectangle(bool flag)
{
    QPainter painter(imageModel->getImage());
    painter.setPen(QPen(penColor, penWidth, Qt::SolidLine, Qt::RoundCap, Qt::RoundJoin));
    if(flag)
        painter.setBrush(QBrush(brushColor));
    if(startPoint != endPoint)
        painter.drawRect(QRect(startPoint, endPoint));
    modifiedFlag = true;
    int rad(penWidth + 5);
    update(QRect(startPoint, endPoint).normalized().adjusted(-rad, -rad, +rad, +rad));
    painter.end();
}

/*!
 \brief метод для обработки инструмента эллипс

 \param flag индикатор для обозначения: рисовать с заливкой или нет
*/
void CImageCV::drawEllipse(bool flag)
{
    QPainter painter(imageModel->getImage());
    painter.setPen(QPen(penColor, penWidth, Qt::SolidLine, Qt::RoundCap, Qt::RoundJoin));
    if(flag)
        painter.setBrush(QBrush(brushColor));
    if(startPoint != endPoint)
        painter.drawEllipse(QRect(startPoint, endPoint));
    modifiedFlag = true;
    int rad(penWidth + 5);
    update(QRect(startPoint, endPoint).normalized().adjusted(-rad, -rad, +rad, +rad));
    painter.end();
}

/*!
 \brief метод для обработки инструмента заливка

 \param flag индикатор для обозначения какой цвет
        penColor или brushColor использовать для заливки
*/
void CImageCV::drawFill(bool flag)
{
    QColor switchColor, oldColor;
    if(!flag)
        switchColor = penColor;
    else
        switchColor = brushColor;
    QRgb pixsel(imageModel->getImage()->pixel(startPoint));
    oldColor.setRgb(pixsel);
   // QImage *tempImage = new QImage();

  /*  if(loupeFlag)
        *tempImage = imageModel->getImage()->scaled(imageModel->getImage()->size() / 4);
    else
        *tempImage = imageModel->getImage()->scaled(imageModel->getImage()->size() / pow(1.2, zoomFactor));
*/
    emit sendWait(true);
    if(switchColor != oldColor)
        drawFillRecurs(startPoint.x(), startPoint.y(), switchColor, oldColor, *imageModel->getImage());
    emit sendWait(false);

    /*if(loupeFlag)
        imageModel->setImage(tempImage->scaled(tempImage->size() * 4));
    else
        imageModel->setImage(tempImage->scaled(tempImage->size() * pow(1.2, zoomFactor)));
    delete tempImage;*/
    modifiedFlag = true;
}

/*!
 \brief метод для обработки инструмента заливка
        рекурсивный, вызывается из darwFill(bool)

 \param x координата пикселя
 \param y координата пикселя
 \param switchColor цвет для заливки
 \param oldColor старый цвет для сравнения
 \param tempImage обрабатываемое изображения
*/
void CImageCV::drawFillRecurs(int x, int y,
                                 QColor switchColor, QColor oldColor,
                                 QImage& tempImage)
{
    int temp_x(x), left_x(0);
    while(1)
    {
        QRgb pixsel(tempImage.pixel(temp_x, y));
        QColor currentColor;
        currentColor.setRgb(pixsel);
        if(currentColor != oldColor)
            break;
        pixsel = switchColor.rgb();
        tempImage.setPixel(temp_x, y, pixsel);
        if(temp_x > 0)
        {
            temp_x--;
            left_x = temp_x;
        }
        else
            break;
    }

    int right_x(0);
    temp_x = x + 1;
    while(1)
    {
        QRgb pixsel(tempImage.pixel(temp_x, y));
        QColor currentColor;
        currentColor.setRgb(pixsel);
        if(currentColor != oldColor)
            break;
        pixsel = switchColor.rgb();
        tempImage.setPixel(temp_x, y, pixsel);
        if(temp_x < tempImage.width() - 1)
        {
            temp_x++;
            right_x = temp_x;
        }
        else
            break;
    }

    for(int x_(left_x+1); x_ < right_x; x_++)
    {
        if(y < 1)
            break;
        if(right_x > tempImage.width())
            break;
        QRgb pixsel(tempImage.pixel(x_, y-1));
        QColor currentColor;
        currentColor.setRgb(pixsel);
        if(currentColor == oldColor && currentColor != switchColor)
            drawFillRecurs(x_, y-1, switchColor, oldColor, tempImage);
    }
    for(int x_(left_x+1); x_ < right_x; x_++)
    {
        if(y >= tempImage.height() - 1)
            break;
        if(right_x > tempImage.width())
            break;
        QRgb pixsel(tempImage.pixel(x_, y+1));
        QColor currentColor;
        currentColor.setRgb(pixsel);
        if(currentColor == oldColor && currentColor != switchColor)
            drawFillRecurs(x_, y+1, switchColor, oldColor, tempImage);
    }
}

/*!
 \brief метод для обработки инструмента спрей

*/
void CImageCV::drawSpray(bool flag)
{
    QPainter painter(imageModel->getImage());
    if(flag)
        painter.setPen(QPen(brushColor, sqrt(penWidth), Qt::SolidLine, Qt::RoundCap, Qt::RoundJoin));
    else
        painter.setPen(QPen(penColor, sqrt(penWidth), Qt::SolidLine, Qt::RoundCap, Qt::RoundJoin));

    for(int i(0); i < 10; i++)
    {
        int x = qrand() % ((7 + 1) + 7)*sqrt(penWidth) - 7*sqrt(penWidth);
        int y = qrand() % ((7 + 1) + 7)*sqrt(penWidth) - 7*sqrt(penWidth);
        painter.drawLine(endPoint.x() + x, endPoint.y() + y,
                         endPoint.x() + x+1, endPoint.y() + y+1);
    }
    update();
    painter.end();
    modifiedFlag = true;
}

/*!
 \brief метод для обработки инструмента лупа

*/
void CImageCV::drawLoupe()
{
    emit sendWait(true);
    if(loupeFlag)
    {
        imageModel->setImage(imageModel->getImage()->scaled(imageModel->getImage()->width() / 4,
                                                            imageModel->getImage()->height() / 4));
        penWidth /= 4;
        loupeFlag = false;
    }
    else
    {
        imageModel->setImage(imageModel->getImage()->scaled(imageModel->getImage()->width() * 4,
                                                            imageModel->getImage()->height() * 4));
        penWidth *= 4;
        loupeFlag = true;
    }
    emit sendWait(false);
}

/*!
 \brief метод для установки пути файла

 \param fileName путь файла
 \return QString обработанное имя файла
*/
QString CImageCV::setFilePath(QString fileName)
{
    imageModel->filePath = fileName;

    QString tmp;
    for(int i(fileName.size()-1); i > 0; i--)
    {
        if(fileName[i] == '/')
            break;
        tmp += fileName[i];
    }
    fileName.clear();
    for(int i(tmp.size()-1); i > -1; i--)
        fileName += tmp[i];
    return fileName;
}

/*!
 \brief метод для получения пути файла

 \return QString путь файла
*/
QString CImageCV::getFilePath()
{
    return imageModel->filePath;
}

/*!
 \brief перегрузка paintEvent

 \param event обрабатываемый эвент
*/
void CImageCV::paintEvent(QPaintEvent *event)
{
    QPainter *painter = new QPainter(this);
    QRect *rect = new QRect(event->rect());
    painter->drawImage(*rect, *imageModel->getImage(), *rect);
    painter->end();
    delete rect;
    delete painter;
}

/*!
 \brief метод для загрузки изображения

 \param fileName имя файла
 \return bool индикатор отображающий удачную\неудачную загрузку
*/
bool CImageCV::setLoadImage(const QString &fileName)
{
    if(imageModel->loadImage(fileName))
    {
        rememberChanges();
        resize(imageModel->getImage()->size());
        modifiedFlag = false;
        return true;
    }
    else
        return false;
}

/*!
 \brief метод для сохранения изображения

 \param fileName имя файла
*/
void CImageCV::setSaveImage(const QString &fileName)
{
    clickToInsert();
    QImage *tempImage = new QImage;
    if(loupeFlag)
        *tempImage = imageModel->getImage()->scaled(imageModel->getImage()->size() / 4);
    else
        *tempImage = imageModel->getImage()->scaled(imageModel->getImage()->size() / pow(1.2, zoomFactor));
    tempImage->save(fileName);
    delete tempImage;

    modifiedFlag = false;
}

/*!
 \brief метод реализующий черно-белый эффект

*/
void CImageCV::setBlackWhite()
{
    clickToInsert();
    for(int i(0); i < imageModel->getImage()->width(); i++)
    {
        for(int y(0); y < imageModel->getImage()->height(); y++)
        {
            QRgb pixsel(imageModel->getImage()->pixel(i, y));
            int rgb = (int)(0.299 * qRed(pixsel) + 0.587 * qGreen(pixsel) + 0.114 * qBlue(pixsel));
            pixsel = qRgb(rgb, rgb, rgb);
            imageModel->getImage()->setPixel(i, y, pixsel);
        }
    }
    rememberChanges();
    update();
    modifiedFlag = true;
}

/*!
 \brief метод реализующий эффект негатива

*/
void CImageCV::setNegative()
{
    clickToInsert();
    imageModel->getImage()->invertPixels(QImage::InvertRgb);
    rememberChanges();
    update();
    modifiedFlag = true;
}

/*!
 \brief метод для поворота изображения

*/
void CImageCV::setRotate()
{
    clickToInsert();
    imageModel->setImage(imageModel->getImage()->mirrored(true, true));
    rememberChanges();
    update();
    modifiedFlag = true;
}

/*!
 \brief метод для изменения размеров изображения

 \param width ширина
 \param height высота
*/
void CImageCV::setResizeImage(int width, int height)
{
    clickToInsert();
    imageModel->setImage(imageModel->getImage()->scaled(width, height));
    resize(imageModel->getImage()->size());
    rememberChanges();
    update();
    modifiedFlag = true;
}

/*!
 \brief метод для изменения размеров зоны

 \param width ширина
 \param height высота
*/
void CImageCV::setResizeArea(int width, int height)
{
    clickToInsert();
    QImage *tempImage = new QImage(width, height, QImage::Format_ARGB32_Premultiplied);
    QPainter painter(tempImage);
    painter.setBrush(QBrush(Qt::white));
    painter.drawRect(QRect(0, 0, width, height));
    painter.drawImage(0, 0, *imageModel->getImage());
    painter.end();

    imageModel->setImage(*tempImage);
    resize(imageModel->getImage()->size());
    rememberChanges();
    update();
    modifiedFlag = true;
    delete tempImage;
}

/*!
 \brief обработчик нажатия левой\правой кнопки мыши

 \param event эвент для обработки
*/
void CImageCV::mousePressEvent(QMouseEvent *event)
{
    if(event->button() == Qt::LeftButton)
    {
        switch(activeTool)
        {
        case NONE:
            dragSelectedArea(event->x(), event->y());
            break;
        case SELECT:
            drawFlag = true;
            clickToInsert();
            isSelecting = true;
            selectImage = *(imageModel->getImage());
            bufImage = *(imageModel->getImage());
            startPoint = endPoint = event->pos() + displacePoint;//??
            break;
        case HAND:
            dragSelectedArea(event->x(), event->y());
            break;
        case PEN: case FILL: case SPRAY: case PIPET: case LOUPE:
            clickToInsert();
            startPoint = endPoint = event->pos() + displacePoint;
            drawFlag = true;
            break;
        case LASTIC:
            clickToInsert();
            tempColor = penColor;
            penColor = Qt::white;
            startPoint = endPoint = event->pos() + displacePoint;
            drawFlag = true;
            break;
        case LINE: case RECTANGLE: case ELLIPSE:
            clickToInsert();
            drawFlag = true;
            bufImage = *(imageModel->getImage());
            startPoint = endPoint = event->pos() + displacePoint;
            break;
        }
    }

    if(event->button() == Qt::RightButton)
    {
        switch(activeTool)
        {
        case NONE: case LASTIC: case LOUPE:
        case SELECT: case HAND:
            break;
        case PEN: case FILL: case SPRAY: case PIPET:
            clickToInsert();
            startPoint = endPoint = event->pos() + displacePoint;
            drawFlag = true;
            break;
        case RECTANGLE: case ELLIPSE: case LINE:
            clickToInsert();
            drawFlag = true;
            bufImage = *(imageModel->getImage());
            startPoint = endPoint = event->pos() + displacePoint;
            break;
        }
    }
}

/*!
 \brief обработчик движения мыши с зажатой левой\правой кнопкой

 \param event эвент для обработки
*/
void CImageCV::mouseMoveEvent(QMouseEvent *event)
{
    if((event->buttons() & Qt::LeftButton) && drawFlag)
    {
        switch(activeTool)
        {
        case NONE: case PIPET: case FILL: case LOUPE:
        case HAND:
            break;
        case SELECT:
            backInTime();
            endPoint = event->pos() + displacePoint;
            drawSelect();
            break;
        case PEN:
            clickToInsert();
            endPoint = event->pos() + displacePoint;
            drawLine(false);
            startPoint = endPoint;
            break;
        case LASTIC:
            clickToInsert();
            endPoint = event->pos() + displacePoint;
            drawLine(false);
            startPoint = endPoint;
            break;
        case SPRAY:
            clickToInsert();
            endPoint = event->pos() + displacePoint;
            drawSpray(false);
            startPoint = endPoint;
            break;
        case LINE:
            clickToInsert();
            backInTime();
            endPoint = event->pos() + displacePoint;
            drawLine(false);
            break;
        case RECTANGLE:
            clickToInsert();
            backInTime();
            endPoint = event->pos() + displacePoint;
            drawRectangle(false);
            break;
        case ELLIPSE:
            clickToInsert();
            backInTime();
            endPoint = event->pos() + displacePoint;
            drawEllipse(false);
            break;
        }
    }

    if((event->buttons() & Qt::RightButton) && drawFlag)
    {
        switch(activeTool)
        {
        case NONE: case LASTIC: case PIPET: case FILL: case LOUPE:
        case SELECT: case HAND:
            break;
        case PEN:
            clickToInsert();
            endPoint = event->pos() + displacePoint;
            drawLine(true);
            startPoint = endPoint;
            break;
        case SPRAY:
            clickToInsert();
            endPoint = event->pos() + displacePoint;
            drawSpray(true);
            startPoint = endPoint;
            break;
        case LINE:
            clickToInsert();
            backInTime();
            endPoint = event->pos() + displacePoint;
            drawLine(true);
            break;
        case RECTANGLE:
            clickToInsert();
            backInTime();
            endPoint = event->pos() + displacePoint;
            drawRectangle(true);
            break;
        case ELLIPSE:
            clickToInsert();
            backInTime();
            endPoint = event->pos() + displacePoint;
            drawEllipse(true);
            break;
        }
    }
}

/*!
 \brief обработчик отжатия левой\правой кнопки мыши

 \param event эвент для обработки
*/
void CImageCV::mouseReleaseEvent(QMouseEvent *event)
{
    if(event->button() == Qt::LeftButton && drawFlag)
    {
        endPoint = event->pos() + displacePoint;
        switch(activeTool)
        {
        case NONE: case HAND:
            break;
        case SELECT:
            backInTime();
            drawSelect();
            update();
            drawFlag = false;
            break;
        case PEN:
            drawLine(false);
            drawFlag = false;
            rememberChanges();
            break;
        case SPRAY:
            drawSpray(false);
            drawFlag = false;
            rememberChanges();
            break;
        case FILL:
            drawFill(false);
            update();
            rememberChanges();
            drawFlag = false;
            break;
        case LASTIC:
            drawLine(false);
            drawFlag = false;
            penColor = tempColor;
            rememberChanges();
            break;
        case PIPET:
            penColor = drawPipet();
            drawFlag = false;
            emit sendPenColor(penColor);
            break;
        case LINE:
            backInTime();
            drawLine(false);
            update();
            drawFlag = false;
            rememberChanges();
            break;
        case RECTANGLE:
            backInTime();
            drawRectangle(false);
            update();
            drawFlag = false;
            rememberChanges();
            break;
        case ELLIPSE:
            backInTime();
            drawEllipse(false);
            update();
            drawFlag = false;
            rememberChanges();
            break;
        case LOUPE:
            drawLoupe();
            resize(imageModel->getImage()->size());
            drawFlag = false;
            if(loupeFlag)
                emit sendScrolBarsPos(event->pos().x()*3, event->pos().y()*3);
            else
                emit sendScrolBarsPos(event->pos().x(), event->pos().y());
            break;
        }
    }

    if(event->button() == Qt::RightButton && drawFlag)
    {
        endPoint = event->pos() + displacePoint;
        switch(activeTool)
        {
        case NONE: case LASTIC: case LOUPE:
        case SELECT: case HAND:
            break;
        case PEN:
            drawLine(true);
            drawFlag = false;
            rememberChanges();
            break;
        case PIPET:
            drawFlag = false;
            brushColor = drawPipet();
            emit sendBrushColor(brushColor);
            break;
        case SPRAY:
            drawSpray(true);
            drawFlag = false;
            rememberChanges();
            break;
        case FILL:
            drawFill(true);
            update();
            drawFlag = false;
            rememberChanges();
            break;
        case LINE:
            backInTime();
            drawLine(true);
            update();
            drawFlag = false;
            rememberChanges();
            break;
        case RECTANGLE:
            backInTime();
            drawRectangle(true);
            update();
            drawFlag = false;
            rememberChanges();
            break;
        case ELLIPSE:
            backInTime();
            drawEllipse(true);
            update();
            drawFlag = false;
            rememberChanges();
            break;
        }
    }
}

/*!
 \brief метод для изменения размера изображения в зивисимости
        от положения слайдера(или фактора)

 \param factor фактор зума
*/
void CImageCV::setZoom(int factor)
{
    clickToInsert();
    double k(1.2);
    imageModel->setImage(imageModel->getImage()->scaled(imageModel->getImage()->size() / pow(k, zoomFactor)));
    penWidth = round(penWidth / pow(k, zoomFactor));
    imageModel->setImage(imageModel->getImage()->scaled(imageModel->getImage()->size() * pow(k, factor)));
    penWidth = round(penWidth * pow(k, factor));
    if(isDraging)
    {
        QPixmap bufCursor = this->cursor().pixmap().scaled(this->cursor().pixmap().size() / pow(k, zoomFactor));
        this->setCursor(QCursor(bufCursor.scaled(bufCursor.size() * pow(k, factor))));
    }
    resize(imageModel->getImage()->size());
    zoomFactor = factor;
    QPainter painter(imageModel->getImage());
    painter.drawImage(imageModel->getImage()->rect(), *imageModel->getImage(), imageModel->getImage()->rect());
    update();
    painter.end();
}

/*!
 \brief метод для печати изображения

*/
void CImageCV::printImage()
{
    QPrinter *printer = new QPrinter();
    QPrintDialog *printDialog = new QPrintDialog(printer);
    if(printDialog->exec())
    {
        QPainter painter(printer);
        QRect rect(painter.viewport());
        QSize size(imageModel->getImage()->size());
        size.scale(rect.size(), Qt::KeepAspectRatio);
        painter.setViewport(rect.x(), rect.y(), size.width(), size.height());
        painter.setWindow(imageModel->getImage()->rect());
        painter.drawImage(0, 0, *imageModel->getImage());
    }
    delete printDialog;
    delete printer;
}

/*!
 \brief метод для выделения области

*/
void CImageCV::drawSelect()
{
    QPainter painter(imageModel->getImage());
    int r = 255 - QColor::fromRgb(imageModel->getImage()->pixel(startPoint)).red();
    int g = 255 - QColor::fromRgb(imageModel->getImage()->pixel(startPoint)).green();
    int b = 255 - QColor::fromRgb(imageModel->getImage()->pixel(startPoint)).blue();
    QColor color = QColor(r, g, b);
    painter.setPen(QPen(color, 1, Qt::DashLine, Qt::RoundCap, Qt::RoundJoin));
    if(startPoint != endPoint)
        painter.drawRect(QRect(startPoint, endPoint));
    QPoint min, max;
    if(startPoint.x() < endPoint.x())
    {
        min.setX(startPoint.x());
        max.setX(endPoint.x());
    }
    else
    {
        min.setX(endPoint.x());
        max.setX(startPoint.x());
    }
    if(startPoint.y() < endPoint.y())
    {
        min.setY(startPoint.y());
        max.setY(endPoint.y());
    }
    else
    {
        min.setY(endPoint.y());
        max.setY(startPoint.y());
    }
    selectRect = QRect(min, max);
    int rad(penWidth + 5);
    update(QRect(startPoint, endPoint).normalized().adjusted(-rad, -rad, +rad, +rad));
    painter.end();
}

/*!
 \brief метод для вставки выделенной области

*/
void CImageCV::clickToInsert()
{
    if(isSelecting)
    {
        isSelecting = false;
        imageModel->setImage(selectImage);
        update();
        //modifiedFlag = true;
    }
}

/*!
 \brief метод реализующий перетаскивание выделенной области

 \param x координата
 \param y координата
*/
void CImageCV::dragSelectedArea(int x, int y)
{
    if(isDraging)
    {
        QPainter painter(imageModel->getImage());
        painter.drawPixmap(x - this->cursor().pixmap().width() / 2,
                           y - this->cursor().pixmap().height() / 2,
                           this->cursor().pixmap().width(),
                           this->cursor().pixmap().height(),
                           this->cursor().pixmap());
        update();
        painter.end();
        if(activeTool == HAND)
            setActiveTool(HAND);
        else
            this->setCursor(*defaultCursor);
        isDraging = false;
        rememberChanges();
    }
    int x1, x2, y1, y2;
    selectRect.getCoords(&x1, &y1, &x2, &y2);
    if(isSelecting && x > x1 && y > y1 && x < x2 && y < y2)
    {
        clickToInsert();

        int newX = this->cursor().pos().x() - (x - (x2 - selectRect.width() / 2));
        int newY = this->cursor().pos().y() - (y - (y2 - selectRect.height() / 2));
        this->cursor().setPos(newX, newY);

        this->setCursor(QCursor(QPixmap::fromImage(imageModel->getImage()->copy(selectRect))));
        QPainter painter(imageModel->getImage());
        painter.setPen(QPen(brushColor, 0, Qt::NoPen));
        painter.setBrush(QBrush(brushColor));
        painter.drawRect(selectRect);
        int rad(penWidth + 5);
        update(QRect(startPoint, endPoint).normalized().adjusted(-rad, -rad, +rad, +rad));
        painter.end();
        isDraging = true;
    }
}

/*!
 \brief метод реализующий копирование выделенной области

 \param img изображение в которое копируем
 \return bool флаг для индикации удачного\неудачного копирования
*/
bool CImageCV::copy(QImage &img)
{
    if(isSelecting)
    {
        img = selectImage.copy(selectRect);
        return true;
    }
    return false;
}

/*!
 \brief метод реализующий вырезание изображения из выделенной области

 \param img изображения в которое вырезаем
 \return bool флаг для индикации удачного\неудачного вырезания
*/
bool CImageCV::cut(QImage &img)
{
    bool temp = copy(img);

    clickToInsert();
    QPainter painter(imageModel->getImage());
    painter.setPen(QPen(brushColor, 0, Qt::NoPen));
    painter.setBrush(QBrush(brushColor));
    painter.drawRect(selectRect);
    int rad(penWidth + 5);
    update(QRect(startPoint, endPoint).normalized().adjusted(-rad, -rad, +rad, +rad));
    painter.end();
    modifiedFlag = true;
    return temp;
}

/*!
 \brief метод реализующий вставку изображения из буфера

 \param img вставляемой изображения
*/
void CImageCV::paste(QImage img)
{
    clickToInsert();
    emit noActiveTool();
    this->setCursor(QCursor(QPixmap::fromImage(img)));
    activeTool = NONE;
    isDraging = true;
    modifiedFlag = true;
}

/*!
 \brief метод для вставки изображения из вне

 \param img вставляемое изображение
*/
void CImageCV::setImage(QImage img)
{
    imageModel->setImage(img);
    modifiedFlag = true;
    update();
}
