// ---------------------------------------------------------------------------
//  PC-8801 emulator
//  Copyright (C) cisc 1999.
// ---------------------------------------------------------------------------
//  $Id: subsys.cpp,v 1.7 1999/04/18 11:20:03 cisc Exp $

#include "headers.h"
//#include <string.h>
#include "device.h"
#include "device_i.h"
#include "subsys.h"
#include "file.h"
#include "status.h"

#define LOGNAME "subsys"
#include "diag.h"

using namespace PC8801;

// ---------------------------------------------------------------------------
//  \zEj
//
SubSystem::SubSystem(const ID& id)
: Device(id)
{
    rom = 0;
    ram = 0;
    dummy = 0;
}

SubSystem::~SubSystem()
{
    delete[] rom;
    delete[] ram;
    delete[] dummy;
}

// ---------------------------------------------------------------------------
//  
//
bool SubSystem::Init(Bus* _bus)
{
    bus = _bus;
    if (!InitMemory())
        return false;
    piom.Connect(&pios);
    pios.Connect(&piom);
    return true;
}

// ---------------------------------------------------------------------------
//  
//
bool SubSystem::InitMemory()
{
    rom = new uint8[0x2000];
    ram = new uint8[0x4000];
    dummy = new uint8[2 << Bus::pagebits];
    if (!rom || !ram || !dummy)
        return false;
    memset(dummy, 0xff, 1 << Bus::pagebits);
    memset(ram, 0x00, 0x4000);

    LoadROM();
    PatchROM();

    // map dummy memory
    for (int i=0; i<0x10000; i+=1<<Bus::pagebits)
    {
        bus->SetReadMemory (i, dummy);
        bus->SetWriteMemory(i, dummy + (1<<Bus::pagebits));
    }

    bus->SetReadMemorys(0, 0x2000, rom);
    bus->SetMemorys(0x4000, 0x4000, ram);
    bus->SetWaits(0x0000, 0x10000, 0);
    return true;
}

// ---------------------------------------------------------------------------
//  ROM ǂݍ
//
bool SubSystem::LoadROM()
{
    memset(rom, 0xff, 0x2000);

    FileIO fio;
    if (fio.Open("PC88.ROM", FileIO::readonly))
    {
        fio.Seek(0x14000, FileIO::begin);
        fio.Read(rom, 0x2000);
        return true;
    }
    if (fio.Open("DISK.ROM", FileIO::readonly))
    {
        fio.Seek(0, FileIO::begin);
        fio.Read(rom, 0x2000);
        return true;
    }
    rom[0] = 0xf3;
    rom[1] = 0x76;
    return false;
}

// ---------------------------------------------------------------------------
//  SubSystem::PatchROM
//  [^[̉]҂ȗpb`𓖂Ă
//  ʂɂĂȂĂNxȂ邾Ȃ񂾂ǂˁD
//
void SubSystem::PatchROM()
{
    if (rom[0xfb] == 0xcd && rom[0xfc] == 0xb4 && rom[0xfd] == 0x02)
    {
        rom[0xfb] = rom[0xfc] = rom[0xfd] = 0;
        rom[0x105] = rom[0x106] = rom[0x107] = 0;
    }
}

// ---------------------------------------------------------------------------
//  Reset
//
void SubSystem::Reset(uint, uint)
{
    piom.Reset();
    pios.Reset();
    idlecount = 0;
}

// ---------------------------------------------------------------------------
//  荞ݎ
//
uint SubSystem::IntAck(uint)
{
    return 0x00;
}

// ---------------------------------------------------------------------------
//  Main  PIO
//
void SubSystem::M_Set0(uint, uint data)
{
    idlecount = 0;
    LOG1(".%.2x ", data);
    piom.SetData(0, data);
}

void SubSystem::M_Set1(uint, uint data)
{
    idlecount = 0;
    LOG1(" %.2x ", data);
    piom.SetData(1, data);
}

void SubSystem::M_Set2(uint, uint data)
{
    idlecount = 0;
    piom.SetData(2, data);
}

void SubSystem::M_SetCW(uint, uint data)
{
    idlecount = 0;
    if (data == 0x0f)
        LOG0("\ncmd: ");
    piom.SetCW(data);
}

uint SubSystem::M_Read0(uint)
{
    idlecount = 0;
    uint d = piom.Read0();
    LOG1(">%.2x ", d);
    return d;
}

uint SubSystem::M_Read1(uint)
{
    idlecount = 0;
    uint d = piom.Read1();
    LOG1(")%.2x ", d);
    return d;
}

uint SubSystem::M_Read2(uint)
{
    statusdisplay.WaitSubSys();
    idlecount = 0;
    return piom.Read2();
}

// ---------------------------------------------------------------------------
//  Sub  PIO
//
void SubSystem::S_Set0(uint, uint data)
{
    idlecount = 0;
//  LOG1("<a %.2x> ", data);
    pios.SetData(0, data);
}

void SubSystem::S_Set1(uint, uint data)
{
    idlecount = 0;
//  LOG1("<b %.2x> ", data);
    pios.SetData(1, data);
}

void SubSystem::S_Set2(uint, uint data)
{
    idlecount = 0;
//  LOG1("<c %.2x> ", data);
    pios.SetData(2, data);
}

void SubSystem::S_SetCW(uint, uint data)
{
    idlecount = 0;
    pios.SetCW(data);
}

uint SubSystem::S_Read0(uint)
{
    idlecount = 0;
    uint d = pios.Read0();
//  LOG1("(a %.2x) ", d);
    return d;
}

uint SubSystem::S_Read1(uint)
{
    idlecount = 0;
    uint d = pios.Read1();
//  LOG1("(b %.2x) ", d);
    return d;
}

uint SubSystem::S_Read2(uint)
{
    idlecount++;
    uint d = pios.Read2();
//  LOG1("(c %.2x) ", d);
    return d;
}


bool SubSystem::IsBusy()
{
    if (idlecount >= 200)
    {
        idlecount = 200;
        return false;
    }
    statusdisplay.WaitSubSys();
    return true;
}


// ---------------------------------------------------------------------------
//  device description
//
const Device::Descriptor SubSystem::descriptor = { indef, outdef };

const Device::InFuncPtr SubSystem::indef[] =
{
#ifndef __OS2__
    static_cast<InFuncPtr> (IntAck),
    static_cast<InFuncPtr> (M_Read0),
    static_cast<InFuncPtr> (M_Read1),
    static_cast<InFuncPtr> (M_Read2),
    static_cast<InFuncPtr> (S_Read0),
    static_cast<InFuncPtr> (S_Read1),
    static_cast<InFuncPtr> (S_Read2),
#else
    (Device::InFuncPtr) (IntAck),
    (Device::InFuncPtr) (M_Read0),
    (Device::InFuncPtr) (M_Read1),
    (Device::InFuncPtr) (M_Read2),
    (Device::InFuncPtr) (S_Read0),
    (Device::InFuncPtr) (S_Read1),
    (Device::InFuncPtr) (S_Read2),
#endif
};

const Device::OutFuncPtr SubSystem::outdef[] =
{
#ifndef __OS2__
    static_cast<OutFuncPtr> (Reset),
    static_cast<OutFuncPtr> (M_Set0),
    static_cast<OutFuncPtr> (M_Set1),
    static_cast<OutFuncPtr> (M_Set2),
    static_cast<OutFuncPtr> (M_SetCW),
    static_cast<OutFuncPtr> (S_Set0),
    static_cast<OutFuncPtr> (S_Set1),
    static_cast<OutFuncPtr> (S_Set2),
    static_cast<OutFuncPtr> (S_SetCW),
#else
    (Device::OutFuncPtr) (Reset),
    (Device::OutFuncPtr) (M_Set0),
    (Device::OutFuncPtr) (M_Set1),
    (Device::OutFuncPtr) (M_Set2),
    (Device::OutFuncPtr) (M_SetCW),
    (Device::OutFuncPtr) (S_Set0),
    (Device::OutFuncPtr) (S_Set1),
    (Device::OutFuncPtr) (S_Set2),
    (Device::OutFuncPtr) (S_SetCW),
#endif
};
