// ---------------------------------------------------------------------------
//  M88 - PC-8801 Emulator
//  Copyright (C) cisc 1997, 1999.
// ---------------------------------------------------------------------------
//  $Id: opnif.cpp,v 1.7 1999/07/15 12:36:40 cisc Exp $

#include "headers.h"
#include "types.h"
#include "misc.h"
#include "schedule.h"
#include "pc88/sound.h"
#include "pc88/opnif.h"
#include "pc88/config.h"

//#define USE_FM_C

#ifdef USE_FM_C
#include "fm.h"
#endif

#define LOGNAME "opnif"
#include "diag.h"

using namespace PC8801;

// ---------------------------------------------------------------------------
//  vXP[̐ݒl
//  static ɂ̂́CFMGen ̐ɂC OPN قȂNbN
//  邱ƂoȂ߁D
//
int OPNIF::prescaler = 0x2d;

// ---------------------------------------------------------------------------
//  Ej
//
OPNIF::OPNIF(const ID& id, int i)
: Device(id)
{
    scheduler = 0;
    sound = 0;
    srate = 0;
    imaskbit = i;
    opnamode = false;
    nextcount = 0;
}

OPNIF::~OPNIF()
{
}

// ---------------------------------------------------------------------------
//  
//
bool OPNIF::Init(Bus* b, int intrport, int io, Scheduler* s, uint c, uint r)
{
    bus = b;
    scheduler = s;
    portio = io;
    opn.SetIntr(bus, intrport);

    if (!opn.Init(c, r, 0))
        return false;
    if (!SetRate(r, c))
        return false;
    prevtime = scheduler->GetTime();
    TimeEvent();

#ifdef USE_FM_C
    void* pcmrom[6] = {0};
    int pcmsize[6] = {0};
    YM2608Init(1, 3993600*2, 44100, pcmrom, pcmsize, 0, 0, 0, 0);
#endif
    return true;
}

// ---------------------------------------------------------------------------
//  EĐ[gݒ
//
bool OPNIF::SetRate(uint _clock, uint rate)
{
    clock = _clock;
    bool fm55k = false;
    if (rate == SOUND_55K)
    {
        fm55k = true;
        rate = 44100;
    }
    srate = rate;
    opn.SetReg(prescaler, 0);
    opn.SetRate(clock, srate, fm55k);

    switch (prescaler & 3)
    {
    case 1: psg.SetClock(clock/4, srate); break;
    case 2: psg.SetClock(clock/2, srate); break;
    case 3: psg.SetClock(clock/1, srate); break;
    }
    return true;
}

// ---------------------------------------------------------------------------
//  
//
void OPNIF::Mix(FM::Sample* dest, int nsamples)
{
    if (enable)
    {
#ifndef USE_FM_C
        opn.Mix(dest, nsamples);
#else
        YM2608UpdateOne(0, dest, nsamples);
#endif
        psg.Mix(dest, nsamples);
    }
}

// ---------------------------------------------------------------------------
//  ʐݒ
//
static inline int ConvertVolume(int volume)
{
    return volume > -40 ? volume : -200;
}

void OPNIF::SetVolume(const Config* config)
{
#ifndef USE_OPN
    opn.SetVolumeFM(ConvertVolume(config->volfm));
    opn.SetVolumeADPCM(ConvertVolume(config->voladpcm));
    opn.SetVolumeRhythmTotal(ConvertVolume(config->volrhythm));
    opn.SetVolumeRhythm(0, ConvertVolume(config->volbd));
    opn.SetVolumeRhythm(1, ConvertVolume(config->volsd));
    opn.SetVolumeRhythm(2, ConvertVolume(config->voltop));
    opn.SetVolumeRhythm(3, ConvertVolume(config->volhh));
    opn.SetVolumeRhythm(4, ConvertVolume(config->voltom));
    opn.SetVolumeRhythm(5, ConvertVolume(config->volrim));
#endif
    psg.SetVolume(ConvertVolume(config->volssg));
}

// ---------------------------------------------------------------------------
//  Reset
//
void OPNIF::Reset(uint, uint)
{
    memset(regs, 0, sizeof(regs));
    opn.Reset();
    psg.Reset();
    psg.SetClock(clock/4, srate);
    opn.SetIntrMask(true);
    prescaler = 0x2d;
#ifdef USE_FM_C
    YM2608ResetChip(0);
#endif
}

// ---------------------------------------------------------------------------
//  荞
//
void OPNIF::OPNUnit::Intr(bool flag)
{
    bool prev = intrpending && intrenabled && bus;
    intrpending = flag;
    LOG3("OPN     :Interrupt %d %d %d\n", intrpending, intrenabled, !prev);
    if (intrpending && intrenabled && bus && !prev)
    {
        bus->Out(pintr, true);
    }
}

// ---------------------------------------------------------------------------
//  荞݋H
//
inline void OPNIF::OPNUnit::SetIntrMask(bool en)
{
    bool prev = intrpending && intrenabled && bus;
    intrenabled = en;
    if (intrpending && intrenabled && bus && !prev)
        bus->Out(pintr, true);
}

void OPNIF::SetIntrMask(uint a, uint intrmask)
{
//  LOG2("Intr enabled (%.2x)[%.2x]\n", a, intrmask);
    opn.SetIntrMask(!(imaskbit & intrmask));
}

// ---------------------------------------------------------------------------
//  SetRegisterIndex
//
void OPNIF::SetIndex0(uint a, uint data)
{
//  LOG2("Index0[%.2x] = %.2x\n", a, data);
    index0 = data;
    if (enable && (data & 0xfc) == 0x2c)
    {
        regs[data] = 1;
        switch (data & 3)
        {
        case 0: return;
        case 1: psg.SetClock(clock/4, srate); break;
        case 2: psg.SetClock(clock/2, srate); break;
        case 3: psg.SetClock(clock/1, srate); break;
        }
        prescaler = data;
        opn.SetReg(data, 0);
    }
#ifdef USE_FM_C
    YM2608Write(0, 0, data);
#endif
}

void OPNIF::SetIndex1(uint a, uint data)
{
//  LOG2("Index1[%.2x] = %.2x\n", a, data);
#ifdef USE_FM_C
    YM2608Write(0, 2, data);
#endif
    index1 = data1 = data;
}

// ---------------------------------------------------------------------------
//  WriteRegister
//
void OPNIF::WriteData0(uint a, uint data)
{
//  LOG2("Write0[%.2x] = %.2x\n", a, data);
    if (enable)
    {
        LOG3("%.8x:OPN[0%.2x] = %.2x\n", scheduler->GetTime(), index0, data);
        TimeEvent(1);
        regs[index0] = data;
#ifdef USE_FM_C
        YM2608Write(0, 1, data);
#endif

        if (index0 < 0x10)
            psg.SetReg(index0, data);
        else
        {
            if (!opnamode)
            {
                if ((index0 & 0xf0) == 0x10) return;
                if (index0 == 0x22) data = 0;
                if (index0 == 0x29) data = 3;
                if (index0 >= 0xb4) data = 0xc0;
            }
            opn.SetReg(index0, data);
            if (index0 == 0x27)
            {
                UpdateTimer();
            }
        }
    }
}

void OPNIF::WriteData1(uint a, uint data)
{
//  LOG2("Write1[%.2x] = %.2x\n", a, data);
    if (enable && opnamode)
    {
        LOG3("%.8x:OPN[1%.2x] = %.2x\n", scheduler->GetTime(), index1, data);
        TimeEvent(1);
        data1 = data;
        regs[0x100 | index1] = data;
#ifdef USE_FM_C
        YM2608Write(0, 3, data);
#endif
        opn.SetReg(0x100 | index1, data);
    }
}

// ---------------------------------------------------------------------------
//  ReadRegister
//
uint OPNIF::ReadData0(uint a)
{
    uint ret;
    if (!enable)
        ret = 0xff;
    else if ((index0 & 0xfe) == 0x0e)
        ret = bus->In(portio + (index0 & 1));
    else if (index0 < 0x10)
        ret = psg.GetReg(index0);
    else if (index0 == 0xff && !opnamode)
        ret = 0;
    else
        ret = opn.GetReg(index0);
//  LOG2("Read0 [%.2x] = %.2x\n", a, ret);
    return ret;
}

uint OPNIF::ReadData1(uint a)
{
    uint ret = 0xff;
    if (enable && opnamode)
    {
        if (index1 == 0x08)
            ret = opn.GetReg(0x100 | index1);
        else
            ret = data1;
    }
//  LOG3("Read1 [%.2x] = %.2x  (d1:%.2x)\n", a, ret, data1);
    return ret;
}

// ---------------------------------------------------------------------------
//  ReadStatus
//
uint OPNIF::ReadStatus(uint a)
{
    uint ret = enable ? opn.ReadStatus() : 0xff;
//  LOG2("status[%.2x] = %.2x\n", a, ret);
    return ret;
}

uint OPNIF::ReadStatusEx(uint a)
{
    uint ret = enable && opnamode ? opn.ReadStatusEx() : 0xff;
//  LOG2("statex[%.2x] = %.2x\n", a, ret);
    return ret;
}

// ---------------------------------------------------------------------------
//  ^C}[XV
//
void OPNIF::UpdateTimer()
{
    static Scheduler::Event* ev;
    if (!ev || !ev->inst)
        LOG1("%.8x:UpdateTimer()", scheduler->GetTime());
    else
        LOG3("%.8x:UpdateTimer() [%.8x/%8d]", scheduler->GetTime(), ev->time, ev->count);

    scheduler->DelEvent(this);
    nextcount = opn.GetNextEvent();
    if (nextcount)
    {
        nextcount = (nextcount+9) / 10;
        ev = scheduler->AddEvent(nextcount, this, static_cast<TimeFunc>(TimeEvent), 0);
        LOG2(" [%.8x] - %8d\n", nextcount + scheduler->GetTime(), nextcount);
    }
    LOG0("\n");
}

// ---------------------------------------------------------------------------
//  ^C}[
//
void OPNIF::TimeEvent(uint rtc)
{
    int currenttime = scheduler->GetTime();
    int diff = currenttime - prevtime;
    prevtime = currenttime;

    if (enable)
    {
        LOG3("%.8x:TimeEvent(%d) : diff:%d\n", currenttime, rtc, diff);

        if (sound)
            sound->Update();
        if (opn.Count(diff * 10) || !rtc)
            UpdateTimer();
    }
}

// ---------------------------------------------------------------------------
//  device description
//
const Device::Descriptor OPNIF::descriptor = { indef, outdef };

const Device::OutFuncPtr OPNIF::outdef[] =
{
#ifndef __OS2__
    static_cast<OutFuncPtr> (Reset),
    static_cast<OutFuncPtr> (SetIndex0),
    static_cast<OutFuncPtr> (SetIndex1),
    static_cast<OutFuncPtr> (WriteData0),
    static_cast<OutFuncPtr> (WriteData1),
    static_cast<OutFuncPtr> (SetIntrMask),
#else
    (Device::OutFuncPtr) (Reset),
    (Device::OutFuncPtr) (SetIndex0),
    (Device::OutFuncPtr) (SetIndex1),
    (Device::OutFuncPtr) (WriteData0),
    (Device::OutFuncPtr) (WriteData1),
    (Device::OutFuncPtr) (SetIntrMask),
#endif
};

const Device::InFuncPtr OPNIF::indef[] =
{
#ifndef __OS2__
    static_cast<InFuncPtr> (ReadStatus),
    static_cast<InFuncPtr> (ReadStatusEx),
    static_cast<InFuncPtr> (ReadData0),
    static_cast<InFuncPtr> (ReadData1),
#else
    (Device::InFuncPtr) (ReadStatus),
    (Device::InFuncPtr) (ReadStatusEx),
    (Device::InFuncPtr) (ReadData0),
    (Device::InFuncPtr) (ReadData1),
#endif
};
