/** ColEm: portable Coleco emulator **************************/
/**                                                         **/
/**                        Coleco.c                         **/
/**                                                         **/
/** This file contains implementation for the Coleco-       **/
/** specific hardware: VDP, etc. Initialization code and    **/
/** definitions needed for the machine-dependent drivers    **/
/** are also here.                                          **/
/**                                                         **/
/** Copyright (C) Marat Fayzullin 1994-1998                 **/
/**     You are not allowed to distribute this software     **/
/**     commercially. Please, notify me, if you make any    **/
/**     changes to this file.                               **/
/*************************************************************/

#include "Coleco.h"
#include "dsp.h"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>

#ifdef ZLIB
#include <zlib.h>
#endif

byte Verbose     = 1;          /* Debug msgs ON/OFF             */
byte UPeriod     = 2;          /* Interrupts/scr. update        */
int  VPeriod     = 60000;      /* Number of cycles per VBlank   */
int  HPeriod     = 215;        /* Number of cycles per HBlank   */
byte AutoA=0,AutoB=0;          /* 1: Autofire for A,B buttons   */
byte LogSnd      = 0;          /* 1: Log soundtrack into a file */
byte Adam        = 0;          /* 1: Emulate Coleco Adam        */

byte *VRAM,*RAM;               /* Main and Video RAMs           */
Z80 CPU;                       /* Z80 CPU registers and state   */
SN76489 PSG;                   /* SN76489 PSG state             */

byte ExitNow;                  /* 1: Exit the emulator          */

byte JoyMode;                  /* Joystick controller mode      */
word JoyState[2];              /* Joystick states               */

char *PrnName = NULL;          /* Printer redirection file      */
FILE *PrnStream;

byte *ChrGen,*ChrTab,*ColTab;  /* VDP tables (screens)          */
byte *SprGen,*SprTab;          /* VDP tables (sprites)          */
pair WVAddr,RVAddr;            /* Storage for VRAM addresses    */
byte VKey;                     /* VDP address latch key         */
byte FGColor,BGColor;          /* Colors                        */
byte ScrMode;                  /* Current screen mode           */
byte CurLine;                  /* Current scanline              */
byte VDP[8],VDPStatus;         /* VDP registers                 */

char *SndName   = "LOG.SND";   /* Soundtrack log file           */
FILE *SndStream = NULL;

// OS/2 specific
extern char *ColecoRom;
// end OS/2 specific

/*** TMS9918/9928 Palette *******************************************/
//struct { byte R,G,B; } Palette[16] =
//{
//  {0x00,0x00,0x00},{0x00,0x00,0x00},{0x20,0xC0,0x20},{0x60,0xE0,0x60},
//  {0x20,0x20,0xE0},{0x40,0x60,0xE0},{0xA0,0x20,0x20},{0x40,0xC0,0xE0},
//  {0xE0,0x20,0x20},{0xE0,0x60,0x60},{0xC0,0xC0,0x20},{0xC0,0xC0,0x80},
//  {0x20,0x80,0x20},{0xC0,0x40,0xA0},{0xA0,0xA0,0xA0},{0xE0,0xE0,0xE0}
//};

/*** Screen handlers and masks for VDP table address registers ******/
struct
{
  void (*Refresh)(byte Y);
  byte R2,R3,R4,R5;
} SCR[MAXSCREEN+1] =
{
  { RefreshLine0,0x7F,0x00,0x3F,0x00 },   /* SCREEN 0:TEXT 40x24    */
  { RefreshLine1,0x7F,0xFF,0x3F,0xFF },   /* SCREEN 1:TEXT 32x24    */
  { RefreshLine2,0x7F,0x80,0x3C,0xFF },   /* SCREEN 2:BLOCK 256x192 */
  { RefreshLine3,0x7F,0x00,0x3F,0xFF }    /* SCREEN 3:GFX 64x48x16  */
};

static void VDPOut(byte Reg,byte Value); /* Write value into VDP */
static void CheckSprites(void);          /* Collisions/5th spr.  */
static void Play(int C,int F,int V);     /* Log and play sound   */

/** StartSMS() ***********************************************/
/** Allocate memory, load ROM image, initialize hardware,   **/
/** CPU and start the emulation. This function returns 0 in **/
/** the case of failure.                                    **/
/*************************************************************/
int StartColeco(char *Cartridge)
{
  FILE *F;
  int *T,I,J;
  char *P;

  /*** VDP control register states: ***/
  static byte VDPInit[8] =
  { 0x00,0x10,0x00,0x00,0x00,0x00,0x00,0x00 };

  /*** STARTUP CODE starts here: ***/
  T=(int *)"\01\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0";
#ifdef LSB_FIRST
  if(*T!=1)
  {
    printf("********** This machine is high-endian. **********\n");
    printf("Take #define LSB_FIRST out and compile ColEm again.\n");
    return(0);
  }
#else
  if(*T==1)
  {
    printf("********* This machine is low-endian. **********\n");
    printf("Insert #define LSB_FIRST and compile ColEm again.\n");
    return(0);
  }
#endif

  /* Calculate IPeriod from VPeriod */
  if(UPeriod<1) UPeriod=1;
  if(VPeriod/HPeriod<256) VPeriod=256*HPeriod;
  CPU.IPeriod=HPeriod;
  CPU.TrapBadOps=Verbose&0x04;
  CPU.IAutoReset=0;

  /* Zero everything */
  VRAM=RAM=NULL;SndStream=NULL;ExitNow=0;

  if(Verbose) printf("Allocating 64kB for CPU address space...");
  if(!(RAM=(byte*)malloc(0x10000))) { if(Verbose) puts("FAILED");return(0); }
  memset(RAM,NORAM,0x10000);
  if(Verbose) printf("OK\nAllocating 16kB for VDP address space...");
  if(!(VRAM=(byte*)malloc(0x4000))) { if(Verbose) puts("FAILED");return(0); }
  memset(VRAM,NORAM,0x4000);

  if(Verbose) printf("OK\nLoading ROMs:\n  Opening COLECO.ROM...");
  P=NULL;
//  F=fopen("COLECO.ROM","rb");
  F=fopen(ColecoRom,"rb");
  if(F)
  {
    if(fread(RAM,1,0x2000,F)!=0x2000) P="SHORT FILE";
    fclose(F);
  }
  else P="NOT FOUND";
  if(P) { if(Verbose) puts(P);return(0); }

  if(Verbose) printf("OK\n  Opening %s...",Cartridge);
  P=NULL;

#ifdef ZLIB
#define fopen          gzopen
#define fclose         gzclose
#define fread(B,N,L,F) gzread(F,B,(L)*(N))
#endif

  F=fopen(Cartridge,"rb");
  if(F)
  {
    if(fread(RAM+0x8000,1,2,F)!=2) P="SHORT FILE";
    else
    {
      I=RAM[0x8000];J=RAM[0x8001];
      if(((I==0x55)&&(J==0xAA))||((I==0xAA)&&(J==0x55)))
      {
        J=2+fread(RAM+0x8002,1,0x7FFE,F);
        if(J<0x1000) P="SHORT FILE";
      }
      else P="INVALID IMAGE";
    }
    fclose(F);
  }
  else P="NOT FOUND";

#ifdef ZLIB
#undef fopen
#undef fclose
#undef fread
#endif

  if(P) { if(Verbose) puts(P);return(0); }
  if(Verbose) printf("%d bytes loaded\n",J);

  if(!PrnName) PrnStream=stdout;
  else
  {
    if(Verbose) printf("Redirecting printer output to %s...",PrnName);
    if(!(PrnStream=fopen(PrnName,"wb"))) PrnStream=stdout;
    if(Verbose) puts(PrnStream==stdout? "FAILED":"OK");
  }

  if(Verbose)
  {
    printf("Initializing CPU and System Hardware:\n");
    printf("  VBlank = %d cycles\n  HBlank = %d cycles\n",VPeriod,HPeriod);
  }

  /* Set up the palette */
//  for(J=0;J<16;J++)
//    SetColor(J,Palette[J].R,Palette[J].G,Palette[J].B);

  /* Initialize VDP registers */
  memcpy(VDP,VDPInit,sizeof(VDP));

  /* Initialize internal variables */
  VKey=1;                              /* VDP address latch key */
  VDPStatus=0x9F;                      /* VDP status register   */
  FGColor=BGColor=0;                   /* Fore/Background color */
  ScrMode=0;                           /* Current screenmode    */
  CurLine=0;                           /* Current scanline      */
  ChrTab=ColTab=ChrGen=VRAM;           /* VDP tables (screen)   */
  SprTab=SprGen=VRAM;                  /* VDP tables (sprites)  */
  JoyMode=0;                           /* Joystick mode key     */
  JoyState[0]=JoyState[1]=0xFFFF;      /* Joystick states       */
  Reset76489(&PSG,Play);               /* Reset SN76489 PSG     */
  Sync76489(&PSG,PSG_SYNC);            /* Make it synchronous   */
  ResetZ80(&CPU);                      /* Reset Z80 registers   */

  if(Verbose) printf("RUNNING ROM CODE...\n");
  J=RunZ80(&CPU);

  if(Verbose) printf("EXITED at PC = %04Xh.\n",J);
  return(1);
}

/** TrashColeco() ********************************************/
/** Free memory allocated by StartColeco().                 **/
/*************************************************************/
void TrashColeco(void)
{
  if(RAM)  free(RAM);
  if(VRAM) free(VRAM);
  if(SndStream) fclose(SndStream);
}

/** WrZ80() **************************************************/
/** Z80 emulation calls this function to write byte V to    **/
/** address A of Z80 address space.                         **/
/*************************************************************/
void WrZ80(register word A,register byte V)
{
  if((A>0x5FFF)&&(A<0x8000))
  {
    A&=0x03FF;
    RAM[0x6000+A]=RAM[0x6400+A]=RAM[0x6800+A]=RAM[0x6C00+A]=
    RAM[0x7000+A]=RAM[0x7400+A]=RAM[0x7800+A]=RAM[0x7C00+A]=V;
  }
}

/** RdZ80() **************************************************/
/** Z80 emulation calls this function to read a byte from   **/
/** address A of Z80 address space. Now  moved to z80.c and **/
/** made inlined to speed things up.                        **/
/*************************************************************/
#ifndef COLEM
byte RdZ80(register word A) { return(RAM[A]); }
#endif

/** PatchZ80() ***********************************************/
/** Z80 emulation calls this function when it encounters a  **/
/** special patch command (ED FE) provided for user needs.  **/
/*************************************************************/
void PatchZ80(Z80 *R) {}

/** InZ80() **************************************************/
/** Z80 emulation calls this function to read a byte from   **/
/** a given I/O port.                                       **/
/*************************************************************/
byte InZ80(register word Port)
{
  static byte KeyCodes[16] =
  {
    0x0A,0x0D,0x07,0x0C,0x02,0x03,0x0E,0x05,
    0x01,0x0B,0x06,0x09,0x08,0x04,0x0F,0x0F,
  };

  switch(Port&0xE0)
  {

case 0x40: /* Printer Status */
  if(Adam&&(Port==0x40)) return(0xFF);
  break;

case 0xE0: /* Joysticks Data */
  Port=(Port>>1)&0x01;
  Port=JoyMode?
    (JoyState[Port]>>8):
    (JoyState[Port]&0xF0)|KeyCodes[JoyState[Port]&0x0F];
  return((Port|0xB0)&0x7F);

case 0xA0: /* VDP Status/Data */
  if(Port&0x01) { Port=VDPStatus;VDPStatus&=0x5F;VKey=1; }
  else { Port=VRAM[RVAddr.W];RVAddr.W=(RVAddr.W+1)&0x3FFF; }
  return(Port);

  }

  /* No such port */
  return(NORAM);
}

/** OutZ80() *************************************************/
/** Z80 emulation calls this function to write a byte to a  **/
/** given I/O port.                                         **/
/*************************************************************/
void OutZ80(register word Port,register byte Value)
{
  static byte SR,VR; /* Sound and VDP register storage */

  switch(Port&0xE0)
  {

case 0x80: JoyMode=0;return;
case 0xC0: JoyMode=1;return;
case 0xE0: Write76489(&PSG,Value);return;

case 0x40:
  if(Adam&&(Port==0x40)) fputc(Value,PrnStream);
  return;

case 0xA0:
  if(Port&0x01)
    if(VKey) { VR=Value;VKey--; }
    else
    {
      VKey++;
      switch(Value&0xC0)
      {
        case 0x80: VDPOut(Value&0x07,VR);break;
        case 0x40: WVAddr.B.l=VR;WVAddr.B.h=Value&0x3F;break;
        case 0x00: RVAddr.B.l=VR;RVAddr.B.h=Value;
      }
    }
  else
    if(VKey)
    { VRAM[WVAddr.W]=Value;WVAddr.W=(WVAddr.W+1)&0x3FFF; }
  return;

  }
}

/** LoopZ80() ************************************************/
/** Z80 emulation calls this function periodically to check **/
/** if the system hardware requires any interrupts.         **/
/*************************************************************/
word LoopZ80(Z80 *R)
{
  static byte UCount=0;
  static byte ACount=0;

  /* Next scanline */
  CurLine=(CurLine+1)%193;

  /* Refresh scanline if needed */
  if(CurLine<192)
  {
    if(!UCount) (SCR[ScrMode].Refresh)(CurLine);
    R->IPeriod=HPeriod;
    return(INT_NONE);
  }

  /* End of screen reached... */

  /* Set IPeriod to the beginning of next screen */
  R->IPeriod=VPeriod-HPeriod*192;

  /* Check joysticks */
  Joysticks();

  /* Autofire emulation */
  ACount=(ACount+1)&0x07;
  if(ACount>3)
  {
    if(AutoA) { JoyState[0]|=0x0040;JoyState[1]|=0x0040; }
    if(AutoB) { JoyState[0]|=0x4000;JoyState[1]|=0x4000; }
  }

  /* Writing interrupt timestamp into sound log */
  if(LogSnd&&SndStream) fputc(0xFF,SndStream);

  /* Flush any accumulated sound changes */
  Sync76489(&PSG,PSG_FLUSH);

  /* Refresh screen if needed */
//  if(UCount) UCount--; else { UCount=UPeriod-1;RefreshScreen(); }
  RefreshScreen();

  /* Setting VDPStatus flags */
  VDPStatus=(VDPStatus&0xDF)|0x80;

  /* Checking sprites: */
  if(ScrMode) CheckSprites();

  /* If exit requested, return INT_QUIT */
  if(ExitNow) return(INT_QUIT);

  /* Generate VDP interrupt */
  return(VKey&&(VDP[1]&0x20)? INT_NMI:INT_NONE);
}

/** VDPOut() *************************************************/
/** Emulator calls this function to write byte V into a VDP **/
/** register R.                                             **/
/*************************************************************/
void VDPOut(register byte R,register byte V)
{
  register byte J;

  switch(R)
  {
    case  0: switch(((V&0x0E)>>1)|(VDP[1]&0x18))
             {
               case 0x10: J=0;break;
               case 0x00: J=1;break;
               case 0x01: J=2;break;
               case 0x08: J=3;break;
               default:   J=ScrMode;
             }
             if(J!=ScrMode)
             {
               ChrTab=VRAM+((long)(VDP[2]&SCR[J].R2)<<10);
               ChrGen=VRAM+((long)(VDP[4]&SCR[J].R4)<<11);
               ColTab=VRAM+((long)(VDP[3]&SCR[J].R3)<<6);
               SprTab=VRAM+((long)(VDP[5]&SCR[J].R5)<<7);
               SprGen=VRAM+((long)VDP[6]<<11);
               ScrMode=J;
             }
             break;
    case  1: switch(((VDP[0]&0x0E)>>1)|(V&0x18))
             {
               case 0x10: J=0;break;
               case 0x00: J=1;break;
               case 0x01: J=2;break;
               case 0x08: J=3;break;
               default:   J=ScrMode;
             }
             if(J!=ScrMode)
             {
               ChrTab=VRAM+((long)(VDP[2]&SCR[J].R2)<<10);
               ChrGen=VRAM+((long)(VDP[4]&SCR[J].R4)<<11);
               ColTab=VRAM+((long)(VDP[3]&SCR[J].R3)<<6);
               SprTab=VRAM+((long)(VDP[5]&SCR[J].R5)<<7);
               SprGen=VRAM+((long)VDP[6]<<11);
               ScrMode=J;
             }
             break;
    case  2: ChrTab=VRAM+((long)(V&SCR[ScrMode].R2)<<10);break;
    case  3: ColTab=VRAM+((long)(V&SCR[ScrMode].R3)<<6);break;
    case  4: ChrGen=VRAM+((long)(V&SCR[ScrMode].R4)<<11);break;
    case  5: SprTab=VRAM+((long)(V&SCR[ScrMode].R5)<<7);break;
    case  6: V&=0x3F;SprGen=VRAM+((long)V<<11);break;
    case  7: FGColor=V>>4;BGColor=V&0x0F;break;

  }
  VDP[R]=V;return;
}

/** CheckSprites() *******************************************/
/** This function is periodically called to check for the   **/
/** sprite collisions and 5th sprite, and set appropriate   **/
/** bits in the VDP status register.                        **/
/*************************************************************/
void CheckSprites(void)
{
  register word LS,LD;
  register byte DH,DV,*PS,*PD,*T;
  byte I,J,N,*S,*D;

  VDPStatus=(VDPStatus&0x9F)|0x1F;
  for(N=0,S=SprTab;(N<32)&&(S[0]!=208);N++,S+=4);

  if(Sprites16x16)
  {
    for(J=0,S=SprTab;J<N;J++,S+=4)
      if(S[3]&0x0F)
        for(I=J+1,D=S+4;I<N;I++,D+=4)
          if(D[3]&0x0F)
          {
            DV=S[0]-D[0];
            if((DV<16)||(DV>240))
            {
              DH=S[1]-D[1];
              if((DH<16)||(DH>240))
              {
                PS=SprGen+((long)(S[2]&0xFC)<<3);
                PD=SprGen+((long)(D[2]&0xFC)<<3);
                if(DV<16) PD+=DV; else { DV=256-DV;PS+=DV; }
                if(DH>240) { DH=256-DH;T=PS;PS=PD;PD=T; }
                while(DV<16)
                {
                  LS=((word)*PS<<8)+*(PS+16);
                  LD=((word)*PD<<8)+*(PD+16);
                  if(LD&(LS>>DH)) break;
                  else { DV++;PS++;PD++; }
                }
                if(DV<16) { VDPStatus|=0x20;return; }
              }
            }
          }
  }
  else
  {
    for(J=0,S=SprTab;J<N;J++,S+=4)
      if(S[3]&0x0F)
        for(I=J+1,D=S+4;I<N;I++,D+=4)
          if(D[3]&0x0F)
          {
            DV=S[0]-D[0];
            if((DV<8)||(DV>248))
            {
              DH=S[1]-D[1];
              if((DH<8)||(DH>248))
              {
                PS=SprGen+((long)S[2]<<3);
                PD=SprGen+((long)D[2]<<3);
                if(DV<8) PD+=DV; else { DV=256-DV;PS+=DV; }
                if(DH>248) { DH=256-DH;T=PS;PS=PD;PD=T; }
                while((DV<8)&&!(*PD&(*PS>>DH))) { DV++;PS++;PD++; }
                if(DV<8) { VDPStatus|=0x20;return; }
              }
            }
          }
  }
}

/** Play() ***************************************************/
/** Log and play sound of given frequency (Hz) and volume   **/
/** (0..255) via given channel (0..3).                      **/
/*************************************************************/
void Play(int C,int F,int V)
{
  static char SndHeader[20] =
    "SND\032\001\005\074\0\0\0\0\0\0\0\0\0\376\003\002";
  static char SndOff[17] =
    "\0\0\0\0\001\0\0\0\002\0\0\0\003\0\0\0";
  static byte OldLogSnd=0;
  byte Buf[4];

  if(LogSnd&&SndName)
  {
    /* If SndStream closed, open it */
    if(!SndStream)
    {
      SndStream=fopen(SndName,"wb");
      if(SndStream)
        fwrite(SndHeader,1,19,SndStream);
      else
        SndName=NULL;
    }
    /* If SndStream ok, log into it */
    if(SndStream)
    {
      /* Log sound change, if SndStream open */
      Buf[0]=C;Buf[1]=F&0xFF;Buf[2]=F>>8;Buf[3]=V;
      fwrite(Buf,1,4,SndStream);
    }
  }
  else if(OldLogSnd&&SndStream) fwrite(SndOff,1,16,SndStream);

  /* Save LogSnd value for the next time */
  OldLogSnd=LogSnd;

#ifdef SOUND
  /* Play actual sound */
  Sound(C,F,V);
#endif
}
