/*************************************************************************/
/*																		 */
/*		 File: indent.cr												 */
/*	  Created: 4 Jul 1994												 */
/* Description: Indentation support macro via BPACKAGES 				 */
/*																		 */
/*************************************************************************/

# include "crisp.h"

# define	MIN_ABBREV	1			  /* Increase this to specify longer */
										/* default minimum abbreviations */

int 	_c_min_abbrev,
		_c_smart,							  /* C smart keyboard handle */
		_c_template,					   /* C template keyboard handle */
		_c_alt_template,					/* C alt template kbd handle */
		_r_keymap,									/* Regular indenting */
		_c_indent_open,
		_c_indent_close,
		_c_indent_first;

void
main()
{
	/*****************************************/
	/*	  Set up smart indenting for C		 */
	/*****************************************/
	keyboard_push();
	assign_to_key("<Enter>",    "_c_indent");
	assign_to_key("<Tab>",      "_slide_in");
	assign_to_key("<Shift-Tab>","_slide_out");
	assign_to_key("<{>",        "_c_open_brace");
	assign_to_key("<}>",        "_c_close_brace");
	assign_to_key("<Alt-=>",	"align_comments");
//	assign_to_key("<Ctrl-{>",   "_just_brace");
//	assign_to_key("<Ctrl-}>",   "_just_cbrace");
	_c_smart = inq_keyboard();
	keyboard_pop(1);
	/*****************************************/
	/*	  Set up template indenting for C	 */
	/*****************************************/
	keyboard_push();
	assign_to_key("<Enter>",    "_c_indent");
	assign_to_key("<Tab>",      "_slide_in");
	assign_to_key("<Shift-Tab>","_slide_out");
	assign_to_key("<Space>",    "_c_abbrev");
	assign_to_key("<{>",        "_brace_expand");
	assign_to_key("<}>",        "_c_close_brace");
	assign_to_key("<Alt-=>",	"align_comments");
//	assign_to_key("<Ctrl-{>",   "_just_brace");
//	assign_to_key("<Ctrl-}>",   "_just_cbrace");
//	assign_to_key("<Ctrl-S>",   "_just_space");
	_c_template = inq_keyboard();
	keyboard_pop(1);
	/*****************************************/
	/*	  Set up alt template C indenting	 */
	/*****************************************/
	keyboard_push();
	assign_to_key("<Enter>",    "_c_indent");
	assign_to_key("<Tab>",      "_c_abbrev");
	assign_to_key("<Shift-Tab>","_slide_out");
	assign_to_key("<{>",        "_brace_expand");
	assign_to_key("<}>",        "_c_close_brace");
	assign_to_key("<Alt-=>",	"align_comments");
//	assign_to_key("<Ctrl-{>",   "_just_brace");
//	assign_to_key("<Ctrl-}>",   "_just_cbrace");
//	assign_to_key("<Ctrl-S>",   "_just_space");
	_c_alt_template = inq_keyboard();
	keyboard_pop(1);
	/*****************************************/
	/*	  Regular indenting 				 */
	/*****************************************/
	keyboard_push();
	assign_to_key("<Enter>",    "_r_indent");
	assign_to_key("<Tab>",      "_slide_in");
	assign_to_key("<Shift-Tab>","_slide_out");
	_r_keymap = inq_keyboard();
	keyboard_pop(1);

}


/*************************************************************************/
/*																		 */
/*	  c_template_first, _cpp_template_first, _c_smart_first,			 */
/*	  _cpp_smart_first, _regular_first: 								 */
/*																		 */
/*	  These macros are called by the BPACKAGES parser in language.cr	 */
/*	  They initialize the local keymaps for the various indenting		 */
/*	  functions, set the abbreviation length and adjust the 			 */
/*	  indenting style.													 */
/*																		 */
/*************************************************************************/

string
_c_template_first()
{
	_c_min_abbrev = MIN_ABBREV;
	get_parm( 0, _c_min_abbrev);
	if (_c_min_abbrev == 0)
	{
		use_local_keyboard(_c_alt_template);
		_c_min_abbrev = 1;
	}
	else
		use_local_keyboard(_c_template);
	_c_indent_open	= 0;
	_c_indent_close = 0;
	_c_indent_first = 0;
	get_parm(1, _c_indent_open);
	get_parm(2, _c_indent_close);
	get_parm(3, _c_indent_first);
	return "";
}

string
_cpp_template_first()
{
	return _c_template_first();
}

string
_cr_template_first()
{
	return _c_template_first();
}

string
_c_smart_first()
{
	use_local_keyboard(_c_smart);
	_c_indent_open	= 0;
	_c_indent_close = 0;
	_c_indent_first = 0;
	get_parm(0, _c_indent_open);
	get_parm(1, _c_indent_close);
	get_parm(2, _c_indent_first);
	return "";
}

string
_cpp_smart_first()
{
	return _c_smart_first();
}

string
_cr_smart_first()
{
	return _c_smart_first();
}

string
_regular_first()
{
	use_local_keyboard(_r_keymap);
	return "";
}



/*************************************************************************/
/*																		 */
/*	  c_indent: 														 */
/*																		 */
/*	  This macro does syntax-sensitive indenting ("smart indenting")	 */
/*	  for C language files. 											 */
/*																		 */
/*************************************************************************/

# define	SEMI_COLON	1						   /* Code for semicolon */
# define	OPEN_BRACE	2						  /* Code for open brace */
# define	CLOSE_BRACE 3						 /* Code for close brace */
# define	FULL_COLON	4						/* Code for colon (case) */
# define	COMMA		5							   /* Code for comma */
# define	CLOSE_PAREN 6					   /* code for closing paren */
# define	EQUALS		7			 /* code for equals (initialisation) */

string	C_TERM_CHR = ";{}:,)=";

# define	IGNORE_LIST_SIZE	3	/* Set to number of elements in list */
list c_ignore_list =
{
	"[ \\t]@{/\\*}+{[~\\*]|[~/]}*{\\*/}",				   /* C comments */
	"[ \\t]@//*$",										 /* C++ comments */
	"[ \\t]@#*$",								  /* Pre-processor lines */
};

string
trm(string src)
{
	return ltrim(trim(src, " \t\n"));
}

/*************************************************************************/
/*    Finds the previous syntactically significant line					 */
/*************************************************************************/
string
_c_prevline()
{
	int		curr_line,
			at_line,
			patt_no,
			done;
	string	line;

	inq_position(curr_line, NULL);
	/*****************************************/
	/*    Move to the line we wish to		 */
	/*    start at							 */
	/*****************************************/
	at_line = curr_line;
	for (done = FALSE; !done; ) 
	{
		/*****************************************/
		/*    Process each line, and find		 */
		/*    one with text that cannot be		 */
		/*    elimited							 */
		/*****************************************/
		beginning_of_line();
		line = read();
		for (patt_no = 0; patt_no < IGNORE_LIST_SIZE; ++patt_no)
		{
			int	patt_len,
				gotcha;
			/*****************************************/
			/*    Keep removing discardable text	 */
			/*****************************************/
			while ((gotcha = search_string(c_ignore_list[patt_no], line, patt_len)))
				line = substr(line,1,gotcha-1) + substr(line,gotcha+patt_len);
		}
		/*****************************************/
		/*    We've now eliminated all			 */
		/*    comment/discardable text. Trim	 */
		/*    the string, and if it's empty		 */
		/*    move back one more line. If		 */
		/*    we've reached the beginning of	 */
		/*    the buffer, simply return an		 */
		/*    empty string						 */
		/*    If there's anything left,			 */
		/*    return that						 */
		/*****************************************/
		line = trm(line);
		if (line != "")
			++done;
		else
		{
			if (!--at_line)
				++done;
			else
				up();
		}
	}
	return line;
}

int
_c_indent()
{
	int 	curr_indent_col,			/* Current unmodified indent col */
			following_position,/* Column of end of line following cursor */
			curr_col,				  /* Column cursor is on when called */
			curr_line,					/* Column line is on when called */
			what_is_char_1,  /* End of first line's character identifier */
			what_is_char_2, 		 /* End of second line's indentifier */
			level,							  /* Current indenting level */
			tmp_level;

	string	following_string,	  /* All characters following the cursor */
			line_end;

	/*****************************************/
	/*	  Gather information on the two 	 */
	/*	  previous non-blank lines. 		 */
	/*****************************************/
	if (!inq_mode())
		end_of_line();
	inq_position(curr_line, curr_col);
	end_of_line();
	inq_position(NULL,following_position);
	/*****************************************/
	/*	  If there are characters			 */
	/*	  following the cursor, save		 */
	/*	  them in following_string. 		 */
	/*****************************************/
	if (following_position > curr_col)
	{
		drop_anchor(MK_NONINC);
		move_abs(0, curr_col);
		following_string = ltrim(read());
		delete_block();
	}

	line_end = _c_prevline(curr_line);				/* Retrieve previous line */
	inq_position(tmp_level, NULL);
	if (line_end != "")
	{
		what_is_char_2 = index(C_TERM_CHR, substr(line_end, strlen(line_end)));
		beginning_of_line();
		re_search(NULL,"[~ \t]");
		inq_position(NULL, curr_indent_col);
		up();
		line_end = _c_prevline();
		inq_position(level, NULL);
		if (line_end != "")
			what_is_char_1 = index(C_TERM_CHR, substr(line_end, strlen(line_end)));
		else
			what_is_char_1 = SEMI_COLON;
	}
	else
	{
		what_is_char_1 = what_is_char_2 = SEMI_COLON;
		curr_indent_col = 1;
	}
	move_abs(curr_line, curr_indent_col);

	/*****************************************/
	/*	  We've determined the last two      */
	/*	  non-blank lines' last              */
	/*	  characters as well as the 		 */
	/*	  column position of the first		 */
	/*	  non-blank character. Now we		 */
	/*	  position the cursor on the new	 */
	/*	  line's proper level.               */
	/*****************************************/
	if (curr_indent_col == 1 && !_c_indent_first && what_is_char_2 == OPEN_BRACE)
		curr_indent_col += distance_to_tab();
	else
	{
		/*****************************************/
		/*    The following switch statement	 */
		/*    is the body of the _c_indent		 */
		/*    macro indenting rules.			 */
		/*    what_is_char_2 is the				 */
		/*    terminator on the immediately		 */
		/*    preceeding line, what_is_char_1	 */
		/*    is the terminator prior that.		 */
		/*    This allows us to make some		 */
		/*    guesses about whether we			 */
		/*    should indent or outdent from		 */
		/*    the current level.				 */
		/*****************************************/
		switch (what_is_char_2)
		{
		case SEMI_COLON:
			if (!what_is_char_1)
				curr_indent_col -= distance_to_tab();
			break;
		case CLOSE_BRACE:
			if (_c_indent_close)
				curr_indent_col -= distance_to_tab();
			break;
		case OPEN_BRACE:
			if (!_c_indent_open)
				curr_indent_col += distance_to_tab();
			break;
		case COMMA:
			if (what_is_char_1 != COMMA)
				curr_indent_col += distance_to_tab();
			break;
		case EQUALS:
		case FULL_COLON:
		case CLOSE_PAREN:
			curr_indent_col += distance_to_tab();
			break;
		default:
			/* Nothing */
			break;
		}
	}

	move_abs(0, curr_col);

	if (get_parm(0, curr_col) == 0)
	{
		if (inq_assignment(inq_command(), 1) == "<Enter>")
			self_insert(key_to_int("<Enter>"));
		else
			self_insert(key_to_int("<Ctrl-M>"));
	}
	beginning_of_line();
	curr_col = distance_to_tab() + 1;

	level = 0;
	while(curr_col <= curr_indent_col)
	{
		move_abs(0, curr_col);
		++level;
		curr_col += distance_to_tab();
	}
	if (following_string != "")
	{
		following_string = substr(following_string, 1, strlen(following_string) - 1);
		save_position();
		insert(following_string);
		restore_position();
	}
	return level;
}



/*************************************************************************/
/*																		 */
/*	  _r_indent:														 */
/*																		 */
/*	  This macro does "standard" style indenting, indenting new          */
/*	  lines to the same column as the previous non-blank line.			 */
/*																		 */
/*************************************************************************/


void
_r_indent()
{
	int 	curr_indent_col,
			following_position,
			curr_col,
			curr_line;
	string	following_string;

	/*****************************************/
	/*	  First, check to see if there		 */
	/*	  are any following characters		 */
	/*	  on the line. If so, read them 	 */
	/*	  into a temporary variable and 	 */
	/*	  delete from the first line.		 */
	/*	  Get column of last non-blank		 */
	/*	  line. 							 */
	/*****************************************/
	if (!inq_mode())
		end_of_line();
	inq_position(curr_line, curr_col);
	end_of_line();
	inq_position(NULL, following_position);

	if (following_position > curr_col)
	{
		drop_anchor(MK_NONINC);
		move_abs(0, curr_col);
		following_string = ltrim(read());
		delete_block();
	}
	if (search_back("<*\\c[~ \\t\\n]"))
		inq_position(NULL, curr_indent_col);
	else
		curr_indent_col = 1;

	move_abs(curr_line, curr_col);

	/*****************************************/
	/*	  We've determined the last          */
	/*	  non-blank lines' indent level      */
	/*	  -- do return, indent, line		 */
	/*	  split, and clean up.				 */
	/*****************************************/
	if (inq_assignment(inq_command(), 1) == "<Enter>")
		self_insert(key_to_int("<Enter>"));
	else
		self_insert(key_to_int("<Ctrl-M>"));

	move_abs(0, curr_indent_col);

	if (following_string != "")
	{
		following_string = substr(following_string, 1, strlen(following_string) - 1);
		insert(following_string);
		move_abs(0, curr_indent_col);
	}
}


/*************************************************************************/
/*																		 */
/*	  _slide_in:														 */
/*																		 */
/*	  This macros slides a marked block of text in one tab stop per 	 */
/*	  press of the tab key (-->|).										 */
/*																		 */
/*************************************************************************/

void
_slide_in()
{
	int 	start_line,
			start_col,
			end_line,
			mark_type;

	mark_type = inq_marked(start_line, start_col, end_line);
	if (mark_type !=MK_NONE)
	{
		int 	cursor_line,
				cursor_col,
				mark_line,
				mark_col,
				old_mode,
				tab_key;

		inq_position(cursor_line, cursor_col);
		swap_anchor();
		inq_position(mark_line, mark_col);

		if (mark_type != MK_COLUMN)
			start_col = 1;
		move_abs(start_line, start_col);
		raise_anchor();

		old_mode = inq_mode();
		insert_mode(TRUE);
		tab_key = key_to_int("<Tab>");

		while (start_line <= end_line)
		{
			search_fwd("[~ \t]");

			if (read(1) != "\n")
			{
				/*****************************************/
				/*	  Note that self_inserting the		 */
				/*	  complete key definition			 */
				/*	  ensures the Tab is converted		 */
				/*	  to spaces if -t is used.			 */
				/*****************************************/
				self_insert(tab_key);
			}
			move_abs(++start_line, start_col);
		}
		move_abs(mark_line, mark_col);
		drop_anchor(mark_type);
		move_abs(cursor_line, cursor_col);
		insert_mode(old_mode);
	}
	else
		self_insert(key_to_int("<Tab>"));
}

/*************************************************************************/
/*																		 */
/*	  _slide_out:														 */
/*																		 */
/*	  This macros slides a marked block of text out one tab stop per	 */
/*	  press of the back-tab key (|<--). 								 */
/*																		 */
/*************************************************************************/

void
_slide_out()
{
	int 	start_line,
			start_col,
			end_line,
			mark_type;

	mark_type = inq_marked(start_line, start_col, end_line);
	if (mark_type != MK_NONE)
	{

		save_position();
		if (mark_type != MK_COLUMN)
			start_col = 1;
		move_abs(start_line, start_col);
		while(start_line <= end_line)
		{
			int 	curr_col;

			search_fwd("[~ \t]");
			inq_position(NULL, curr_col);
			if (curr_col > start_col)
			{
				drop_anchor(MK_NONINC);
				_back_tab();
				inq_position(NULL, curr_col);
				if (curr_col < start_col)
					move_abs(0, start_col);
				delete_block();
			}
			move_abs(++start_line, start_col);
		}
		restore_position();
	}
	else
		_back_tab();
}


/*************************************************************************/
/*																		 */
/*	  Template editing macros:											 */
/*																		 */
/*	  These macro performs simple template editing for C programs.		 */
/*	  Each time the space bar is pressed, the characters before the 	 */
/*	  cursor are checked to see if they are "if", "else if",             */
/*	  "while", "for", "do", switch" or "case" (or abbreviations for      */
/*	  them). These keywords must only be preceded with spaces or		 */
/*	  tabs, and be typed at the end of a line. If a match is found, 	 */
/*	  the remainder of the statement is filled in automatically.		 */
/*																		 */
/*	  In addition, a brace pairer is included -- this automatically 	 */
/*	  inserts a matching brace at the proper indentation level when 	 */
/*	  an opening brace is typed in. To insert a brace without a 		 */
/*	  matching brace (it attempts to be smart about matching braces,	 */
/*	  but you never can make this type of thing quite smart enough),	 */
/*	  type either Ctrl-{ or quote the brace with Alt-q. 				 */
/*																		 */
/*************************************************************************/

# define	DO_BRACE	1
# define	DO_CLOSE	2
# define	DO_WHILE	4

# define	ABBR_COMPL	1
# define	ABBR_FLAGS	2
# define	ABBR_POSL	3
# define	ABBR_POSC	4

list c_abbrev_list =
{
	/* match,	completion, flags, rel col shift, rel line shift */
	"\xffreturn",	"return ;", 					0,				 0, -1,
	"\xffelse if",	"else if ()",					0,				 0, -1,
	"\xffelse", 	"else ",						0,				 0,  0,
	"\xffif ",		"if ()",						0,				 0, -1,
	"\xffwhile",	"while ()", 					0,				 0, -1,
	"\xfffor",		"for (;;)", 					0,				 0, -3,
	"\xffdo",		"do",							DO_WHILE,		 2,  2,
	"\xffswitch",	"switch ()",					DO_BRACE,		 0, -1,
	"\xffcase", 	"case :",						0,				 0, -1,
	"\xffdefault",	"default:\n",					0,				 0, -1,
	"\xffmain", 	"main (int argc, char *argv[])",DO_BRACE,		 2,-25,
	"\xff#if",		"# if ",						0,				 0,  0,
	"\xff#ifdef",	"# if defined(	)", 			0,				 0, -2,
	"\xff#endif",	"# endif\t",					0,				 0, -1,
	"\xff#include", "# include <>", 				0,				 0, -1,
	"\xff#define",	"# define\t",					0,				 0,  0,
# if MIN_ABBREV > 1 												
	"\xff# if", 	"# if ",						0,				 0,  0,
	"\xff# ifdef",	"# if defined(	)", 			0,				 0, -2,
	"\xff# endif",	"# endif\t",					0,				 0, -1,
	"\xff# include","# include <>", 				0,				 0, -1,
	"\xff# define", "# define\t",					0,				 0,  0,
# endif
};

/*************************************************************************/
/*																		 */
/*	  _c_abbrev:														 */
/*																		 */
/*	  This function checks to see if the characters before the space	 */
/*	  just typed (and actually inserted by this function) are			 */
/*	  destined to be followed by the remainder of a C construct.		 */
/*																		 */
/*************************************************************************/

void
_c_abbrev()
{
	int 	done = FALSE;

	if (read(1) == "\n")
	{
		int 	length, atstart;
		string	line;

		save_position();
		beginning_of_line();
		line = read(1);
		atstart = (line == " " || line == "\t");
		line = trm(read());
		restore_position();
		length = strlen(line);

		if (length >= _c_min_abbrev)
		{
			int 	loc;

			line = "\xff" + line;
			if ((loc = re_search(NULL, line, c_abbrev_list)) >= 0)
			{
				string		completion;

				prev_char(length);
				delete_char(length);
				completion = c_abbrev_list[loc+ABBR_COMPL];
				insert(completion);
				save_position();
				if (c_abbrev_list[loc+ABBR_FLAGS] & DO_WHILE)
				{
					_open_line();
					_brace_expand();
					move_rel(1, 0);
					_open_line();
					insert("while();");
				}
				if (c_abbrev_list[loc+ABBR_FLAGS] & DO_BRACE)
				{
					_open_line();
					_brace_expand();
				}
				restore_position();
				move_rel(c_abbrev_list[loc+ABBR_POSL], c_abbrev_list[loc+ABBR_POSC]);
				done = TRUE;
			}
		}
	}
	if (!done)
	{
		if (inq_local_keyboard() == _c_alt_template)
			slide_in();
		else
			self_insert();
	}
}


/*************************************************************************/
/*																		 */
/*	  brace_expand: 													 */
/*																		 */
/*	  This function checks to see if the typed brace should be			 */
/*	  indented and matched to the current indenting level.				 */
/*																		 */
/*************************************************************************/


void
_brace_expand()
{
	string	line;

	save_position();
	beginning_of_line();
	line = read();
	restore_position();
	_c_open_brace();
	save_position();
	_c_indent();
	_c_close_brace();
	restore_position();
}


void
_c_open_brace()
{
	if (trm(read()) != "")
		self_insert('{');
	else
	{
		int		curr_line,
				curr_col;

		inq_position(curr_line, curr_col);
		beginning_of_line();
		if (trm(read()) == "")
		{
			/*****************************************/
			/*    Align open brace at previous		 */
			/*    indent level						 */
			/*****************************************/
			_c_prevline();
			beginning_of_line();
			re_search(NULL,"[~ \t]");
			inq_position(NULL, curr_col);
		}
		move_abs(curr_line, curr_col);
		self_insert('{');
		_c_indent();
	}
}


void
_c_close_brace()
{
	if (trm(read()) == "")
	{
		int		curr_line,
				curr_col,
				do_backtab = FALSE;

		inq_position(curr_line, curr_col);
		beginning_of_line();
		if (trm(read()) == "")
		{
			if (!_c_indent_first || !_c_indent_close)
				++do_backtab;
		}
		move_abs(curr_line, curr_col);
		if (do_backtab && curr_col > 1)
			_back_tab();
	}
	self_insert('}');
}


/*************************************************************************/
/*																		 */
/*	  _just_brace, _just_cbrace, _just_space:							 */
/*																		 */
/*	  These functions insert unexpanded and expanded braces and 		 */
/*	  spaces.															 */
/*																		 */
/*************************************************************************/


void
_just_brace()
{
	insert("{");
}

void
_just_cbrace()
{
	insert("}");
}

void
_just_space()
{
	insert(" ");
}


/*************************************************************************/
/*    																	 */
/*    _c_comment_align													 */
/*    																	 */
/*    Aligns C/C++ comments on the right side for easy readin'			 */
/*    																	 */
/*************************************************************************/

int
	c_align_column = 76;					 /* column to right align on */

int
indexn (string s, int start, string subs)
{
	int		i;
	if (start < 1)
		start = 1;
	else
		s = substr(s, start);
	i = index(s, subs);
	if (i)
		i += start - 1;
	return i;
}

void
_c_comment_align()
{
	int		cur_line,
			cur_col;
	string	line;

	inq_position(cur_line, cur_col);
	beginning_of_line();
	line = detab(read());		 			 /* Read all of current line */
	if (line != "")
	{
		int		posc,
				posp,
				done,
				toadd;
		string	work,
				comment;

		work = line;
		for (toadd = done = 0; !done; )
		{
			int pquote,
				nquote;

			/*****************************************/
			/*	  Handle C style comments			 */
			/*****************************************/
			posc = re_search(SF_LENGTH,"{/\\*}+{[~\\*]|[~/]}*{\\*/}$", work);
			/*****************************************/
			/*    Handle C++ style comments			 */
			/*****************************************/
			posp = re_search(SF_LENGTH, "//*$", work);
			if (posp > posc)				  /* Select last one in line */
				posc = posp;
			else
				posp = 0;			  /* posp is non-zero if c++ comment */

			if (!posc)									/* Nothing found */
				++done;
			else
			{
				/*****************************************/
				/*    Make sure they aren't in quotes	 */
				/*****************************************/
				while ((pquote = indexn(work, pquote+1, "\"")) > 0 && pquote < posc)
				{
					if (pquote > 1)
					{
						string ch = substr(work, pquote-1, 1);
						if (ch == "\\" || ch == "'")
							continue;			 /* Ignore escaped quote */
					}
					++nquote;
				}
				if (!(nquote & 1))					/* Even number is ok */
					++done;
				else
				{
					/*****************************************/
					/*    At this point we know comment		 */
					/*    is quoted							 */
					/*****************************************/
					if (!pquote)
					{
						/*****************************************/
						/*    No quote after comment mark?		 */
						/*    Don't touch this line				 */
						/*****************************************/
						posc = 0;
						++done;
					}
					toadd = pquote;
					work = substr(work, pquote+1);
				}
			}
		}
		if (posc > 0)
		{
			/*****************************************/
			/*    Now, reposition the line by		 */
			/*    splitting the line and			 */
			/*    comment, stripping spaces off		 */
			/*    the end of the code line,			 */
			/*    deleting the line and				 */
			/*    reinserting it and positioning	 */
			/*    the comment at the right			 */
			/*    position aligned on the right		 */
			/*    side								 */
			/*****************************************/
			posc += toadd;
			comment = substr(line, posc);
			line = trim(substr(line, 1, posc - 1), " \t\n");
			/*****************************************/
			/*    We have to be careful if the		 */
			/*    line + comment length exceeds		 */
			/*    the column at the right			 */
			/*****************************************/
			if ((strlen(line) + strlen(comment)) >= c_align_column)
			{
				line += comment;
				comment = "";
			}
			/*****************************************/
			/*    Now, remove the line and			 */
			/*    insert a newline					 */
			/*****************************************/
			delete_line();
			insert("\n");
			/*****************************************/
			/*    Move back up and reinsert the		 */
			/*    line								 */
			/*****************************************/
			prev_char();
			insert(line);
			if (comment != "")
			{
				/*****************************************/
				/*    Place the comment in virtual		 */
				/*    space and insert					 */
				/*****************************************/
				move_abs(NULL, c_align_column - strlen(comment));
				insert(comment);
			}
			beginning_of_line();
			/*****************************************/
			/*    Entab the line again				 */
			/*****************************************/
			line = entab(read());
			delete_line();
			insert(line);
		}
	}
	move_abs(cur_line, cur_col);
}


/*************************************************************************/
/*    Align comments for a bounded region of lines						 */
/*************************************************************************/
void
c_align_comments(int startline, int endline)
{
	save_position();
	while (startline <= endline)
	{
		move_abs(startline++, 1);
		_c_comment_align();
	}
	restore_position();
}

/*************************************************************************/
/*    Align comments for marked region or entire buffer					 */
/*************************************************************************/
void
align_comments()
{
	int		mark_start,
			mark_end;

	if (!inq_marked(mark_start, NULL, mark_end, NULL))
	{
		mark_start = 1;
		mark_end = inq_lines();
	}
	c_align_comments(mark_start, mark_end);
}
