/* ------------------------------------------------------------------------
 *
 *       BBBBBBBB    IIIIIIIIII  NN      NN  DDDDDDDD
 *       BB     BB       II      NNN     NN  DD     DD
 *       BB      BB      II      NNNN    NN  DD      DD
 *       BB     BB       II      NN NN   NN  DD      DD
 *       BBBBBBBB        II      NN  NN  NN  DD      DD
 *       BB     BB       II      NN   NN NN  DD      DD
 *       BB      BB      II      NN    NNNN  DD      DD
 *       BB     BB       II      NN     NNN  DD     DD
 *       BBBBBBBB    IIIIIIIIII  NN      NN  DDDDDDDD
 *
 *       Copyright 1998 by ELSA AG, Aachen (Germany)
 *
 *       @Project : Win32 uart routinen
 *       @System  :
 *       @Modul   : w32uart.c
 *       @Prefix  : xW32Uart
 *       @Hardware: Intel
 *       @Compiler: MSVC 6.0
 *       @Library : s. #include...
 *       @Author  : wf
 *       @Info    : Erstellt fr serielle CAPI
 *       @Status  : %
 *
 * ------------------------------------------------------------------------
 *       @History :
 * 141298 /wf       - Angelegt
 * ------------------------------------------------------------------------ */

/** -----------------------------------------------------------------------
 * Ansteuerung des VCOMM unter WIN32
 * ------------------------------------------------------------------------ */

/*---- include area --------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <windows.h>
#include <process.h>

#define _gsp
//#include "p:\bind\master\ios\common.src\iosdef.h"
//#include "..\common.src\bastypes.h"
#include "w32uart.h"

/*---- define area ---------------------------------------------------------*/

/*---- macro area ----------------------------------------------------------*/

/*---- typedef area --------------------------------------------------------*/

typedef struct
{
  HANDLE Port;
  HANDLE W32UartEventThreadHandle;

  void (*ProcAddrCts     )(void);
  void (*ProcAddrCtsOn   )(void);
  void (*ProcAddrCtsOff  )(void);

  void (*ProcAddrDsr     )(void);
  void (*ProcAddrDsrOn   )(void);
  void (*ProcAddrDsrOff  )(void);

  void (*ProcAddrRing    )(void);
  void (*ProcAddrRingOn  )(void);
  void (*ProcAddrRingOff )(void);

  void (*ProcAddrDcd     )(void);
  void (*ProcAddrDcdOn   )(void);
  void (*ProcAddrDcdOff  )(void);

  void (*ProcAddrBreak   )(void);
  void (*ProcAddrError   )(void);
  void (*ProcAddrRxChar  )(void);
  void (*ProcAddrRxFlag  )(void);
  void (*ProcAddrRx80Full)(void);
  void (*ProcAddrTxEmpty )(void);
} tW32UartData;

/*---- data area -----------------------------------------------------------*/

/*---- function area -------------------------------------------------------*/

/* ------------------------------------------------------------------------
 * @Function:   W32UartError
 * @Job     :   %
 * @Info    :   dient zur anbindung an ein debugsystem
 * @Result  :   %
 * @Family  :   %
 * @Example :   %
 * @Status  :   %
 * ------------------------------------------------------------------------ */

void W32UartError
(tU8 * ErrorMsg)                            /* ^Error-Message               */

{
  printf ("%s",ErrorMsg);
}

/* ------------------------------------------------------------------------
 * @Function:
 * @Job     :
 * @Info    :
 * @Result  :
 * @Family  :
 * @Example :
 * @Status  :
 * ------------------------------------------------------------------------ */

HANDLE W32UartStartThread
(void(*StartAddr)(void *),                  /* Zu startende Function        */
 tU32 pPortData)

{
  tU32   ID;

  return (CreateThread(NULL, 0, (LPTHREAD_START_ROUTINE)StartAddr,
          (void*)pPortData, 0, &ID));
}

/* ------------------------------------------------------------------------
 * @Function:
 * @Job     :
 * @Info    :
 * @Result  :
 * @Family  :
 * @Example :
 * @Status  :
 * ------------------------------------------------------------------------ */

void W32UartStopThread
(HANDLE ThreadHandle)                       /* Handle Thread zum stoppen.   */

{
  TerminateThread (ThreadHandle, 0);
  CloseHandle     (ThreadHandle);
}

/* ------------------------------------------------------------------------
 * @Function:
 * @Job     :
 * @Info    :
 * @Result  :
 * @Family  :
 * @Example :
 * @Status  :
 * ------------------------------------------------------------------------ */

void HandleEvent
(tU32 EventMask,                            /* Maske der Events             */
 tU32 pPortData)

{
  if (EventMask & EV_CTS)
  {
    if (((tW32UartData*)pPortData)->ProcAddrCts)
      ((tW32UartData*)pPortData)->ProcAddrCts   ();
    if (((tW32UartData*)pPortData)->ProcAddrCtsOn   &&   xW32UartCts(pPortData))
      ((tW32UartData*)pPortData)->ProcAddrCtsOn ();
    if (((tW32UartData*)pPortData)->ProcAddrCtsOff  && (!xW32UartCts(pPortData)))
      ((tW32UartData*)pPortData)->ProcAddrCtsOff();
  }

  if (EventMask & EV_DSR)
  {
    if (((tW32UartData*)pPortData)->ProcAddrDsr)
      ((tW32UartData*)pPortData)->ProcAddrDsr   ();
    if (((tW32UartData*)pPortData)->ProcAddrDsrOn   &&   xW32UartDsr(pPortData))
      ((tW32UartData*)pPortData)->ProcAddrDsrOn ();
    if (((tW32UartData*)pPortData)->ProcAddrDsrOff  && (!xW32UartDsr(pPortData)))
      ((tW32UartData*)pPortData)->ProcAddrDsrOff();
  }

  if (EventMask & EV_RLSD)
  {
    if (((tW32UartData*)pPortData)->ProcAddrDcd)
      ((tW32UartData*)pPortData)->ProcAddrDcd   ();
    if (((tW32UartData*)pPortData)->ProcAddrDcdOn   &&   xW32UartDcd(pPortData))
      ((tW32UartData*)pPortData)->ProcAddrDcdOn ();
    if (((tW32UartData*)pPortData)->ProcAddrDcdOff  && (!xW32UartDcd(pPortData)))
      ((tW32UartData*)pPortData)->ProcAddrDcdOff();
  }

  if (EventMask & EV_RING)
  {
    if (((tW32UartData*)pPortData)->ProcAddrRing) ((tW32UartData*)pPortData)->ProcAddrRing   ();
  }

  if (EventMask & EV_BREAK)
  {
    if (((tW32UartData*)pPortData)->ProcAddrBreak) ((tW32UartData*)pPortData)->ProcAddrBreak();
  }

  if (EventMask & EV_ERR)
  {
    if (((tW32UartData*)pPortData)->ProcAddrError) ((tW32UartData*)pPortData)->ProcAddrError();
  }

  if (EventMask & EV_RXCHAR)
  {
    if (((tW32UartData*)pPortData)->ProcAddrRxChar) ((tW32UartData*)pPortData)->ProcAddrRxChar();
  }

  if (EventMask & EV_RXFLAG)
  {
    if (((tW32UartData*)pPortData)->ProcAddrRxFlag) ((tW32UartData*)pPortData)->ProcAddrRxFlag();
  }

  if (EventMask & EV_RX80FULL)
  {
    if (((tW32UartData*)pPortData)->ProcAddrRx80Full) ((tW32UartData*)pPortData)->ProcAddrRx80Full();
  }

  if (EventMask & EV_TXEMPTY)
  {
    if (((tW32UartData*)pPortData)->ProcAddrTxEmpty) ((tW32UartData*)pPortData)->ProcAddrTxEmpty();
  }
}

/* ------------------------------------------------------------------------
 * @Function:
 * @Job     :
 * @Info    :
 * @Result  :
 * @Family  :
 * @Example :
 * @Status  :
 * ------------------------------------------------------------------------ */

void W32UartEventThread
( void *arg)                                /* Dummy                        */

{
  tU32       EventMask;
  OVERLAPPED OsStatus = {0};
  tU32       Dummy;
  tU32       Rc=0;
  tW32UartData *pPortData = arg;

  OsStatus.hEvent = CreateEvent(NULL, FALSE, FALSE, NULL);

  while (1) {

    if (WaitCommEvent (pPortData->Port, &EventMask, &OsStatus))
    {
      HandleEvent (EventMask, (tU32)pPortData);
    }
    else
    {
      if ((Rc = GetLastError()) == ERROR_IO_PENDING)
      {
        WaitForSingleObject(OsStatus.hEvent, INFINITE);
        if (GetOverlappedResult(pPortData->Port, &OsStatus, &Dummy, FALSE))
        {
          HandleEvent (EventMask, (tU32)pPortData);
        }
      }
      else
      {
        W32UartError ("Error 1 in W32UartEventThread()\r\n");
      }
    }
  }
}

/* ------------------------------------------------------------------------
 * @Function:
 * @Job     :
 * @Info    :
 * @Result  :
 * @Family  :
 * @Example :
 * @Status  :
 * ------------------------------------------------------------------------ */

tU32 xW32UartPurgeRxBuff
(tU32 pPortData)                  /*                              */

{
  tU32 Rc = W32UART_PURGE_RX_BUFF_ERROR;

  if (PurgeComm(((tW32UartData*)pPortData)->Port, PURGE_RXCLEAR)) Rc = W32UART_PURGE_RX_BUFF_SUCCESS;
  return Rc;
}

/* ------------------------------------------------------------------------
 * @Function:
 * @Job     :
 * @Info    :
 * @Result  :
 * @Family  :
 * @Example :
 * @Status  :
 * ------------------------------------------------------------------------ */

tU32 xW32UartPurgeTxBuff
(tU32 pPortData)                  /*                              */

{
  tU32 Rc = W32UART_PURGE_TX_BUFF_ERROR;

  if (PurgeComm(((tW32UartData*)pPortData)->Port, PURGE_TXCLEAR)) Rc = W32UART_PURGE_TX_BUFF_SUCCESS;
  return Rc;
}

/* ------------------------------------------------------------------------
 * @Function:
 * @Job     :
 * @Info    :
 * @Result  :
 * @Family  :
 * @Example :
 * @Status  :
 * ------------------------------------------------------------------------ */

tU32 xW32UartPurgeRxTxBuff
(tU32 pPortData)                  /*                              */

{
  tU32 Rc = W32UART_PURGE_RXTX_BUFF_ERROR;

  if (PurgeComm(((tW32UartData*)pPortData)->Port, PURGE_RXCLEAR | PURGE_TXCLEAR))
    Rc = W32UART_PURGE_RXTX_BUFF_SUCCESS;
  return Rc;
}

/* ------------------------------------------------------------------------
 * @Function:
 * @Job     :
 * @Info    :
 * @Result  :
 * @Family  :
 * @Example :
 * @Status  :
 * ------------------------------------------------------------------------ */

tU32 xW32UartOpen
(tU8*  ComPort,                              /* "COM1:" oder "COM2:"...      */
 tU32  BitRate,                              /* gueltige Bitrate             */
 tU32  ByteSize,                             /* 7, 8, ...                    */
 tU32  Parity,                               /* no, odd, even, mark, space   */
 tU32  StopBit,                              /* 1, 1.5 oder 2 StopBits       */
 tU32* pHandle)                             /* Handle                       */

{
  DCB           Dcb;
  tW32UartData* pPortData = malloc (sizeof(tW32UartData));

  memset (pPortData, 0, sizeof(tW32UartData));
  *pHandle = (tU32)pPortData;

  pPortData->Port = CreateFile(ComPort, GENERIC_READ|GENERIC_WRITE, 0, 0,
                               OPEN_EXISTING, FILE_FLAG_OVERLAPPED, 0);

  if(pPortData->Port == INVALID_HANDLE_VALUE)
  {
    return W32UART_OPEN_ERROR_CREATE_FILE;
  }

  if(!GetCommState(pPortData->Port, &Dcb))
  {
    CloseHandle(pPortData->Port);
    return W32UART_OPEN_ERROR_GET_COMM_STATE;
  }

  Dcb.DCBlength    = sizeof(Dcb);
  Dcb.BaudRate     = BitRate;
  Dcb.fBinary      = 1;
  Dcb.fOutxCtsFlow = 1;
  Dcb.fOutxDsrFlow = 0;
  Dcb.fDtrControl  = DTR_CONTROL_ENABLE;
  Dcb.fOutX        = 0;
  Dcb.fInX         = 0;
  Dcb.fRtsControl  = RTS_CONTROL_HANDSHAKE;
  Dcb.ByteSize     = (tU8)ByteSize;
  Dcb.Parity       = (tU8)Parity;
  Dcb.StopBits     = (tU8)StopBit;
  Dcb.EvtChar      = 0;

  if(!SetCommState(pPortData->Port, &Dcb))
  {
    CloseHandle(pPortData->Port);
    return W32UART_OPEN_ERROR_SET_COMM_STATE;
  }

  if (xW32UartPurgeRxTxBuff((tU32)pPortData))
  {
    CloseHandle(pPortData->Port);
    return W32UART_OPEN_PURGE_RXTX_BUFF_ERROR;
  }

  SetCommMask (pPortData->Port, EV_BREAK   | EV_CTS      |  EV_DSR     |
                     EV_ERR     | EV_RING     |  EV_RLSD    |
                   /*  EV_RXCHAR  | EV_RXFLAG |*/EV_TXEMPTY |
                     EV_RX80FULL );

  pPortData->W32UartEventThreadHandle = W32UartStartThread (W32UartEventThread, (tU32)pPortData);

  return W32UART_OPEN_SUCCESS;
}

/* ------------------------------------------------------------------------
 * @Function:
 * @Job     :
 * @Info    :
 * @Result  :
 * @Family  :
 * @Example :
 * @Status  :
 * ------------------------------------------------------------------------ */

tU32 W32UartModemStatus
(tU32 ModemStatusMask,                      /*                              */
 tU32 pPortData)                  /*                              */

{
  tU32 ModemStatus;

  GetCommModemStatus (((tW32UartData*)pPortData)->Port, &ModemStatus);
  return (ModemStatus & ModemStatusMask);
}

/* ------------------------------------------------------------------------
 * @Function:
 * @Job     :
 * @Info    :
 * @Result  :
 * @Family  :
 * @Example :
 * @Status  :
 * ------------------------------------------------------------------------ */

tU32 xW32UartDcd
(tU32 pPortData)                  /*                              */

{
  return (W32UartModemStatus(MS_RLSD_ON, pPortData));
}

/* ------------------------------------------------------------------------
 * @Function:
 * @Job     :
 * @Info    :
 * @Result  :
 * @Family  :
 * @Example :
 * @Status  :
 * ------------------------------------------------------------------------ */

tU32 xW32UartDsr
(tU32 pPortData)                  /*                              */

{
  return (W32UartModemStatus(MS_DSR_ON, pPortData));
}

/* ------------------------------------------------------------------------
 * @Function:
 * @Job     :
 * @Info    :
 * @Result  :
 * @Family  :
 * @Example :
 * @Status  :
 * ------------------------------------------------------------------------ */

tU32 xW32UartRing
(tU32 pPortData)                  /*                              */

{
  return (W32UartModemStatus(MS_RING_ON, pPortData));
}

/* ------------------------------------------------------------------------
 * @Function:
 * @Job     :
 * @Info    :
 * @Result  :
 * @Family  :
 * @Example :
 * @Status  :
 * ------------------------------------------------------------------------ */

tU32 xW32UartCts
(tU32 pPortData)                  /*                              */
{
  return (W32UartModemStatus(MS_CTS_ON, pPortData));
}

/* ------------------------------------------------------------------------
 * @Function:
 * @Job     :
 * @Info    :
 * @Result  :
 * @Family  :
 * @Example :
 * @Status  :
 * ------------------------------------------------------------------------ */

tU32 xW32UartWriteChar
(tU8 WriteChar,                             /* zu schreibender character    */
 void (*ProcCallBack)(void *),              /* callback, wenn fertig (opt.) */
 tU32 pPortData)                  /*                              */
{
  return (xW32UartWriteBlock(&WriteChar, 1, ProcCallBack, pPortData));
}

/* ------------------------------------------------------------------------
 * @Function:
 * @Job     :
 * @Info    :
 * @Result  :
 * @Family  :
 * @Example :
 * @Status  :
 * ------------------------------------------------------------------------ */

tU32 xW32UartWriteBlock
(tU8 *Buff,                                 /* ^block zu versenden          */
 tU32 Len,                                  /* laenge der daten             */
 void (*ProcCallBack)(void *),              /* callback, wenn fertig (opt.) */
 tU32 pPortData)                  /*                              */

{
  OVERLAPPED OsStatus = {0};
  tU32       Dummy         ;
  tU32       Rc       = 0  ;
  tU32       Count    = 0  ;

  OsStatus.hEvent = CreateEvent(NULL, FALSE, FALSE, NULL);

  if (WriteFile(((tW32UartData*)pPortData)->Port, Buff, Len, &Count, &OsStatus))
  {
    if (ProcCallBack) W32UartStartThread (ProcCallBack, (tU32)pPortData);
  }
  else
  {
    if ((Rc = GetLastError()) == ERROR_IO_PENDING)
    {
      WaitForSingleObject(OsStatus.hEvent, INFINITE);
      if (GetOverlappedResult(((tW32UartData*)pPortData)->Port, &OsStatus, &Dummy, FALSE))
      {
        if (ProcCallBack) W32UartStartThread (ProcCallBack, (tU32)pPortData);
      }
    }
    else
    {
      CloseHandle (OsStatus.hEvent);
      W32UartError ("Error 1 in W32UartWriteBlock()\r\n");
      return W32UART_WRITE_BLOCK_ERROR;
    }
  }
  CloseHandle (OsStatus.hEvent);
  return W32UART_WRITE_BLOCK_SUCCESS;
}

/* ------------------------------------------------------------------------
 * @Function:
 * @Job     :
 * @Info    :
 * @Result  :
 * @Family  :
 * @Example :
 * @Status  :
 * ------------------------------------------------------------------------ */

tU32 xW32UartDataAvailable
(tU32 pPortData)                  /*                              */

{
  tU32    Error;
  COMSTAT Status;

  ClearCommError(((tW32UartData*)pPortData)->Port, &Error, &Status);
  return Status.cbInQue;
}

/* ------------------------------------------------------------------------
 * @Function:
 * @Job     :
 * @Info    :
 * @Result  :
 * @Family  :
 * @Example :
 * @Status  :
 * ------------------------------------------------------------------------ */

tU32  xW32UartReadChar
(tU8 *ReadChar,                             /* ^ziel                        */
 void (*ProcCallBack)(void *),              /* callback, wenn fertig (opt.) */
 tU32 pPortData)                  /*                              */

{
  return (xW32UartReadBlock(ReadChar, 1, ProcCallBack, pPortData));
}

/* ------------------------------------------------------------------------
 * @Function:
 * @Job     :
 * @Info    :
 * @Result  :
 * @Family  :
 * @Example :
 * @Status  :
 * ------------------------------------------------------------------------ */

tU32  xW32UartReadBlock
(tU8 *Buff,                                 /* ^ziel                        */
 tU32 Len,                                  /* max. laenge                  */
 void (*ProcCallBack)(void *),              /* callback, wenn fertig (opt.) */
 tU32 pPortData)                  /*                              */

{
  OVERLAPPED OsStatus = {0};
  tU32       Dummy;
  tU32       Rc       = 0;
  tU32       Count    = 0;

  OsStatus.hEvent = CreateEvent(NULL, FALSE, FALSE, NULL);

  if (ReadFile(((tW32UartData*)pPortData)->Port, Buff, Len, &Count, &OsStatus))
  {
    if (ProcCallBack) W32UartStartThread(ProcCallBack, (tU32)pPortData);
  }
  else
  {
    if ((Rc = GetLastError()) == ERROR_IO_PENDING)
    {
      WaitForSingleObject(OsStatus.hEvent, INFINITE);
      if (GetOverlappedResult(((tW32UartData*)pPortData)->Port, &OsStatus, &Dummy, FALSE))
      {
        if (ProcCallBack) W32UartStartThread(ProcCallBack, (tU32)pPortData);
      }
    }
    else
    {
      CloseHandle (OsStatus.hEvent);
      W32UartError ("Error 1 in W32UartReadBlock()\r\n");
      return W32UART_READ_BLOCK_ERROR;
    }
  }
  CloseHandle (OsStatus.hEvent);
  return W32UART_READ_BLOCK_SUCCESS;
}

/* ------------------------------------------------------------------------
 * @Function:
 * @Job     :
 * @Info    :
 * @Result  :
 * @Family  :
 * @Example :
 * @Status  :
 * ------------------------------------------------------------------------ */

tU32 xW32UartClose
(tU32 pPortData)                  /*                              */

{
  if (((tW32UartData*)pPortData)->W32UartEventThreadHandle)
  {
    W32UartStopThread (((tW32UartData*)pPortData)->W32UartEventThreadHandle);
  }

  if (xW32UartPurgeRxTxBuff(pPortData))
  {
    free ((void*)pPortData);
    return W32UART_CLOSE_PURGE_RXTX_BUFF_ERROR;
  }

  if (!CloseHandle(((tW32UartData*)pPortData)->Port))
  {
    free ((void*)pPortData);
    return W32UART_CLOSE_CLOSE_HANDLE_ERROR;
  }

  free ((void*)pPortData);
  return W32UART_CLOSE_SUCCESS;
}

/* ------------------------------------------------------------------------
 * @Function:
 * @Job     :
 * @Info    :
 * @Result  :
 * @Family  :
 * @Example :
 * @Status  :
 * ------------------------------------------------------------------------ */

tU32 xW32UartSetEventProc
(tU32 EventID,                              /* callback-event               */
 void (*ProcAddrParm)(void),                /* callback-adresse             */
 tU32 pPortData)                  /*                              */

{
  switch (EventID)
  {
    case W32UART_EVENT_CTS       : ((tW32UartData*)pPortData)->ProcAddrCts      = ProcAddrParm; break;
    case W32UART_EVENT_CTS_ON    : ((tW32UartData*)pPortData)->ProcAddrCtsOn    = ProcAddrParm; break;
    case W32UART_EVENT_CTS_OFF   : ((tW32UartData*)pPortData)->ProcAddrCtsOff   = ProcAddrParm; break;

    case W32UART_EVENT_DSR       : ((tW32UartData*)pPortData)->ProcAddrDsr      = ProcAddrParm; break;
    case W32UART_EVENT_DSR_ON    : ((tW32UartData*)pPortData)->ProcAddrDsrOn    = ProcAddrParm; break;
    case W32UART_EVENT_DSR_OFF   : ((tW32UartData*)pPortData)->ProcAddrDsrOff   = ProcAddrParm; break;

    case W32UART_EVENT_DCD       : ((tW32UartData*)pPortData)->ProcAddrDcd      = ProcAddrParm; break;
    case W32UART_EVENT_DCD_ON    : ((tW32UartData*)pPortData)->ProcAddrDcdOn    = ProcAddrParm; break;
    case W32UART_EVENT_DCD_OFF   : ((tW32UartData*)pPortData)->ProcAddrDcdOff   = ProcAddrParm; break;

    case W32UART_EVENT_RING      : ((tW32UartData*)pPortData)->ProcAddrRing     = ProcAddrParm; break;
    case W32UART_EVENT_BREAK     : ((tW32UartData*)pPortData)->ProcAddrBreak    = ProcAddrParm; break;
    case W32UART_EVENT_ERROR     : ((tW32UartData*)pPortData)->ProcAddrError    = ProcAddrParm; break;
    case W32UART_EVENT_RX_CHAR   : ((tW32UartData*)pPortData)->ProcAddrRxChar   = ProcAddrParm; break;
    case W32UART_EVENT_RX_FLAG   : ((tW32UartData*)pPortData)->ProcAddrRxFlag   = ProcAddrParm; break;
    case W32UART_EVENT_RX80_FULL : ((tW32UartData*)pPortData)->ProcAddrRx80Full = ProcAddrParm; break;
    case W32UART_EVENT_TX_EMPTY  : ((tW32UartData*)pPortData)->ProcAddrTxEmpty  = ProcAddrParm; break;

    default            : return (W32UART_SET_EVENT_PROC_ERROR);
  }
  return (W32UART_SET_EVENT_PROC_SUCCESS);
}

/*--------------------------------------------------------------------------*/

void    xW32UartClearDtr
(tU32 pPortData)                  /*                              */
{
  DCB Dcb;

  GetCommState(((tW32UartData*)pPortData)->Port, &Dcb);

  Dcb.DCBlength    = sizeof(Dcb);
  Dcb.fDtrControl  = DTR_CONTROL_DISABLE;

  SetCommState(((tW32UartData*)pPortData)->Port, &Dcb);
}

/*--------------------------------------------------------------------------*/

void    xW32UartSetDtr
(tU32 pPortData)                  /*                              */
{
  DCB Dcb;

  GetCommState(((tW32UartData*)pPortData)->Port, &Dcb);

  Dcb.DCBlength    = sizeof(Dcb);
  Dcb.fDtrControl  = DTR_CONTROL_ENABLE;

  SetCommState(((tW32UartData*)pPortData)->Port, &Dcb);
}

/*--------------------------------------------------------------------------*/

void    xW32UartClearRts
(tU32 pPortData)                  /*                              */
{
//  EscapeCommFunction (((tW32UartData*)pPortData)->Port,CLRRTS);
}

/*--------------------------------------------------------------------------*/

void    xW32UartSetRts
(tU32 pPortData)                  /*                              */
{
//  EscapeCommFunction (((tW32UartData*)pPortData)->Port,SETRTS);
}

/*--------------------------------------------------------------------------*/


HANDLE  xW32UartGetPortHandle
(tU32 pPortData)
{
  return (((tW32UartData*)pPortData)->Port);
}


/*--------------------------------------------------------------------------*/

