/*-
 * Javoids -- Javoids is an asteroids based game (that look nothing like the original).
 * 
 * Copyright (C) 1999-2006 Patrick Mallette
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * I can be reached at parickmallette@rogers.com
 */
package javoids;

import static javoids.ImageMap.Name.NO_IMAGE;

/* Weapon--------------------- */
/**
 * A class to represent a game weapon.
 * @author mallette
 */
public class Weapon extends Item
{
  /** This is the version used for serializing/deserializing (storing/retrieving) this object */
  private static final long serialVersionUID    = 1L;
  /** the weapon's regular image */
  private ImageMap.Name     bulletImage         = NO_IMAGE;
  /** the shielded image (for display in the panel) */
  private ImageMap.Name     shieldedImage       = NO_IMAGE;
  /** the homing image (for display in the panel) */
  private ImageMap.Name     homingImage         = NO_IMAGE;
  /** the shielded homing image (for display in the panel) */
  private ImageMap.Name     shieldedHomingImage = NO_IMAGE;
  /** the weapon damage per bullet */
  private int               damage;
  /** the maximum number of times the weapon can be fired */
  private int               maximumFired;
  /** the range of the weapon */
  private int               range;
  /** the velocity of the bullets fired from this weapon */
  private int               velocity;
  /** the number of directions the weapon fires */
  private int               directions;
  /** the number of sheidled bullets the weapon has */
  private int               shields;
  /** the number of hoing bullets the weapon has */
  private int               homings;
  /** the number of bullets fired */
  private int               fired;

  /**
   * Constructor (default)
   */
  public Weapon()
  {
    super.reset();
  }

  /**
   * Constructor (copy)
   * @param weapon the wepon to be copied
   */
  public Weapon(Weapon weapon)
  {
    super(weapon);
    if (Item.Type.GUN1.equals( weapon.getType() ))
      this.setLimitless(true);
    this.setBulletImage(weapon.getBulletImage());
    this.setShieldedImage(weapon.getShieldedImage());
    this.setHomingImage(weapon.getHomingImage());
    this.setShieldedHomingImage(weapon.getShieldedHomingImage());
    this.setDamage(weapon.getDamage());
    this.setMaximumFired(weapon.getMaximumFired());
    this.setRange(weapon.getRange());
    this.setVelocity(weapon.getVelocity());
    this.setDirections(weapon.getDirections());
    this.setShields(weapon.getShields());
    this.setHomings(weapon.getHomings());
    this.setFired(weapon.getFired());
  }

  /**
   * @return the bullet's image
   */
  public ImageMap.Name getBulletImage()
  {
    return this.bulletImage;
  }

  /**
   * @return the bullet's shielded image
   */
  public ImageMap.Name getShieldedImage()
  {
    return this.shieldedImage;
  }

  /**
   * @return the homing image
   */
  public ImageMap.Name getHomingImage()
  {
    return this.homingImage;
  }

  /**
   * @return the shielded homing image
   */
  public ImageMap.Name getShieldedHomingImage()
  {
    return this.shieldedHomingImage;
  }

  /**
   * @return the base damage this weapon causes per hit
   */
  public int getDamage()
  {
    return this.damage;
  }

  /**
   * @return the maximum number of times it can be fired before needing to pause
   */
  public int getMaximumFired()
  {
    return this.maximumFired;
  }

  /**
   * @return the weapon's range
   */
  public int getRange()
  {
    return this.range;
  }

  /**
   * @return the bullet's velocity
   */
  public int getVelocity()
  {
    return this.velocity;
  }

  /**
   * @return the number of directions the weapon fires in
   */
  public int getDirections()
  {
    return this.directions;
  }

  /**
   * @return the number of bullets that will fire with a shield on them
   */
  public int getShields()
  {
    return this.shields;
  }

  /**
   * @return the number of bullets that will fire with the homing property
   */
  public int getHomings()
  {
    return this.homings;
  }

  /**
   * @return the number of times this weapon has been fired already (resets after bullets die)
   */
  public int getFired()
  {
    return this.fired;
  }

  /**
   * @return the name of the item (weapon name in this case)
   */
  /*
   * (non-Javadoc)
   * @see javoids.Item#getImage()
   */
  @Override
  public ImageMap.Name getImage()
  {
    ImageMap.Name image = this.getDefaultImage();
    boolean shielded = this.shields > 0;
    boolean homing = this.homings > 0;
    if (shielded && homing)
      image = this.shieldedHomingImage;
    else if (shielded)
      image = this.shieldedImage;
    else if (homing)
      image = this.homingImage;
    return image;
  }

  /**
   * @param _type the item name
   */
  /*
   * (non-Javadoc)
   * @see javoids.Item#setType(javoids.Item.Type)
   */
  @Override
  public void setType(Type _type)
  {
    switch (_type)
    {
      case SHIELD :
      case AFTERBURNER :
      case JUMP :
      case BOMB1 :
      case BOMB2 :
      case GUN1 :
      case GUN2 :
      case GUN3 :
      case MACHINEGUN1 :
      case MACHINEGUN2 :
      case MACHINEGUN3 :
      case MULTIGUN1 :
      case MULTIGUN2 :
      case MULTIGUN3 :
      case MULTIGUN4 :
      case MULTIGUN5 :
      case MULTIGUN6 :
      case ROCKET1 :
      case ROCKET2 :
        super.setType(_type);
        break;
      default :
        System.out.printf(Messages.getString("Weapon.ItemMissing"),_type); //$NON-NLS-1$
        break;
    }
  }

  /**
   * @param _bulletImage the image to give this weapon's bullets
   */
  public void setBulletImage(ImageMap.Name _bulletImage)
  {
    this.bulletImage = _bulletImage;
  }

  /**
   * @param _shieldedImage the image to give this weapon's shielded bullets (or to use in the toolbar)
   */
  public void setShieldedImage(ImageMap.Name _shieldedImage)
  {
    this.shieldedImage = _shieldedImage;
  }

  /**
   * @param _homingImage the image to give this weapon's homing bullets (or to use in the toolbar)
   */
  public void setHomingImage(ImageMap.Name _homingImage)
  {
    this.homingImage = _homingImage;
  }

  /**
   * @param _shieldedHomingImage the image to give this weapon's shielded homing bullets (or to use in the toolbar)
   */
  public void setShieldedHomingImage(ImageMap.Name _shieldedHomingImage)
  {
    this.shieldedHomingImage = _shieldedHomingImage;
  }

  /**
   * @param _damage the base damage to inflict per hit
   */
  public void setDamage(int _damage)
  {
    this.damage = _damage > 0 ? _damage : 0;
  }

  /**
   * @param _maximumFired the maximum number of times the weapon can be fired before needing to paused
   */
  public void setMaximumFired(int _maximumFired)
  {
    this.maximumFired = _maximumFired > 0 ? _maximumFired : 0;
  }

  /**
   * @param _range the range the fired bullets can travel
   */
  public void setRange(int _range)
  {
    this.range = _range > 0 ? _range : 0;
  }

  /**
   * @param _velocity the bullet velocity
   */
  public void setVelocity(int _velocity)
  {
    this.velocity = _velocity > 0 ? _velocity : 0;
  }

  /**
   * @param _directions the number of directions to shoot in
   */
  public void setDirections(int _directions)
  {
    this.directions = _directions > 0 ? _directions : 0;
  }

  /**
   * @param _shields the number of shots that will be shielded
   */
  public void setShields(int _shields)
  {
    this.shields = _shields > 0 ? _shields : 0;
  }

  /**
   * @param _homings the number of shots that will be homing
   */
  public void setHomings(int _homings)
  {
    this.homings = _homings > 0 ? _homings : 0;
  }

  /**
   * @param _fired the number of shots that can be fired at once
   */
  public void setFired(int _fired)
  {
    this.fired = _fired > 0 ? _fired : 0;
  }

  /**
   * Reset the item properties to the default values.
   */
  /*
   * (non-Javadoc)
   * @see javoids.Item#reset()
   */
  @Override
  public void reset()
  {
    super.reset();
    this.shields = 0;
    this.homings = 0;
    this.fired = 0;
  }

  /**
   * @param _count the additional number of shots that will be shielded
   */
  public void modifyShields(int _count)
  {
    this.setShields(this.getShields() + _count);
  }

  /**
   * @param _count the additional number of shots that will be homing
   */
  public void modifyHomings(int _count)
  {
    this.setHomings(this.getHomings() + _count);
  }

  /**
   * @param _count the additional number of shots that were fired
   */
  public void modifyFired(int _count)
  {
    this.setFired(this.getFired() + _count);
  }

  /**
   * Provide a String representation of this object.
   * @return String A representation of the object for debugging.
   */
  @Override
  public String toString()
  {
    return String.format(Messages.getString("Weapon.ToString"),super.toString(),this.shieldedImage,this.homingImage,this.shieldedHomingImage,Integer.valueOf(this.damage),Integer.valueOf(this.maximumFired),Integer.valueOf(this.range),Integer.valueOf(this.velocity),Integer.valueOf(this.directions),Integer.valueOf(this.shields),Integer.valueOf(this.homings)); //$NON-NLS-1$
  }
}
/* Weapon--------------------- */
