/*-
 * Javoids -- Javoids is an asteroids based game (that look nothing like the original).
 * 
 * Copyright (C) 1999-2006 Patrick Mallette
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * I can be reached at parickmallette@rogers.com
 */
package javoids;

import java.awt.Graphics2D;
import java.util.Collection;
import java.util.Vector;

/* SpriteVector--------------- */
/**
 * A vecotr class to hold many sprites.
 * @author mallette
 * @param <E> the type of sprite being stored
 */
public class SpriteVector<E extends BasicSprite> extends Vector<E>
{
  /** This is the version used for serializing/deserializing (storing/retrieving) this object */
  private static final long serialVersionUID = 1L;

  /**
   * @param maxParts
   */
  public SpriteVector(int maxParts)
  {
    super(maxParts);
  }

  /**
   * move all the sprites in the vector
   */
  public void moveSprites()
  {
    for (BasicSprite basicSprite : this)
    {
      if (basicSprite.isAlive())
      {
        if (basicSprite.isAutomaticMove())
        {
          double amount = Math.random() * Move.MAX_VELOCITY;
          double direction = Math.random() * 90.0;
          // occasionally turn, accelerate, move
          if (Math.random() > 0.5) // only turn some of the times
          {
            direction = Math.random() * 90.0;
            direction = (direction - 45.0) / 2.0;
            if (direction >= 0)
              basicSprite.turn(Move.RIGHT,direction);
            else
              basicSprite.turn(Move.LEFT,-direction);
          }
          amount -= Move.MAX_VELOCITY / 2.0;
          if (amount > 0)
            basicSprite.accelerate(amount);
        }
        basicSprite.move();
      }
    }
  }

  /**
   * Move the sprites in the vector to the target sprites in another vector (if they are homing) going to the closest one.
   * @param sprites the other sprites that are targets.
   */
  public void moveSpritesHoming(Collection<? extends BasicSprite> sprites) // what about allowing all sprites to be guided? Seeker Roids or Mines? Ok!
  {
    if (this.size() > 0 && sprites != null && sprites.size() > 0)
    {
      double distance;
      double closestDistance;
      BasicSprite target;
      
      for (BasicSprite basicSprite1 : this)
      {
        if (basicSprite1.isAlive() && basicSprite1.isHoming())
        {
          // find closest Sprite of another type
          target = null;
          closestDistance = Double.MAX_VALUE;
          for (BasicSprite basicSprite2 : sprites)
          {
            if (basicSprite2.isAlive() && basicSprite1.getParent() != null && !basicSprite1.getParent().equals( basicSprite2 ))
            {
              distance = Common.getDistanceM(basicSprite1.getX(),basicSprite1.getY(),basicSprite2.getX(),basicSprite2.getY(),basicSprite1.getScreen().width,basicSprite1.getScreen().height);
              if (distance < closestDistance)
              {
                closestDistance = distance;
                target = basicSprite2;
              }
            }
          }
          if (target != null)
          {
            Common.target(basicSprite1.getMove(),target.getMove());
            basicSprite1.accelerate(basicSprite1.getMaxVelocity());
            basicSprite1.setMove(basicSprite1.getMove()); // keeps halo or shield in line with sprite!
          }
        }
      }
    }
  }

  /**
   * Move the sprites in the vector toward/away from the sprites in another vector (if the other sprites are gravity sources). Applying the effect from each gravity source/sink.
   * @param sprites the other sprites that are targets.
   */
  public void moveSpritesGravity(Collection<? extends BasicSprite> sprites) // what about allowing all sprites to be guided? Seeker Roids or Mines? Ok!
  {
    if (this.size() > 0 && sprites != null && sprites.size() > 0)
    {
      for (BasicSprite basicSprite1 : this)
      {
        if (basicSprite1.isAlive() && basicSprite1.isGravityWell() || basicSprite1.isGravitySource())
        {
          int minSide = Math.min(basicSprite1.getScreen().width,basicSprite1.getScreen().height);
          int sizeSideRatio = minSide / Shapes.DEFAULT_SIZE;
          double modifiedMass1 = (basicSprite1.isGravityWell() ? 8 : 12) * basicSprite1.getMass();
          
          for (BasicSprite basicSprite2 : sprites)
          {
            if (!basicSprite1.equals( basicSprite2 ) && (basicSprite2.getParent() == null || basicSprite2.getParent() != null && !basicSprite2.getParent().equals( basicSprite1 )))
            {
              double amount = 0;
              double direction = basicSprite2.getDirection();
              int distanceRating = (int)(Common.getDistanceM(basicSprite2.getMove(),basicSprite1.getMove()) / sizeSideRatio);
              double maxVelocity = basicSprite2.getMaxVelocity();
              double modifiedMass2 = basicSprite2.isGravitySource() ? basicSprite2.getMass() : 1;
              Common.target(basicSprite2.getMove(),basicSprite1.getMove());
              if (basicSprite1.isGravitySource() && !basicSprite2.isGravityWell())
                basicSprite2.getMove().turn(Move.RIGHT,180);
              amount = modifiedMass1 * modifiedMass2 / Math.pow(2,distanceRating);
              basicSprite2.setMaxVelocity(Move.MAX_VELOCITY);
              basicSprite2.accelerate(amount);
              basicSprite2.setDirection(direction);
              basicSprite2.setMaxVelocity(maxVelocity);
              basicSprite2.setMove(basicSprite2.getMove()); // keeps halo or shield in line with sprite!
            }
          }
        }
      }
    }
  }

  /**
   * Remove the dead sprites from the vector
   */
  synchronized public void removeDead()
  {
    SpriteVector<E> sprites = new SpriteVector<E>(this.size());
    
    for (E basicSprite : this)
      if (!basicSprite.isAlive() && !basicSprite.isPlayer())
        sprites.add(basicSprite);
    this.removeAll(sprites);
  }

  /**
   * Draw all of the sprites in this vector.
   * @param g2d the graphics context
   * @param foregroundImage the foreground image to redraw the sprites on
   */
  synchronized public void drawSprites(Graphics2D g2d,Graphics2D foregroundImage)
  {
    for (BasicSprite basicSprite : this)
      basicSprite.draw(g2d,foregroundImage);
  }

  /**
   * add a sprite to this vector.
   * @param basicSprite the sprite to add
   * @return true if successful, otherwise return false
   */
  /*
   * (non-Javadoc)
   * @see java.util.Vector#add(E)
   */
  @Override
  synchronized public boolean add(E basicSprite)
  {
    boolean returnValue = false;
    if (Debug.debugOneSprite)
    {
      if (this.size() <= 0)
        returnValue = super.add(basicSprite);
    }
    else
    {
      if (this.size() < this.capacity())
        returnValue = super.add(basicSprite);
    }
    return returnValue;
  }

  /**
   * add all the sprites from another SpriteVector to this vector.
   * @param sprites the sprites to add to this vector
   * @return true if successful, otherwise return false
   */
  synchronized public boolean addAll(SpriteVector<E> sprites)
  {
    boolean returnValue = false;
    if (Debug.debugOneSprite)
    {
      if (this.size() < 1 && sprites.size() > 0)
        returnValue = this.add(sprites.get(0));
    }
    else
    {
      returnValue = sprites.size() > 0 && this.size() < this.capacity();
      for (E sprite : sprites)
        if (this.size() < this.capacity())
          this.add(sprite);
        else
          break;
    }
    return returnValue;
  }

  /**
   * Provide a String representation of this object.
   * @return String A representation of the object for debugging.
   */
  @Override
  public synchronized String toString()
  {
    return super.toString();
  }
}
/* SpriteVector--------------- */
