/*-
 * Javoids -- Javoids is an asteroids based game (that look nothing like the original).
 * 
 * Copyright (C) 1999-2006 Patrick Mallette
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * I can be reached at parickmallette@rogers.com
 */
package javoids;

import static javoids.BasicSprite.Gravity.NONE;
import static javoids.BasicSprite.Gravity.SINK;
import static javoids.BasicSprite.Gravity.SOURCE;
import static javoids.Health.DurationType.NORMAL;
import static javoids.ImageMap.Name.PACMAN_POWERPILL;
import static javoids.ImageMap.Name.POWERUP1;
import static javoids.Shapes.Shape.POINT;
import static javoids.SoundMap.Sound.POWERUPDIE;

import java.util.HashMap;

/* PowerUp-------------------- */
/**
 * A sprite that grants Ship sprites special items, bonus points, or effects like repairing damage
 * @author mallette
 */
public final class PowerUp extends ShieldedSprite
{
  /** This is the version used for serializing/deserializing (storing/retrieving) this object */
  private static final long       serialVersionUID = 1L;
  /** The default number of points this is worth */
  private final static int        DEFAULT_POINTS   = 10000;
  /** The maximum number of these sprites allowed in the game */
  protected final static int      MAX_NUMBER       = 5;
  /** the sprite's maximum size */
  protected final static int      MAX_SIZE         = 100;
  /** the sprite's minimum size */
  protected final static int      MIN_SIZE         = 30;
  /** the sprite's default size */
  protected final static int      DEFAULT_SIZE     = 30;   // This should match the size of the image!
  /** the chance this sprite is shielded */
  private final static double     CHANCE_SHIELDED  = 0.0;
  /** the chance this sprite is homing */
  private final static double     CHANCE_HOMING    = 0.35;
  /** the chance this sprite will rotate */
  private final static double     CHANCE_ROTATE    = 0.30;
  /** the amount to rotate */
  private final static double     ROTATION         = 30;
  /** a mapping of item names to items */
  private HashMap<Item.Type,Item> items;
  /** extra health information */
  private Health                  extraAlive;
  /** extra gravity related information */
  private Gravity                 extraGravity;
  /** extra gravity duration information (how long a gravity effect will last) */
  private int                     extraGravityDuration;

  /**
   * Constructor
   * @param parent the parent of this sprite
   * @param health the health information
   * @param move the movment information
   * @param _size the size this should be
   */
  public PowerUp(BasicSprite parent,Health health,Move move,int _size)
  {
    super(parent,health,move,null,_size,POWERUP1,Shapes.getAreas(POINT,_size,PowerUp.MAX_SIZE),Shapes.getColors(POINT));
    this.items = Media.getItems();
    this.extraGravity = NONE;
    this.extraGravityDuration = 0;
    this.modifyRandomItems(); // this should be done last or just before the accelerate
    if (Math.random() < PowerUp.CHANCE_ROTATE)
      this.setRotation(Math.random() * PowerUp.ROTATION * (Math.random() < 0.5 ? -1 : 1));
    this.accelerate(this.getMaxVelocity());
    this.setShape(POINT);
    this.setDisplayAreas(false);
  }

  /**
   * @return the maximum size
   */
  public int getMaximumSize()
  {
    return PowerUp.MAX_SIZE;
  }

  /**
   * @return the minimum size
   */
  public int getMinimumSize()
  {
    return PowerUp.MIN_SIZE;
  }

  /**
   * @return the default size
   */
  public int getDefaultSize()
  {
    return PowerUp.DEFAULT_SIZE;
  }

  /**
   * @param _size the size to set the sprite's size to
   */
  /*
   * (non-Javadoc)
   * @see javoids.Sprite#setSize(int)
   */
  @Override
  public void setSize(int _size)
  {
    super.setSize(_size >= PowerUp.MIN_SIZE ? _size <= PowerUp.MAX_SIZE ? _size : PowerUp.MAX_SIZE : PowerUp.MIN_SIZE);
  }

  /**
   * @return a hashmap of items (key=item type or name, value=item object)
   */
  public HashMap<Item.Type,Item> getItems()
  {
    return this.items;
  }

  /**
   * @return get any health related information that should be given to the capturing sprite
   */
  public Health getExtraHealth()
  {
    return this.extraAlive;
  }

  /**
   * @return get any gravity related information that should be given to the capturing sprite
   */
  public Gravity getExtraGravity()
  {
    return this.extraGravity;
  }

  /**
   * @return get any gravity duration related information that should be given to the capturing sprite (only limited durations)
   */
  public int getExtraGravityDuration()
  {
    return this.extraGravityDuration;
  }

  /**
   * @param _damage the damage the sprite sustained
   * @return the new damage level
   */
  /*
   * (non-Javadoc)
   * @see javoids.BasicSprite#modifyDamage(int)
   */
  @Override
  public int modifyDamage(int _damage)
  {
    int damage;

    this.setPoints(-this.getDamage() * Health.DEFAULT_DAMAGE_CAPACITY / this.getMaxDamage());
    damage = super.modifyDamage(_damage);
    if (!this.isAlive())
      Media.play(POWERUPDIE);
    return damage;
  }

  /**
   * Bbuild a powerup with raqndom effects and attributes to apply to a ship when the two collide.
   */
  private void modifyRandomItems()
  {
    double chance = Math.random();
    if (chance < 0.55)
      this.modifyRandomItem();
    else if (chance <= 0.8)
      this.modifyRandomAttribute();
    else if (chance <= 0.9)
    {
      chance = Math.random();
      if (chance <= 0.4)
        this.extraGravity = SINK;
      else if (chance <= 0.8)
        this.extraGravity = SOURCE;
      else
        this.extraGravity = NONE;
      this.extraGravityDuration = (int)(Math.random() * 100 * this.extraGravity.ordinal());
    }
    // else don't modify anything (a dud)
  }

  /**
   * Pick an item this power up has and change a property of it (count, points, duration, etc).
   */
  private void modifyRandomItem()
  {
    Item.Type selectedItem = Item.Type.values()[(int)(Math.random() * Item.Type.values().length)];
    double chance = 0;
    Item item = null;
    for (Item.Type name : Item.Type.values())
    {
      if (!Item.Type.NO_TOOL.equals( name ))
      {
        item = this.items.get(name);
        if (name.equals(selectedItem))
        {
          if (item.isLimitless())
            item.setCount(item.getDefaultCount());
          else
            item.modifyCount((int)(Math.random() * item.getDefaultCount() + 1));
          if (item instanceof Weapon)
          {
            Weapon weapon2 = (Weapon)item;
            chance = Math.random();
            if (chance <= PowerUp.CHANCE_SHIELDED)
              weapon2.modifyShields(weapon2.getDefaultCount() / 10);
            chance = Math.random();
            if (chance <= PowerUp.CHANCE_HOMING)
              weapon2.modifyHomings(weapon2.getDefaultCount() / 10);
          }
        }
        else
        {
          item.setCount(0);
          if (this.items.get(name) instanceof Weapon)
          {
            ((Weapon)item).setShields(0);
            ((Weapon)item).setHomings(0);
          }
        }
      }
    }
  }

  /**
   * Change a random attribute like points or health information.
   */
  private void modifyRandomAttribute()
  {
    double n = Math.random();
    if (n <= 0.4)
      this.setPoints((int)(Math.random() * PowerUp.DEFAULT_POINTS + 1));
    else if (n <= 0.7)
      this.extraAlive = new Health(-(int)(Math.random() * Health.DEFAULT_DEATHS + 1),-(int)(Math.random() * Health.DEFAULT_DAMAGE_CAPACITY * 2 + 1),-(int)(Math.random() * Health.DEFAULT_DURATION * 2 + 1),PowerUp.DEFAULT_SIZE,NORMAL);
    else
      this.extraAlive = new Health(0,-(int)(Math.random() * Health.DEFAULT_DAMAGE_CAPACITY * 2 + 1),-(int)(Math.random() * Health.DEFAULT_DURATION * 2 + 1),PowerUp.DEFAULT_SIZE,NORMAL);
  }

  /**
   * @return the image name this sprite is using for drawing purposes
   */
  /*
   * (non-Javadoc)
   * @see javoids.Sprite#getImageNumber()
   */
  @Override
  public ImageMap.Name getImageNumber()
  {
    return BasicSprite.getPacmanGame() ? PACMAN_POWERPILL : this.getOriginalImage();
  }

  /**
   * Provide a String representation of this object.
   * @return String A representation of the object for debugging.
   */
  @Override
  public String toString()
  {
    return String.format(Messages.getString("PowerUp.ToString"),super.toString(),this.items,this.extraAlive,this.extraGravity,Integer.valueOf(this.extraGravityDuration)); //$NON-NLS-1$
  }
}
/* PowerUp-------------------- */
