/*-
 * Javoids -- Javoids is an asteroids based game (that look nothing like the original).
 * 
 * Copyright (C) 1999-2006 Patrick Mallette
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * I can be reached at parickmallette@rogers.com
 */
package javoids;

import static javoids.KeyMap.Action.ACTION_UNKNOWN;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Frame;
import java.awt.Insets;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.InputEvent;
import java.awt.event.MouseEvent;
import java.awt.event.MouseMotionAdapter;
import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Collections;
import java.util.Vector;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JFileChooser;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.ScrollPaneConstants;
import javax.swing.filechooser.FileFilter;

/**
 * A class to allow the user to respecify the action taken by the keys on the user's keyboard.
 * @author mallette
 */
public class KeyDialog extends JDialog
{
  /** This is the version used for serializing/deserializing (storing/retrieving) this object */
  private static final long serialVersionUID = 1L;
  /** Was the result successful? */
  private boolean           successStatus;
  /** the main menu for this dialog */
  private JMenuBar          KeyMainMenu      = new JMenuBar();
  /** the file menu */
  private JMenu             keyFileMenu      = new JMenu();
  /** the file | open menu item */
  private JMenuItem         keyFileOpenItem  = new JMenuItem();
  /** the file | save menu item */
  private JMenuItem         keyFileSaveItem  = new JMenuItem();
  /** the file | exit menu item */
  private JMenuItem         keyFileExitItem  = new JMenuItem();
  /** a scroll pane to controll scrolling in the dialog */
  private JScrollPane       scrollPane       = new JScrollPane(ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS,ScrollPaneConstants.HORIZONTAL_SCROLLBAR_ALWAYS);
  /** a panel to hold the screen controls */
  private JPanel            panel            = new JPanel();
  /** a panel to hold the keyboar buttons */
  private JPanel            tablePanel       = new JPanel();
  /** a layout for panels */
  private BorderLayout      borderLayout1    = new BorderLayout();
  /** aa mapping of key names to key values */
  private KeyMap            keyMap;
  /** a panel to hold the okay/cancel buttons */
  private JPanel            buttonPanel      = new JPanel();
  /** okay button */
  private JButton           okButton         = new JButton();
  /** cancel button */
  private JButton           cancelButton     = new JButton();
  /** popup menu */
  private JPopupMenu        menuPopup;
  /** the currently selected key */
  private String            currentKey;
  /** a list of all keyboard buttons in the panel */
  private transient JButton[][]       buttonList       = new JButton[6][21];

  /**
   * Constructor
   * @param frame the owning frame for this dialog
   * @param title the title of this frame
   * @param modal is this a modal dialog?
   */
  public KeyDialog(Frame frame,String title,boolean modal)
  {
    super(frame,title,modal);
    this.successStatus = false; // set result to CANCEL to insure results aren't used unless OK chosen
    this.setJMenuBar(this.KeyMainMenu);
    this.setModal(true);
    this.setTitle(Messages.getString("KeyDialog.KeyboarLayoutTitle")); //$NON-NLS-1$
    this.keyFileMenu.setToolTipText(Messages.getString("KeyDialog.Empty")); //$NON-NLS-1$
    this.keyFileMenu.setFocusPainted(true);
    this.keyFileMenu.setMnemonic('F');
    this.keyFileMenu.setText(Messages.getString("KeyDialog.File")); //$NON-NLS-1$
    this.keyFileOpenItem.setMnemonic('O');
    this.keyFileOpenItem.setText(Messages.getString("KeyDialog.Open")); //$NON-NLS-1$
    this.keyFileOpenItem.setAccelerator(javax.swing.KeyStroke.getKeyStroke(79,InputEvent.ALT_MASK,false));
    this.keyFileOpenItem.addActionListener(new ActionListener()
    {
      public void actionPerformed(ActionEvent actionEvent)
      {
        KeyDialog.this.keyFileOpenItem_actionPerformed(actionEvent);
      }
    });
    this.keyFileSaveItem.setMnemonic('S');
    this.keyFileSaveItem.setText(Messages.getString("KeyDialog.Save")); //$NON-NLS-1$
    this.keyFileSaveItem.setAccelerator(javax.swing.KeyStroke.getKeyStroke(83,InputEvent.ALT_MASK,false));
    this.keyFileSaveItem.addActionListener(new ActionListener()
    {
      public void actionPerformed(ActionEvent actionEvent)
      {
        KeyDialog.this.keyFileSaveItem_actionPerformed(actionEvent);
      }
    });
    this.keyFileExitItem.setActionCommand(Messages.getString("KeyDialog.Exit")); //$NON-NLS-1$
    this.keyFileExitItem.setMnemonic('E');
    this.keyFileExitItem.setText(Messages.getString("KeyDialog.Exit")); //$NON-NLS-1$
    this.keyFileExitItem.setAccelerator(javax.swing.KeyStroke.getKeyStroke(88,InputEvent.ALT_MASK,false));
    this.keyFileExitItem.addActionListener(new ActionListener()
    {
      public void actionPerformed(ActionEvent actionEvent)
      {
        KeyDialog.this.keyFileExitItem_actionPerformed(actionEvent);
      }
    });
    this.scrollPane.setColumnHeader(null);
    this.scrollPane.setBorder(BorderFactory.createRaisedBevelBorder());
    this.scrollPane.getHorizontalScrollBar().setUnitIncrement(this.scrollPane.getHorizontalScrollBar().getMaximum());
    this.scrollPane.getVerticalScrollBar().setUnitIncrement(this.scrollPane.getVerticalScrollBar().getMaximum());
    this.okButton.addActionListener(new ActionListener()
    {
      public void actionPerformed(ActionEvent actionEvent)
      {
        KeyDialog.this.okButton_actionPerformed(actionEvent);
      }
    });
    this.keyFileMenu.add(this.keyFileOpenItem);
    this.keyFileMenu.add(this.keyFileSaveItem);
    this.keyFileMenu.add(this.keyFileExitItem);
    this.KeyMainMenu.add(this.keyFileMenu);
    this.tablePanel.setLayout(null);
    this.scrollPane.setAutoscrolls(true);
    this.scrollPane.setMinimumSize(this.tablePanel.getMinimumSize());
    this.scrollPane.setViewportView(this.tablePanel);
    this.setButtonValues();
    this.resizeButtons();
    this.okButton.setSelected(true);
    this.okButton.setText(Messages.getString("KeyDialog.Ok")); //$NON-NLS-1$
    this.cancelButton.setText(Messages.getString("KeyDialog.Cancel")); //$NON-NLS-1$
    this.cancelButton.addActionListener(new ActionListener()
    {
      public void actionPerformed(ActionEvent actionEvent)
      {
        KeyDialog.this.cancelButton_actionPerformed(actionEvent);
      }
    });
    this.buttonPanel.add(this.okButton,null);
    this.buttonPanel.add(this.cancelButton,null);
    this.panel.setMinimumSize(new Dimension(500,this.tablePanel.getMinimumSize().height + this.buttonPanel.getMinimumSize().height + this.scrollPane.getHorizontalScrollBar().getMinimumSize().height + this.scrollPane.getBorder().getBorderInsets(this.scrollPane).top + this.scrollPane.getBorder().getBorderInsets(this.scrollPane).bottom));
    this.panel.setPreferredSize(new Dimension(750,this.tablePanel.getPreferredSize().height + this.buttonPanel.getPreferredSize().height + this.scrollPane.getHorizontalScrollBar().getPreferredSize().height + this.scrollPane.getBorder().getBorderInsets(this.scrollPane).top + this.scrollPane.getBorder().getBorderInsets(this.scrollPane).bottom));
    this.panel.setLayout(this.borderLayout1);
    this.panel.add(this.scrollPane);
    this.panel.add(this.buttonPanel,BorderLayout.SOUTH);
    this.add(this.panel,BorderLayout.CENTER);
    this.setSize(new Dimension(750,300));
    this.pack();
  }

  /**
   * Constructor (default)
   */
  public KeyDialog()
  {
    this(null,Messages.getString("KeyDialog.Empty"),false); //$NON-NLS-1$
  }

  /**
   * @return a mapping of key names to key values in the form of a KeyMap object.
   */
  KeyMap getKeyMap()
  {
    return this.keyMap;
  }

  /**
   * @return whether or not the action of the save was successful
   */
  public boolean getSuccessStatus()
  {
    return this.successStatus;
  }

  /**
   * @param _keyMap the keymap to use for the dialog
   */
  void setKeyMap(KeyMap _keyMap)
  {
    this.keyMap = new KeyMap(_keyMap); // store a copy to modify
    this.successStatus = false;
  }

  /**
   * Set the values for each button to the keyboard character and the action associated with them.
   */
  private void setButtonValues()
  {
    this.buttonList[0][0] = new JButton(Messages.getString("KeyMap.Escape")); //$NON-NLS-1$
    this.buttonList[0][1] = new JButton(Messages.getString("KeyMap.F1")); //$NON-NLS-1$
    this.buttonList[0][2] = new JButton(Messages.getString("KeyMap.F2")); //$NON-NLS-1$
    this.buttonList[0][3] = new JButton(Messages.getString("KeyMap.F3")); //$NON-NLS-1$
    this.buttonList[0][4] = new JButton(Messages.getString("KeyMap.F4")); //$NON-NLS-1$
    this.buttonList[0][5] = new JButton(Messages.getString("KeyMap.F5")); //$NON-NLS-1$
    this.buttonList[0][6] = new JButton(Messages.getString("KeyMap.F6")); //$NON-NLS-1$
    this.buttonList[0][7] = new JButton(Messages.getString("KeyMap.F7")); //$NON-NLS-1$
    this.buttonList[0][8] = new JButton(Messages.getString("KeyMap.F8")); //$NON-NLS-1$
    this.buttonList[0][9] = new JButton(Messages.getString("KeyMap.F9")); //$NON-NLS-1$
    this.buttonList[0][10] = new JButton(Messages.getString("KeyMap.F10")); //$NON-NLS-1$
    this.buttonList[0][11] = new JButton(Messages.getString("KeyMap.F11")); //$NON-NLS-1$
    this.buttonList[0][12] = new JButton(Messages.getString("KeyMap.F12")); //$NON-NLS-1$
    this.buttonList[0][13] = new JButton(Messages.getString("KeyMap.PrintScreen")); //$NON-NLS-1$
    this.buttonList[0][14] = new JButton(Messages.getString("KeyMap.ScrollLock")); //$NON-NLS-1$
    this.buttonList[0][15] = new JButton(Messages.getString("KeyMap.Break")); //$NON-NLS-1$
    this.buttonList[0][16] = new JButton(""); //$NON-NLS-1$
    this.buttonList[0][17] = new JButton(""); //$NON-NLS-1$
    this.buttonList[0][18] = new JButton(""); //$NON-NLS-1$
    this.buttonList[0][19] = new JButton(""); //$NON-NLS-1$
    this.buttonList[0][20] = new JButton(""); //$NON-NLS-1$
    this.buttonList[1][0] = new JButton(Messages.getString("KeyMap.BackQuote")); //$NON-NLS-1$
    this.buttonList[1][1] = new JButton(Messages.getString("KeyMap.1")); //$NON-NLS-1$
    this.buttonList[1][2] = new JButton(Messages.getString("KeyMap.2")); //$NON-NLS-1$
    this.buttonList[1][3] = new JButton(Messages.getString("KeyMap.3")); //$NON-NLS-1$
    this.buttonList[1][4] = new JButton(Messages.getString("KeyMap.4")); //$NON-NLS-1$
    this.buttonList[1][5] = new JButton(Messages.getString("KeyMap.5")); //$NON-NLS-1$
    this.buttonList[1][6] = new JButton(Messages.getString("KeyMap.6")); //$NON-NLS-1$
    this.buttonList[1][7] = new JButton(Messages.getString("KeyMap.7")); //$NON-NLS-1$
    this.buttonList[1][8] = new JButton(Messages.getString("KeyMap.8")); //$NON-NLS-1$
    this.buttonList[1][9] = new JButton(Messages.getString("KeyMap.9")); //$NON-NLS-1$
    this.buttonList[1][10] = new JButton(Messages.getString("KeyMap.0")); //$NON-NLS-1$
    this.buttonList[1][11] = new JButton(Messages.getString("KeyMap.Minus")); //$NON-NLS-1$
    this.buttonList[1][12] = new JButton(Messages.getString("KeyMap.Equals")); //$NON-NLS-1$
    this.buttonList[1][13] = new JButton(Messages.getString("KeyMap.BackSpace")); //$NON-NLS-1$
    this.buttonList[1][14] = new JButton(Messages.getString("KeyMap.Insert")); //$NON-NLS-1$
    this.buttonList[1][15] = new JButton(Messages.getString("KeyMap.Home")); //$NON-NLS-1$
    this.buttonList[1][16] = new JButton(Messages.getString("KeyMap.PageUp")); //$NON-NLS-1$
    this.buttonList[1][17] = new JButton(Messages.getString("KeyMap.NumLock")); //$NON-NLS-1$
    this.buttonList[1][18] = new JButton(Messages.getString("KeyMap.Slash")); //$NON-NLS-1$
    this.buttonList[1][19] = new JButton(Messages.getString("KeyMap.Multiply")); //$NON-NLS-1$
    this.buttonList[1][20] = new JButton(Messages.getString("KeyMap.Minus")); //$NON-NLS-1$
    this.buttonList[2][0] = new JButton(Messages.getString("KeyMap.TabText")); //$NON-NLS-1$
    this.buttonList[2][1] = new JButton(Messages.getString("KeyMap.Q")); //$NON-NLS-1$
    this.buttonList[2][2] = new JButton(Messages.getString("KeyMap.W")); //$NON-NLS-1$
    this.buttonList[2][3] = new JButton(Messages.getString("KeyMap.E")); //$NON-NLS-1$
    this.buttonList[2][4] = new JButton(Messages.getString("KeyMap.R")); //$NON-NLS-1$
    this.buttonList[2][5] = new JButton(Messages.getString("KeyMap.T")); //$NON-NLS-1$
    this.buttonList[2][6] = new JButton(Messages.getString("KeyMap.Y")); //$NON-NLS-1$
    this.buttonList[2][7] = new JButton(Messages.getString("KeyMap.U")); //$NON-NLS-1$
    this.buttonList[2][8] = new JButton(Messages.getString("KeyMap.I")); //$NON-NLS-1$
    this.buttonList[2][9] = new JButton(Messages.getString("KeyMap.O")); //$NON-NLS-1$
    this.buttonList[2][10] = new JButton(Messages.getString("KeyMap.P")); //$NON-NLS-1$
    this.buttonList[2][11] = new JButton(Messages.getString("KeyMap.LeftBracket")); //$NON-NLS-1$
    this.buttonList[2][12] = new JButton(Messages.getString("KeyMap.RightBracket")); //$NON-NLS-1$
    this.buttonList[2][13] = new JButton(Messages.getString("KeyMap.DoubleBackSlash")); //$NON-NLS-1$
    this.buttonList[2][14] = new JButton(Messages.getString("KeyMap.Delete")); //$NON-NLS-1$
    this.buttonList[2][15] = new JButton(Messages.getString("KeyMap.End")); //$NON-NLS-1$
    this.buttonList[2][16] = new JButton(Messages.getString("KeyMap.PageDown")); //$NON-NLS-1$
    this.buttonList[2][17] = new JButton(Messages.getString("KeyMap.NumPad7")); //$NON-NLS-1$
    this.buttonList[2][18] = new JButton(Messages.getString("KeyMap.NumPad8")); //$NON-NLS-1$
    this.buttonList[2][19] = new JButton(Messages.getString("KeyMap.NumPad9")); //$NON-NLS-1$
    this.buttonList[2][20] = new JButton(Messages.getString("KeyMap.Plus")); //$NON-NLS-1$
    this.buttonList[3][0] = new JButton(Messages.getString("KeyMap.CapsLock")); //$NON-NLS-1$
    this.buttonList[3][1] = new JButton(Messages.getString("KeyMap.A")); //$NON-NLS-1$
    this.buttonList[3][2] = new JButton(Messages.getString("KeyMap.S")); //$NON-NLS-1$
    this.buttonList[3][3] = new JButton(Messages.getString("KeyMap.D")); //$NON-NLS-1$
    this.buttonList[3][4] = new JButton(Messages.getString("KeyMap.F")); //$NON-NLS-1$
    this.buttonList[3][5] = new JButton(Messages.getString("KeyMap.G")); //$NON-NLS-1$
    this.buttonList[3][6] = new JButton(Messages.getString("KeyMap.H")); //$NON-NLS-1$
    this.buttonList[3][7] = new JButton(Messages.getString("KeyMap.J")); //$NON-NLS-1$
    this.buttonList[3][8] = new JButton(Messages.getString("KeyMap.K")); //$NON-NLS-1$
    this.buttonList[3][9] = new JButton(Messages.getString("KeyMap.L")); //$NON-NLS-1$
    this.buttonList[3][10] = new JButton(Messages.getString("KeyMap.SemiColon")); //$NON-NLS-1$
    this.buttonList[3][11] = new JButton(Messages.getString("KeyMap.Quote")); //$NON-NLS-1$
    this.buttonList[3][12] = new JButton(Messages.getString("KeyMap.Enter")); //$NON-NLS-1$
    this.buttonList[3][13] = new JButton(Messages.getString("KeyMap.Enter")); //$NON-NLS-1$
    this.buttonList[3][14] = new JButton(""); //$NON-NLS-1$
    this.buttonList[3][15] = new JButton(""); //$NON-NLS-1$
    this.buttonList[3][16] = new JButton(""); //$NON-NLS-1$
    this.buttonList[3][17] = new JButton(Messages.getString("KeyMap.NumPad4")); //$NON-NLS-1$
    this.buttonList[3][18] = new JButton(Messages.getString("KeyMap.NumPad5")); //$NON-NLS-1$
    this.buttonList[3][19] = new JButton(Messages.getString("KeyMap.NumPad6")); //$NON-NLS-1$
    this.buttonList[3][20] = new JButton(Messages.getString("KeyMap.Plus")); //$NON-NLS-1$
    this.buttonList[4][0] = new JButton(Messages.getString("KeyMap.Shift")); //$NON-NLS-1$
    this.buttonList[4][1] = new JButton(Messages.getString("KeyMap.Z")); //$NON-NLS-1$
    this.buttonList[4][2] = new JButton(Messages.getString("KeyMap.X")); //$NON-NLS-1$
    this.buttonList[4][3] = new JButton(Messages.getString("KeyMap.C")); //$NON-NLS-1$
    this.buttonList[4][4] = new JButton(Messages.getString("KeyMap.V")); //$NON-NLS-1$
    this.buttonList[4][5] = new JButton(Messages.getString("KeyMap.B")); //$NON-NLS-1$
    this.buttonList[4][6] = new JButton(Messages.getString("KeyMap.N")); //$NON-NLS-1$
    this.buttonList[4][7] = new JButton(Messages.getString("KeyMap.M")); //$NON-NLS-1$
    this.buttonList[4][8] = new JButton(Messages.getString("KeyMap.Comma")); //$NON-NLS-1$
    this.buttonList[4][9] = new JButton(Messages.getString("KeyMap.Period")); //$NON-NLS-1$
    this.buttonList[4][10] = new JButton(Messages.getString("KeyMap.Slash")); //$NON-NLS-1$
    this.buttonList[4][11] = new JButton(Messages.getString("KeyMap.Shift")); //$NON-NLS-1$
    this.buttonList[4][12] = new JButton(""); //$NON-NLS-1$
    this.buttonList[4][13] = new JButton(""); //$NON-NLS-1$
    this.buttonList[4][14] = new JButton(""); //$NON-NLS-1$
    this.buttonList[4][15] = new JButton(Messages.getString("KeyMap.Up")); //$NON-NLS-1$
    this.buttonList[4][16] = new JButton(""); //$NON-NLS-1$
    this.buttonList[4][17] = new JButton(Messages.getString("KeyMap.NumPad1")); //$NON-NLS-1$
    this.buttonList[4][18] = new JButton(Messages.getString("KeyMap.NumPad2")); //$NON-NLS-1$
    this.buttonList[4][19] = new JButton(Messages.getString("KeyMap.NumPad3")); //$NON-NLS-1$
    this.buttonList[4][20] = new JButton(Messages.getString("KeyMap.Enter")); //$NON-NLS-1$
    this.buttonList[5][0] = new JButton(Messages.getString("KeyMap.Control")); //$NON-NLS-1$
    this.buttonList[5][1] = new JButton(Messages.getString("KeyMap.Windows")); //$NON-NLS-1$
    this.buttonList[5][2] = new JButton(Messages.getString("KeyMap.Alt")); //$NON-NLS-1$
    this.buttonList[5][3] = new JButton(Messages.getString("KeyMap.Space")); //$NON-NLS-1$
    this.buttonList[5][4] = new JButton(Messages.getString("KeyMap.Space")); //$NON-NLS-1$
    this.buttonList[5][5] = new JButton(Messages.getString("KeyMap.Space")); //$NON-NLS-1$
    this.buttonList[5][6] = new JButton(Messages.getString("KeyMap.Space")); //$NON-NLS-1$
    this.buttonList[5][7] = new JButton(Messages.getString("KeyMap.Space")); //$NON-NLS-1$
    this.buttonList[5][8] = new JButton(Messages.getString("KeyMap.Space")); //$NON-NLS-1$
    this.buttonList[5][9] = new JButton(Messages.getString("KeyMap.Space")); //$NON-NLS-1$
    this.buttonList[5][10] = new JButton(Messages.getString("KeyMap.Alt")); //$NON-NLS-1$
    this.buttonList[5][11] = new JButton(Messages.getString("KeyMap.Windows")); //$NON-NLS-1$
    this.buttonList[5][12] = new JButton(Messages.getString("KeyMap.Menu")); //$NON-NLS-1$
    this.buttonList[5][13] = new JButton(Messages.getString("KeyMap.Control")); //$NON-NLS-1$
    this.buttonList[5][14] = new JButton(Messages.getString("KeyMap.Left")); //$NON-NLS-1$
    this.buttonList[5][15] = new JButton(Messages.getString("KeyMap.Down")); //$NON-NLS-1$
    this.buttonList[5][16] = new JButton(Messages.getString("KeyMap.Right")); //$NON-NLS-1$
    this.buttonList[5][17] = new JButton(Messages.getString("KeyMap.NumPad0")); //$NON-NLS-1$
    this.buttonList[5][18] = new JButton(Messages.getString("KeyMap.NumPad0")); //$NON-NLS-1$
    this.buttonList[5][19] = new JButton(Messages.getString("KeyMap.Period")); //$NON-NLS-1$
    this.buttonList[5][20] = new JButton(Messages.getString("KeyMap.Enter")); //$NON-NLS-1$
    MouseMotionAdapter mouseMotionAdapter = new MouseMotionAdapter()
    {
      private final static long serialVersionUID = 1L;

      @Override
      public void mouseMoved(MouseEvent mouseEvent)
      {
        KeyDialog.this.mouseMovedButton(mouseEvent);
      }
    };
    for (int j = 0;j < 21;j++)
    {
      for (int i = 0;i < 6;i++)
      {
        Action action = new AbstractAction(this.buttonList[i][j].getText())
        {
          private final static long serialVersionUID = 1L;

          public void actionPerformed(ActionEvent actionEvent)
          {
            KeyDialog.this.actionPerformedButton(actionEvent);
          }
        };
        this.buttonList[i][j].setAction(action);
        this.buttonList[i][j].addMouseMotionListener(mouseMotionAdapter);
        this.tablePanel.add(this.buttonList[i][j]);
      }
    }
  }

  /**
   * @param button the button to set the bounds of
   * @param insets the inset attributes
   * @param location the location of the button
   * @param dimension the size of the button (width/height)
   */
  private void setButtonBounds(JButton button,Insets insets,Point location,Dimension dimension) // pass new point dimension
  {
    button.setMargin(insets);
    button.setLocation(location.x,location.y);
    button.setPreferredSize(dimension);
    button.setSize(dimension);
  }

  /**
   * Resize all of the buttons.
   */
  private void resizeButtons()
  {
    Insets insets = new Insets(0,0,0,0);
    int left = 0;
    int width = 78;
    // do one column at a time to have even width for all buttons in column
    this.setButtonBounds(this.buttonList[0][0],insets,new Point(left,0),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[1][0],insets,new Point(left,30),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[2][0],insets,new Point(left,60),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[3][0],insets,new Point(left,90),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[4][0],insets,new Point(left,120),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[5][0],insets,new Point(left,150),new Dimension(width,30));
    left += width;
    width = 44;
    this.setButtonBounds(this.buttonList[0][1],insets,new Point(left,0),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[1][1],insets,new Point(left,30),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[2][1],insets,new Point(left,60),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[3][1],insets,new Point(left,90),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[4][1],insets,new Point(left,120),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[5][1],insets,new Point(left,150),new Dimension(width,30));
    left += width;
    width = 28;
    this.setButtonBounds(this.buttonList[0][2],insets,new Point(left,0),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[1][2],insets,new Point(left,30),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[2][2],insets,new Point(left,60),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[3][2],insets,new Point(left,90),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[4][2],insets,new Point(left,120),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[5][2],insets,new Point(left,150),new Dimension(width,30));
    left += width;
    width = 28;
    this.setButtonBounds(this.buttonList[0][3],insets,new Point(left,0),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[1][3],insets,new Point(left,30),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[2][3],insets,new Point(left,60),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[3][3],insets,new Point(left,90),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[4][3],insets,new Point(left,120),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[5][3],insets,new Point(left,150),new Dimension(196,30));
    left += width;
    width = 28;
    this.setButtonBounds(this.buttonList[0][4],insets,new Point(left,0),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[1][4],insets,new Point(left,30),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[2][4],insets,new Point(left,60),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[3][4],insets,new Point(left,90),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[4][4],insets,new Point(left,120),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[5][4],insets,new Point(left,150),new Dimension(width,30));
    left += width;
    width = 28;
    this.setButtonBounds(this.buttonList[0][5],insets,new Point(left,0),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[1][5],insets,new Point(left,30),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[2][5],insets,new Point(left,60),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[3][5],insets,new Point(left,90),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[4][5],insets,new Point(left,120),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[5][5],insets,new Point(left,150),new Dimension(width,30));
    left += width;
    width = 28;
    this.setButtonBounds(this.buttonList[0][6],insets,new Point(left,0),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[1][6],insets,new Point(left,30),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[2][6],insets,new Point(left,60),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[3][6],insets,new Point(left,90),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[4][6],insets,new Point(left,120),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[5][6],insets,new Point(left,150),new Dimension(width,30));
    left += width;
    width = 28;
    this.setButtonBounds(this.buttonList[0][7],insets,new Point(left,0),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[1][7],insets,new Point(left,30),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[2][7],insets,new Point(left,60),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[3][7],insets,new Point(left,90),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[4][7],insets,new Point(left,120),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[5][7],insets,new Point(left,150),new Dimension(width,30));
    left += width;
    width = 28;
    this.setButtonBounds(this.buttonList[0][8],insets,new Point(left,0),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[1][8],insets,new Point(left,30),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[2][8],insets,new Point(left,60),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[3][8],insets,new Point(left,90),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[4][8],insets,new Point(left,120),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[5][8],insets,new Point(left,150),new Dimension(width,30));
    left += width;
    width = 28;
    this.setButtonBounds(this.buttonList[0][9],insets,new Point(left,0),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[1][9],insets,new Point(left,30),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[2][9],insets,new Point(left,60),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[3][9],insets,new Point(left,90),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[4][9],insets,new Point(left,120),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[5][9],insets,new Point(left,150),new Dimension(width,30));
    left += width;
    width = 28;
    this.setButtonBounds(this.buttonList[0][10],insets,new Point(left,0),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[1][10],insets,new Point(left,30),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[2][10],insets,new Point(left,60),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[3][10],insets,new Point(left,90),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[4][10],insets,new Point(left,120),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[5][10],insets,new Point(left,150),new Dimension(width,30));
    left += width;
    width = 44;
    this.setButtonBounds(this.buttonList[0][11],insets,new Point(left,0),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[1][11],insets,new Point(left,30),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[2][11],insets,new Point(left,60),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[3][11],insets,new Point(left,90),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[4][11],insets,new Point(left,120),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[5][11],insets,new Point(left,150),new Dimension(width,30));
    left += width;
    width = 47;
    this.setButtonBounds(this.buttonList[0][12],insets,new Point(left,0),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[1][12],insets,new Point(left,30),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[2][12],insets,new Point(left,60),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[3][12],insets,new Point(left,90),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[4][12],insets,new Point(left,120),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[5][12],insets,new Point(left,150),new Dimension(width,30));
    left += width;
    width = 93;
    this.setButtonBounds(this.buttonList[0][13],insets,new Point(left,0),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[1][13],insets,new Point(left,30),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[2][13],insets,new Point(left,60),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[3][13],insets,new Point(left,90),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[4][13],insets,new Point(left,120),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[5][13],insets,new Point(left,150),new Dimension(width,30));
    left += width;
    width = 95;
    this.setButtonBounds(this.buttonList[0][14],insets,new Point(left,0),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[1][14],insets,new Point(left,30),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[2][14],insets,new Point(left,60),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[3][14],insets,new Point(left,90),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[4][14],insets,new Point(left,120),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[5][14],insets,new Point(left,150),new Dimension(width,30));
    left += width;
    width = 46;
    this.setButtonBounds(this.buttonList[0][15],insets,new Point(left,0),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[1][15],insets,new Point(left,30),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[2][15],insets,new Point(left,60),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[3][15],insets,new Point(left,90),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[4][15],insets,new Point(left,120),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[5][15],insets,new Point(left,150),new Dimension(width,30));
    left += width;
    width = 83;
    this.setButtonBounds(this.buttonList[0][16],insets,new Point(left,0),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[1][16],insets,new Point(left,30),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[2][16],insets,new Point(left,60),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[3][16],insets,new Point(left,90),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[4][16],insets,new Point(left,120),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[5][16],insets,new Point(left,150),new Dimension(width,30));
    left += width;
    width = 73;
    this.setButtonBounds(this.buttonList[0][17],insets,new Point(left,0),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[1][17],insets,new Point(left,30),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[2][17],insets,new Point(left,60),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[3][17],insets,new Point(left,90),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[4][17],insets,new Point(left,120),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[5][17],insets,new Point(left,150),new Dimension(width,30));
    left += width;
    width = 64;
    this.setButtonBounds(this.buttonList[0][18],insets,new Point(left,0),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[1][18],insets,new Point(left,30),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[2][18],insets,new Point(left,60),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[3][18],insets,new Point(left,90),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[4][18],insets,new Point(left,120),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[5][18],insets,new Point(left,150),new Dimension(width,30));
    left += width;
    width = 64;
    this.setButtonBounds(this.buttonList[0][19],insets,new Point(left,0),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[1][19],insets,new Point(left,30),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[2][19],insets,new Point(left,60),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[3][19],insets,new Point(left,90),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[4][19],insets,new Point(left,120),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[5][19],insets,new Point(left,150),new Dimension(width,30));
    left += width;
    width = 47;
    this.setButtonBounds(this.buttonList[0][20],insets,new Point(left,0),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[1][20],insets,new Point(left,30),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[2][20],insets,new Point(left,60),new Dimension(width,60));
    this.setButtonBounds(this.buttonList[3][20],insets,new Point(left,90),new Dimension(width,30));
    this.setButtonBounds(this.buttonList[4][20],insets,new Point(left,120),new Dimension(width,60));
    this.setButtonBounds(this.buttonList[5][20],insets,new Point(left,150),new Dimension(width,30));
    left += width;
    this.tablePanel.setMinimumSize(new Dimension(left,180));
    this.tablePanel.setPreferredSize(new Dimension(left,180));
  }

  /**
   * Open a keymap and update the key dialog with those values.
   * @param actionEvent the action to perform
   */
  void keyFileOpenItem_actionPerformed(@SuppressWarnings("unused") ActionEvent actionEvent)
  {
    JFileChooser chooser = new JFileChooser();
    KeyboardFileFilter filter = new KeyboardFileFilter();
    chooser.setFileFilter(filter);
    chooser.setAcceptAllFileFilterUsed(false);
    chooser.setDialogTitle(Messages.getString("KeyDialog.LoadKeyboardMessage")); //$NON-NLS-1$
    if (chooser.showOpenDialog(this.getParent()) == JFileChooser.APPROVE_OPTION)
    {
      try
      {
        this.keyMap.load(chooser.getSelectedFile().toURL());
      }
      catch (MalformedURLException malformedURLException)
      {
        System.out.printf(Messages.getString("KeyDialog.ErrorProblemSavingFile"),chooser.getSelectedFile().toString(),malformedURLException.getMessage()); //$NON-NLS-1$
      }
    }
  }

  /**
   * Save the keymap from this dialog.
   * @param actionEvent the action to perform
   */
  void keyFileSaveItem_actionPerformed(@SuppressWarnings("unused") ActionEvent actionEvent)
  {
    URL url = null;
    JFileChooser chooser = new JFileChooser();
    KeyboardFileFilter filter = new KeyboardFileFilter();
    chooser.setFileFilter(filter);
    chooser.setAcceptAllFileFilterUsed(false);
    chooser.setDialogTitle(Messages.getString("KeyDialog.SaveKeyboardMessage")); //$NON-NLS-1$
    if (chooser.showSaveDialog(this.getParent()) == JFileChooser.APPROVE_OPTION)
    {
      try
      {
        String filename = chooser.getSelectedFile().toString();
        if (KeyboardFileFilter.getExtension(chooser.getSelectedFile()) != null)
        {
          if (!KeyboardFileFilter.getExtension(chooser.getSelectedFile()).equals(Messages.getString("KeyDialog.Properties"))) //$NON-NLS-1$
            filename = filename + Messages.getString("KeyDialog.DotProperties"); //$NON-NLS-1$
        }
        else
        {
          filename = filename + Messages.getString("KeyDialog.DotProperties"); //$NON-NLS-1$
        }
        url = new URL(Messages.getString("KeyDialog.FileUrl") + filename); //$NON-NLS-1$
        this.keyMap.save(url);
      }
      catch (MalformedURLException malformedURLException)
      {
        System.out.printf(Messages.getString("KeyDialog.ErrorProblemSavingFile"),url,malformedURLException.getMessage()); //$NON-NLS-1$
      }
    }
  }

  /**
   * Exit the dialog and abandon your changes.
   * @param actionEvent the action to perform
   */
  void keyFileExitItem_actionPerformed(@SuppressWarnings("unused") ActionEvent actionEvent)
  {
    this.setVisible(false);
  }

  /**
   * The dialog ended successfully so exit the dialog.
   * @param actionEvent the action to perform
   */
  void okButton_actionPerformed(@SuppressWarnings("unused") ActionEvent actionEvent)
  {
    this.successStatus = true;
    this.setVisible(false);
  }

  /**
   * Cancel your changes.
   * @param actionEvent the action to perform
   */
  void cancelButton_actionPerformed(@SuppressWarnings("unused") ActionEvent actionEvent)
  {
    this.setVisible(false);
  }

  /**
   * Upodate the tooltip when the mouse moves.
   * @param mouseEvent the mouse event to handle
   */
  void mouseMovedButton(MouseEvent mouseEvent)
  {
    if (mouseEvent.getSource() instanceof JButton)
    {
      String mnemonic = ((JButton)mouseEvent.getSource()).getText();
      if (!mnemonic.equalsIgnoreCase("")) //$NON-NLS-1$
      {
        if (KeyMap.containsMnemonicKey(mnemonic))
        {
          Integer value1 = KeyMap.getKey(mnemonic);
          KeyMap.Action value2 = this.keyMap.getAction(value1);
          String toolTip = KeyMap.getMnemonic(value2);
          ((JButton)mouseEvent.getSource()).setToolTipText(toolTip);
        }
      }
      else
        ((JButton)mouseEvent.getSource()).setToolTipText(null);
    }
  }

  /**
   * Perform the action specified.
   * @param actionEvent the event to process
   */
  public void actionPerformedButton(ActionEvent actionEvent)
  {
    this.currentKey = actionEvent.getActionCommand();
    if (!this.currentKey.equalsIgnoreCase("")) //$NON-NLS-1$
    {
      Vector<String> actionList = new Vector<String>();
      for (KeyMap.Action value : this.keyMap.getActions())
        if (!ACTION_UNKNOWN.equals( value ) && KeyMap.containsActionMnemonic(value))
          if (KeyMap.getMnemonic(value) != null)
            if (!actionList.contains(KeyMap.getMnemonic(value)))
              actionList.add(KeyMap.getMnemonic(value));
      Collections.sort(actionList);
      this.menuPopup = new JPopupMenu();
      for (String value : actionList)
      {
        Action action = new AbstractAction(value)
        {
          private final static long serialVersionUID = 1L;

          public void actionPerformed(ActionEvent _actionEvent)
          {
            KeyDialog.this.actionPerformedMenu(_actionEvent);
          }
        };
        this.menuPopup.add(action);
      }
      // add the popup here to get the new action!
      this.menuPopup.setInvoker(this);
      this.menuPopup.show((Component)actionEvent.getSource(),25,20);
    }
  }

  /**
   * Store an action for a key to perform when pressed.
   * @param actionEvent the event to process
   */
  public void actionPerformedMenu(ActionEvent actionEvent)
  {
    this.menuPopup.setVisible(false);
    String action = KeyMap.getMnemonic(KeyMap.getAction(actionEvent.getActionCommand()));
    this.keyMap.loadMnemonic(this.currentKey,action);
  }
}

/**
 * A class to handle the input of actions for key presses.
 * @author mallette
 */
class KeyboardFileFilter extends FileFilter
{
  /**
   * Determine whether or not to accept the changes.
   * @param file the file to load from, and save to.
   * @return were the changes successful?
   */
  /*
   * (non-Javadoc)
   * @see javax.swing.filechooser.FileFilter#accept(java.io.File)
   */
  @Override
  public boolean accept(File file)
  {
    boolean returnValue = false;
    if (file.isDirectory())
    {
      returnValue = true;
    }
    else
    {
      String extension = KeyboardFileFilter.getExtension(file);
      if (extension != null)
        returnValue = extension.equals(Messages.getString("KeyDialog.Properties")); //$NON-NLS-1$
    }
    return returnValue;
  }

  /**
   * Get the description of the file filter.
   * @return the description for the file type
   */
  /*
   * (non-Javadoc)
   * @see javax.swing.filechooser.FileFilter#getDescription()
   */
  @Override
  public String getDescription()
  {
    return Messages.getString("KeyDialog.KeyboardPropertiesFile"); //$NON-NLS-1$
  }

  /**
   * @param file the file name for the keyboard information
   * @return the extension on the file
   */
  public static String getExtension(File file)
  {
    String ext = null;
    String string = file.getName();
    int position = string.lastIndexOf('.');
    if (position > 0 && position < string.length() - 1)
      ext = string.substring(position + 1).toLowerCase();
    return ext;
  }
}
