/******************************************************************************
 * Project JKobo
 * XKobo arcade style game
 *
 * Copyright (C) 1997 John "Jake" Wetzel All Rights Reserved
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 *****************************************************************************/

/***************************** FILE INFORMATION ******************************
 * NAME OF FILE 	  : Tiff.java
 * CONCEPT & ARCHITECTURE : Akira Higuchi
 * MAIN PROGRAMMER	  : Jake Wetzel
 * DATE CREATED 	  : 10-25-98
 * DESCRIPTION		  :
 * CHANGE LOG		  :
 *	  WHO		  WHEN			  WHY
 *	  ---		  ----			  ---
 *
 *****************************************************************************/

/******************************* IMPORT FILES ********************************/

import java.awt.Image;
import java.awt.Toolkit;
import java.awt.image.ColorModel;
import java.awt.image.ImageObserver;
import java.awt.image.IndexColorModel;
import java.awt.image.MemoryImageSource;
import java.awt.image.PixelGrabber;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.text.MessageFormat;
import java.util.MissingResourceException;
import java.util.ResourceBundle;


/********    tiny TIFF loader  ( 256 colors, not compressed )	********/

class Tiff
{

  static final int CM_MIN   =	 0;
  static final int CM_MAX   =  255;
  static final int BUFF_MAX = 1024;

  static String FindError;
  static String ReadError;
  static String CloseError;
  static String FormatError;
  static String WriteError;
  static String FlushError;
  static String SpriteName;
  static String TiffName;

/******************************** GLOBAL DATA ********************************/

  int pos_image;
  int pos_cmap;
  int sizex;
  int sizey;
  int endianness;
  byte data[];
  IndexColorModel cmap;
  MemoryImageSource tif;

/**************************** FUNCTION DECLARATIONS  *************************/

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : Tiff
 * DESCRIPTION	  :
 *****************************************************************************/
  Tiff()
  {
    long len;
    File f;
    FileInputStream fis;
    String s;
    Object objs[];


    fis = null;
    data = null;
    f = new File(JKobo.Context.getAppPath(), "xkp256.tif");
    if(f.exists() == false)
      return;

    len = f.length();
    if(len == 0)
      return;

    objs = new Object[2];
    objs[0] = (Object) SpriteName;
    objs[1] = (Object) f.getAbsolutePath();
    data = new byte[(int) len];
    fis = null;

    try
      {
	fis = new FileInputStream(f);
	fis.read(data);
      }

    catch(FileNotFoundException e1)
      {
	System.err.println(MessageFormat.format(FindError, objs));
      }

    catch(IOException e2)
      {
	System.err.println(MessageFormat.format(ReadError, objs));
      }

    finally
      {
	if(fis != null)
	  {
	    try
	      {
		fis.close();
	      }
	    catch(IOException e2)
	      {
		System.err.println(MessageFormat.format(CloseError, objs));
	      }
	  }
      }

  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : get16Bits
 * DESCRIPTION	  :
 *****************************************************************************/
  private int get16Bits(byte p[], int n)
  {
    int c;


    if(endianness == 0x49)
      {
	c = ((int) p[n]) & 0xff;
	c |= (((int) p[n+1]) << 8);
      }
    else
      {
	c = ((int) p[n+1]) & 0xff;
	c |= (((int) p[n]) << 8);
      }

    return(c);

  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : get32Bits
 * DESCRIPTION	  :
 *****************************************************************************/
  private int get32Bits(byte p[], int n)
  {
    int c;


    if(endianness == 0x49)
      {
	c = ((int) p[n]) & 0xff;
	c |= ((((int) p[n+1]) << 8) & 0xff00);
	c |= ((((int) p[n+2]) << 16) & 0xff0000);
	c |= (((int) p[n+3]) << 24);
      }
    else
      {
	c = ((int) p[n+3]) & 0xff;
	c |= ((((int) p[n+2]) << 8) & 0xff00);
	c |= ((((int) p[n+1]) << 16) & 0xff0000);
	c |= (((int) p[n]) << 24);
      }

    return(c);

  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : readTag
 * DESCRIPTION	  :
 *****************************************************************************/
  private int readTag(byte data[], int n)
  {
    byte p[] = data;
    int dir_p = 0;
    int tag_max;
    int tag_type;
    int i, m;

    endianness = (int) p[n];
    if((endianness != 0x49) && (endianness != 0x4d))
       return(1);
    if(endianness != ((int) p[n+1]))
      return(2);
    if(get16Bits(p, 2) != 0x2a)
      return(3);
    m = n + 4;

    while(true)
      {
	dir_p = get32Bits(p, m);
	if(dir_p == 0)
	  break;
	m = dir_p;

	tag_max = get16Bits(p, m);
	m += 2;
	for(i=0; i<tag_max; i++)
	  {
	    tag_type = get16Bits(p, m);
	    switch(tag_type)
	      {
	      case 0x0100:
		/* image width */
		sizex = get16Bits(p, m + 8);
		break;
	      case 0x0101:
		/* image length */
		sizey = get16Bits(p, m + 8);
		break;
	      case 0x0102:
		/* bits per sample */
		if(get16Bits(p, m + 8) != 8)
		  return(4);
		break;
	      case 0x0103:
		/* compression */
		if(get16Bits(p, m + 8) != 1)
		  return(5);
		break;
	      case 0x0111:
		/* strip offset */
		pos_image = get32Bits(p, m + 8);
		break;
	      case 0x0140:
		/* color map */
		pos_cmap = get32Bits(p, m + 8);
		break;
	    }
	    m += 12;
	  }
      }

    if((pos_image == 0) || (pos_cmap == 0) || (sizex == 0) || (sizey == 0))
      return(6);

    /* if big endian, point to least significant byte */
    if(endianness == 0x4d)
      pos_cmap++;

    return(0);

  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : readTiff
 * DESCRIPTION	  :
 *****************************************************************************/
  MemoryImageSource readTiff()
  {
    int x, y, m, n, trans, ps;
    byte r[], g[], b[], t[];
    boolean u[];



    if(data == null)
      return(null);

    if(readTag(data, 0) != 0)
      {
	 System.err.println(MessageFormat.format(FormatError, new Object[0]));
	 data = null;
	 return(null);
      }

    r = new byte[256];
    g = new byte[256];
    b = new byte[256];
    u = new boolean[256];

    n = pos_cmap;
    for(trans=x=0; x<256; x++)
      {
	r[x] = (byte) (get16Bits(data, n) >>> 8);
	g[x] = (byte) (get16Bits(data, n + 512) >>> 8);
	b[x] = (byte) (get16Bits(data, n + 1024) >>> 8);
	if((r[x] == ((byte) 0xff)) && (g[x] == 0) && (b[x] == 0))
	  trans = x;
	n += 2;
      }

    /* check if all palette entries are used */
    n = pos_image;
    for(y=0; y<sizey; y++)
      {
	for(x=0; x<sizex; x++)
	  {
	    m = ((int) data[n + x]) & 0xff;
	    u[m] = true;
	  }
	n += sizex;
      }

    for(x=0; x<256; x++)
      if(u[x] == false)
	break;

    if(x < 256)
      {
	t = new byte[256];
	for(x=y=0; x<256; x++)
	  {
	    if(u[x] == false)
	      continue;
	    t[x] = (byte) y;
	    if(x > y)
	      {
		r[y] = r[x];
		g[y] = g[x];
		b[y] = b[x];
	      }
	    y++;
	  }
	ps = y;
	n = pos_image;
	for(y=0; y<sizey; y++)
	  {
	    for(x=0; x<sizex; x++)
	      {
		m = ((int) data[n + x]) & 0xff;
		data[n + x] = t[m];
	      }
	    n += sizex;
	  }
	cmap = new IndexColorModel(8, ps, r, g, b, (int) t[trans]);
      }
    else
      cmap = new IndexColorModel(8, 256, r, g, b, trans);

    tif = new MemoryImageSource(sizex, sizey, (ColorModel) cmap, data,
				pos_image, sizex);
    data = null;

    return(tif);

  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : writeTiff
 * DESCRIPTION	  :
 *****************************************************************************/
  static void writeTiff(String filename, Image image)
  {
    int i, j, k, n, w, h, off;
    int pix[], size, rows, strips;
    byte t[], r[], g[], b[], buf[];
    FileOutputStream fos;
    BufferedOutputStream bos;
    PixelGrabber pg;
    Object objs[];


    if(image == null)
      return;

    if(filename == null)
      return;

    objs = new Object[2];
    objs[0] = (Object) TiffName;
    objs[1] = (Object) filename;
    t = new byte[3];
    w = image.getWidth(null);
    h = image.getHeight(null);
    buf = new byte[BUFF_MAX];
    rows = 8192 / w;
    if(rows > h)
      rows = h;
    strips = h / rows;
    if((h % rows) > 0)
      strips++;
    pix = new int[w * h];
    pg = new PixelGrabber(image, 0, 0, w, h, pix, 0, w);
    pg.startGrabbing();
    fos = null;
    bos = null;
    try {
      fos = new FileOutputStream(filename);
      bos = new BufferedOutputStream((OutputStream) fos);
      off = 0;

      /* write endian tag */
      buf[0] = buf[1] = 0x4d;
      off += 2;

      /* write TIFF file identifier */
      writeShort(buf, off, (short) 0x2a);
      off += 2;

      /* write offset of first (and only) directory */
      writeInt(buf, off, 8);
      off += 4;

      /* write number of tags in this directory */
      writeShort(buf, off, (short) 15);
      off += 2;

      /*
       * offset = 10
       */

      /* write subfile type */
      writeShort(buf, off, (short) 0xfe);
      off += 2;
      writeShort(buf, off, (short) 4);
      off += 2;
      writeInt(buf, off, 1);
      off += 4;
      writeInt(buf, off, 0);
      off += 4;

      /* write image width */
      writeShort(buf, off, (short) 0x100);
      off += 2;
      writeShort(buf, off, (short) 4);
      off += 2;
      writeInt(buf, off, 1);
      off += 4;
      writeInt(buf, off, w);
      off += 4;

      /* write image height */
      writeShort(buf, off, (short) 0x101);
      off += 2;
      writeShort(buf, off, (short) 4);
      off += 2;
      writeInt(buf, off, 1);
      off += 4;
      writeInt(buf, off, h);
      off += 4;

      /* write image bits per sample */
      writeShort(buf, off, (short) 0x102);
      off += 2;
      writeShort(buf, off, (short) 3);
      off += 2;
      writeInt(buf, off, 1);
      off += 4;
      writeShort(buf, off, (short) 8);	  /* 256 color palette */
      off += 2;
      writeShort(buf, off, (short) 0);
      off += 2;

      /* write image compression */
      writeShort(buf, off, (short) 0x103);
      off += 2;
      writeShort(buf, off, (short) 3);
      off += 2;
      writeInt(buf, off, 1);
      off += 4;
      writeShort(buf, off, (short) 1);
      off += 2;
      writeShort(buf, off, (short) 0);
      off += 2;

      /* write image photometric interpretation */
      writeShort(buf, off, (short) 0x106);
      off += 2;
      writeShort(buf, off, (short) 3);
      off += 2;
      writeInt(buf, off, 1);
      off += 4;
      writeShort(buf, off, (short) 3);
      off += 2;
      writeShort(buf, off, (short) 0);
      off += 2;

      /* write image strip offsets */
      writeShort(buf, off, (short) 0x111);
      off += 2;
      writeShort(buf, off, (short) 4);
      off += 2;
      if(strips > 1)
	{
	  writeInt(buf, off, strips);
	  off += 4;
	  writeInt(buf, off, 210);
	  off += 4;
	}
      else
	{
	  writeInt(buf, off, 1);
	  off += 4;
	  writeInt(buf, off, 210);
	  off += 4;
	}

      /* write image samples per pixel */
      writeShort(buf, off, (short) 0x115);
      off += 2;
      writeShort(buf, off, (short) 3);
      off += 2;
      writeInt(buf, off, 1);
      off += 4;
      writeShort(buf, off, (short) 1);	  /* palette color */
      off += 2;
      writeShort(buf, off, (short) 0);
      off += 2;

      /* write image rows per strip */
      writeShort(buf, off, (short) 0x116);
      off += 2;
      writeShort(buf, off, (short) 4);
      off += 2;
      writeInt(buf, off, 1);
      off += 4;
      writeInt(buf, off, rows);
      off += 4;

      /* write image strip byte counts */
      writeShort(buf, off, (short) 0x117);
      off += 2;
      writeShort(buf, off, (short) 4);
      off += 2;
      if(strips > 1)
	{
	  writeInt(buf, off, strips);
	  off += 4;
	  writeInt(buf, off, 210 + (strips << 2));
	  off += 4;
	}
      else
	{
	  writeInt(buf, off, 1);
	  off += 4;
	  writeInt(buf, off, w*rows);
	  off += 4;
	}

      /* write image x resolution */
      writeShort(buf, off, (short) 0x11a);
      off += 2;
      writeShort(buf, off, (short) 5);
      off += 2;
      writeInt(buf, off, 1);
      off += 4;
      writeInt(buf, off, 194);
      off += 4;

      /* write image y resolution */
      writeShort(buf, off, (short) 0x11b);
      off += 2;
      writeShort(buf, off, (short) 5);
      off += 2;
      writeInt(buf, off, 1);
      off += 4;
      writeInt(buf, off, 202);
      off += 4;

      /* write image planar configuration */
      writeShort(buf, off, (short) 0x11c);
      off += 2;
      writeShort(buf, off, (short) 3);
      off += 2;
      writeInt(buf, off, 1);
      off += 4;
      writeShort(buf, off, (short) 1);
      off += 2;
      writeShort(buf, off, (short) 0);
      off += 2;

      /* write image resolution unit */
      writeShort(buf, off, (short) 0x128);
      off += 2;
      writeShort(buf, off, (short) 3);
      off += 2;
      writeInt(buf, off, 1);
      off += 4;
      writeShort(buf, off, (short) 2);
      off += 2;
      writeShort(buf, off, (short) 0);
      off += 2;

      /* write image color map */
      writeShort(buf, off, (short) 0x140);
      off += 2;
      writeShort(buf, off, (short) 3);
      off += 2;
      writeInt(buf, off, 3 * 256);
      off += 4;
      if(strips > 1)
	{
	  writeInt(buf, off, 210 + (strips << 3) + (strips * rows * w));
	  off += 4;
	}
      else
	{
	  writeInt(buf, off, 210 + (h * w));
	  off += 4;
	}

      /* write offset of next directory */
      writeInt(buf, off, 0);
      off += 4;

      /*
       * offset = 194
       */

      /* write image x resolution */
      writeInt(buf, off, Toolkit.getDefaultToolkit().getScreenResolution());
      off += 4;
      writeInt(buf, off, 1);
      off += 4;

      /*
       * offset = 202
       */

      /* write image y resolution */
      writeInt(buf, off, Toolkit.getDefaultToolkit().getScreenResolution());
      off += 4;
      writeInt(buf, off, 1);
      off += 4;

      bos.write(buf, 0, off);
      off = 0;

      /*
       * offset = 210
       */

      /* write strip offsets */
      n = 210;
      if(strips > 1)
	{
	  n += (strips << 3);
	  for(i=0; i<strips; i++)
	    {
	      if((off + 4) > BUFF_MAX)
		{
		  bos.write(buf, 0, off);
		  off = 0;
		}
	      writeInt(buf, off, n);
	      off += 4;
	      n += (w * rows);
	    }
	  bos.write(buf, 0, off);
	  off = 0;
	}

      /* image strip byte counts */
      if(strips > 1)
	{
	  n = w * rows;
	  for(i=0; i<strips; i++)
	    {
	      if((off + 4) > BUFF_MAX)
		{
		  bos.write(buf, 0, off);
		  off = 0;
		}
	      writeInt(buf, off, n);
	      off += 4;
	    }
	  bos.write(buf, 0, off);
	  off = 0;
	}

      /* write image palette index data */
      r = new byte[256];
      g = new byte[256];
      b = new byte[256];
      size = 0;
      for(i=0; i<h; i++)
	{
	  for(j=0; j<w; j++)
	    {
	      n = (i * w) + j;
	      t[0] = (byte) pix[n];
	      t[1] = (byte) (pix[n] >>> 8);
	      t[2] = (byte) (pix[n] >>> 16);
	      for(k=0; k<size; k++)
		{
		  if(t[2] != r[k])
		    continue;
		  if(t[1] != g[k])
		    continue;
		  if(t[0] != b[k])
		    continue;
		  break;
		}
	      if(k == size)
		{
		  r[size] = t[2];
		  g[size] = t[1];
		  b[size] = t[0];
		  size++;
		}
	      if(off == BUFF_MAX)
		{
		  bos.write(buf, 0, off);
		  off = 0;
		}
	      buf[off] = (byte) k;
	      off++;
	    }
	}

      /* fill out remainder of strip */
      for(i=h*w; i<strips*rows*w; i++)
	{
	  if(off == 1024)
	    {
	      bos.write(buf, 0, off);
	      off = 0;
	    }
	  buf[off] = 0;
	  off++;
	}

      if(off > 0)
	{
	  bos.write(buf, 0, off);
	  off = 0;
	}

      /* write red palette table */
      for(i=0; i<size; i++)
	{
	  buf[off] = r[i];
	  off++;
	  buf[off] = r[i];
	  off++;
	}
      for(; i<256; i++)
	{
	  buf[off] = 0;
	  off++;
	  buf[off] = 0;
	  off++;
	}
      bos.write(buf, 0, off);
      off = 0;

      /* write green palette table */
      for(i=0; i<size; i++)
	{
	  buf[off] = g[i];
	  off++;
	  buf[off] = g[i];
	  off++;
	}
      for(; i<256; i++)
	{
	  buf[off] = 0;
	  off++;
	  buf[off] = 0;
	  off++;
	}
      bos.write(buf, 0, off);
      off = 0;

      /* write blue palette table */
      for(i=0; i<size; i++)
	{
	  buf[off] = b[i];
	  off++;
	  buf[off] = b[i];
	  off++;
	}
      for(; i<256; i++)
	{
	  buf[off] = 0;
	  off++;
	  buf[off] = 0;
	  off++;
	}
      bos.write(buf, 0, off);
      off = 0;
    }
    catch(IOException e) {
      System.err.println(MessageFormat.format(WriteError, objs));
    }

  if(bos != null)
    {
      try {
	bos.flush();
      }
      catch(IOException e) {
	System.err.println(MessageFormat.format(FlushError, objs));
      }
    }

  if(fos != null)
    {
      try {
	fos.close();
      }
      catch(IOException e) {
	System.err.println(MessageFormat.format(CloseError, objs));
      }
    }

  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : writeShort
 * DESCRIPTION	  :
 *****************************************************************************/
  static void writeShort(byte buf[], int offset, short data)
  {

    buf[offset] = (byte) (data >>> 8);
    buf[offset+1] = (byte) data;

  }  /* END OF MODULE : writeShort */

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : writeInt
 * DESCRIPTION	  :
 *****************************************************************************/
  static void writeInt(byte buf[], int offset, int data)
  {

    buf[offset] = (byte) (data >>> 24);
    buf[offset+1] = (byte) (data >>> 16);
    buf[offset+2] = (byte) (data >>> 8);
    buf[offset+3] = (byte) data;

  }  /* END OF MODULE : writeInt */

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : static
 * DESCRIPTION	  :
 *****************************************************************************/
  static
  {
    ResourceBundle bundle;


    bundle = JKobo.Context.getResourceBundle();

    try {
      FindError = bundle.getString("FindFile.message");
    }
    catch(MissingResourceException e) {
      FindError = "Unable to find {0} file \"{1}\"";
    }
    try {
      ReadError = bundle.getString("ReadFile.message");
    }
    catch(MissingResourceException e) {
      ReadError = "Error reading {0} file \"{1}\"";
    }
    try {
      CloseError = bundle.getString("CloseFile.message");
    }
    catch(MissingResourceException e) {
      CloseError = "Unable to close {0} file \"{1}\"";
    }
    try {
      FormatError = bundle.getString("TiffFormatFile.message");
    }
    catch(MissingResourceException e) {
      FormatError = "Fatal error (Illegal format)";
    }
    try {
      WriteError = bundle.getString("WriteFile.message");
    }
    catch(MissingResourceException e) {
      WriteError = "Error writing {0} file \"{1}\"";
    }
    try {
      FlushError = bundle.getString("FlushFile.message");
    }
    catch(MissingResourceException e) {
      FlushError = "Error flushing {0} file \"{1}\"";
    }
    try {
      SpriteName = bundle.getString("SpriteFileName.message");
    }
    catch(MissingResourceException e) {
      SpriteName = "sprite";
    }
    try {
      TiffName = bundle.getString("TiffFileName.message");
    }
    catch(MissingResourceException e) {
      TiffName = "tiff";
    }

  }

}  /* END OF CLASS : Tiff */


/* END OF FILE : Tiff.java */

