/**
MultiLineLabel custom component
Original version is an extract from the book Java in a Nutshell
Contents have been modifyed so that the line breaks are calculated
automatically rather than from the newline chars in the text.
Copyright (c) 1996 IBM Corp.
@version 1.0
@author Philip Taunton
*/

import java.awt.*;
import java.util.*;

/**
A custom component to replace the Label component which arranges the text
on multiple lines where necessary.
*/
public class MultiLineLabel extends Canvas
{
  public static final int LEFT=0;         // alignment constants
  public static final int CENTER=1;
  public static final int RIGHT=2;
  protected String[] lines;               // the lines of text to display
  protected String[] words;               // the words of text to display
  protected int num_lines;                // the number of lines
  protected int num_words;                // the number of words
  protected int margin_width;             // left and right margins
  protected int margin_height;            // top and bottom margins
  protected int line_height;              // total height of the font
  protected int line_ascent;              // font height above baseline
  protected int[] line_widths;            // how wide each line is
  protected int max_width;                // the width of the widest line
  protected int alignment=LEFT;           // the alignment of the text

/**
Breaks a specified label up into an array of words. It uses the
StringTokenizer utility class.
@param label The text to break up into a string array.
*/
  protected void newLabel(String label)
  {
    StringTokenizer t=new StringTokenizer(label," ");
    num_words=t.countTokens();
    words=new String[num_words];
    for (int i=0;i<num_words;i++)
    {
      words[i]=t.nextToken();
    } // end for
  }

/**
Works out how large the font is, and how wide each line of the label is,
and how wide the widest line is.
*/
  protected void measure()
  {
    int actual_lines, sizeof_space, line_width, cur_width;
    String local=new String("");
    FontMetrics fm=this.getFontMetrics(this.getFont());
    if (fm==null)
      return;                             // bail out if we don't yet have any metrics
    Dimension d=this.size();
    if ((d.height==0)||(d.width==0)) return;
    actual_lines=0;
    line_height=fm.getHeight();
    line_ascent=fm.getAscent();
    num_lines=d.height/line_height+1;     // maximum possible number of lines
    lines=new String[num_lines];          // get some storage for the lines
    line_widths=new int[num_lines];
    max_width=d.width-2*margin_width;
    line_width=0;
    sizeof_space=fm.stringWidth(" ");     // get the size of a space
    for (int i=0;i<num_words && actual_lines<num_lines;i++)
    {
      cur_width=fm.stringWidth(words[i]);
      line_width+=cur_width+sizeof_space;
      if ((line_width-sizeof_space)>max_width)
      {                                   // need a new line
        line_widths[actual_lines]=line_width-cur_width-sizeof_space;
        lines[actual_lines]=local;
        actual_lines+=1;
        local=new String(words[i]);             // start a new line
        line_width=cur_width+sizeof_space;
      }
      else
      {
        if (i>0)                          // not for the first word
          local=local.concat(" ");
        local=local.concat(words[i]);
      }
    } // end for
    if (actual_lines<num_lines)
    {                                     // only if there is still space
      line_widths[actual_lines]=line_width-sizeof_space;  // catch last line
      lines[actual_lines]=local;
      actual_lines+=1;
      num_lines=actual_lines;
    }
  }

/**
Constructs a MultiLineLabel from the text, margin width, margin height
and the text alignment.
@param label The text to display
@param margin_width The margin width
@param margin_height The margin height
@param alignment The alignment of the text within the MultiLineLabel
*/
  public MultiLineLabel(String label, int margin_width, int margin_height,
      int alignment)
  {
    newLabel(label);
    this.margin_width=margin_width;
    this.margin_height=margin_height;
    this.alignment=alignment;
  }

/**
Constructs a MultiLineLabel from the text, margin width and margin height.
Defaults to LEFT alignment.
@param label The text to display
@param margin_width The margin width
@param margin_height The margin height
*/
  public MultiLineLabel(String label, int margin_width, int margin_height)
  {
    this(label,margin_width,margin_height,LEFT);
  }

/**
Constructs a MultiLineLabel from the text and text alignment.
Defaults to a margin width and margin height of 10.
@param label The text to display
@param alignment The alignment of the text within the MultiLineLabel
*/
  public MultiLineLabel(String label, int alignment)
  {
    this(label,10,10,alignment);
  }

/**
Constructs a MultiLineLabel from the text only.
Defaults to a margin width and margin height of 10 and LEFT alignment.
@param label The text to display
*/
  public MultiLineLabel(String label)
  {
    this(label,10,10,LEFT);
  }

/**
Set the text value of the MultiLineLabel.
@param label The text to set for the label.
*/
  public void setLabel(String label)
  {
    newLabel(label);
    measure();
    repaint();
  }

/**
Set the current font for the text of the label.
@param f The new font to set.
*/
  public void setFont(Font f)
  {
    super.setFont(f);
    measure();
    repaint();
  }

/**
Set the curent foreground colour.
@param c The new colour to set.
*/
  public void setForeground(Color c)
  {
    super.setForeground(c);
    repaint();
  }

/**
Set the current text alignment.
@param a The new text alignment.
*/
  public void setAlignment(int a)
  {
    alignment=a;
    repaint();
  }

/**
Set the current margin width.
@param mw The new margin width.
*/
  public void setMarginWidth(int mw)
  {
    margin_width=mw;
    repaint();
  }

/**
Set the current margin height
@param mh The new margin height.
*/
  public void setMarginHeight(int mh)
  {
    margin_height=mh;
    repaint();
  }

/**
Get the current text alignment.
@return the current text alignment.
*/
  public int getAlignment()
  {
    return(alignment);
  }

/**
Get the current margin width.
@return the current margin width.
*/
  public int getMarginWidth()
  {
    return(margin_width);
  }

/**
Get the current margin height.
@return the current margin height.
*/
  public int getMarginHeight()
  {
    return(margin_height);
  }

/**
Calls the line measurement function after font metrics are available
*/
// This method is involked after our canvas is first created but before
// it can actually be displayed. After we've involker our superclass's
// addNotify method, we have font metric and can successfully call
// measure() to figure out how big the label is.
  public void addNotify()
  {
    super.addNotify();
    measure();
  }

/**
Get the preferred size for this component.
@return the preferred Dimension of the component.
*/
  public Dimension perferredSize()
  {
    return new Dimension(max_width+2*margin_width,
        num_lines*line_height+2*margin_height);
  }

/**
Get the minimum size for this component.
@return the minimum Dimension of the component.
*/
  public Dimension minimumSize()
  {
    return new Dimension(max_width,num_lines*line_height);
  }

/**
Draw the lines of text on the canvas
@param g The graphics context for this component.
*/
  public void paint(Graphics g)
  {
    int x, y;
    Dimension d=this.size();
    if ((line_ascent==0)||(line_height==0)) measure();
    y=line_ascent+(d.height-num_lines*line_height)/2;
    for (int i=0;i<num_lines;i++,y+=line_height)
    {
      switch(alignment)
      {
        case LEFT:
          x=margin_width;
          break;
        case CENTER:
        default:
          x=(d.width-line_widths[i])/2;
          break;
        case RIGHT:
          x=d.width-margin_width-line_widths[i];
          break;
      } // end switch
      g.drawString(lines[i],x,y);
    } // end for
  }
}
