MODULE ObjectHash;

IMPORT 
    Error,
    ST := AscString;

(* ============================================================ *)

TYPE
    Object* = POINTER TO EXTENSIBLE RECORD END;

    HashTable* = POINTER TO 
        RECORD
            name    : POINTER TO ARRAY OF ST.CharOpen;
            object  : POINTER TO ARRAY OF Object;
            size-   : INTEGER;
            entries-: INTEGER;
        END;

(* ============================================================ *)
PROCEDURE^ (h: HashTable) enterStr*(IN str : ARRAY OF CHAR) : INTEGER, NEW;
(* ============================================================ *)

PROCEDURE (h: HashTable) Reset, NEW;
VAR i : INTEGER;
BEGIN
    FOR i := 0 TO h.size-1 DO h.name[i] := NIL; h.object[i] := NIL END;
END Reset;

(* -------------------------------------------- *)

PROCEDURE (h: HashTable) InitObjectHash*(nElem : INTEGER), NEW;
BEGIN
    IF    nElem <=  4099 THEN nElem :=  4099;
    ELSIF nElem <=  8209 THEN nElem :=  8209;
    ELSIF nElem <= 12289 THEN nElem := 12289;
    ELSIF nElem <= 18433 THEN nElem := 18433;
    ELSIF nElem <= 32833 THEN nElem := 32833;
    ELSIF nElem <= 46691 THEN nElem := 46691;
    ELSE  nElem := 65521;
    END;
    IF (h.name # NIL) & (h.size >= nElem) THEN
        h.Reset();
    ELSE
        h.size := nElem;
        NEW(h.name, nElem); 
        NEW(h.object, nElem); 
    END; (* IF *)
    h.entries := 0;
END InitObjectHash;

(* ============================================================ *)

PROCEDURE (h: HashTable) hashStr(IN str : ARRAY OF CHAR) : INTEGER, NEW;
VAR
    tot : INTEGER;
    idx : INTEGER;
    len : INTEGER;
BEGIN [UNCHECKED_ARITHMETIC]
    (* need to turn of overflow checking *)
    len := LEN(str$);
    tot := 0;
    FOR idx := 0 TO len-1 DO
        INC(tot, tot);
        IF tot < 0 THEN INC(tot) END;
        INC(tot, ORD(str[idx]));
    END; (* FOR *)
    RETURN tot MOD h.size; 
END hashStr;

(* -------------------------------------------- *)

PROCEDURE equalStr(val : ST.CharOpen; IN str : ARRAY OF CHAR) : BOOLEAN;
VAR i : INTEGER;
BEGIN
   (*
    * LEN(val) includes the terminating nul character.
    * LEN(str$) does not include the nul character.
    *)
    IF LEN(val) # LEN(str$)+1 THEN RETURN FALSE END;
    FOR i := 0 TO LEN(val)-1 DO
        IF str[i] # val[i] THEN RETURN FALSE END;
    END; (* FOR *)
    RETURN TRUE;
END equalStr;

(* -------------------------------------------- *)

PROCEDURE (h: HashTable) IsNamedObjExist*(str : ARRAY OF CHAR; OUT key : INTEGER) : BOOLEAN, NEW;
BEGIN
    (* evaluate the key value of the input 'str',
     * the key value is dedicated to the 'str' after evaluation
     *)
    key := h.enterStr(str);
    RETURN h.object[key] # NIL;
END IsNamedObjExist;

(* -------------------------------------------- *)

PROCEDURE (h: HashTable) enterObj*(key : INTEGER; obj: Object), NEW;
BEGIN
    ASSERT(h.name[key] # NIL);
    h.object[key] := obj;
END enterObj;

(* -------------------------------------------- *)

PROCEDURE (h: HashTable) enterStr*(IN str : ARRAY OF CHAR) : INTEGER, NEW;
VAR
    step : INTEGER;
    key  : INTEGER;
    val  : ST.CharOpen;
BEGIN
    step := 1;
    key  := h.hashStr(str);
    val  := h.name[key];
    WHILE (val # NIL) & ~equalStr(val,str) DO
        INC(key, step);
        INC(step,2);
        IF (step >= h.size) THEN
	    Error.WriteString("Hash table overflow - current size = "); Error.WriteInt(h.size,1); Error.WriteLn;
	    ASSERT(FALSE);
        END; (* IF *)
        IF key >= h.size THEN DEC(key, h.size) END; (* wrap-around *)
        val := h.name[key];
    END; (* WHILE *)
    (* Loop has been exitted. But for which reason? *)
    IF val = NIL THEN
        INC(h.entries);
        h.name[key] := ST.ToChrOpen(str);
    END; (* IF *)                       (* ELSE val already in table ... *) 
    RETURN key;
END enterStr;

(* -------------------------------------------- *)

PROCEDURE (h: HashTable) charOpenOfHash*(hsh : INTEGER) : ST.CharOpen, NEW;
BEGIN
    RETURN h.name[hsh];
END charOpenOfHash;

(* -------------------------------------------- *)

PROCEDURE (h: HashTable) KeyToObject*(key: INTEGER): Object, NEW;
BEGIN
    RETURN h.object[key];
END KeyToObject;

(* ============================================================ *)
BEGIN (* ====================================================== *)
END ObjectHash.  (* =========================================== *)
(* ============================================================ *)

