/* $Id: MsqlQueryData.java,v 2.7 1999/07/22 20:57:56 borg Exp $ */
/* Copyright  1998 George Reese, All Rights Reserved */
package com.imaginary.sql.msql;
import com.imaginary.util.Encoder;
import com.imaginary.util.NoSuchEncoderException;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.sql.Blob;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;

/**
 * The MsqlQueryData class is new to mSQL-JDBC 2.0 and replaces the
 * functionality of the old MsqlResultSet class.  That is, it represents
 * the results of a SQL query sent to mSQL.
 * <BR>
 * Last modified $Date: 1999/07/22 20:57:56 $
 * @version $Revision: 2.7 $
 * @author George Reese (borg@imaginary.com)
 */
public class MsqlQueryData extends MsqlResultSet {
    // The number of columns in this result set 
    private         int               columnCount;
    // Maps column names to numbers
    private         HashMap           columnMap        = null;
    // A signal that the loading of this result set is complete
    private         boolean           complete         = false;
    // The connection
    private         MsqlConnection    connection       = null;
    // The data for the current row
    private         MsqlRow           currentRow       = null;
    // The data from the last column read 
    protected       String            lastColumn       = null;
    // The meta data associated with this object
    private         ResultSetMetaData metaData         = null;
    // This is an exception generated while reading the data
    private         SQLException      readException    = null;
    // The number of the current row
    private         int               rowNumber        = -1;
    // The rows associated with this result set
    private         ArrayList         rows             = new ArrayList();

    /**
     * Constructs a result set. The results are loaded in a separate thread.
     * @param stmt the statement owning this result set
     * @param count the number of columns in the result set
     * @param ll the logging level
     * @exception java.sql.SQLException an error occurred constructing the
     * result set
     */
    MsqlQueryData(MsqlStatement stmt, int count, int ll) throws SQLException {
        this(null, stmt, count, ll, false);
    }
    
    /**
     * Constructs a result set.
     * @param conn the connection owning this result set
     * @param count the number of columns in the result set
     * @param ll the logging level
     * @exception java.sql.SQLException an error occurred constructing the
     * result set
     */
    MsqlQueryData(MsqlConnection conn, int count, int ll) 
        throws SQLException {
        this(conn, null, count, ll, false);
    }
    
    /**
     * Constructs a result set.
     * @param conn the connection owning this result set
     * @param count the number of columns in the result set
     * @param ll the logging level
     * @param nt don't call loadResults()
     * @exception java.sql.SQLException an error occurred constructing the
     * result set
     */
    MsqlQueryData(MsqlConnection conn, int count, int ll,
                  boolean nt)
        throws SQLException {
        this(conn, null, count, ll, nt);
    }
    
    /**
     * Constructs a result set.
     * @param conn the connection owning this result set
     * @param stmt the statement owning this result set
     * @param count the number of columns in the result set
     * @param ll the logging level
     * @param nt don't call loadResults()
     * @exception java.sql.SQLException an error occurred constructing the
     * result set
     */
    MsqlQueryData(MsqlConnection conn, MsqlStatement stmt, int count, int ll,
                  boolean nt)
        throws SQLException {
        super(stmt, ll);
        connection = conn;
        columnCount = count;
        if( !nt ) {
            Thread t = new Thread() {
                public void run() {
                    try {
                        loadResults();
                    }
                    catch( SQLException e ) {
                        log.log("MsqlQueryData()", MsqlLog.ERROR,
                                "Results load failed: " + e.getMessage());
                        readException = e;
                        complete();
                    }
                }
            };

            t.setPriority(Thread.NORM_PRIORITY-1);
            t.start();
        }
    }
    
    /**
     * Positions the result set to an absolute position relative to
     * either the beginning or end of the result set.
     * @param row positive indicates the absolute row number from the
     * start, negative indicates the absolute row number from the end
     * @return true if the positioning points to a row
     * @exception java.sql.SQLException a database error occurred,
     * the result set is TYPE_FORWARD_ONLY, or row is 0
     */
    public boolean absolute(int row) throws SQLException {
        log.log("absolute()", MsqlLog.JDBC, "Absolute " + row + ".");
        if( getType() == ResultSet.TYPE_FORWARD_ONLY ) {
            log.log("absolute()", MsqlLog.ERROR,"Result set is forward only.");
            throw new MsqlException("Result set is TYPE_FORWARD_ONLY.");
        }
        if( row > 0 ) {
            rowNumber = (row-1);
        }
        else if( row < 0 ) {
            synchronized( rows ) {
                while( !complete ) {
                    try { rows.wait(1500); }
                    catch( InterruptedException e ) { }
                }
            }
            rowNumber = rows.size() + row;
        }
        else {
            log.log("absolute()", MsqlLog.ERROR, "Cannot move to 0th row.");
            throw new MsqlException("Cannot move to the 0th row.");
        }
        try {
            currentRow = getRowData(rowNumber);
        }
        catch( SQLException e ) {
            if( rowNumber < 1 ) {
                rowNumber = 0;
            }
            else {
                rowNumber = -2;
            }
            return false;
        }
        return true;
    }

    protected void addRow(MsqlRow row) {
        synchronized( rows ) {
            rows.add(row);
            rows.notifyAll();
        }
    }
    
    /**
     * Clears any changes you have made to the current row.
     * @throws java.sql.SQLException a database error occurred
     */
    public void cancelRowUpdates() throws SQLException {
        super.cancelRowUpdates();
        currentRow.refresh();
    }
    
    /**
     * Closes the result set.
     * @exception java.sql.SQLException thrown for errors on closing
     */
    public void close() throws SQLException {
        super.close();
        synchronized( rows ) {
            while( !complete ) {
                try { rows.wait(1500); }
                catch( InterruptedException e ) { }
            }
        }
    }

    /**
     * Marks the load as completed.
     */
    public void complete() {
        super.complete();
        synchronized( rows ) {
            complete = true;
            rows.notifyAll();
        }
    }
    
    /**
     * Attempts to find the column number associated with the name
     * given.  It is recommended to avoid this method (as well as
     * any method accessing a column by name) as it is very, very slow.
     * It will first search for a match in the form of table.column.
     * If you specify only the column, however, it will then look for
     * a match solely on column name.
     * @param name the name of the desired column
     * @return the column number for the specified column name
     * @exception java.sql.SQLException thrown on a read error
     */
    public int findColumn(String name) throws SQLException {
        Integer num;

        log.log("findColumn()", MsqlLog.JDBC, "Finding column " + name);
        if( columnMap == null ) {
            ResultSetMetaData meta;

            // Thanks to Joern Kellermann for this fix
            meta = getMetaData();
            columnMap = new HashMap();
            for(int i=1; i<=columnCount; i++) {
                String lb = meta.getTableName(i) + "." +meta.getColumnName(i);
		
                columnMap.put(lb, new Integer(i));
            }
        }
        if( !columnMap.containsKey(name) ) {
            Iterator names = columnMap.keySet().iterator();
            int dot;

            while( names.hasNext() ) {
                String label = (String)names.next();
                String nom;
		
                dot = label.indexOf(".");
                if( dot == -1 ) {
                    nom = label;
                }
                else if( dot >= label.length() -1 ) {
                    nom = label.substring(0, dot);
                }
                else {
                    nom = label.substring(dot+1);
                }
                if( nom.equals(name) ) {
                    return ((Integer)columnMap.get(label)).intValue();
                }
            }
            log.log("findcolumn()", MsqlLog.ERROR, "Invalid column name.");
            throw new MsqlException("Invalid column name: " + name);
        }
        else {
            return ((Integer)columnMap.get(name)).intValue();
        }
    }

    /**
     * @param column the column number for the desired column
     * @return an ASCII input stream for the desired column
     * @exception java.sql.SQLException thrown when the column cannot be read
     */
    public InputStream getAsciiStream(int column) throws SQLException {
        getColumn(column);
        try {
            return new MsqlAsciiInputStream(lastColumn);
        }
        catch( UnsupportedEncodingException e ) {
            throw new MsqlException(e);
        }
    }

    /**
     * @param column the number of the desired column
     * @return the column as an InputStream
     * @exception java.sql.SQLException thrown in the event of an error
     * reading the column
     */
    public InputStream getBinaryStream(int column) throws SQLException {
        try {
            Encoder enc = Encoder.getInstance(Encoder.BASE64);
	    
            getColumn(column);
            return new ByteArrayInputStream(enc.decode(lastColumn));
        }
        catch( NoSuchEncoderException e ) {
            throw new MsqlException(e);
        }
    }

    /**
     * @param ccolumn the column number of the desired column
     * @return the column as a JDBC 2.0 Blob
     * @throws java.sql.SQLException a database error occurred
     */
    public Blob getBlob(int column) throws SQLException {
        return new MsqlBlob(getBytes(column));
    }
    
    /**
     * @param column the number of the desired column
     * @return the column as a boolean
     * @exception java.sql.SQLException thrown in the event of an error
     * reading the column
     */
    public boolean getBoolean(int column) throws SQLException {
        getColumn(column);
        if( wasNull() ) {
            return false;
        }
        if( lastColumn.length() == 0 ) {
            return false;
        }
        else {
            char c = lastColumn.charAt(0);
	    
            if( c == '0' || c == '\0' || c == 'n' || c == 'N' ) {
                return false;
            }
            else {
                return true;
            }
        }
    }

    /**
     * @param column the number of the desired column
     * @return the column as a byte
     * @exception java.sql.SQLException thrown in the event of an error
     * reading the column
     */
    public byte getByte(int column) throws SQLException {
        getColumn(column);
        if( wasNull() || lastColumn.length() == 0 ) {
            return (byte)0;
        }
        else {
            try {
                return (lastColumn.getBytes("8859_1"))[0];
            }
            catch( UnsupportedEncodingException e ) {
                throw new MsqlException(e);
            }
        }
    }

    /**
     * @param column the number of the desired column
     * @return the column as a byte array
     * @exception java.sql.SQLException thrown in the event of an error
     * reading the column
     */
    public byte[] getBytes(int column) throws SQLException {
        String data = getString(column);
	
        try {
            Encoder enc = Encoder.getInstance(Encoder.BASE64);
	    
            return enc.decode(data);
        }
        catch( NoSuchEncoderException e ) {
            throw new MsqlException(e);
        }
    }

    /**
     * Retrieves the specified column and puts it into the lastColumn
     * attribute.
     * @param column the column being retrieved
     * @exception java.sql.SQLException the cursor is on a non-existent row
     */
    protected void getColumn(int column) throws SQLException {
        log.log("getColumn()", MsqlLog.DRIVER, "Getting column " + column);
        try {
            lastColumn = currentRow.getColumn(column-1, getType());
        }
        catch( Exception e ) {
            if( currentRow == null ) {
                throw new MsqlException("Result set cursor is " +
                                        "positioned outside of the " +
                                        "result set.");
            }
            throw new MsqlException(e);
        }
    }

    /**
     * @return the MsqlConnection that created this result set
     */
    MsqlConnection getConnection() throws SQLException {
        if( connection == null ) {
            return (MsqlConnection)getStatement().getConnection();
        }
        else {
            return connection;
        }
    }
    
    /**
     * @return the meta data associated with this result set
     * @exception java.sql.SQLException the meta data could not be loaded
     */
    public ResultSetMetaData getMetaData() throws SQLException {
        log.log("getMetaData()", MsqlLog.JDBC, "Getting meta data.");
        synchronized( rows ) {
            if( metaData != null ) {
                return metaData;
            }
            while( metaData == null ) {
                try { rows.wait(1500); }
                catch( InterruptedException e ) { }
            }
            return metaData;
        }
    }

    /**
     * @return the current row number
     * @exception java.sql.SQLException this is never thrown
     */
    public int getRow() throws SQLException {
        return (rowNumber+1);
    }
    
    /**
     * Reads the specified row number.
     * @return the data for each column in the specified row
     * @exception java.sql.SQLException an attempt was made to access
     * non-existent rows
     */
    private MsqlRow getRowData(int row) throws SQLException {
        clearWarnings();
        if( row < 0 ) {
            throw new MsqlException("Attempt to access a non-existent row.");
        }
        synchronized( rows ) {
            if( readException != null ) {
                throw readException;
            }
            while( rows.size() <= row ) {
                if( complete ) {
                    throw new MsqlException("Attempt to access a " +
                                            "non-existent row.");
                }
                else {
                    try { rows.wait(1500); }
                    catch( InterruptedException e ) { }
                }
            }
        }
        return (MsqlRow)rows.get(row);
    }

    /**
     * @param column the number of the desired column
     * @return the column as a Java String
     * @exception java.sql.SQLException thrown in the event of an error
     * reading the column
     */
    public String getString(int column) throws SQLException {
        getColumn(column);
        if( wasNull() ) {
            return null;
        }
        else {
            return lastColumn;
        }
    }

    /**
     * @param column the number of the desired column
     * @return the column as an InputStream
     * @exception java.sql.SQLException thrown in the event of an error
     * reading the column
     * @deprecated use getCharacterStream()
     */
    public InputStream getUnicodeStream(int column) throws SQLException {
        getColumn(column);
        try {
            return new MsqlUnicodeInputStream(lastColumn);
        }
        catch( UnsupportedEncodingException e ) {
            throw new MsqlException(e);
        }
    }

    /**
     * Provides the update value for the specified column.
     * @param col the column whose updated values are sought
     * @return the update value for the specified column
     * @throws java.sql.SQLException a database error occurred or an
     * incorrect encoding is specified for this result set
     */
    protected String getUpdate(int col) throws SQLException {
        String data;

        super.getUpdate(col); // does logging
        return currentRow.getColumn(col-1, ResultSet.TYPE_SCROLL_SENSITIVE);
    }

    /**
     * This is an expensive operation and should be avoided.
     * @return true if the current row is the last row
     * @exception java.sql.SQLException this is never thrown
     */
    public boolean isLast() throws SQLException {
        synchronized( rows ) {
            while( !complete ) {
                try { rows.wait(1500); }
                catch( InterruptedException e ) { }
            }
        }
        if( rowNumber == (rows.size()-1) ) {
            return true;
        }
        else {
            return false;
        }
    }

    protected ResultSetMetaData loadMetaData() throws SQLException {
        MsqlConnection conn = getConnection();
        ArrayList cols = new ArrayList();
        String cat = conn.getCatalog();

        while( true ) {
            byte[] data;
	    
            try {
                data = conn.getInputStream().read();
            }
            catch( IOException e ) {
                log.log("loadMetaData()", MsqlLog.ERROR,
                        "Error reading meta data from stream: " +
                        e.getMessage());
                throw new MsqlException(e);
            }
            if( data.length > 2 && data[0] == '-' && data[1] == '1' &&
                data[2] == ':' ) {
                String tmp;
		    
                try {
                    tmp = new String(data, getEncoding());
                }
                catch( UnsupportedEncodingException e ) {
                    log.log("loadMetaData()", MsqlLog.ERROR,
                            "Error with encoding on read: " +
                            e.getMessage());
                    throw new MsqlException(e);
                }
                log.log("loadMetaData()", MsqlLog.ERROR, tmp);
                throw new MsqlException(tmp);
            }
            else if( data.length > 4 && data[0] == '-' && data[1] == '1' &&
                     data[2] == '0' && data[3] == '0' && data[4] == ':' ) {
                return new MsqlResultSetMetaData(cat, cols);
            }
            else {
                RowTokenizer rd = new RowTokenizer(data, getEncoding(),log.getLevel());

                cols.add(rd);
            }
        }
    }
    
    /**
     * Runs the thread for loading data from the database.
     */
    protected void loadResults() throws SQLException {
        MsqlConnection conn;

        log.log("loadResults()", MsqlLog.DRIVER, "Loading results.");
        conn = getConnection();
        while( true ) {
            byte[] data;

            try {
                MsqlInputStream input = conn.getInputStream();

                data = input.read();
            }
            catch( Exception e ) {
                throw new MsqlException(e);
            }
            if( data.length > 2 && data[0] == '-' && data[1] == '1' &&
                data[2] == ':' ) {
                String tmp;
		
                try {
                    tmp = new String(data, getEncoding());
                }
                catch( UnsupportedEncodingException e ) {
                    throw new MsqlException(e);
                }
                throw new MsqlException(tmp);
            }
            else if( data.length > 4 && data[0] == '-' && data[1] == '1' &&
                     data[2] == '0' && data[3] == '0' && data[4] == ':' ) {
                break;
            }
            else {
                try {
                    MsqlRow row = readRow(data);

                    if( row != null ) {
                        addRow(row);
                    }
                }
                catch( Exception e ) {
                    throw new MsqlException(e);
                }
            }
        }
        try {
            metaData = loadMetaData();
            complete();
        }
        catch( SQLException e ) {
            metaData = null;
            throw e;
        }
    }

    /**
     * Moves to the next row of data for processing.  If there are no
     * more rows to be processed, then it will return false.
     * @return true if there are results to be processed, false otherwise
     * @exception java.sql.SQLException thrown if a read error occurs
     */
    public boolean next() throws SQLException {
        log.log("next()", MsqlLog.JDBC,
                "Moving from row " + rowNumber + " to row " + (rowNumber+1) +
                ".");
        rowNumber++;
        try {
            currentRow = getRowData(rowNumber);
        }
        catch( SQLException e ) {
            rowNumber = -2;
            return false;
        }
        return true;
    }

    /**
     * Moves to the previous row of data for processing.  If there is no
     * previous row to be processed, then it will return false.  An exception
     * will be thrown if the result set type equals TYPE_FORWARD_ONLY which
     * is the default.
     * @return true if there are results to be processed, false otherwise
     * @exception java.sql.SQLException thrown if a read error occurs
     */
    public boolean previous() throws SQLException {
        log.log("previous()", MsqlLog.JDBC, "Moving to previous row.");
        if( getType() == ResultSet.TYPE_FORWARD_ONLY ) {
            log.log("previous()", MsqlLog.ERROR, "Result set forward-only.");
            throw new MsqlException("ResultSet is forward-only.");
        }
        if( isAfterLast() ) {
            return last();
        }
        else if( rowNumber > 0 ) {
            rowNumber--;
            try {
                currentRow = getRowData(rowNumber);
            }
            catch( SQLException e ) {
                rowNumber = 0;
                return false;
            }
            return true;
        }
        else {
            return false;
        }
    }
    
    /*
     * Reads a single row of data.  A row comes in the format:
     * <PLAINTEXT>
     * COLUMN_LENGTH:DATACOLUMN_LENGTH:DATA...COLUMN_LENGTH:DATA
     * </PLAINTEXT>
     * @param data the data from the database to be formed into a row
     * @exception java.sql.SQLException the row could not be read
     */
    protected MsqlRow readRow(byte[] data) throws SQLException {
        RowTokenizer parser = new RowTokenizer(data, getEncoding(),
                                               log.getLevel());

        return new MsqlRow(parser, getEncoding());    
    }

    /**
     * Restores the current row to its original values.  This does not
     * go back to the database, so any modifications to the row by
     * other transactions will not be noted.
     * @throws java.sql.SQLException could not refresh row
     */
    public void refreshRow() throws SQLException {
        super.refreshRow();
        currentRow.refresh();
    }
    
    /**
     * Moves the result set forward the specified number of rows relative
     * to the current row.
     * @param count the number of rows to move forward
     * @return true if the result set remains on a row, false otherwise
     * @exception java.sql.SQLException a database error occurred, the
     * result set is TYPE_FORWARD_ONLY, or there is no current row
     */
    public boolean relative(int count) throws SQLException {
        log.log("relative()", MsqlLog.JDBC,
                "Moving relative " + count + " rows.");
        if( getType() == ResultSet.TYPE_FORWARD_ONLY ) {
            log.log("relative()", MsqlLog.ERROR, "Row set is forward-only.");
            throw new MsqlException("Row set is TYPE_FORWARD_ONLY.");
        }
        if( rowNumber < 1 ) {
            log.log("relative()", MsqlLog.ERROR, "No current row set.");
            throw new MsqlException("No current row is set.");
        }
        rowNumber += count;
        try {
            currentRow = getRowData(rowNumber);
        }
        catch( SQLException e ) {
            rowNumber = -1;
            return false;
        }
        return true;
    }

    /**
     * @return true if the current row is marked for deletion
     * @throws java.sql.SQLException a database error occurred
     */
    public boolean rowDeleted() throws SQLException {
        return currentRow.isDeleted();
    }
    
    /**
     * @return true if the current row is an insert
     * @throws java.sql.SQLException a database error occurred
     */
    public boolean rowInserted() throws SQLException {
        return currentRow.isInserted();
    }
    
    /**
     * @return true if the current row has been updated
     * @throws java.sql.SQLException a database error occurred
     */
    public boolean rowUpdated() throws SQLException {
        return currentRow.isUpdated();
    }
    
    /**
     * Changes the specified column.
     * @param column the column being retrieved
     * @throws java.sql.SQLException the cursor is on a non-existent row or
     * the result set is not updatable
     */
    protected void setColumn(int column, String value) throws SQLException {
        if( getConcurrency() != ResultSet.CONCUR_UPDATABLE ) {
            log.log("setColumn()", MsqlLog.ERROR, "Result set not updatable.");
            throw new MsqlException("Result set is not updatable.");
        }
        currentRow.setColumn(column-1, value);
    }

    /**
     * Adds to the functionality in <CODE>MsqlResultSet</CODE> by
     * resetting the row state.
     * @throws java.sql.SQLException result set is read only
     */
    public void updateRow() throws SQLException {
        log.log("updateRow()", MsqlLog.JDBC, "Updating row.");
        if( !currentRow.isUpdated() ) {
            log.log("updateRow()", MsqlLog.ERROR,
                    "Attempt to update an unmodified row.");
            throw new MsqlException("Attempt to update an unmodified row.");
        }
        super.updateRow();
        currentRow.update();
    }
    
    /**
     * @return true if the last value read was null
     * @exception java.sql.SQLException this is never thrown
     */
    public boolean wasNull() throws SQLException {
        if( lastColumn == null ) {
            return true;
        }
        else {
            return false;
        }
    }
}
