/* $Id: MsqlPreparedStatement.java,v 2.7 1999/08/03 01:26:02 borg Exp $ */
/* Copyright  1998-1999 George Reese, All Rights Reserved */
package com.imaginary.sql.msql;

import com.imaginary.util.Encoder;
import com.imaginary.util.NoSuchEncoderException;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectOutputStream;
import java.io.Reader;
import java.io.Serializable;
import java.io.UnsupportedEncodingException;
import java.math.BigDecimal;
import java.sql.Array;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.Date;
import java.sql.PreparedStatement;
import java.sql.Ref;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Time;
import java.sql.Timestamp;
import java.sql.Types;
import java.text.FieldPosition;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Locale;

/**
 * The MsqlPreparedStatement is an mSQL implementation of the 
 * JDBC PreparedStatement interface.  Specifically, it enables an
 * application to execute the same SQL over and over again without
 * repeatedly writing logic to build the SQL statements.  Instead,
 * the application just passes new inputs.  Because mSQL is completely
 * unaware of the concept of a PreparedStatement, the mSQL driver
 * basically hacks it by doing its own parsing and management.
 * There is still a huge advantage to using the PreparedStatement over
 * a regular statement in that you never have to worry about date or
 * String formatting.
 * <BR>
 * Last modified $Date: 1999/08/03 01:26:02 $
 * @version $Revision: 2.7 $
 * @author George Reese (borg@imaginary.com)
 */
public class MsqlPreparedStatement extends MsqlStatement
    implements PreparedStatement {
    /**
     * Runs through the string and escapes all significant characters like
     * "'".
     * @param str the string to be fixed
     * @return the fixed string
     */
    static String fixString(String str) {
        if( str.indexOf("'") != -1 ) {
            StringBuffer buff = new StringBuffer();
            
            for(int i=0; i<str.length(); i++) {
                char c = str.charAt(i);

                switch( c ) {
                case '\'': case '%': case '_':
                    buff.append('\\');
                    break;
                default:
                    break;
                }
                buff.append(c);
            }
            str = buff.toString();
        }
        return str;
    }

    // The constant part of the SQL statement
    private String[]  constants = null;
    // A flag indicating that the prepared statement is parsing the SQL still
    // 0 = unparsed, 1 = parsing, 2 = parsed
    private int       parsing   = 0;
    // The unparsed prepared statement
    private String    statement = null;
    // The values to assign to the prepared statement
    private String[]  values    = null;

    /**
     * Constructs a new prepared statement owned by the specified connection
     * and supporting the specified SQL.  The constructed prepared statement
     * is TYPE_FORWARD_ONLY and CONCUR_READ_ONLY.
     * @param c the owning connection
     * @param sql the prepared statement
     */
    MsqlPreparedStatement(MsqlConnection c, String sql, int ll) {
        this(c, sql, ResultSet.TYPE_FORWARD_ONLY,
             ResultSet.CONCUR_READ_ONLY, ll);
    }

    /**
     * Constructs a new prepared statement owned by the specified connection
     * and supporting the specified SQL.
     * @param c the owning connection
     * @param sql the prepared statement
     * @param t the type for the result sets generated by this statement
     * @param concur the result set concurrency
     */
    MsqlPreparedStatement(MsqlConnection c, String sql, int t, int concur,
                          int ll) {
        super(c, t, concur, ll);
        statement = sql;
        Thread th = new Thread() {
            public void run() {
                parseSQL();
            }
        };

        th.setPriority(Thread.NORM_PRIORITY-1);
        th.start();
    }	

    /**
     * Adds the current parameter values to the batch list and then clears
     * the parameters for the next set of parameters.
     * @exception java.sql.SQLException could not add the current parameters
     * to the batch list
     */
    public void addBatch() throws SQLException {
        addBatch(getSQL());
        clearParameters();
    }

    /**
     * Clears the current parameters.
     * @exception java.sql.SQLException this is never thrown
     */
    public void clearParameters() throws SQLException {
        synchronized( this ) {
            while( parsing != 2 ) {
                try { wait(1500); }
                catch( InterruptedException e ) { }
            }
        }
        for(int i=0; i<values.length; i++) {
            values[i] = null;
        }
    }

    /**
     * Executes the stored procedure with its current values.
     * @return true if the stored procedure generated a result set
     * @exception java.sql.SQLException an error occurred executing the SQL
     */
    public boolean execute() throws SQLException {
        return execute(getSQL());
    }

    /**
     * Executes the stored query with its current values.
     * @return the results of the stored procedure
     * @exception java.sql.SQLException an error occurred executing the SQL
     */
    public ResultSet executeQuery() throws SQLException {
        return executeQuery(getSQL());
    }

    /**
     * Executes the stored update with its current values.
     * @return the number of rows affected by the update
     * @exception java.sql.SQLException an error occurred executing the SQL
     */
    public int executeUpdate() throws SQLException {
        return executeUpdate(getSQL());
    }

    /**
     * This errors out because I have not yet figured out a good way to
     * implement this for mSQL.
     * @return ResultSetMetaData representing result sets for this stored
     * procedure
     * @exception java.sql.SQLException this is always thrown
     */
    public ResultSetMetaData getMetaData() throws SQLException {
        throw new MsqlException("This operation is not yet supported.");
    }
    
    /**
     * Translates the stored procedure into valid SQL using the current
     * parameters.
     * @return valid mSQL SQL
     * @exception java.sql.SQLException one or more parameters were not set
     */
    private String getSQL() throws SQLException {
        String sql;
        int i;

        synchronized( this ) {
            while( parsing != 2 ) {
                try { wait(1500); }
                catch( InterruptedException e ) { }
            }
        }
        sql = constants[0];
        for(i=0; i<values.length; i++) {
            if( values[i] == null ) {
                throw new MsqlException("No value set for parameter " + (i+1) +
                                        ".");
            }
            sql += values[i];
            if( (i+1) < constants.length ) {
                sql += constants[i+1];
            }
        }
        return sql;
    }

    /**
     * Parses the prepared statement so SQL can be easily generated.
     */
    private void parseSQL() {
        ArrayList consts = new ArrayList();
        String sql = statement;
        int ind, count = 0;

        synchronized( this ) {
            if( parsing != 0 ) {
                return;
            }
            parsing = 1;
        }
        while( true ) {
            ind = sql.indexOf("?");
            if( ind < 0 ) {
                consts.add(sql);
                break;
            }
            count++;
            if( ind == 0 ) {
                consts.add("");
            }
            else {
                consts.add(sql.substring(0, ind));
            }
            if( ind < (sql.length() -1) ) {
                sql = sql.substring(ind+1);
            }
            else {
                sql = "";
            }
        }
        values = new String[count];
        constants = new String[consts.size()];
        consts.toArray(constants);
        synchronized( this ) {
            parsing = 2;
            notifyAll();
        }
    }

    /**
     * This is not really supported, but it tries.
     * @param ind the parameter to be set
     * @param arr the array value
     * @exception java.sql.SQLException an error occurred setting the parameter
     */
    public void setArray(int ind, Array arr) throws SQLException {
        validateIndex(ind);
        values[ind-1] = arr.toString();
    }

    /**
     * Sets the parameter to the data in the specified stream.
     * @param ind the parameter to set
     * @param in the stream containing the data
     * @param len the number of bytes in the stream
     * @exception java.sql.SQLException an error occurred setting the parameter
     */
    public void setAsciiStream(int ind, InputStream in, int len)
        throws SQLException {
        byte[] str;

        validateIndex(ind);
        str = new byte[len];
        try {
            in.read(str, 0, len);
            values[ind-1] = new String(str, "8859_1");
        }
        catch( IOException e ) {
            throw new MsqlException(e);
        }
    }

    /**
     * Sets the specified parameter to a BigDecimal value.
     * @param ind the parameter to be set
     * @param db the BigDecimal value
     * @exception java.sql.SQLException an error occurred setting the parameter
     */
    public void setBigDecimal(int ind, BigDecimal bd) throws SQLException {
        validateIndex(ind);
        values[ind-1] = bd.toString();
    }
    
    /**
     * Sets the parameter to the data in the specified stream.
     * @param ind the parameter to set
     * @param in the stream containing the data
     * @param len the number of bytes in the stream
     * @exception java.sql.SQLException this is always thrown as mSQL has
     * no binary object support
     */
    public void setBinaryStream(int ind, InputStream in, int len)
        throws SQLException {
        byte[] data = new byte[len];
	
        try {
            in.read(data, 0, len);
        }
        catch( IOException e ) {
            throw new MsqlException(e);
        }
        setBytes(ind, data);
    }

    /**
     * Sets the specified parameter to a Blob value.
     * @param ind the parameter to be set
     * @param db the Blob value
     * @exception java.sql.SQLException an error occurred setting the parameter
     */
    public void setBlob(int ind, Blob b) throws SQLException {
        long len = b.length();

        if( len > Integer.MAX_VALUE ) {
            throw new MsqlException("Binary length too long for mSQL.");
        }
        setBinaryStream(ind, b.getBinaryStream(), (int)len);
    }
    
    /**
     * Sets the specified parameter to a boolean value.  Specifically, it
     * will set the column to the int 1 foor true or 0 for false.
     * @param ind the parameter to set
     * @param b the value to set
     * @exception java.sql.SQLException an error occurred setting the parameter
     */
    public void setBoolean(int ind, boolean b) throws SQLException {
        validateIndex(ind);
        if( b ) {
            values[ind-1] = "1";
        }
        else {
            values[ind-1] = "0";
        }
    }

    /**
     * Sets the specified parameter to the specified byte value.
     * @param ind the paramter to be set
     * @param b the byte value
     * @exception java.sql.SQLException an error occurred setting the paramter
     */
    public void setByte(int ind, byte b) throws SQLException {
        validateIndex(ind);
        values[ind-1] = ("" + b);
    }

    /**
     * Sets the specified parameter to the specified byte value.
     * @param ind the paramter to be set
     * @param data the byte array value
     * @exception java.sql.SQLException an error occurred setting the paramter
     */
    public void setBytes(int ind, byte[] data) throws SQLException {
        String value;
	
        validateIndex(ind);
        try {
            Encoder enc = Encoder.getInstance(Encoder.BASE64);
	    
            // encode the binary data using base 64
            data = enc.encode(data);
            // now, convert the encoded array to a String
            // base 64 is an ASCII encoding
            setString(ind, new String(data, "8859_1"));
        }
        catch( UnsupportedEncodingException e ) {
            throw new MsqlException(e);
        }
        catch( NoSuchEncoderException e ) {
            throw new MsqlException(e);
        }
    }

    /**
     * Sets the parameter to the data in the specified stream.
     * @param ind the parameter to set
     * @param in the stream containing the data
     * @param len the number of bytes in the stream
     * @exception java.sql.SQLException an error occurred setting the parameter
     */
    public void setCharacterStream(int ind, Reader in, int len)
        throws SQLException {
        char[] str = new char[len];
	
        validateIndex(ind);
        try {
            in.read(str, 0, len);
        }
        catch( IOException e ) {
            throw new MsqlException(e);
        }
        values[ind-1] = MsqlPreparedStatement.fixString(new String(str));
    }

    /**
     * Sets the specified parameter to a Clob value.
     * @param ind the parameter to be set
     * @param c the Clob value
     * @exception java.sql.SQLException an error occurred setting the parameter
     */
    public void setClob(int ind, Clob c) throws SQLException {
        throw new MsqlException("CLOBs are not supported in mSQL.");
    }
    
    /**
     * Sets the specified parameter to a date value stored in the mSQL
     * database as a string in the form dd-MMM-yyyy.
     * @param ind the parameter to be set
     * @param d the Date value to set
     * @exception java.sql.SQLException an error occurred setting the parameter
     */
    public void setDate(int ind, Date d) throws SQLException {
        SimpleDateFormat fmt;
	
        validateIndex(ind);
        fmt = new SimpleDateFormat("dd-MMM-yyyy", Locale.US);
        values[ind-1] = "'" + fmt.format(d) + "'";
    }

    /**
     * Sets the specified parameter to a date value stored in the mSQL
     * database as a string in the form dd-MMM-yyyy.
     * @param ind the parameter to be set
     * @param d the Date value to set
     * @param cal the Calendar to use
     * @exception java.sql.SQLException an error occurred setting the parameter
     */
    public void setDate(int ind, Date d, Calendar cal) throws SQLException {
        SimpleDateFormat fmt;
	
        validateIndex(ind);
        fmt = new SimpleDateFormat("dd-MMM-yyyy", Locale.US);
        values[ind-1] = "'" + fmt.format(d) + "'";
    }

    /**
     * Sets the specified parameter to a double value.
     * @param ind the parameter to be set
     * @param d the value to set
     * @exception java.sql.SQLException an error occurred setting the parameter
     */
    public void setDouble(int ind, double d) throws SQLException {
        validateIndex(ind);
        values[ind-1] = ("" + d);
    }
    
    /**
     * Sets the specified parameter to a float value.
     * @param ind the parameter to be set
     * @param f the value to set
     * @exception java.sql.SQLException an error occurred setting the parameter
     */
    public void setFloat(int ind, float f) throws SQLException {
        validateIndex(ind);
        values[ind-1] = ("" + f);
    }
    
    /**
     * Sets the specified parameter to an int value.
     * @param ind the parameter to be set
     * @param x the value to set
     * @exception java.sql.SQLException an error occurred setting the parameter
     */
    public void setInt(int ind, int x) throws SQLException {
        validateIndex(ind);
        values[ind-1] = ("" + x);
    }

    /**
     * Sets the specified parameter to a long value.
     * @param ind the parameter to be set
     * @param l the value to set
     * @exception java.sql.SQLException an error occurred setting the parameter
     */
    public void setLong(int ind, long l) throws SQLException {
        validateIndex(ind);
        values[ind-1] = ("" + l);
    }
    
    /**
     * Sets the specified parameter to a null value.
     * @param ind the parameter to be set
     * @param type the SQL type of the value to be set
     * @exception java.sql.SQLException an error occurred setting the parameter
     */
    public void setNull(int ind, int type) throws SQLException {
        validateIndex(ind);
        values[ind-1] = "NULL";
    }

    /**
     * Sets the specified parameter to a null value.
     * @param ind the parameter to be set
     * @param type the SQL type of the value to be set
     * @param tname the name of the UDT
     * @exception java.sql.SQLException an error occurred setting the parameter
     */
    public void setNull(int ind, int type, String tname) throws SQLException {
        validateIndex(ind);
        values[ind-1] = "NULL";
    }

    /**
     * Sets the specified parameter to a Java object value by calling
     * ob.toString(). 
     * @param ind the parameter to be set
     * @param ob the object to be set
     * @exception java.sql.SQLException an error occurred setting the parameter
     */
    public void setObject(int ind, Object ob) throws SQLException {
        if( ob instanceof Date ) {
            setDate(ind, (Date)ob);
        }
        else if( ob instanceof String ) {
            setString(ind, ob.toString());
        }
        else if( ob instanceof StringBuffer ) {
            setString(ind, ob.toString());
        }
        else {
            setObject(ind, ob, Types.BINARY);
        }
    }
    
    /**
     * Sets the specified parameter to a Java object value according to the
     * SQL type specified.
     * @param ind the parameter to be set
     * @param ob the object to be set
     * @param type the target SQL type
     * @exception java.sql.SQLException an error occurred setting the parameter
     */
    public void setObject(int ind, Object ob, int type) throws SQLException {
        String value;
	
        switch(type) {
        case Types.BIT:
            {
                value = ob.toString();
                if( value.length() < 1 ) {
                    setByte(ind, (byte)0);
                }
                else {
                    byte b = (byte)value.charAt(0);
		    
                    setByte(ind, b);
                }
                return;
            }
	    
        case Types.TINYINT: case Types.SMALLINT:
            {
                setShort(ind, Short.parseShort(ob.toString()));
                return;
            }

        case Types.INTEGER:
            {
                setInt(ind, Integer.parseInt(ob.toString()));
                return;
            }
	    
        case Types.BIGINT: 
            {
                setLong(ind, Long.parseLong(ob.toString()));
                return;
            }
	    
        case Types.FLOAT: case Types.REAL:
            {
                float f = Float.valueOf(ob.toString()).floatValue();
		
                setFloat(ind, f);
                return;
            }
	    
        case Types.DOUBLE:
            {
                double d = Double.valueOf(ob.toString()).doubleValue();
		
                setDouble(ind, d);
                return;
            }
	    
        case Types.NUMERIC: case Types.DECIMAL:
            {
                setBigDecimal(ind, new BigDecimal(ob.toString()));
                return;
            }

        case Types.CHAR: case Types.VARCHAR: case Types.LONGVARCHAR:
            {
                setString(ind, ob.toString());
                return;
            }
	    
        case Types.DATE: case Types.TIME: 
            {
                String tmp = ob.toString();
                String fmtstr;

                if( type == Types.DATE ) {
                    fmtstr = "dd-MMM-yyyy";
                }
                else {
                    fmtstr = "HH:mm:ss";
                }
                try {
                    SimpleDateFormat fmt = new SimpleDateFormat(fmtstr,
                                                                Locale.US);

                    if( type == Types.DATE ) {
                        setDate(ind, (java.sql.Date)fmt.parse(tmp));
                    }
                    else {
                        setTime(ind, (java.sql.Time)fmt.parse(tmp));
                    }
                }
                catch( ParseException e ) {
                    throw new MsqlException(e);
                }
                return;
            }
	    
        case Types.TIMESTAMP:
            {
                Timestamp ts = new Timestamp(Long.parseLong(ob.toString()));

                setTimestamp(ind, ts);
                return;
            }

        case Types.BINARY: case Types.VARBINARY: case Types.LONGVARBINARY:
        case Types.BLOB:
            {
                if( ob instanceof Blob ) {
                    setBlob(ind, (Blob)ob);
                }
                else if( ob instanceof byte[] ) {
                    setBytes(ind, (byte[])ob);
                }
                else if( ob instanceof Serializable ) {
                    try {
                        ByteArrayOutputStream baos;
                        ObjectOutputStream oos;
			
                        baos = new ByteArrayOutputStream();
                        oos = new ObjectOutputStream(baos);
                        oos.writeObject(ob);
                        oos.flush();
                        setBytes(ind, baos.toByteArray());
                    }
                    catch( IOException e ) {
                        throw new MsqlException(e);
                    }
                }
                else {
                    throw new MsqlException("Invalid binary object type.");
                }
                return;
            }

        case Types.OTHER: case Types.JAVA_OBJECT: case Types.DISTINCT:
        case Types.STRUCT: case Types.ARRAY: 
        case Types.CLOB: case Types.REF:
            {
                throw new MsqlException("UDTs are not supported.");
            }
        }
    }

    /**
     * Sets the specified parameter to a Java object value according to the
     * SQL type specified.
     * @param ind the parameter to be set
     * @param ob the object to be set
     * @param type the target SQL type
     * @param scale this is ignored
     * @exception java.sql.SQLException an error occurred setting the parameter
     */
    public void setObject(int ind, Object ob, int type, int scale)
        throws SQLException {
        setObject(ind, ob, type);
    }

    /**
     * Sets the specified parameter to a Ref value.
     * @param ind the parameter to be set
     * @param r the Ref value
     * @exception java.sql.SQLException an error occurred setting the parameter
     */
    public void setRef(int ind, Ref r) throws SQLException {
        throw new MsqlException("Refs are not supported in mSQL.");
    }
    
    /**
     * Sets the specified parameter to a short value.
     * @param ind the parameter to be set
     * @param s the value to set
     * @exception java.sql.SQLException an error occurred setting the parameter
     */
    public void setShort(int ind, short s) throws SQLException {
        validateIndex(ind);
        values[ind-1] = ("" + s);
    }

    /**
     * Sets the specified parameter to a String value.
     * @param ind the parameter to be set
     * @param str the value to set
     * @exception java.sql.SQLException an error occurred setting the parameter
     */
    public void setString(int ind, String str) throws SQLException {
        validateIndex(ind);
        values[ind-1] = "'" + MsqlPreparedStatement.fixString(str) + "'";
    }

    /**
     * Sets the specified parameter to a Time value.
     * @param ind the parameter to be set
     * @param t the value to set
     * @exception java.sql.SQLException an error occurred setting the parameter
     */
    public void setTime(int ind, Time t) throws SQLException {
        SimpleDateFormat fmt;

        validateIndex(ind);
        fmt = new SimpleDateFormat("HH:mm:ss", Locale.US);	
        values[ind-1] = "'" + fmt.format(t) + "'";
    }

    /**
     * Sets the specified parameter to a Time value.
     * @param ind the parameter to be set
     * @param t the value to set
     * @param cal the Calendar to use
     * @exception java.sql.SQLException an error occurred setting the parameter
     */
    public void setTime(int ind, Time t, Calendar cal) throws SQLException {
        SimpleDateFormat fmt;

        validateIndex(ind);
        fmt = new SimpleDateFormat("HH:mm:ss", Locale.US);	
        values[ind-1] = "'" + fmt.format(t) + "'";
    }

    /**
     * Sets the specified parameter to a Timestamp value.
     * @param ind the parameter to be set
     * @param t the value to set
     * @exception java.sql.SQLException an error occurred setting the parameter
     */
    public void setTimestamp(int ind, Timestamp t) throws SQLException {
        setLong(ind, t.getTime());
    }

    /**
     * Sets the specified parameter to a Timestamp value.
     * @param ind the parameter to be set
     * @param t the value to set
     * @param cal the Calendar to use
     * @exception java.sql.SQLException an error occurred setting the parameter
     */
    public void setTimestamp(int ind, Timestamp t, Calendar cal)
        throws SQLException {
        setLong(ind, t.getTime());
    }

    /**
     * Sets the specified parameter to the data contained in the specified
     * unicode stream.
     * @param ind the parameter being set
     * @param in the stream with the data
     * @param len the number of bytes to be read
     * @exception java.sql.SQLException an error occurred setting the
     * parameter
     * @deprecated use setCharacterStream()
     */
    public void setUnicodeStream(int ind, InputStream in, int len)
        throws SQLException {
        byte[] str;

        validateIndex(ind);
        str = new byte[len];
        try {
            in.read(str, 0, len);
            values[ind-1] = new String(str, "UTF8");
        }
        catch( IOException e ) {
            throw new MsqlException(e);
        }
    }

    /**
     * Waits for any parsing to finish and then checks to see that
     * the specified index is valid for this stored procedure.
     * @param ind the parameter number to be tested
     * @exception java.sql.SQLException the specified parameter is invalid
     */
    private void validateIndex(int ind) throws SQLException {
        synchronized( this ) {
            while( parsing != 2 ) {
                try { wait(1500); }
                catch( InterruptedException e ) { }
            }
        }
        if( ind < 1 ) {
            throw new MsqlException("Cannot address an index less than 1.");
        }
        else if( ind > values.length ) {
            throw new MsqlException("Attempted to assign a value to " +
                                    "parameter " + ind + " when there are " +
                                    "only " + values.length + " parameters.");
        }
    }
}
